//-----------------------------------------------------------------------------
// Watch_XTAL_F30x.c
//-----------------------------------------------------------------------------
// Copyright 2003 Cygnal Integrated Products, Inc.
//
// AUTH: FB
// DATE: 23 JAN 03
//
// This example shows how to start an external 32.768 kHz watch crystal in low
// power applications. Since a watch crystal can take longer than one second
// to start, the device goes into Idle mode after turning on the external
// oscillator. Timer2, configured to generate an interrupt every 100 ms using 
// a timebase derived from the internal oscillator, wakes the device to check
// the XTLVLD flag. Once the watch crystal has started, the internal oscillator
// is disabled and the system uses the crystal as the system clock source.
//
// Target: C8051F30x
//
// Tool chain: KEIL Eval 'c'
//

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <c8051f300.h>                    // SFR declarations

//-----------------------------------------------------------------------------
// 16-bit SFR Definitions for 'F30x
//-----------------------------------------------------------------------------

sfr16 DP       = 0x82;                    // data pointer
sfr16 TMR2RL   = 0xca;                    // Timer2 reload value
sfr16 TMR2     = 0xcc;                    // Timer2 counter
sfr16 PCA0CP1  = 0xe9;                    // PCA0 Module 1 Capture/Compare
sfr16 PCA0CP2  = 0xeb;                    // PCA0 Module 2 Capture/Compare
sfr16 PCA0     = 0xf9;                    // PCA0 counter
sfr16 PCA0CP0  = 0xfb;                    // PCA0 Module 0 Capture/Compare

//-----------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------

#define INTCLK       24500000 / 8         // Internal Oscillator frequency
                                          // in Hz (divide by 8 mode)
#define EXTCLK       32768                // Frequency for 32.768 kHz External
                                          // crystal oscillator


//-----------------------------------------------------------------------------
// Global VARIABLES
//-----------------------------------------------------------------------------
bit OSC_READY = 0;                        // flag to indicate when external
                                          // oscillator is ready

//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------
void SYSCLK_Init (void);
void PORT_Init (void);
void Timer2_Init (int counts);
void Timer2_ISR (void);

//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------
void main (void) {

   // disable watchdog timer
   PCA0MD &= ~0x40;                       // WDTE = 0 (clear watchdog timer
                                          // enable)

   PORT_Init();                           // initialize the crossbar and GPIO
   SYSCLK_Init();                         // start external oscillator


   // The system should be running from the external oscillator at this point

   while(1){
      PCON |= 0x01;                       // put the device in Idle mode
   }
}



//-----------------------------------------------------------------------------
// SYSCLK_Init
//-----------------------------------------------------------------------------
//
// This routine starts the external 32.768 kHz watch crystal and puts the system
// in Idle mode. Timer 2 interrupts check the status of the XTLVLD bit and
// switches the system clock to the external oscillator when it is ready. The
// system remains in Idle mode until the oscillator starts.
//
void SYSCLK_Init (void)
{

   OSCXCN = 0x61;                         // start external oscillator

   Timer2_Init(INTCLK/12/10);             // configure Timer2 to overflow
                                          // at 10 Hz (every 100 ms)
                                      
   EA = 1;                                // enable global interrupts

   while(!OSC_READY){
      PCON |= 1;                          // put device in Idle mode
   }

}


//-----------------------------------------------------------------------------
// PORT_Init
//-----------------------------------------------------------------------------
//
// Configure the Crossbar and GPIO ports.
// P0.0 -
// P0.1 -
// P0.2 - XTAL1
// P0.3 - XTAL2
// P0.4 -
// P0.5 -
// P0.6 -
// P0.7 - C2D
//
void PORT_Init (void)
{
   XBR0     = 0x0c;                       // skip crystal pins
   XBR2     = 0x40;                       // enable crossbar and weak pull-ups

   P0MDIN  &= ~0x0c;                      // configure XTAL1 and XTAL2 as analog
                                          // inputs
}

//-----------------------------------------------------------------------------
// Timer2_Init
//-----------------------------------------------------------------------------
//
// Configure Timer2 to auto-reload at interval specified by <counts>
// using the system clock / 12 as its time base.
//
void Timer2_Init (int counts)
{
   TMR2CN = 0x00;                         // Stop Timer0;
                                          // Timer2 timebase is SYSCLK/12

   TMR2RL = -counts;                      // Init reload value
   TMR2 = TMR2RL;                         // Init Timer2
   ET2 = 1;                               // enable Timer2 interrupts
   TR2 = 1;                               // start Timer2
}



//-----------------------------------------------------------------------------
// Timer2_ISR
//-----------------------------------------------------------------------------
//
// This interrupt service routine is called on Timer2 overflows
//
void Timer2_ISR (void) interrupt 5
{
   TF2H = 0;                              // clear Timer2 overflow flag

   if(OSCXCN & 0x80)                      // if crystal osc. has settled
   {
      OSCXCN = 0x60;                      // decrease crystal drive current
      RSTSRC = 0x04;                      // enable missing clock detector
      OSCICN = 0x08;                      // switch to external oscillator
                                          // and disable internal oscillator

      TR2 = 0;                            // stop Timer2

      OSC_READY = 1;                      // indicate that the external osc.
                                          // is ready

   }
}
