//-----------------------------------------------------------------------------
// CP0_Stop_F30x.c
//-----------------------------------------------------------------------------
// Copyright 2003 Cygnal Integrated Products, Inc.
//
// AUTH: FB / GV
// DATE: 23 JAN 03
//
// This example shows how a sleep mode can be implemented in a system using
// the Stop mode of the CPU. This application implements a software counter
// that is incremented approximately every second when the device is in
// Normal mode. If the system is powered down when it is in Stop mode, the
// counter resumes counting the next time it enters Normal mode. If the
// system is powered down while it is in Normal mode, the counter will reset
// to zero. Every time the counter is updated, the current value of the
// counter is printed to the UART.
//
// The S2 switch toggles the system back and forth between Stop mode and 
// Normal mode. On power up, the system is in Stop mode. The system enters 
// Normal mode when the S2 switch on the target board is pressed, causing
// a Comparator 0 reset. 
//
// Upon entering Normal mode, External Interrupt 0 (/INT0) is activated
// to sense the S2 switch and Comparator 0 is disabled as a reset source.
// Pressing S2 in Normal mode will cause the INT0_ISR to put the system
// in Stop mode. 
//
// This program uses Comparator 0 as a reset source.  When S2 on the target
// board is pressed, the the CP0+ input drops below CP0- (VDD/2).  This causes
// Comparator 0 to issue a system reset.
//
// For this example, it is necessary to make the following connections:
// 1. P0.3_SW  -> P0.0
// 2. P0.2     -> P0.1
// 3. P0.2_LED -> P0.2
//
// P0.1 is used as a reference voltage for the comparator and should be
// approximately halfway between VDD and GND. When the LED is connected
// to P0.2 (high-impedance with weak pull-up), the voltage on P0.2 is
// around 1.7 Volts.
//
// Since this program writes to FLASH, the VDD monitor is enabled.
//
// Target: C8051F30x
// Tool Chain: KEIL C51 6.03 / KEIL EVAL C51
//

//-----------------------------------------------------------------------------
// Include Files
//-----------------------------------------------------------------------------

#include <c8051f300.h>
#include <stdio.h>

//-----------------------------------------------------------------------------
// 16-bit SFR Definitions for 'F30x
//-----------------------------------------------------------------------------

sfr16 DP       = 0x82;                    // data pointer
sfr16 TMR2RL   = 0xca;                    // Timer2 reload value
sfr16 TMR2     = 0xcc;                    // Timer2 counter
sfr16 PCA0CP1  = 0xe9;                    // PCA0 Module 1 Capture/Compare
sfr16 PCA0CP2  = 0xeb;                    // PCA0 Module 2 Capture/Compare
sfr16 PCA0     = 0xf9;                    // PCA0 counter
sfr16 PCA0CP0  = 0xfb;                    // PCA0 Module 0 Capture/Compare

//-----------------------------------------------------------------------------
// Global Constants
//-----------------------------------------------------------------------------

#define SYSCLK   3062500                  // System Clock Frequency in Hz
#define BAUDRATE 9600                     // UART Baud Rate in bps

sbit S2 = P0^0;                           // Switch on Target Board

//-----------------------------------------------------------------------------
// Global Variables
//-----------------------------------------------------------------------------
long COUNT = 0;                           // Software Counter

char code SLEEP _at_ 0x1000;              // Flag that indicates device
                                          // is in a low-power state. The
                                          // flag is TRUE when it contains
                                          // an 0x55 pattern. Any other
                                          // pattern indicates FALSE.

long code COUNT_SAVE _at_ 0x1001;         // Non-volatile storage for
                                          // the current count



//-----------------------------------------------------------------------------
// Function Prototypes
//-----------------------------------------------------------------------------

void SYSCLK_Init (void);
void PORT_Init (void);
void CPT0_Init (void);
void ResetSRC_Init(void);
void EX0_Init(void);
void UART0_Init (void);
void Check_Reset_Source(void);
void wait_ms(int ms);

//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------

void main (void)
{

   PCA0MD &= ~0x40;                       // disable the watchdog timer
   RSTSRC = 0x02;                         // enable VDD monitor

   EX0_Init();                            // initialize External Interrupt 0
   PORT_Init ();                          // initialize crossbar and GPIO
   SYSCLK_Init();                         // initialize the system clock
   UART0_Init();                          // initialize UART communication

   EA = 1;                                // Enable global interrupts



   Check_Reset_Source();                  // check whether the source of
                                          // of the last reset was due to
                                          // a power-on condition or due to
                                          // a comparator



   while(1){

      // print current count
      printf("Current Count: %ld\n", COUNT);
      COUNT++;

      // wait for one second
      wait_ms(1000);

   }
}

//-----------------------------------------------------------------------------
// Check_Reset_Source
//-----------------------------------------------------------------------------
//
// This routine is called on every device reset.
//
// On each comparator reset, it restores the value of <COUNT> from the
// <COUNT_SAVE> variable stored in FLASH if the device was in a low-power state
// prior to the reset(i.e. the SLEEP flag in FLASH was set to an 0x55 pattern).
// If the <SLEEP> flag was not set then the <COUNT> variable is set to zero and
// the device starts normal mode operation.
//
// On each power-on reset or HW pin reset, the device goes into a low power
// mode waiting for a comparator reset.
//
void Check_Reset_Source(void)
{
   char EA_SAVE;                          // interrupt state preservation
   char xdata * idata ptrSLEEP = &SLEEP;  // FLASH write pointer

   printf("\nDevice Reset -- RESET SOURCE = 0x%02bX\n\n", RSTSRC);

   // check for power-on, HW pin, watchdog timer or missing clock detector reset
   if(RSTSRC & 0x4F){
      CPT0_Init();                        // initialize comparator 0
      ResetSRC_Init();                    // set comparator 0 as a reset source

      printf("Entering Stop Mode\n\n");
      PCON |= 0x02;                       // put device in stop mode
   }

   // check for a comparator reset
   else if( RSTSRC & 0x20){

      while(!S2);                         // wait while switch down
      wait_ms(5);                         // wait until switch stabilizes


      // if the device was in a low-power state ( <SLEEP> flag is set to TRUE),
      // then resume counting, otherwise start counting from zero
      if(SLEEP == 0x55){

         // 1. restore <COUNT>
         COUNT = COUNT_SAVE;

         // 2. Set <SLEEP> flag to FALSE by erasing the FLASH page containing
         //    the variable
         EA_SAVE = EA;                    // preserve interrupt state
         EA = 0;                          // disable interrupts
         PSCTL = 0x01;                    // MOVX writes write FLASH byte
         FLKEY = 0xA5;                    // FLASH lock and key sequence 1
         FLKEY = 0xF1;                    // FLASH lock and key sequence 2
         *ptrSLEEP = 0x00;                // clear SLEEP flag to indicate device
                                          // is no longer in Stop mode
         PSCTL = 0x00;                    // disable FLASH writes/erases
         EA = EA_SAVE;                    // restore interrupt state
         
         // 3. Enable External Interrupt 0
         EX0 = 1;
      }

      // otherwise start counting at zero
      else{
         // 1. Set <COUNT> to zero
         COUNT = 0;

         // 2. Enable External Interrupt 0
         EX0 = 1;                         // Enable External Interrupt0
      }

   }

   // handle error condition for unrecognized reset source
   else {
      printf("\n**UNRECOGNIZED RESET SOURCE = 0x%02bX\n", RSTSRC);
      PCON |= 0x02;                       //  place device in Stop mode
   }


}

//-----------------------------------------------------------------------------
// wait_ms
//-----------------------------------------------------------------------------
//
// This routine inserts a delay of <ms> milliseconds.
//
void wait_ms(int ms)
{
   TMR2CN = 0x00;                         // Configure Timer 2 as a 16-bit
                                          // timer counting SYSCLKs/12
   TMR2RL = -(SYSCLK/1000/12);            // Timer 2 overflows at 1 kHz
   TMR2 = TMR2RL;

   TR2 = 1;                               // Start Timer 2

   while(ms){
      TF2H = 0;
      while(!TF2H);                       // wait until timer overflows
      ms--;                               // decrement ms
   }

   TR2 = 0;                               // Stop Timer 2
}


//-----------------------------------------------------------------------------
// Interrupt Service Routines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// INT0_ISR
//-----------------------------------------------------------------------------

void INT0_ISR (void) interrupt 0
{  
   // pointer to COUNT
   unsigned char* ptrCOUNT = &COUNT;                
   
   // FLASH write pointer 
   char xdata * idata ptrCOUNT_SAVE = &COUNT_SAVE;   
   
   // FLASH write pointer
   char xdata * idata ptrSLEEP = &SLEEP;             
   
   char EA_SAVE = EA;                     // save interrupt status

   printf("Entering Stop Mode\n\n");

   EA = 0;                                // disable interrupts
   
   PSCTL = 0x03;                          // MOVX writes erase FLASH page
   FLKEY = 0xA5;                          // FLASH lock and key sequence 1
   FLKEY = 0xF1;                          // FLASH lock and key sequence 2
   *ptrCOUNT_SAVE = 0;                    // initiate page erase

   PSCTL = 0x01;                          // MOVX writes write FLASH byte

   // copy <COUNT> to the <COUNT_SAVE> variable in FLASH

   FLKEY = 0xA5;                          // FLASH lock and key sequence 1
   FLKEY = 0xF1;                          // FLASH lock and key sequence 2
   ptrCOUNT_SAVE[0] = ptrCOUNT[0];        // copy first byte

   FLKEY = 0xA5;                          // FLASH lock and key sequence 1
   FLKEY = 0xF1;                          // FLASH lock and key sequence 2
   ptrCOUNT_SAVE[1] = ptrCOUNT[1];        // copy second byte

   FLKEY = 0xA5;                          // FLASH lock and key sequence 1
   FLKEY = 0xF1;                          // FLASH lock and key sequence 2
   ptrCOUNT_SAVE[2] = ptrCOUNT[2];        // copy third byte

   FLKEY = 0xA5;                          // FLASH lock and key sequence 1
   FLKEY = 0xF1;                          // FLASH lock and key sequence 2
   ptrCOUNT_SAVE[3] = ptrCOUNT[3];        // copy fourth byte

   FLKEY = 0xA5;                          // FLASH lock and key sequence 1
   FLKEY = 0xF1;                          // FLASH lock and key sequence 2
   *ptrSLEEP = 0x55;                      // set SLEEP flag to indicate device
                                          // is in Stop mode and <COUNT> has
                                          // been saved in FLASH

   PSCTL = 0x00;                          // disable FLASH writes and erases

   EX0 = 0;                               // disable External Interrupt 0

   EA = EA_SAVE;                          // restore interrupt status

   while(!S2);                            // wait while switch down

   CPT0_Init();                           // initialize comparator 0
   ResetSRC_Init();                       // set comparator 0 as a reset source

   PCON |= 0x02;                          // put the device in Stop mode


}

//-----------------------------------------------------------------------------
// Initialization Subroutines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// SYSCLK_Init
//-----------------------------------------------------------------------------
// This routine initializes the system clock to use the precision internal
// oscillator divided by 8 as its clock source.
//
void SYSCLK_Init (void)
{

   OSCICN = 0x04;                         // SYSCLK is internal osc.
                                          // in divide by 8 mode running
                                          // at 3.0625 MHz
}

//-----------------------------------------------------------------------------
// PORT Initialization
//-----------------------------------------------------------------------------
//
// Configure the Crossbar and GPIO ports
//
// P0.0 - CP0+ input (connected to S2)
// P0.1 - CP0- input
// P0.2 - Comparator Voltage Reference  (connected to P0.1)
// P0.3 - Used as a weak pull-up for P0.0
//
void PORT_Init (void)
{
   XBR0     = 0x07;                       // skip P0.0 - P0.2 in crossbar


   XBR1     = 0x03;                       // Enable UART0
   XBR2     = 0x40;                       // Enable crossbar and weak pull-ups

   P0MDOUT |= 0x10;                       // TX0 is a push-pull output

}


//-----------------------------------------------------------------------------
// Comparator0 Initialization
//-----------------------------------------------------------------------------
//
// Initialize Comparator 0 to detect when the SW2 switch is pressed.
//
void CPT0_Init(void)
{

   P0MDIN  &= ~0x03;                      // Comparator 0 inputs (P0.0
                                          // and P0.1) are analog inputs.

   CPT0CN = 0x8F;                         // Comparator enabled with maximum
                                          // positive and negative hysteresis

   CPT0MX = 0x00;                         // P0.1 = Inverting Input for
                                          // the comparator
                                          // P0.0 = Non-Inverting Input for the
                                          // comparator 

   wait_ms(500);                          // wait for comparator inputs to settle

   CPT0CN &= ~0x30;                       // clear interrupt flags


}

//-----------------------------------------------------------------------------
// Reset Source Initialization
//-----------------------------------------------------------------------------
//
// Configure Comparator 0 as a reset source.
//
void ResetSRC_Init(void)
{
   RSTSRC = 0x22;                         // Comparator 0 is a reset source
                                          // VDD Monitor enabled
}

//-----------------------------------------------------------------------------
// External Interrupt 0 Initialization
//-----------------------------------------------------------------------------
//
// Configure External Interrupt 0 to generate an interrupt on the falling
// edge of P0.0.
//
void EX0_Init(void)
{
   IT01CF = 0x00;                         // Configure External Interrupt 0 to
                                          // generate an interrupt on the falling
                                          // edge of P0.0 (S2 switch)
}
//-----------------------------------------------------------------------------
// UART0_Init
//-----------------------------------------------------------------------------
//
// Configure the UART0 using Timer1, for <BAUDRATE> and 8-N-1.
//
void UART0_Init (void)
{
   SCON0 = 0x10;                          // SCON0: 8-bit variable bit rate
                                          //        level of STOP bit is ignored
                                          //        RX enabled
                                          //        ninth bits are zeros
                                          //        clear RI0 and TI0 bits
   if (SYSCLK/BAUDRATE/2/256 < 1) {
      TH1 = -(SYSCLK/BAUDRATE/2);
      CKCON |= 0x10;                      // T1M = 1; SCA1:0 = xx
   } else if (SYSCLK/BAUDRATE/2/256 < 4) {
      TH1 = -(SYSCLK/BAUDRATE/2/4);
      CKCON &= ~0x13;
      CKCON |=  0x01;                     // T1M = 0; SCA1:0 = 01
   } else if (SYSCLK/BAUDRATE/2/256 < 12) {
      TH1 = -(SYSCLK/BAUDRATE/2/12);
      CKCON &= ~0x13;                     // T1M = 0; SCA1:0 = 00
   } else {
      TH1 = -(SYSCLK/BAUDRATE/2/48);
      CKCON &= ~0x13;
      CKCON |=  0x02;                     // T1M = 0; SCA1:0 = 10
   }

   TL1 = TH1;                             // set Timer1 to reload value
   TMOD &= ~0xf0;                         // TMOD: timer 1 in 8-bit autoreload
   TMOD |=  0x20;
   TR1 = 1;                               // START Timer1
   TI0 = 1;                               // Indicate TX0 ready

}