//-----------------------------------------------------------------------------
// ADC_B_F30x.c
//-----------------------------------------------------------------------------
// Copyright 2003 Cygnal Integrated Products, Inc.
//
// AUTH: FB
// DATH: 23 JAN 03
//
// This example captures ADC samples at a rate of 10 Hz from P0.0 and is clocked
// from a 32.768 kHz watch crystal. This program keeps the CPU in Idle mode
// until a Timer2 overflow. The Timer2 interrupt turns on the ADC, takes a 
// sample, then turns it off to save power. 
//
// This program is meant to be used as a comparison to ADC_A_F30x to show that
// reducing the peak current required to take an ADC sample does not always
// save power. In this case, decreasing the SYSCLK frequency increased the 
// average system current because the ADC was "on" for a longer period of time.
//
// Target: C8051F30x
//
// Tool chain: KEIL Eval 'c'
//

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <c8051f300.h>                    // SFR declarations
#include <math.h>

//-----------------------------------------------------------------------------
// 16-bit SFR Definitions for 'F30x
//-----------------------------------------------------------------------------

sfr16 DP       = 0x82;                    // data pointer
sfr16 TMR2RL   = 0xca;                    // Timer2 reload value
sfr16 TMR2     = 0xcc;                    // Timer2 counter
sfr16 PCA0CP1  = 0xe9;                    // PCA0 Module 1 Capture/Compare
sfr16 PCA0CP2  = 0xeb;                    // PCA0 Module 2 Capture/Compare
sfr16 PCA0     = 0xf9;                    // PCA0 counter
sfr16 PCA0CP0  = 0xfb;                    // PCA0 Module 0 Capture/Compare

//-----------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------

#define INTCLK       24500000 / 8         // Internal Oscillator frequency  
                                          // in Hz (divide by 8 mode) 
#define EXTCLK       32768                // Frequency for 32.768 kHz External
                                          // crystal oscillator
#define SAMPLERATE   10                   // ADC Sampling Rate in Hz

sbit LED = P0^2;                          // LED='1' means ON
sbit SW2 = P0^3;                          // SW2='0' means switch pressed

//-----------------------------------------------------------------------------
// Global VARIABLES
//-----------------------------------------------------------------------------
char ADC_READING = 0;

//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------
void SYSCLK_Init (void);
void PORT_Init (void);
void Crystal_Stabilize (void);
void Timer2_Init (int counts);
void Timer2_ISR (void);
//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------
void main (void) {
   
   // disable watchdog timer
   PCA0MD &= ~0x40;                       // WDTE = 0 (clear watchdog timer 
                                          // enable)
   
   PORT_Init();                           // initialize the Crossbar and GPIO
   SYSCLK_Init();                         // start external oscillator
   Timer2_Init(EXTCLK/8/SAMPLERATE);      // configure Timer2 to overflow at 
                                          // <SAMPLERATE> times per second
   
   EA = 1;                                // enable global interrupts

   while(1){
      
      PCON |= 0x01;                       // put the device in idle mode
   }
}



//-----------------------------------------------------------------------------
// SYSCLK_Init
//-----------------------------------------------------------------------------
//
// This routine initializes the system clock to use the external 32.768 kHz 
// watch crystal as its clock source and disables the internal oscillator.
//
void SYSCLK_Init (void)
{
   
   int i;                                 // delay counter

   OSCXCN = 0x61;                         // start external oscillator

   for (i=0; i < 256; i++) ;              // wait for osc to start up

   while (!(OSCXCN & 0x80)) ;             // wait for crystal osc. to settle
   
   
   Crystal_Stabilize();
   
   
   OSCXCN = 0x60;                         // decrease XFCN (crystal drive current)
   RSTSRC = 0x04;                         // enable missing clock detector
   OSCICN = 0x08;                         // switch to external oscillator

   
}

//-----------------------------------------------------------------------------
// Crystal_Stabilize
//-----------------------------------------------------------------------------
//
// Low-frequency crystal stabilization wait routine:
//
// This routine measures the period of the external oscillator with respect
// to the internal oscillator and loops until the external oscillator period is
// measured to be within 4 internal oscillator periods for 500 cycles in
// a row.  This is only necessary for tuning fork crystals, which have 
// abnormally long stabilization times (on the order of seconds).
//
// Assumes that the internal oscillator operating in divide-by-8 mode is
// selected as the system clock source.  Also assumes that the external
// oscillator has been enabled, configured, and is oscillating.
//
// Here we measure the number of system clocks in 8 "EXTCLK/8" periods.
// We compare successive measurements.  When we obtain 500 measurements
// in a row that are all within 4 system clocks of each other the
// routine will exit.  This condition will only occur once the crystal
// oscillator has fully stabilized at its resonant frequency.  
// 
// Note that this can take several seconds.
//
void Crystal_Stabilize (void)
{
   int current, last;                     // used in osc. stabilization check
   int tolerance_count;

   // init PCA0
   PCA0CN = 0x00;                         // Stop counter; clear all flags
   PCA0MD = 0x0b;                         // PCA counts in IDLE mode;
                                          // EXTCLK / 8 is time base;
                                          // overflow interrupt is enabled

   // init Timer0
   TCON &= ~0x30;                         // Stop timer; clear TF0
   TMOD &= ~0x0f;                         // Timer0 in 16-bit counter mode
   TMOD |=  0x01;
   CKCON |= 0x08;                         // Timer0 counts SYSCLKs

   tolerance_count = 500;                 // wait for 500 external cycles in a row
                                          // to lie within 4 internal clocks of each
                                          // other
   current = 0;

   do {
      PCA0CN = 0x00;
      PCA0L = 0xFF;                       // set PCA time base to '-1'
      PCA0H = 0xFF;
      TCON &= ~0x30;
      TH0 = 0x00;                         // init T0 time base
      TL0 = 0x00;

      // start PCA0
      CR = 1;
      while (CF == 0);                    // wait for edge
      TR0 = 1;                            // Start Timer0
      CF = 0;                             // clear PCA overflow
      PCA0L = -8;                         // set PCA to overflow in 8 cycles
      PCA0H = (-8) >> 8;
      while (CF == 0);
      TR0 = 0;
      last = current;
      current = (TH0 << 8) | TL0;
      if (abs (current - last) > 4) {
         tolerance_count = 500;           // falls outside bounds; reset
                                          // counter
      } else {
         tolerance_count--;               // in-bounds; update counter
      }
      
   } while (tolerance_count != 0);

}

//-----------------------------------------------------------------------------
// PORT_Init
//-----------------------------------------------------------------------------
//
// Configure the Crossbar and GPIO ports.
// P0.0 - ADC Input
// P0.1 -
// P0.2 - XTAL1
// P0.3 - XTAL2
// P0.4 - 
// P0.5 - 
// P0.6 - 
// P0.7 - C2D
//
void PORT_Init (void)
{
   XBR0     = 0x0d;                       // skip crystal pins and P0.0 in crossbar 
   XBR2     = 0x40;                       // enable crossbar and weak pull-ups
    
   P0MDIN  &= ~0x0c;                      // configure XTAL1 and XTAL2 as analog
                                          // inputs
   P0MDIN  &= ~0x01;                      // configure P0.0 as an analog input
}

//-----------------------------------------------------------------------------
// Timer2_Init
//-----------------------------------------------------------------------------
//
// Configure Timer2 to auto-reload at interval specified by <counts> 
// using EXTCLK / 8 as its time base.
//
void Timer2_Init (int counts)
{
   TMR2CN = 0x01;                         // Stop Timer2; 
                                          // Timer2 timebase is EXTCLK/8
   
   TMR2RL = -counts;                      // Init reload value
   TMR2 = TMR2RL;                         // Init Timer2
   ET2 = 1;                               // enable Timer2 interrupts
   TR2 = 1;                               // start Timer2
}


//-----------------------------------------------------------------------------
// Timer2_ISR
//-----------------------------------------------------------------------------
//
// This ISR is called at <SAMPLERATE> Hz on Timer2 overflows
//
void Timer2_ISR (void) interrupt 5
{
   
    
   TF2H = 0;                              // clear Timer2 overflow flag

   ADC0CN = 0x80;                         // enable ADC

   REF0CN |= 0x0A;                        // Select voltage reference and enable
                                          // bias generator

   AMX0SL = 0x80;                         // ADC in single-ended mode sampling P0.0
   
   ADC0CF = (EXTCLK << 3);                // Set SAR clock frequency to ~32kHz 
   
   ADC0CF |= 0x01;                        // Set PGA gain
   
   // settling time starts at this point, sampling should not start until
   // the appropriate settling time has passed. At this point, we using
   // a 32.768 kHz so each SYSCLK cycle is 30.5 us. 
   
   AD0INT = 0;                            // Clear conversion complete flag
   AD0BUSY = 1;                           // Start a conversion 
   while(!AD0INT);                        // Wait until conversion complete

   AD0EN = 0;                             // Disable ADC
   REF0CN &= ~0x02;                       // Turn off bias generator 
   ADC_READING = ADC0;                    // Capture ADC Reading

}
