//-----------------------------------------------------------------------------
// Cache_Example_4.c
//-----------------------------------------------------------------------------
// Copyright 2002 Cygnal Integrated Products, Inc.
//
// AUTH: FB
// DATE: 21 NOV 02
//
// This example combines several of the techniques discussed in AN034 to
// store ADC samples in FLASH. The maximum sampling rate in this example
// is limited by FLASH write time.
// 
// A FLASH block write takes less than 60 s to write four consecutive bytes 
// to FLASH. The maximum sampling rate is (1/60 s x 4 = 66.6 kHz). We will 
// choose 65 kHz for simplicity. To achieve a 65 kHz FLASH write rate, the next 
// 4 ADC samples should be ready to write to FLASH as soon as the current FLASH 
// operation is complete. This is accomplished by executing code exclusively 
// from the cache once the capture operation has started. This will allow FLASH 
// write operations to occur back to back.
// 
// This program uses a 22.1184 Mhz crystal oscillator multiplied by (9/4) 
// for an effective SYSCLK of 49.7664 Mhz. This program also initializes and 
// uses UART0 at <BAUDRATE> bits per second.  
//
// Target: C8051F12x
// Tool chain: KEIL C51 6.03 / KEIL EVAL C51
//

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <c8051f120.h>                 // SFR declarations
#include <stdio.h>                     // printf() and getchar()

//-----------------------------------------------------------------------------
// 16-bit SFR Definitions for 'F12x
//-----------------------------------------------------------------------------

sfr16 DP       = 0x82;                 // data pointer
sfr16 ADC0     = 0xbe;                 // ADC0 data
sfr16 ADC0GT   = 0xc4;                 // ADC0 greater than window
sfr16 ADC0LT   = 0xc6;                 // ADC0 less than window
sfr16 RCAP2    = 0xca;                 // Timer2 capture/reload
sfr16 RCAP3    = 0xca;                 // Timer3 capture/reload
sfr16 RCAP4    = 0xca;                 // Timer4 capture/reload
sfr16 TMR2     = 0xcc;                 // Timer2
sfr16 TMR3     = 0xcc;                 // Timer3
sfr16 TMR4     = 0xcc;                 // Timer4
sfr16 DAC0     = 0xd2;                 // DAC0 data
sfr16 DAC1     = 0xd2;                 // DAC1 data
sfr16 PCA0CP5  = 0xe1;                 // PCA0 Module 5 capture
sfr16 PCA0CP2  = 0xe9;                 // PCA0 Module 2 capture
sfr16 PCA0CP3  = 0xeb;                 // PCA0 Module 3 capture
sfr16 PCA0CP4  = 0xed;                 // PCA0 Module 4 capture
sfr16 PCA0     = 0xf9;                 // PCA0 counter
sfr16 PCA0CP0  = 0xfb;                 // PCA0 Module 0 capture
sfr16 PCA0CP1  = 0xfd;                 // PCA0 Module 1 capture


//-----------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------
#define TRUE         1
#define FALSE        0

#define EXTCLK       22118400          // External oscillator frequency in Hz.
#define SYSCLK       49766400          // Output of PLL derived from 
                                       // (EXTCLK*9/4).

#define BAUDRATE     115200            // Baud rate of UART in bps
                                       // Note: The minimum standard baud rate
                                       // supported by the UART0_Init routine
                                       // in this file is 19,200 bps when
                                       // SYSCLK = 49.76MHz. 

#define SAMPLERATE   65000             // Approximate ADC0 Sampling Rate, 
                                       // limited by FLASH write rates.  
                                       

sbit LED = P1^6;                       // LED='1' means ON.
sbit SW2 = P3^7;                       // SW2='0' means switch pressed.


#define ADC0_ISR_LEN 0x63              // This macro defines the length (in 
                                       // bytes) of the ADC0_ISR routine. It
                                       // is used to determine how many bytes
                                       // to push into the cache. This 
                                       // information can be obtained from 
                                       // the *.M51 file associated with this
                                       // project.

#define ADC0_ISR_NUM 15                // This macro defines the ADC0 End of 
                                       // conversion interrupt number.

#define while_TR3_NonBlocking_LEN 0x04 // Length of the while_TR3_NonBlocking()
                                       // function in bytes.


#define LOG_START 0x1000               // Starting address of LOG.
#define LOG_END 0x2000                 // Ending address of LOG.
#define FLASH_PAGESIZE 1024            // Number of bytes in a FLASH page.

//-----------------------------------------------------------------------------
// Global VARIABLES
//-----------------------------------------------------------------------------

#define input_str_len 4                // Buffer to hold characters entered
char input_str[input_str_len];         // at the command prompt.
 
//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------
void main(void);

// Initialization Routines
void SYSCLK_Init(void);
void PORT_Init(void);
void UART0_Init (void);
void ADC0_Init (void);
void CACHE_Init (void);
void Timer3_Init(int counts);

// Support Routines
void print_menu(void);
void print_data(char print_raw_data);
void FLASH_Erase(void);
void Cache_Function( unsigned int start_address, unsigned int length );
void Cache_ISR(   unsigned int start_address, 
                  unsigned int interrupt_length,
                  unsigned int interrupt_number );


// Cached Routines
void while_TR3_NonBlocking(void);
void ADC0_ISR (void);


//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------

void main (void) 
{

   WDTCN = 0xde;                       // disable watchdog timer
   WDTCN = 0xad;

   PORT_Init ();                       // initialize crossbar and GPIO
   SYSCLK_Init ();                     // initialize oscillator
   UART0_Init ();                      // initialize UART0
   ADC0_Init ();                       // initialize ADC0   
   CACHE_Init ();   

   Timer3_Init(SYSCLK/SAMPLERATE);     // initialize Timer4 to overflow
                                       // and generate interrupts at 
                                       // <SAMPLERATE> Hz
   
   // Cache the ADC0 End of Conversion ISR
   Cache_ISR( (unsigned int) ADC0_ISR, ADC0_ISR_LEN, ADC0_ISR_NUM );
   
   // Cache the while_TR3_NonBlocking() routine
   Cache_Function( (unsigned int) while_TR3_NonBlocking, 
                                  while_TR3_NonBlocking_LEN);
 
   EA = 1;                             // Enable Global Interrupts

   print_menu();                       // print the command menu

   while (1){

      SFRPAGE = UART0_PAGE;         
      printf("\nEnter a command > ");
      gets(input_str, input_str_len);
      
      switch ( input_str[0] ){
                      
         // Capture new data
         case '1': FLASH_Erase();               // erase FLASH log

                   SFRPAGE = TMR3_PAGE;   
                   TR3 = 1;                     // start Timer3    
                   while_TR3_NonBlocking();     // execute: while(TR3);  
                                                // Timer3 will be stopped
                                                // after data has been captured.
               
                   SFRPAGE = UART0_PAGE;
                   printf("\nA new data set has been captured.\n");
                   break;

         // Print Data (Press CTRL+C to stop)
         case '2': print_data(TRUE);
                   break;
         
         // Print Average ADC Reading
         case '3': print_data(FALSE);
                   break;
         
         // Print Command List
         case '?': print_menu(); 
                   break;    

         default:  printf("\n*** Unknown Command.\n");
                   break;
      }   

   } // end while

} 

//-----------------------------------------------------------------------------
// Cached Routines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// while_TR3_NonBlocking
//-----------------------------------------------------------------------------
//
// This function polls the TR3 bit and returns when TR3 = 0. This 
// function should be cached during device initialization to allow servicing
// interrupts during FLASH write operations.
//
void while_TR3_NonBlocking(void)
{
   while(TR3);
}

//-----------------------------------------------------------------------------
// ADC0_ISR
//-----------------------------------------------------------------------------
//
// This ISR is called on the end of an ADC0 conversion.
//
void ADC0_ISR (void) interrupt 15 
{
   // FLASH write pointer
   static unsigned char xdata * idata pwrite = LOG_START;

   AD0INT = 0;                       // clear ADC0 conversion complete
                                     // interrupt flag
   SFRPAGE = LEGACY_PAGE;
   FLSCL |= 0x01;                    // enable FLASH writes/erases
   PSCTL  = 0x01;                    // MOVX writes FLASH byte
   
   SFRPAGE = ADC0_PAGE;
   *pwrite = ADC0H;                  // store the first 8-bits of the 
                                     // 12-bit left-justified ADC0
                                     // result to FLASH
   SFRPAGE = LEGACY_PAGE;
   FLSCL &= ~0x01;                   // disable FLASH writes/erases
   PSCTL  = 0x00;                    // MOVX writes target XRAM

   pwrite++;                         // increment FLASH write pointer

   // Check for the end of the log. If <LOG_END> has been reached
   // disable Timer3 and reset it to its reload value. Also set the 
   // FLASH write pointer to <LOG_START>
   if( pwrite >= LOG_END ){
      
      SFRPAGE = TMR3_PAGE;
      TR3 = 0;                       // Disable Timer3
      TMR3 = RCAP3;                  // Reset Timer3 to its reload value.      
      pwrite = LOG_START;            // Reset FLASH write pointer
      return;                        // Exit ISR
   
   }   
}
//-----------------------------------------------------------------------------
// Support Routines
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// print_menu
//-----------------------------------------------------------------------------
//
// This routine prints the command menu to the UART.
//
void print_menu(void)
{
   SFRPAGE = UART0_PAGE;
   printf("\n\nC8051F12x Non-Volatile Data Capture Example\n");
   printf("----------------------------------------------\n");
   printf("1. Capture new data\n");
   printf("2. Print Data (Press CTRL+C to stop)\n");
   printf("3. Print Average ADC Reading\n");
   printf("?. Print Command List\n");
}

//-----------------------------------------------------------------------------
// print_data
//-----------------------------------------------------------------------------
//
//
void print_data(char print_raw_data)
{
   long i;
   long result = 0;
   

   char code * idata pread = LOG_START;         // FLASH read pointer
  
   // cycle through LOG from start to end
   for( i = 0; i < (LOG_END - LOG_START); i++ ){
         
         // add up all the samples for an average
         result += pread[i];
          
         // print each sample to the UART if <print_raw_data> is set
         if(print_raw_data){
            printf("0x%02bX ", pread[i]);
         }      
    }
      
    
   // print the average to the UART   
   SFRPAGE = UART0_PAGE;
   printf("\n\nThe average ADC reading is 0x%02bX\n", (char)(result/i));


}

//-----------------------------------------------------------------------------
// FLASH_Erase
//-----------------------------------------------------------------------------
void FLASH_Erase(void)
{
   char SFRPAGE_SAVE = SFRPAGE;        // Preserve current SFR page
   bit EA_SAVE;                        // Preserve interrupt state
   int i;                              // Software counter   
   char xdata * idata pwrite;          // FLASH write/erase pointer

   // Set the FLASH write/erase pointer to the beginning of the FLASH log
   pwrite = LOG_START;

   // Erase all FLASH pages in the FLASH log
   for( i = 0; i < (LOG_END - LOG_START); i += FLASH_PAGESIZE ){
      
      SFRPAGE = LEGACY_PAGE;
      
      EA_SAVE = EA;
      EA = 0;                          // Disable interrupts    
      FLSCL |= 0x01;                   // Enable FLASH writes/erases
      PSCTL = 0x03;                    // MOVX writes FLASH byte
   
      pwrite[i] = 0;                   // Initiate FLASH page erase     
      
      FLSCL &= ~0x01;                  // Disable FLASH writes/erases
      PSCTL = 0x00;                    // MOVX targets XRAM
      EA = EA_SAVE;                    // Restore interrupt state      
   }
  
   SFRPAGE = SFRPAGE_SAVE;             // Restore SFR page
}

//-----------------------------------------------------------------------------
// Cache_ISR
//-----------------------------------------------------------------------------
//
// This routine pushes and locks an entire ISR and its interrupt vector into 
// the cache. This function assumes that the interrupt vector and ISR is
// located in the common area.
//
void Cache_ISR(   unsigned int start_address, 
                  unsigned int interrupt_length,
                  unsigned int interrupt_number   )
{
   char SFRPAGE_SAVE = SFRPAGE;        // Preserve current SFR page
   char EA_SAVE = EA;                  // Preserve interrupt state

   unsigned char code* pread;          // Pointer used to generate MOVC 
                                       // instructions to initiate a cache
                                       // push operation

   unsigned char temp_char;            // Temporary char.
   
   SFRPAGE = CONFIG_PAGE;              // Set SFR page
   
   // Check if there is enough room to cache the ISR. When the number of
   // slots required by the ISR + interrupt vector are subtracted from
   // the CHSLOT pointer, the result is the number of unlocked slots available.
   if( ((CCH0LC & 0x3F) - (interrupt_length/4 + 2)) <= 0){
      while(1);                        // Handle Error Condition.
                                       // Not enough room to cache ISR
   }    

   // Set the <pread> pointer to the address of the interrupt vector.
   pread = ((interrupt_number * 8) + 3 ); 
   
   EA = 0;                             // Disable Interrupts

   CCH0LC &= ~0xC0;                    // Clear the CHPUSH and CHPOP bits
   CCH0LC |=  0x80;                    // Enable cache pushes, MOVC instructions
                                       // will push 4-byte FLASH segments into
                                       // the cache
                                       
   // Push the interrupt vector in the cache slot pointed to by CHSLOT.
   // An interrupt vector consists of a 3-byte LJMP instruction and the
   // LSBs of the vector address are always 10b. One byte of the LJMP
   // instruction fall in the first 4-byte FLASH segment and the remaining
   // two bytes of the instruction falls in the second 4-byte segment. 
   // Two cache pushes are neccessary.
   
   temp_char = *pread;                 // Push the first 4-byte segment (MOVC)
   temp_char = *(pread + 4);           // Push the second 4-byte segment (MOVC)

   // Push the interrupt service routine in the cache. The interrupt starts
   // at <START_LABEL> and is <interrupt_length> bytes long.
   
   pread = (unsigned char code*) start_address;
                                       // Set <pread> to the address of the
                                       // interrupt service routine.
   
   while( pread < start_address + interrupt_length ){       
      
      temp_char = *pread;              // Push a 4-byte segment (MOVC)
      
      pread += 4;                      // Increment <pread> to the next 
                                       // segment.
   }                                        
   

   CCH0LC &= ~0x80;                    // Clear CHPUSH to disable cache pushes.
   SFRPAGE = SFRPAGE_SAVE;             // Restore SFR page.
   EA = EA_SAVE;                       // Restore interrupt state.

}

//-----------------------------------------------------------------------------
// Cache_Function
//-----------------------------------------------------------------------------
//
// This routine pushes and locks a function into the cache. This routine 
// assumes that the function is located in the common area.
//
//
void Cache_Function( unsigned int start_address, unsigned int length )
{
   char SFRPAGE_SAVE = SFRPAGE;        // Preserve current SFR page
   char EA_SAVE = EA;                  // Preserve interrupt state

   unsigned char code* pread;          // Pointer used to generate MOVC 
                                       // instructions to initiate a cache
                                       // push operation

   unsigned char temp_char;            // Temporary char.
   
   SFRPAGE = CONFIG_PAGE;              // Set SFR page

   // Check if there is enough room to cache the function. When the number of
   // slots required by the function are subtracted from
   // the CHSLOT pointer, the result is number of unlocked slots available.
   if( (CCH0LC & 0x3F) - (length/4) <= 0){
      while(1);                        // Handle Error Condition.
                                       // Not enough room to cache function
   }
 
   // Set <pread> to the address of the function
   pread = (unsigned char code*) start_address;
                                      
   EA = 0;                             // Disable Interrupts

   CCH0LC &= ~0xC0;                    // Clear the CHPUSH and CHPOP bits
   CCH0LC |=  0x80;                    // Enable cache pushes, MOVC instructions
                                       // will push 4-byte FLASH segments into
                                       // the cache
                                       
   // Push the function in the cache. The interrupt starts
   // at <START_LABEL> and is <length> bytes long.
   while( pread < start_address + length ){       
      
      temp_char = *pread;              // Push a 4-byte segment (MOVC)
      
      pread += 4;                      // Increment <pread> to the next 
                                       // segment.
   }                                        
   

   CCH0LC &= ~0x80;                    // Clear CHPUSH to disable cache pushes.
   SFRPAGE = SFRPAGE_SAVE;             // Restore SFR page.
   EA = EA_SAVE;                       // Restore interrupt state.

}

//-----------------------------------------------------------------------------
// Initialization Routines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// SYSCLK_Init
//-----------------------------------------------------------------------------
//
// This routine initializes the system clock to use an external 22.1184 MHz 
// crystal oscillator multiplied by a factor of 9/4 using the PLL as its 
// clock source. The resulting frequency is 22.1184 MHz * 9/4 = 49.7664 MHz
//
void SYSCLK_Init (void)
{
   int i;                           // delay counter
   
   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page

   SFRPAGE = CONFIG_PAGE;           // set SFR page

   OSCXCN = 0x67;                   // start external oscillator with
                                    // 22.1184MHz crystal

   for (i=0; i < 256; i++) ;        // Wait for osc. to start up
   
   while (!(OSCXCN & 0x80)) ;       // Wait for crystal osc. to settle

   CLKSEL = 0x01;                   // Select the external osc. as
                                    // the SYSCLK source

   OSCICN = 0x00;                   // Disable the internal osc.
   
   //Turn on the PLL and increase the system clock by a factor of M/N = 9/4
   SFRPAGE = CONFIG_PAGE;
   
   PLL0CN  = 0x04;                  // Set external osc. as PLL source 
   SFRPAGE = LEGACY_PAGE;
   FLSCL   = 0x10;                  // Set FLASH read time for 50MHz clk 
                                    // or less 
   SFRPAGE = CONFIG_PAGE;
   PLL0CN |= 0x01;                  // Enable Power to PLL
   PLL0DIV = 0x04;                  // Set Pre-divide value to N (N = 4)
   PLL0FLT = 0x01;                  // Set the PLL filter register for 
                                    // a reference clock from 19 - 30 MHz
                                    // and an output clock from 45 - 80 MHz 
   PLL0MUL = 0x09;                  // Multiply SYSCLK by M (M = 9)
   
   for (i=0; i < 256; i++) ;        // Wait at least 5us
   PLL0CN  |= 0x02;                 // Enable the PLL
   while(!(PLL0CN & 0x10));         // Wait until PLL frequency is locked
   CLKSEL  = 0x02;                  // Select PLL as SYSCLK source

   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page
}


//-----------------------------------------------------------------------------
// PORT_Init
//-----------------------------------------------------------------------------
//
// This routine configures the crossbar and GPIO ports.
//
void PORT_Init (void)
{
   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page
   
   SFRPAGE = CONFIG_PAGE;           // set SFR page

   XBR0     = 0x04;                 // Enable UART0
   XBR1     = 0x00;
   XBR2     = 0x40;                 // Enable crossbar and weak pull-up
                                    

   P0MDOUT |= 0x01;                 // Set TX0 pin to push-pull
   P1MDOUT |= 0x40;                 // Set P1.6(LED) to push-pull
   
   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page

}

//-----------------------------------------------------------------------------
// UART0_Init
//-----------------------------------------------------------------------------
//
// Configure the UART0 using Timer1, for <baudrate> and 8-N-1. In order to
// increase the clocking flexibility of Timer0, Timer1 is configured to count
// SYSCLKs. 
// 
// To use this routine SYSCLK/BAUDRATE/16 must be less than 256. For example,
// if SYSCLK = 50 MHz, the lowest standard baud rate supported by this
// routine is 19,200 bps. 
//
void UART0_Init (void)
{
   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page
   
   SFRPAGE = UART0_PAGE;

   SCON0  = 0x50;                   // SCON0: mode 0, 8-bit UART, enable RX
   SSTA0  = 0x10;                   // Timer 1 generates UART0 baud rate and
                                    // UART0 baud rate divide by two disabled
   SFRPAGE = TIMER01_PAGE;
   TMOD   &= ~0xF0;                  
   TMOD   |=  0x20;                 // TMOD: timer 1, mode 2, 8-bit reload
   
   TH1 = -(SYSCLK/BAUDRATE/16);     // Set the Timer1 reload value
                                    // When using a low baud rate, this equation
                                    // should be checked to ensure that the
                                    // reload value will fit in 8-bits.
                                                           
   CKCON |= 0x10;                   // T1M = 1; SCA1:0 = xx
   

   TL1 = TH1;                       // initialize Timer1
   TR1 = 1;                         // start Timer1  
   
   SFRPAGE = UART0_PAGE;
   TI0 = 1;                         // Indicate TX0 ready
   
   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page
}

//-----------------------------------------------------------------------------
// ADC0_Init
//-----------------------------------------------------------------------------
//
// Configure ADC0 to use Timer 3 overflows as a conversion source and to
// use left-justified output mode. 
//
void ADC0_Init (void)
{
   char SFRPAGE_SAVE = SFRPAGE;     // Preserve current SFR page

   SFRPAGE = ADC0_PAGE;

   ADC0CN = 0x85;                   // ADC0 enabled; normal tracking
                                    // mode; ADC0 conversions are initiated
                                    // on Timer3 overflows; ADC0 data is
                                    // left-justified

   REF0CN = 0x07;                   // enable temp sensor, on-chip VREF,
                                    // and VREF output buffer
   AMX0SL = 0x0F;                   // Select TEMP sens as ADC mux output

   ADC0CF = ((SYSCLK/2500000) << 3);// ADC conversion clock = 2.5MHz

   ADC0CF |= 0x01;                  // PGA gain = 2
   
   EIE2 |= 0x02;                    // Enable ADC0 End-of-conversion 
                                    // interrupts
   
   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page.
}


//-----------------------------------------------------------------------------
// CACHE_Init
//-----------------------------------------------------------------------------
//
// This routine initializes the cache by enabling FLASH block writes
// 
void CACHE_Init (void)
{  
   char SFRPAGE_SAVE = SFRPAGE;     // Preserve current SFR page
   
   SFRPAGE = CONFIG_PAGE;
   
   CCH0CN |= 0x01;                  // Enable Block Writes to FLASH

   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page.

}

//-----------------------------------------------------------------------------
// Timer3_Init
//-----------------------------------------------------------------------------
// This routine initializes Timer3 in auto-reload mode to generate interrupts
// at intervals specified in <counts>.
//
void Timer3_Init (int counts)
{
   char SFRPAGE_SAVE = SFRPAGE;     // Preserve current SFR page

   SFRPAGE = TMR3_PAGE;
   
   TMR3CN = 0;                      // STOP timer; set to auto-reload mode
   TMR3CF = 0x08  ;                 // Timer3 counts SYSCLKs
   RCAP3 = -counts;                 // set reload value
   
   TMR3 = RCAP3;                    // Initialize Timer3 to its reload value  
   
   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page.      
}



