//-----------------------------------------------------------------------------
// Cache_Example_3.c
//-----------------------------------------------------------------------------
// Copyright 2002 Cygnal Integrated Products, Inc.
//
// AUTH: FB
// DATE: 14 OCT 02
//
// This example demonstrates using FLASH block writes to copy an array of data
// from RAM to non-volatile FLASH memory.
//
// This program uses the the 24.5 MHz internal oscillator multiplied by two 
// for an effective SYSCLK of 49 MHz. 
//
// Target: C8051F12x
// Tool chain: KEIL C51 6.03 / KEIL EVAL C51
//

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <c8051f120.h>                 // SFR declarations
#include <stdio.h>                     // printf() and getchar()

//-----------------------------------------------------------------------------
// 16-bit SFR Definitions for 'F12x
//-----------------------------------------------------------------------------

sfr16 DP       = 0x82;                 // data pointer
sfr16 ADC0     = 0xbe;                 // ADC0 data
sfr16 ADC0GT   = 0xc4;                 // ADC0 greater than window
sfr16 ADC0LT   = 0xc6;                 // ADC0 less than window
sfr16 RCAP2    = 0xca;                 // Timer2 capture/reload
sfr16 RCAP3    = 0xca;                 // Timer3 capture/reload
sfr16 RCAP4    = 0xca;                 // Timer4 capture/reload
sfr16 TMR2     = 0xcc;                 // Timer2
sfr16 TMR3     = 0xcc;                 // Timer3
sfr16 TMR4     = 0xcc;                 // Timer4
sfr16 DAC0     = 0xd2;                 // DAC0 data
sfr16 DAC1     = 0xd2;                 // DAC1 data
sfr16 PCA0CP5  = 0xe1;                 // PCA0 Module 5 capture
sfr16 PCA0CP2  = 0xe9;                 // PCA0 Module 2 capture
sfr16 PCA0CP3  = 0xeb;                 // PCA0 Module 3 capture
sfr16 PCA0CP4  = 0xed;                 // PCA0 Module 4 capture
sfr16 PCA0     = 0xf9;                 // PCA0 counter
sfr16 PCA0CP0  = 0xfb;                 // PCA0 Module 0 capture
sfr16 PCA0CP1  = 0xfd;                 // PCA0 Module 1 capture

//-----------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------
#define TRUE         1
#define FALSE        0

#define INTCLK       24500000          // Internal oscillator frequency in Hz
#define SYSCLK       49000000          // Output of PLL derived from (INTCLK*2)

sbit LED = P1^6;                       // LED='1' means ON
sbit SW2 = P3^7;                       // SW2='0' means switch pressed

#define FLASH_PAGESIZE 1024            // The size of a FLASH page in bytes
#define IMAGE_BUFF_LEN 4096            // The size of <IMAGE_BUFF> in bytes 
#define XRAM_BUFF_LEN 4096             // The size of <XRAM_BUFF> in bytes 

//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------
void main(void);
void SYSCLK_Init(void);
void PORT_Init(void);
void IMAGE_Erase(void);
void IMAGE_Capture(void);
void IMAGE_Restore(void);

//-----------------------------------------------------------------------------
// Global VARIABLES
//-----------------------------------------------------------------------------
char xdata XRAM_BUFF[XRAM_BUFF_LEN] _at_ 0x0000;
char code IMAGE_BUFF[IMAGE_BUFF_LEN] _at_ 0x1000;


//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------

void main (void) 
{
   int i;                              // software counter   

   WDTCN = 0xde;                       // disable watchdog timer
   WDTCN = 0xad;

   PORT_Init ();                       // initialize crossbar and GPIO
   SYSCLK_Init ();                     // initialize oscillator
     
   // Fill XRAM with 0xAA
   for( i = 0; i < XRAM_BUFF_LEN; i++){
      XRAM_BUFF[i] = 0xAA;
   }

   // Erase the XRAM image buffer in FLASH and capture the current
   // XRAM contents
   IMAGE_Erase();
   IMAGE_Capture();

   // Fill XRAM with 0xBB
   for( i = 0; i < XRAM_BUFF_LEN; i++){
      XRAM_BUFF[i] = 0xBB;
   }

   // Restore the original XRAM image.
   IMAGE_Restore();
   
} 

//-----------------------------------------------------------------------------
// Support Routines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// IMAGE_Erase
//-----------------------------------------------------------------------------
// 
// This function erases the FLASH pages reserved for capturing an image of XRAM.
//
//
void IMAGE_Erase(void)
{
   char SFRPAGE_SAVE = SFRPAGE;        // Preserve current SFR page
   bit EA_SAVE;                        // Preserve interrupt state
   int i;                              // Software counter   
   char xdata * idata pwrite;          // FLASH write/erase pointer

   // Set the FLASH write/erase pointer to the beginning of <IMAGE_BUFF>
   pwrite = IMAGE_BUFF;

   // Erase all FLASH pages in the XRAM image buffer
   for( i = 0; i < IMAGE_BUFF_LEN; i += FLASH_PAGESIZE ){
      
      SFRPAGE = LEGACY_PAGE;
      
      EA_SAVE = EA;
      EA = 0;                          // Disable interrupts                 
      FLSCL |= 0x01;                   // Enable FLASH writes/erases
      PSCTL = 0x03;                    // MOVX writes FLASH byte
   
      pwrite[i] = 0;                   // Initiate FLASH page erase                  
      
      FLSCL &= ~0x01;                  // Disable FLASH writes/erases
      PSCTL = 0x00;                    // MOVX targets XRAM
      EA = EA_SAVE;                    // Restore interrupt state      
   }
  
   SFRPAGE = SFRPAGE_SAVE;             // Restore SFR page
}

//-----------------------------------------------------------------------------
// IMAGE_Capture
//-----------------------------------------------------------------------------
//
// This function captures an image of XRAM and stores it in FLASH.
//
//
void IMAGE_Capture(void)
{
   char SFRPAGE_SAVE = SFRPAGE;        // Preserve current SFR page
   bit EA_SAVE;                        // Preserve interrupt state
   int i;                              // Software counter   
   char xdata * idata pwrite;          // FLASH write/erase pointer
   unsigned char dat;                  // Holds data to copy   

   // Set the FLASH write/erase pointer to the beginning of <IMAGE_BUFF>
   pwrite = IMAGE_BUFF;
   
   // Enable FLASH block writes
   SFRPAGE = CONFIG_PAGE;
   CCH0CN |= 0x01;                     // Set the CHBLKW bit to enable
                                       // FLASH block writes

   // Copy an image of the entire XRAM to <IMAGE_BUFF> in FLASH
   for( i = 0; i < IMAGE_BUFF_LEN; i++ ){
      
      dat = XRAM_BUFF[i];              // Read one byte from XRAM
           
      SFRPAGE = LEGACY_PAGE;
      
      EA_SAVE = EA;
      EA = 0;                          // Disable interrupts                 
      FLSCL |= 0x01;                   // Enable FLASH writes/erases
      PSCTL = 0x01;                    // MOVX writes write FLASH byte
                                       // MOVX reads still target XRAM

      pwrite[i] = dat;                 // Write Data to FLASH                   
      
      FLSCL &= ~0x01;                  // Disable FLASH writes/erases
      PSCTL = 0x00;                    // MOVX targets XRAM
      EA = EA_SAVE;                    // Restore interrupt state      
   }
   
   SFRPAGE = CONFIG_PAGE;
   CCH0CN &= ~0x01;                    // Clear the CHBLKW bit

   SFRPAGE = SFRPAGE_SAVE;             // Restore SFR page
}

//-----------------------------------------------------------------------------
// IMAGE_Restore
//-----------------------------------------------------------------------------
//
// This function restores the contents of XRAM from a memory image stored
// in FLASH.
//
void IMAGE_Restore(void)
{
   int i;                              // Software counter   

   // Restore the image in <IMAGE_BUFF> in to XRAM
   for( i = 0; i < IMAGE_BUFF_LEN; i++ ){

      XRAM_BUFF[i] = IMAGE_BUFF[i];    // Restore one byte of XRAM      
   }
  
}

//-----------------------------------------------------------------------------
// Initialization Routines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// SYSCLK_Init
//-----------------------------------------------------------------------------
//
// This routine initializes the system clock to use the internal oscillator
// at 24.5 MHz multiplied by two using the PLL.
//
void SYSCLK_Init (void)
{
   int i;                           // software timer

   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page

   SFRPAGE = CONFIG_PAGE;           // set SFR page

   OSCICN = 0x83;                   // set internal oscillator to run
                                    // at its maximum frequency

   CLKSEL = 0x00;                   // Select the internal osc. as
                                    // the SYSCLK source
   
   //Turn on the PLL and increase the system clock by a factor of M/N = 2
   SFRPAGE = CONFIG_PAGE;
   
   PLL0CN  = 0x00;                  // Set internal osc. as PLL source
   SFRPAGE = LEGACY_PAGE;
   FLSCL   = 0x10;                  // Set FLASH read time for 50MHz clk
                                    // or less
   SFRPAGE = CONFIG_PAGE;
   PLL0CN |= 0x01;                  // Enable Power to PLL
   PLL0DIV = 0x01;                  // Set Pre-divide value to N (N = 1)
   PLL0FLT = 0x01;                  // Set the PLL filter register for 
                                    // a reference clock from 19 - 30 MHz
                                    // and an output clock from 45 - 80 MHz 
   PLL0MUL = 0x02;                  // Multiply SYSCLK by M (M = 2)
   
   for (i=0; i < 256; i++) ;        // Wait at least 5us
   PLL0CN  |= 0x02;                 // Enable the PLL
   while(!(PLL0CN & 0x10));         // Wait until PLL frequency is locked
   CLKSEL  = 0x02;                  // Select PLL as SYSCLK source

   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page
}


//-----------------------------------------------------------------------------
// PORT_Init
//-----------------------------------------------------------------------------
//
// This routine configures the crossbar and GPIO ports.
//
void PORT_Init (void)
{
   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page
   
   SFRPAGE = CONFIG_PAGE;           // set SFR page

   XBR0     = 0x00;                 
   XBR1     = 0x00;
   XBR2     = 0x40;                 // Enable crossbar and weak pull-ups
                                    

   P1MDOUT |= 0x40;                 // Set P1.6(LED) to push-pull
   
   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page
}



