//-----------------------------------------------------------------------------
// Cache_Example_2.c
//-----------------------------------------------------------------------------
// Copyright 2002 Cygnal Integrated Products, Inc.
//
// AUTH: FB
// DATE: 21 NOV 02
//
// This example shows how to service interrupts during FLASH write operations.
// During device initialization, the FLASH_ByteWrite_NonBlocking() function
// and the Timer0_ISR() are locked in the cache. This allows Timer0 interrupts
// to be serviced while a FLASH write is taking place. 
//
// This program uses the the 24.5 MHz internal oscillator multiplied by two 
// for an effective SYSCLK of 49 MHz. This program also initializes and uses
// UART1 at <BAUDRATE> bits per second.
//
//
// Target: C8051F12x
// Tool chain: KEIL C51 6.03 / KEIL EVAL C51
//

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <c8051f120.h>                 // SFR declarations
#include <stdio.h>                     // printf() and getchar()

//-----------------------------------------------------------------------------
// 16-bit SFR Definitions for 'F12x
//-----------------------------------------------------------------------------

sfr16 DP       = 0x82;                 // data pointer
sfr16 ADC0     = 0xbe;                 // ADC0 data
sfr16 ADC0GT   = 0xc4;                 // ADC0 greater than window
sfr16 ADC0LT   = 0xc6;                 // ADC0 less than window
sfr16 RCAP2    = 0xca;                 // Timer2 capture/reload
sfr16 RCAP3    = 0xca;                 // Timer3 capture/reload
sfr16 RCAP4    = 0xca;                 // Timer4 capture/reload
sfr16 TMR2     = 0xcc;                 // Timer2
sfr16 TMR3     = 0xcc;                 // Timer3
sfr16 TMR4     = 0xcc;                 // Timer4
sfr16 DAC0     = 0xd2;                 // DAC0 data
sfr16 DAC1     = 0xd2;                 // DAC1 data
sfr16 PCA0CP5  = 0xe1;                 // PCA0 Module 5 capture
sfr16 PCA0CP2  = 0xe9;                 // PCA0 Module 2 capture
sfr16 PCA0CP3  = 0xeb;                 // PCA0 Module 3 capture
sfr16 PCA0CP4  = 0xed;                 // PCA0 Module 4 capture
sfr16 PCA0     = 0xf9;                 // PCA0 counter
sfr16 PCA0CP0  = 0xfb;                 // PCA0 Module 0 capture
sfr16 PCA0CP1  = 0xfd;                 // PCA0 Module 1 capture

//-----------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------
#define TRUE         1
#define FALSE        0

#define INTCLK       24500000          // Internal oscillator frequency in Hz
#define SYSCLK       49000000          // Output of PLL derived from (INTCLK*2)

sbit LED = P1^6;                       // LED='1' means ON
sbit SW2 = P3^7;                       // SW2='0' means switch pressed

#define Timer0_ISR_NUM 0x01            // Timer0 interrupt number
#define Timer0_ISR_LEN 0x05            // Length of Timer0_ISR in bytes
                                       // Check the MAP (*.M51) file for 
                                       // this project.

#define FLASH_ByteWrite_NonBlocking_LEN 0x22
                                       // The number of bytes that will be 
                                       // cached in the FLASH write routine.

//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------
void main(void);
void SYSCLK_Init(void);
void PORT_Init(void);
void Timer0_Init (void);
void Timer0_ISR (void);
void Cache_ISR(   unsigned int start_address, 
                  unsigned int interrupt_length,
                  unsigned int interrupt_number );
void Cache_Function( unsigned int start_address, unsigned int length );
void FLASH_ByteWrite_NonBlocking ( unsigned char xdata* pwrite, unsigned char dat); 
void FLASH_PageErase(unsigned int address);


//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------

void main (void) 
{
   unsigned char xdata* data pwrite;   // FLASH write and erase pointer
	unsigned char temp_char = 0x00;     // Temporary char
   int i;                              // Software counter
               
   WDTCN = 0xde;                       // Disable watchdog timer
   WDTCN = 0xad;

   PORT_Init ();                       // Initialize crossbar and GPIO
   SYSCLK_Init ();                     // Initialize oscillator
   Timer0_Init ();                     // Initialize Timer0;
   
   Cache_Function( (unsigned int) FLASH_ByteWrite_NonBlocking, 
                                  FLASH_ByteWrite_NonBlocking_LEN ); 
   
      
   Cache_ISR( (unsigned int) Timer0_ISR, Timer0_ISR_LEN, Timer0_ISR_NUM); 
      
   FLASH_PageErase(0x1000);            // Erase the FLASH page at 0x1000
   pwrite = 0x1000;                    // Set FLASH write pointer to 0x1000

   EA = 1;
   
   for( i = 0; i < 100; i++ ){
      
      // Write a byte to FLASH without blocking any cached ISRs (eg. Timer0_ISR)
      FLASH_ByteWrite_NonBlocking( pwrite++, temp_char++);   
   }
   
   while(1);

} 


//-----------------------------------------------------------------------------
// Timer0_ISR
//-----------------------------------------------------------------------------
// 
// This interrupt service routine toggles the LED when Timer0 overflows.

void Timer0_ISR (void) interrupt 1 {
    
   TF0 = 0;                            // Clear Timer0 overflow flag
   LED = ~LED;                         // Toggle LED
   

} 

//-----------------------------------------------------------------------------
// Cache_ISR
//-----------------------------------------------------------------------------
//
// This routine pushes and locks an entire ISR and its interrupt vector into 
// the cache. This function assumes that the interrupt vector and ISR is
// located in the common area.
//
void Cache_ISR(   unsigned int start_address, 
                  unsigned int interrupt_length,
                  unsigned int interrupt_number   )
{
   char SFRPAGE_SAVE = SFRPAGE;        // Preserve current SFR page
   char EA_SAVE = EA;                  // Preserve interrupt state

   unsigned char code* pread;          // Pointer used to generate MOVC 
                                       // instructions to initiate a cache
                                       // push operation

   unsigned char temp_char;            // Temporary char.
   
   SFRPAGE = CONFIG_PAGE;              // Set SFR page
   
   // Check if there is enough room to cache the ISR. When the number of
   // slots required by the ISR + interrupt vector are subtracted from
   // the CHSLOT pointer, the result is the number of unlocked slots available.
   if( ((CCH0LC & 0x3F) - (interrupt_length/4 + 2)) <= 0){
      while(1);                        // Handle Error Condition.
                                       // Not enough room to cache ISR
   }    

   // Set the <pread> pointer to the address of the interrupt vector.
   pread = ((interrupt_number * 8) + 3 ); 
   
   EA = 0;                             // Disable Interrupts

   CCH0LC &= ~0xC0;                    // Clear the CHPUSH and CHPOP bits
   CCH0LC |=  0x80;                    // Enable cache pushes, MOVC instructions
                                       // will push 4-byte FLASH segments into
                                       // the cache
                                       
   // Push the interrupt vector in the cache slot pointed to by CHSLOT.
   // An interrupt vector consists of a 3-byte LJMP instruction and the
   // LSBs of the vector address are always 10b. One byte of the LJMP
   // instruction fall in the first 4-byte FLASH segment and the remaining
   // two bytes of the instruction falls in the second 4-byte segment. 
   // Two cache pushes are neccessary.
   
   temp_char = *pread;                 // Push the first 4-byte segment (MOVC)
   temp_char = *(pread + 4);           // Push the second 4-byte segment (MOVC)

   // Push the interrupt service routine in the cache. The interrupt starts
   // at <START_LABEL> and is <interrupt_length> bytes long.
   
   pread = (unsigned char code*) start_address;
                                       // Set <pread> to the address of the
                                       // interrupt service routine.
   
   while( pread < start_address + interrupt_length ){       
      
      temp_char = *pread;              // Push a 4-byte segment (MOVC)
      
      pread += 4;                      // Increment <pread> to the next 
                                       // segment.
   }                                        
   

   CCH0LC &= ~0x80;                    // Clear CHPUSH to disable cache pushes.
   SFRPAGE = SFRPAGE_SAVE;             // Restore SFR page.
   EA = EA_SAVE;                       // Restore interrupt state.

}

//-----------------------------------------------------------------------------
// Cache_Function
//-----------------------------------------------------------------------------
//
// This routine pushes and locks a function into the cache. This routine 
// assumes that the function is located in the common area.
//
//
void Cache_Function( unsigned int start_address, unsigned int length )
{
   char SFRPAGE_SAVE = SFRPAGE;        // Preserve current SFR page
   char EA_SAVE = EA;                  // Preserve interrupt state

   unsigned char code* pread;          // Pointer used to generate MOVC 
                                       // instructions to initiate a cache
                                       // push operation

   unsigned char temp_char;            // Temporary char.
   
   SFRPAGE = CONFIG_PAGE;              // Set SFR page

   // Check if there is enough room to cache the function. When the number of
   // slots required to cache the function is subtracted from
   // the CHSLOT pointer, the result is the number of unlocked slots available.
   if( (CCH0LC & 0x3F) - (length/4) <= 0){
      while(1);                        // Handle Error Condition.
                                       // Not enough room to cache function
   }

   // Set <pread> to the address of the function
   pread = (unsigned char code*) start_address;
                                      
   

   EA = 0;                             // Disable Interrupts

   CCH0LC &= ~0xC0;                    // Clear the CHPUSH and CHPOP bits
   CCH0LC |=  0x80;                    // Enable cache pushes, MOVC instructions
                                       // will push 4-byte FLASH segments into
                                       // the cache
                                       
   // Push the function in the cache. The interrupt starts
   // at <START_LABEL> and is <length> bytes long.
   while( pread < start_address + length ){       
      
      temp_char = *pread;              // Push a 4-byte segment (MOVC)
      
      pread += 4;                      // Increment <pread> to the next 
                                       // segment.
   }                                        
   

   CCH0LC &= ~0x80;                    // Clear CHPUSH to disable cache pushes.
   SFRPAGE = SFRPAGE_SAVE;             // Restore SFR page.
   EA = EA_SAVE;                       // Restore interrupt state.

}

//-----------------------------------------------------------------------------
// FLASH_ByteWrite_NonBlocking
//-----------------------------------------------------------------------------
//
// This routine writes one byte to FLASH and allows interrupts with cached ISRs
// to be serviced during the FLASH write operation.
//
void FLASH_ByteWrite_NonBlocking ( unsigned char xdata* pwrite, unsigned char dat ) 
{

   EA = 0;                           // Disable Global Interrupts
   SFRPAGE = LEGACY_PAGE;
   FLSCL |= 0x01;                    // Enable FLASH writes/erases
   PSCTL  = 0x01;                    // MOVX writes FLASH byte
      	
   *pwrite++ = dat;                  // Initiate FLASH write (generate MOVX write)
      
   SFRPAGE = LEGACY_PAGE;
   PSCTL  = 0x00;                    // MOVX targets XRAM
   FLSCL &= ~0x01;                   // Disable FLASH writes/erases
   EA = 1;                           // Enable global interrupts
      
   SFRPAGE = CONFIG_PAGE;                               
   while(FLBUSY);                    // Any interrupts with preloaded ISRs may be 
                                     // serviced while the CIP-51 is in this loop
}

//-----------------------------------------------------------------------------
// FLASH_PageErase
//-----------------------------------------------------------------------------
//
// This routine erases the FLASH page located at <address>;
//
void FLASH_PageErase(unsigned int address)
{
   char SFRPAGE_SAVE = SFRPAGE;        // Preserve current SFR page
   bit EA_SAVE;                        // Preserve interrupt state
   char xdata * idata pwrite;          // FLASH write/erase pointer

   // Set the FLASH write/erase pointer to the FLASH page at 0x1000
   pwrite = (char xdata *) address;

   SFRPAGE = LEGACY_PAGE;
      
   EA_SAVE = EA;
   EA = 0;                             // Disable interrupts                 
   FLSCL |= 0x01;                      // Enable FLASH writes/erases
   PSCTL = 0x03;                       // MOVX writes FLASH byte
   
   *pwrite = 0;                        // Initiate FLASH page erase                  
      
   FLSCL &= ~0x01;                     // Disable FLASH writes/erases
   PSCTL = 0x00;                       // MOVX targets XRAM
   EA = EA_SAVE;                       // Restore interrupt state      
   
   SFRPAGE = SFRPAGE_SAVE;             // Restore SFR page

}

//-----------------------------------------------------------------------------
// Initialization Routines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// SYSCLK_Init
//-----------------------------------------------------------------------------
//
// This routine initializes the system clock to use the internal oscillator
// at 24.5 MHz multiplied by two using the PLL.
//
void SYSCLK_Init (void)
{
   int i;                           // software timer

   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page

   SFRPAGE = CONFIG_PAGE;           // set SFR page

   OSCICN = 0x83;                   // set internal oscillator to run
                                    // at its maximum frequency

   CLKSEL = 0x00;                   // Select the internal osc. as
                                    // the SYSCLK source
   
   //Turn on the PLL and increase the system clock by a factor of M/N = 2
   SFRPAGE = CONFIG_PAGE;
   
   PLL0CN  = 0x00;                  // Set internal osc. as PLL source
   SFRPAGE = LEGACY_PAGE;
   FLSCL   = 0x10;                  // Set FLASH read time for 50MHz clk
                                    // or less
   SFRPAGE = CONFIG_PAGE;
   PLL0CN |= 0x01;                  // Enable Power to PLL
   PLL0DIV = 0x01;                  // Set Pre-divide value to N (N = 1)
   PLL0FLT = 0x01;                  // Set the PLL filter register for 
                                    // a reference clock from 19 - 30 MHz
                                    // and an output clock from 45 - 80 MHz 
   PLL0MUL = 0x02;                  // Multiply SYSCLK by M (M = 2)
   
   for (i=0; i < 256; i++) ;        // Wait at least 5us
   PLL0CN  |= 0x02;                 // Enable the PLL
   while(!(PLL0CN & 0x10));         // Wait until PLL frequency is locked
   CLKSEL  = 0x02;                  // Select PLL as SYSCLK source

   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page
}


//-----------------------------------------------------------------------------
// PORT_Init
//-----------------------------------------------------------------------------
//
// This routine configures the crossbar and GPIO ports.
//
void PORT_Init (void)
{
   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page
   
   SFRPAGE = CONFIG_PAGE;           // set SFR page

   XBR0     = 0x00;                 
   XBR1     = 0x00;
   XBR2     = 0x40;                 // Enable crossbar and weak pull-up
                                    

   P1MDOUT |= 0x40;                 // Set P1.6(LED) to push-pull
   
   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page
}


//-----------------------------------------------------------------------------
// Timer0_Init
//-----------------------------------------------------------------------------
//
// Configure Timer0 to count SYSCLKs in Mode1(16-bit counter timer).
//
void Timer0_Init (void)
{
   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page
   
   SFRPAGE = TIMER01_PAGE;
   
   TMOD   &= ~0x0F;
   TMOD   |=  0x01;                 // TMOD: Timer0, mode 1, 16-bit timer

   CKCON  |= 0x08;                  // Timer0 counts SYSCLKs

   ET0 = 1;                         // Enable Timer0 overflow interrupts
   
   TR0 = 1;                         // Start Timer0

   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page

}
