//-----------------------------------------------------------------------------
// Cache_Example_1.c
//-----------------------------------------------------------------------------
// Copyright 2002 Cygnal Integrated Products, Inc.
//
// AUTH: FB
// DATE: 21 NOV 02
//
// This example shows how to cache an ISR entry point to achieve deterministic 
// latency.  The Cache_ISR_Entry() function call may be commented out or compiled 
// to compare the difference between cached ISR entry points verses ISRs which
// do not have their entry points cached.
//
// Timer0  is configured to count SYSCLKs, and service an interrupt on
// overflow. If a breakpoint is placed on the first instruction of Timer0_ISR,
// the value of the timer when the breakpoint is reached is the 
// number of SYSCLK cycles taken to enter the ISR after the overflow event
// has occurred.
//
// This program uses the the 24.5 MHz internal oscillator multiplied by two 
// for an effective SYSCLK of 49 MHz. 
//
// Target: C8051F12x
// Tool chain: KEIL C51 6.03 / KEIL EVAL C51
//

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <c8051f120.h>                 // SFR declarations
#include <stdio.h>                     // printf() and getchar()

//-----------------------------------------------------------------------------
// 16-bit SFR Definitions for 'F12x
//-----------------------------------------------------------------------------

sfr16 DP       = 0x82;                 // data pointer
sfr16 ADC0     = 0xbe;                 // ADC0 data
sfr16 ADC0GT   = 0xc4;                 // ADC0 greater than window
sfr16 ADC0LT   = 0xc6;                 // ADC0 less than window
sfr16 RCAP2    = 0xca;                 // Timer2 capture/reload
sfr16 RCAP3    = 0xca;                 // Timer3 capture/reload
sfr16 RCAP4    = 0xca;                 // Timer4 capture/reload
sfr16 TMR2     = 0xcc;                 // Timer2
sfr16 TMR3     = 0xcc;                 // Timer3
sfr16 TMR4     = 0xcc;                 // Timer4
sfr16 DAC0     = 0xd2;                 // DAC0 data
sfr16 DAC1     = 0xd2;                 // DAC1 data
sfr16 PCA0CP5  = 0xe1;                 // PCA0 Module 5 capture
sfr16 PCA0CP2  = 0xe9;                 // PCA0 Module 2 capture
sfr16 PCA0CP3  = 0xeb;                 // PCA0 Module 3 capture
sfr16 PCA0CP4  = 0xed;                 // PCA0 Module 4 capture
sfr16 PCA0     = 0xf9;                 // PCA0 counter
sfr16 PCA0CP0  = 0xfb;                 // PCA0 Module 0 capture
sfr16 PCA0CP1  = 0xfd;                 // PCA0 Module 1 capture

//-----------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------
#define TRUE         1
#define FALSE        0

#define INTCLK       24500000          // Internal oscillator frequency in Hz
#define SYSCLK       49000000          // Output of PLL derived from (INTCLK*2)

sbit LED = P1^6;                       // LED='1' means ON
sbit SW2 = P3^7;                       // SW2='0' means switch pressed

#define Timer0_ISR_NUM 1               // Timer0 interrupt number

//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------
void main(void);
void SYSCLK_Init(void);
void PORT_Init(void);
void Timer0_Init (void);
void Timer0_ISR (void);
void Cache_ISR_Entry( unsigned int start_address, 
                      unsigned int interrupt_number );

//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------

void main (void) 
{

   WDTCN = 0xde;                       // disable watchdog timer
   WDTCN = 0xad;

   PORT_Init ();                       // initialize crossbar and GPIO
   SYSCLK_Init ();                     // initialize oscillator
   Timer0_Init ();                     // initialize Timer0;
      
   //Cache_ISR_Entry( (unsigned int) Timer0_ISR, Timer0_ISR_NUM); 

   EA = 1;

   while(1){
        
      PCON |= 0x01;                    // Put the CPU in idle mode. The CPU
                                       // will only enter idle mode when 
                                       // running at full speed and will return
                                       // to full operation when Timer0 
                                       // overflows. This instruction has no
                                       // effect if the CIP-51 is in debug
                                       // mode.
   }

} 


//-----------------------------------------------------------------------------
// Timer0_ISR
//-----------------------------------------------------------------------------
// 
// This interrupt service routine toggles the LED when Timer0 overflows.
//  
void Timer0_ISR (void) interrupt 1 {
    
   TF0 = 0;                            // Clear Timer0 overflow flag
   LED = ~LED;                         // Toggle LED
} 

//-----------------------------------------------------------------------------
// Cache_ISR_Entry
//-----------------------------------------------------------------------------
//
// This routine pushes and locks an interrupt vector and the first one or two
// 4-byte FLASH segments of an ISR entry point into the cache. 
//
void Cache_ISR_Entry (   unsigned int start_address, 
                         unsigned int interrupt_number   )
{
   char SFRPAGE_SAVE = SFRPAGE;        // Preserve current SFR page
   char EA_SAVE = EA;                  // Preserve interrupt state

   unsigned char code* pread;          // Pointer used to generate MOVC 
                                       // instructions to initiate a cache
                                       // push operation

   unsigned char temp_char;            // Temporary char.

   // Set the <pread> pointer to the address of the interrupt vector.
   pread = ((interrupt_number * 8) + 3 ); 
   

   SFRPAGE = CONFIG_PAGE;              // Set SFR page

   EA = 0;                             // Disable Interrupts

   CCH0LC &= ~0xC0;                    // Clear the CHPUSH and CHPOP bits
   CCH0LC |=  0x80;                    // Enable cache pushes, MOVC instructions
                                       // will push 4-byte FLASH segments into
                                       // the cache
                                       
   // Check if there is enough room to cache the interrupt vector and the first
   // two 4-byte segments of the ISR entry point.  When the two (or one) slot(s)
   // required to cache the ISR entry point and the two slots needed to cache the
   // interrupt vector (Total 4 slots) are subtracted from the CHSLOT pointer,
   // the result is the number of unlocked slots available.
   if( ((CCH0LC & 0x3F) - 4) <= 0 ){
      while(1);                        // Handle Error Condition.
                                       // Not enough room to cache ISR Entry Point
   }    
   
   // Push the interrupt vector in the cache slot pointed to by CHSLOT.
   // An interrupt vector generated by the KEIL C51 compiler consists of a
   // 3-byte LJMP instruction. Since the LSBs of an interrupt vector address are 
   // always 10b, the first two bytes of the LJMP instruction fall in the first
   // 4-byte FLASH segment and the remaining byte of the instruction falls in the
   // second 4-byte segment.  Two cache pushes are neccessary.
   
   temp_char = *pread;                 // Push the first 4-byte segment (MOVC)
   temp_char = *(pread + 4);           // Push the second 4-byte segment (MOVC)

   // If the first byte of the ISR is aligned on a multiple of 4 (i.e. the LSBs
   // of the first byte are 00b), then cache the first 4-byte segment. Otherwise,
   // cache the first two 4-byte segments.
   pread = (unsigned char code*) start_address;
                                       // Set <pread> to the address of the
                                       // interrupt service routine.
   
   temp_char = *pread;                 // Push the first 4-byte segment (MOVC)
      
   if(start_address & 0x03){
      temp_char = *(pread + 4);        // Push the second 4-byte segment (MOVC)
   }                                   // if the two LSBs of <start_address>
                                       // are not zero

   CCH0LC &= ~0x80;                    // Clear CHPUSH to disable cache pushes.
   SFRPAGE = SFRPAGE_SAVE;             // Restore SFR page.
   EA = EA_SAVE;                       // Restore interrupt state.

}

//-----------------------------------------------------------------------------
// Initialization Routines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// SYSCLK_Init
//-----------------------------------------------------------------------------
//
// This routine initializes the system clock to use the internal oscillator
// at 24.5 MHz multiplied by two using the PLL.
//
void SYSCLK_Init (void)
{
   int i;                           // software timer

   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page

   SFRPAGE = CONFIG_PAGE;           // set SFR page

   OSCICN = 0x83;                   // set internal oscillator to run
                                    // at its maximum frequency

   CLKSEL = 0x00;                   // Select the internal osc. as
                                    // the SYSCLK source
   
   //Turn on the PLL and increase the system clock by a factor of M/N = 2
   SFRPAGE = CONFIG_PAGE;
   
   PLL0CN  = 0x00;                  // Set internal osc. as PLL source
   SFRPAGE = LEGACY_PAGE;
   FLSCL   = 0x10;                  // Set FLASH read time for 50MHz clk
                                    // or less
   SFRPAGE = CONFIG_PAGE;
   PLL0CN |= 0x01;                  // Enable Power to PLL
   PLL0DIV = 0x01;                  // Set Pre-divide value to N (N = 1)
   PLL0FLT = 0x01;                  // Set the PLL filter register for 
                                    // a reference clock from 19 - 30 MHz
                                    // and an output clock from 45 - 80 MHz 
   PLL0MUL = 0x02;                  // Multiply SYSCLK by M (M = 2)
   
   for (i=0; i < 256; i++) ;        // Wait at least 5us
   PLL0CN  |= 0x02;                 // Enable the PLL
   while(!(PLL0CN & 0x10));         // Wait until PLL frequency is locked
   CLKSEL  = 0x02;                  // Select PLL as SYSCLK source

   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page
}


//-----------------------------------------------------------------------------
// PORT_Init
//-----------------------------------------------------------------------------
//
// This routine configures the crossbar and GPIO ports.
//
void PORT_Init (void)
{
   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page
   
   SFRPAGE = CONFIG_PAGE;           // set SFR page

   XBR0     = 0x00;                 
   XBR1     = 0x00;
   XBR2     = 0x40;                 // Enable crossbar and weak pull-up
                                    

   P1MDOUT |= 0x40;                 // Set P1.6(LED) to push-pull
   
   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page
}


//-----------------------------------------------------------------------------
// Timer0_Init
//-----------------------------------------------------------------------------
//
// Configure Timer0 to count SYSCLKs in Mode 1 (16-bit timer).
//
void Timer0_Init (void)
{
   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page
   
   SFRPAGE = TIMER01_PAGE;
   
   TMOD   &= ~0x0F;
   TMOD   |=  0x01;                 // TMOD: Timer0, mode 1, 16-bit timer

   CKCON  |= 0x08;                  // Timer0 counts SYSCLKs

   ET0 = 1;                         // Enable Timer0 overflow interrupts
   
   TR0 = 1;                         // Start Timer0

   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page

}
