//-----------------------------------------------------------------------------
// AN018_SW.c
//-----------------------------------------------------------------------------
// Copyright 2001 Cygnal Integrated Products, Inc.
//
// AUTH: BW
//
// This program outputs the C8051Fxxx die temperature out the hardware 
// UART at 115.2kbps. Assumes an 18.432MHz crystal is attached between 
// XTAL1 and XTAL2.
//
// The ADC is configured to look at the on-chip temp sensor.  The sampling
// rate of the ADC is determined by the constant <SAMPLE_RATE>, which is given
// in Hz.  The maximum value of <SAMPLE_RATE> is limited to ~86kHz due to
// the choice of 18.432MHz crystal (SAR clock = SYSCLK / 16 = 1.152MHz. One
// conversion takes 16 SAR clocks --> 72kHz sampling rate).
// 
// The ADC End of Conversion Interrupt Handler retrieves the sample
// from the ADC and adds it to a running accumulator.  Every 256 
// samples, the ADC updates and stores its result in the global variable
// <result>.  The sampling technique of adding a set of values and
// decimating them (posting results every 256th sample) is called accumulate
// and dump.  It is easy to implement and requires very few resources.
//
// For each power of 4, you gain 1 bit of effective resolution.
// For a factor of 256, gain you 4 bits of resolution: 4^4 = 256.
// Also, to properly scale the result back to 16-bits, perform a right
// shift of 4 bits.
//
// Target: C8051F00x or C8051F01x
// Tool chain: KEIL C51 6.03 / KEIL C51 EVAL version
//

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------

#include <stdio.h>



#include <c8051f000.h>                 // SFR declarations

//-----------------------------------------------------------------------------
// 16-bit SFR Definitions for F00x, F01x
//-----------------------------------------------------------------------------

sfr16 DP       = 0x82;                 // data pointer
sfr16 TMR3RL   = 0x92;                 // Timer3 reload value
sfr16 TMR3     = 0x94;                 // Timer3 counter
sfr16 ADC0     = 0xbe;                 // ADC0 data
sfr16 ADC0GT   = 0xc4;                 // ADC0 greater than window
sfr16 ADC0LT   = 0xc6;                 // ADC0 less than window
sfr16 RCAP2    = 0xca;                 // Timer2 capture/reload
sfr16 T2       = 0xcc;                 // Timer2
sfr16 DAC0     = 0xd2;                 // DAC0 data
sfr16 DAC1     = 0xd5;                 // DAC1 data

//-----------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------

#define SYSCLK       18432000          // SYSCLK frequency in Hz
#define BAUDRATE     115200            // Baud rate of UART in bps
#define SAMPLE_RATE  100000            // Sample frequency in Hz

#define LED          P1.6              // LED=1 means ON


//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------

void SYSCLK_Init (void);
void PORT_Init (void);
void UART_Init (void);
void ADC_Init (void);
void TIMER3_Init (int counts);
void ADC_ISR (void);

//-----------------------------------------------------------------------------
// Global VARIABLES
//-----------------------------------------------------------------------------

long result;                            // Output result from oversmapling and 
                                       // averaging 256 samples from the ADC for
                                       // 16-bit measurement resolution

//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------

void main (void) {
   long temp_copy;
   int temp_int;                       // integer portion of temperature
   int temp_frac;                      // fractional portion of temperature (in 
                                       // hundredths of a degree)

   WDTCN = 0xde;                       // disable watchdog timer
   WDTCN = 0xad;

   SYSCLK_Init ();                     // initialize oscillator
   PORT_Init ();                       // initialize crossbar and GPIO
   UART_Init ();                       // initialize UART
   TIMER3_Init (SYSCLK/SAMPLE_RATE);   // initialize Timer3 to overflow at
                                       // sample rate
   ADC_Init ();                        // init ADC

	ADCEN = 1;                          // enable ADC

   result = 0L;                        // initialize temperature variable

   EA = 1;                             // Enable global interrupts

	while (1) {
      temp_copy = result;              // Get most recent sample to convert
                                       //  the ADC code to a temperature
      temp_copy -= 0xa381;             // correct offset to 0deg, 0V
      temp_copy *= 0x01a9;             // 2.86mV/degree C
      temp_copy *= 100;                // convert result to 100ths of a degree C
      temp_copy = temp_copy >> 16;     // divide by 2^16
      temp_int = temp_copy / 100;      // Seperate integer and fractional components
      temp_frac = temp_copy - (100 * temp_int);
		printf ("Temperature is %d.%d\n", (int) temp_int, (int) temp_frac);
	}
}

//-----------------------------------------------------------------------------
// Initialization Subroutines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// SYSCLK_Init
//-----------------------------------------------------------------------------
//
// This routine initializes the system clock to use an 18.432MHz crystal
// as its clock source.
//
void SYSCLK_Init (void)
{
   int i;                              // delay counter

   OSCXCN = 0x67;                      // start external oscillator with
                                       // 18.432MHz crystal

   for (i=0; i < 256; i++) ;           // XTLVLD blanking interval (>1ms)

   while (!(OSCXCN & 0x80)) ;          // Wait for crystal osc. to settle

   OSCICN = 0x88;                      // select external oscillator as SYSCLK
                                       // source and enable missing clock
                                       // detector
}

//-----------------------------------------------------------------------------
// PORT_Init
//-----------------------------------------------------------------------------
//
// Configure the Crossbar and GPIO ports
//
void PORT_Init (void)
{
   XBR0    = 0x07;                     // Enable I2C, SPI, and UART
   XBR1    = 0x00;
   XBR2    = 0x40;                     // Enable crossbar and weak pull-ups
   PRT0CF |= 0xff;                     // enable all outputs on P0 as push-pull
                                       // push-pull; let xbar configure pins 
                                       // as inputs as necessary
   PRT1CF |= 0x40;                     // enable P1.6 (LED) as push-pull output
}

//-----------------------------------------------------------------------------
// PORT_Init
//-----------------------------------------------------------------------------
//
// Configure the UART using Timer1, for <baudrate> and 8-N-1.
//
void UART_Init (void)
{
   SCON   = 0x50;                      // SCON: mode 1, 8-bit UART, enable RX
   TMOD   = 0x20;                      // TMOD: timer 1, mode 2, 8-bit reload
   TH1    = -(SYSCLK/BAUDRATE/16);     // set Timer1 reload value for baudrate
   TR1    = 1;                         // start Timer1
   CKCON |= 0x10;                      // Timer1 uses sysclk as time base
   PCON  |= 0x80;                      // SMOD = 1
   TI     = 1;                         // Indicate TX ready
}

//-----------------------------------------------------------------------------
// ADC_Init
//-----------------------------------------------------------------------------
//
// Configure A/D converter to use Timer3 overflows as conversion source, to
// generate an interrupt on conversion complete, and to use right-justified
// output mode.  Enables ADC end of conversion interrupt. Leaves ADC disabled.
//
void ADC_Init (void)
{
   ADC0CN = 0x04;                      // ADC disabled; normal tracking
                                       // mode; ADC conversions are initiated 
                                       // on overflow of Timer3; ADC data is
                                       // right-justified
   REF0CN = 0x07;                      // enable temp sensor, on-chip VREF,
                                       // and VREF output buffer
   AMX0SL = 0x0f;                      // Select TEMP sens as ADC mux output
   ADC0CF = 0x61;                      // ADC conversion clock = sysclk/8

   EIE2 |= 0x02;                       // enable ADC interrupts
}

//-----------------------------------------------------------------------------
// TIMER3_Init
//-----------------------------------------------------------------------------
//
// Configure Timer3 to auto-reload at interval specified by <counts> (no 
// interrupt generated) using SYSCLK as its time base.
//
void TIMER3_Init (int counts)
{
   TMR3CN = 0x02;                      // Stop Timer3; Clear TF3;
                                       // use SYSCLK as timebase
   TMR3RL  = -counts;                  // Init reload values
   TMR3    = 0xffff;                   // set to reload immediately
   EIE2   &= ~0x01;                    // disable Timer3 interrupts
   TMR3CN |= 0x04;                     // start Timer3
}

//-----------------------------------------------------------------------------
// Interrupt Service Routines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// ADC_ISR
//-----------------------------------------------------------------------------
//
// ADC end-of-conversion ISR 
// Here we take the ADC sample, add it to a running total <accumulator>, and
// decrement our local decimation counter <int_dec>.  When <int_dec> reaches
// zero, we calculate the new value of the global variable <result>,
// which stores the accumulated ADC result.
//
void ADC_isr (void) interrupt 15
{
   static unsigned int_dec=256;        // integrate/decimate counter
                                       // we post a new result when
                                       // int_dec = 0
   static long accumulator=0L;         // heres where we integrate the
                                       // ADC samples             

   ADCINT = 0;                         // clear ADC conversion complete
                                       // indicator

	accumulator += ADC0;                // read ADC value and add to running
                                       // total
   int_dec--;                          // update decimation counter

   if (int_dec == 0) {                 // if zero, then decimate
      int_dec = 256;                   // reset counter
      result = accumulator >> 4;       // Shift to perform the divide operation
      accumulator = 0L;                // dump accumulator
   }
}
