;-----------------------------------------------------------------------------
;	COPYRIGHT 2001 CYGNAL INTEGRATED PRODUCTS, INC.
;
; 	FILE NAME  	: an015_2.ASM 
; 	TARGET MCU	: C8051F000
; 	DESCRIPTION	: Example source code for a software UART
;
; IMPLEMENTATION NOTES:
;  - uses Timer0 in Mode3 (two 8-bit timers)
;  - Timer0 run/overflow is used for RX state machine
;  - Timer1 overflow is used for TX state machine
;  - 8N1, no Framing error detection
;  - uses IE7 as user-level interrupt
;  - uses single-byte PC offset for state table implementation
;  - uses /INT0 falling as START bit detect (primitive START bit verification)
;
;-----------------------------------------------------------------------------

;-----------------------------------------------------------------------------
; EQUATES
;-----------------------------------------------------------------------------

$MOD8F000

; SW UART constants
SW_TX_GPIO	EQU	P0.4				; SW UART TX GPIO pin (can be any GPIO pin)
SW_RX_GPIO	EQU	P0.2				; SW UART RX GPIO pin (must be /INT0)

TIME_COUNT	EQU	320				; = SYSCLK_FREQ / BAUD_RATE (MIN = 128 HD / 
											;  480 FD)
											; Note: 320 is the limit for reliable
											;  FD operation...
											; FOR SYSCLK = 18.432 MHz:
											;  115200 = 160
											;  57600 = 320
											;  38400 = 480
											;  19200 = 960
											;  9600 = 1920
											;  4800 = 3840
											;  2400 = 7680

TX_CORR		EQU	41					; (41) Timer preload correction value in 
											;  cycles for TX
RX_CORR		EQU	47					; (47) Timer preload correction value in 
											;  cycles for RX
THALF_CORR	EQU	113				; (113)Timer preload correction value for 
											;  3/2 RX

TX_BT		EQU TIME_COUNT - TX_CORR	; actual 16-bit bit counter cycle value 
												;  TX
RX_BT		EQU TIME_COUNT - RX_CORR	; actual 16-bit bit counter cycle value 
												;  RX
THALF_BT	EQU TIME_COUNT*3/2 - THALF_CORR ; actual 16-bit 1.5 bit cycle value 
												;  RX

RX_BUFSIZE	EQU	16					; size of RX buffer in chars
;-----------------------------------------------------------------------------
; VARIABLES
;-----------------------------------------------------------------------------

BSEG
			org 0h

SRI:		DBIT		1					; SW UART Receive complete flag
STI:		DBIT		1					; SW UART Transmit complete flag
STXBSY:	DBIT		1					; SW UART TX BSY flag
SREN:		DBIT		1					; SW UART RX Enable
SES:		DBIT		1					; SW UART user-Interrup Support Enable

DSEG at 30h

TDR:		DS			1					; SW UART TX data register
RDR:		DS			1					; SW UART RX data register
RXSHIFT:	DS			1					; SW UART RX shift register
SURXST:	DS			1					; SW UART RX state variable
SUTXST:	DS			1					; SW UART TX state variable
BCRHI:	DS			1					; MSB of 16-bit bit timer for SW UART RX
BCTHI:	DS			1					; MSB of 16-bit bit timer for SW UART TX

;test variables
RX_TAIL:	DS			1					; write pointer to RX message buffer
TX_VAL:	DS			1					; value to transmit
;-------------------
; Indirect address space variables

ISEG at 80h

RX_BUF:	DS			RX_BUFSIZE		; RX message buffer

;-------------------
; STACK

STACK_TOP:	DS		1					; placeholder in symbol table for 
											;  beginning of hardware stack

;-----------------------------------------------------------------------------
; MACRO DEFINITIONS
;-----------------------------------------------------------------------------

;-----------------------------------------------------------------------------
; RESET AND INTERRUPT VECTOR TABLE
;-----------------------------------------------------------------------------

CSEG
			org	00h					
			ljmp	Reset					; RESET initialization vector

			org	03h
			ljmp	INT0_ISR				; Software UART RX START bit detect

			org	0bh
			ljmp	Timer0_ISR			; Software UART RX state machine interrupt

			org	1bh
			ljmp	Timer1_ISR			; Software UART TX state machine interrupt

			org	9bh
			ljmp	IE7_ISR				; user-level Software UART interrupt
			
;-----------------------------------------------------------------------------
; MAIN PROGRAM CODE
;-----------------------------------------------------------------------------

			org	0B3h

Main:
;			ajmp	PolledRX_PolledTX	; leave one of these lines uncommented

			ajmp	InterruptRX_InterruptTX	; leave one of these lines uncommented

			sjmp	$						; spin forever...

;-----------------------------------------------------------------------------
; MAIN SUBROUTINES
;-----------------------------------------------------------------------------

;-----------------------------------------------------------------------------
; PolledRX_PolledTX
;-----------------------------------------------------------------------------
; This routine demonstrates polled access to the SW UART.
;
; The transmitter transmits a sequence from $00 to $ff
;
; The receiver receives characters and stores them in a circular buffer.
;
PolledRX_PolledTX:
			acall	SW_UART_Init		; initialize SW UART (leave in a
											;  disabled state)

			setb	SREN					; enable SW UART receiver
			clr	SES					; disable user-level interrupt
											;  support
			acall	SW_UART_Enable		; enable SW UART

			; transmit message -- polled mode
			jb		STXBSY, $			; wait for SW TX available

			; transmit chars $00 to $ff
			clr	a
TX_LOOP:	setb  STXBSY            ; Claim SW UART Transmitter
         mov	TDR, a				; write char to transmit data reg
			setb	ET1					; initiate SW TX operation
			inc	a						; set next value to write
			jnb	STI, $				; wait for TX complete
			clr	STI					; clear TX complete indicator
			jnz	TX_LOOP
TX_LOOP_END:

			mov	RX_TAIL, #RX_BUF	; init TAIL pointer

			; receive message -- polled mode
RX_LOOP:	mov	r0, RX_TAIL			; indirect address to write character to
			jnb	SRI, $				; wait for RX character
			clr	SRI					; clear RX complete indicator
			mov	@r0, RDR				; store it
			inc	RX_TAIL				; advance TAIL pointer
			mov	a, RX_TAIL			; handle TAIL wrapping
			add	a, #-(RX_BUF + RX_BUFSIZE)
			jnc	RX_LOOP
			mov	RX_TAIL, #RX_BUF	; wrap TAIL

			sjmp	RX_LOOP				; repeat forever...

;-----------------------------------------------------------------------------
; InterruptRX_InterruptTX
;-----------------------------------------------------------------------------
; This routine demonstrates interrupt access to the SW UART.
;
; The receiver receives characters and stores them in a circular buffer.
; Both the transmit and receive routines are located in the IE7_ISR handler.
;
InterruptRX_InterruptTX:

			acall	SW_UART_Init		; initialize SW UART (leave in a
											; disabled state)
			setb	SES					; Enable user-level interrupt support
			setb	SREN					; Enable SW UART receiver
			
			mov	RX_TAIL, #RX_BUF	; init TAIL pointer

			acall	SW_UART_Enable		; enable SW UART

			setb	STI					; kick-start SW UART transmitter
			orl	EIE2, #00100000b	; by enabling IE7
			orl	PRT1IF, #10000000b; and activating IE7

			sjmp	$

;-----------------------------------------------------------------------------
; INTERRUPT VECTORS
;-----------------------------------------------------------------------------
;-----------------------------------------------------------------------------
; Reset Interrupt Vector
;
; This routine initializes the device and all peripherals and variables.
;   - External oscillator started (sysclk will be switched to external osc. 
;     once XTLVLD goes high)
;   - Watchdog timer is disabled
;   - Crossbar and GPIO output modes are defined
;   - H/W stack pointer is initialized
;   - interrupt priorities and enables are initialized here
;     - /INT0
;     - Timer0
;     - Timer1

Reset:
         mov	OSCXCN, #01100110b; Enable Crystal osc., divide by 1 mode
											;  XFCN = '110' for 18.432 MHz crystal
											;  External oscillator will be selected
											;  below after XTLVLD has gone high
											;  indicating that the external osc
											;  has started and settled (several
											;  hundred microseconds from now).

			mov	WDTCN, #0deh		; disable watchdog timer
			mov	WDTCN, #0adh

			; set up Crossbar and Port I/O
			mov	XBR0, #00000100b	; enable HW UART on P0.0 (TX), P0.1 (RX)
			mov	XBR1, #10000100b	; enable /INT0 on P0.2; /SYSCLK on P0.3
			mov	XBR2, #01000000b	; enable crossbar w/ pull-ups enabled
			orl	PRT0CF, #00011101b; enable P0.0, 0.2, 0.3, and 0.4 as push-pull
											; P0.4 is SW UART TX pin
											; P0.2 is SW UART RX pin
			orl	PRT1CF, #01000000b; enable P1.6 (LED on target board) as 
											;  push-pull

			mov	SP, #STACK_TOP		; init stack pointer to end of allocated RAM

		; Wait >1 ms before checking external crystal for stability
		   clr 	a
		   mov	r0, a		          ; Clear r0

		   djnz	r0, $		          ; Delay ~380 s
		   djnz	r0, $	             ; Delay ~380 s
		   djnz	r0, $			       ; Delay ~380 s				
											

OSC_WAIT:
			mov	a, OSCXCN			; spin here until crystal osc is stable
			jnb	acc.7, OSC_WAIT

			orl	OSCICN, #00001000b; Select external osc. as
											;  system clock source
			anl	OSCICN, #NOT(00000100b); Disable internal oscillator
			orl	OSCICN, #10000000b; Enable missing clock detector
											;  this must be done AFTER
											;  selecting ext osc as system
											;  clock source

			setb	EA						; enable GLOBAL interrupts

			ljmp 	Main

;-----------------------------------------------------------------------------
; Timer0_ISR / INT0_ISR
;
; These interrupts start and drive the SW UART receive state machine
;
SWRX_STATE_TABLE:							; each table entry is 1 byte
	DB		SWRX_S0 - SWRX_STATE_TABLE	; IDLE / START detect
	DB		SWRX_S1 - SWRX_STATE_TABLE	; b0
	DB		SWRX_S2 - SWRX_STATE_TABLE	; b1
	DB		SWRX_S3 - SWRX_STATE_TABLE	; b2
	DB		SWRX_S4 - SWRX_STATE_TABLE	; b3
	DB		SWRX_S5 - SWRX_STATE_TABLE	; b4
	DB		SWRX_S6 - SWRX_STATE_TABLE	; b5
	DB		SWRX_S7 - SWRX_STATE_TABLE	; b6
	DB		SWRX_S8 - SWRX_STATE_TABLE	; b7
	DB		SWRX_S9 - SWRX_STATE_TABLE	; STOP bit capture

INT0_ISR:
Timer0_ISR:
			push	PSW					; resource preservation
			push	acc

			mov	a, BCRHI				; if BCRHI is non-zero, we need to roll 
											;  through the timer again...
			jz		SWRX_PROCESS_STATE
			dec	BCRHI
			ajmp	Timer0_ISR_EXIT

SWRX_PROCESS_STATE:
			push	DPH					; resource preservation
			push	DPL

			mov	a, SURXST			; read state offset from table
			mov	DPTR, #SWRX_STATE_TABLE
			movc	a, @A+DPTR			; 'a' now contains state offset (PC)
			jmp	@A+DPTR				; execute state

Timer0_ISR_END:						; ALL RX states return here

			pop	DPL					; resource restoration
			pop	DPH
Timer0_ISR_EXIT:
			pop	acc					; resource restoration
			pop	PSW
			reti

;SWRX_S0: RX IDLE state
; At this point, a falling edge has been detected on /INT0.  
; We first check to see if the SW UART receiver is enabled.  If it is, we check
; once to see if the RX pin is still low (START bit valid).  If it is, we set up 
; Timer0 to count for 3/2 bit time in order to capture the LSB.  Here, we also 
; disable /INT0 interrupts.
;	- Check for SREN = '1':  IF '1':
;		- Load TL0 with 3/2 bit time value
;		- Start Timer
;		- Enable TF0 interrupt
;		- Disable /INT0 interrupt
;		- INC state variable to S1
;	- IF SREN = '0' (SW UART RX disabled)
;		- exit gracefully, next state is S0
;
SWRX_S0:
			jnb	SREN, SWRX_S0_END	; Check to see if SW UART RX is enabled
											;  if not, exit and remain at IDLE state

			jb		SW_RX_GPIO, SWRX_S0_END	; check to see if START bit is good

			clr	EX0					; disable /INT0

			clr	TR0					; Stop Timer0 (low)
			clr	TF0					; Clear any pending interrupts

			mov	BCRHI, #HIGH(THALF_BT)	; set Timer0 (low) + BCRHI for 1.5 bit 
			mov	TL0, #-LOW(THALF_BT)	;  times from now (we assume the start
											;  bit is good)

			setb	ET0					; enable Timer0 interrupts
			setb	TR0					; Start Timer0L

			inc	SURXST				; next state is SWRX_S1 (we assume START bit 
											;  is good)

SWRX_S0_END:
			ajmp	Timer0_ISR_END

;SWRX_S1 thru SWRX_S8: Capture b0..b7
; At this point, we've determined that the START bit is valid, and we're going to 
; query RX_GPIO at bit intervals, shifting the results into RXSHIFT.
;	- If BCRHI is non-zero, then we need to spin through the timer again
;		- DEC BCRHI
;		- let timer roll over on its own
;		- leave state as is
;	- If BCRHI is zero:
;		- stop timer
;		- Move RX_GPIO state into Carry
;		- Right shift Carry into RXSHIFT 
;		- set up timer to capture the next bit
;		- enable timer
;		- advance state variable
;
SWRX_S1:
SWRX_S2:
SWRX_S3:
SWRX_S4:
SWRX_S5:
SWRX_S6:
SWRX_S7:
SWRX_S8:
			clr	TR0					; Stop Timer0 (low)
			clr	TF0					; Clear any pending interrupts

			mov	BCRHI, #HIGH(RX_BT); load bit time value into 16-bit virtual 
											;  counter
			mov	TL0, #-LOW(RX_BT)

			setb	TR0					; START RX bit timer

			mov	C, SW_RX_GPIO		; Move RX state into Carry prior to rshift

			mov	a, RXSHIFT
			rrc	a						; right shift Carry into shift register
			mov	RXSHIFT, a			; re-store

			inc	SURXST				; advance state variable

SWRX_S2_END:
			ajmp	Timer0_ISR_END

;SWRX_S9: Capture STOP bit
; At this point, we've shifted all the data bits into RXSHIFT, and we're ready to 
; sample the STOP bit. Here, we indicate that we've received a character, and reset 
; the state machine back to IDLE. In this implementation, we don't actually capture 
; the STOP bit; we assume it's good.  Here's where we would add support for Framing 
; Error detection.
;	- If BCRHI is non-zero, then we need to spin through the timer again
;		- DEC BCRHI
;		- let timer roll over on its own
;		- leave state as is
;	- If BCRHI is zero:
;		- stop timer
;		- Move RXSHIFT into RDR
;		- Set SRI
;		- Disable timer interrupt
;		- Enable /INT0 interrupt
;		- Reset state variable to IDLE
;		- Check to see if User-level interrupt support is enabled (EIS): If so:
;			- Enable IE7
;			- Toggle P1.7 to activate IE7
;
SWRX_S9:
			clr	TR0					; Stop Timer0L
			mov	RDR, RXSHIFT		; move data from shift reg to data reg
			
			setb	SRI					; set SW UART SRI bit to indicate RX complete
			clr	ET0					; Disable Timer0L interrupt
			clr	IE0					; Disable pending /INT0 interrupts
			setb	EX0					; Enable /INT0 interrupt
			mov	SURXST, #00			; reset RX state to IDLE

			jnb	SES, SWRX_S9_END	; check to see if user-level interrupt 
											;  support is enabled
			orl	EIE2, #00100000b	; enable IE7; leave priority alone
			orl	PRT1IF, #10000000b; activate IE7
SWRX_S9_END:
			ajmp	Timer0_ISR_END

;-----------------------------------------------------------------------------
; Timer1_ISR (note that this is actually called by the upper-half of Timer0
;  which is operating in Mode 3)
;
; This interrupt drives the SW UART transmit state machine
;
SWTX_STATE_TABLE:						; each table entry is 1 byte; 11 entries
											;  total
	DB		SWTX_S0  - SWTX_STATE_TABLE	; START bit
	DB		SWTX_S1  - SWTX_STATE_TABLE	; b0
	DB		SWTX_S2  - SWTX_STATE_TABLE	; b1
	DB		SWTX_S3  - SWTX_STATE_TABLE	; b2
	DB		SWTX_S4  - SWTX_STATE_TABLE	; b3
	DB		SWTX_S5  - SWTX_STATE_TABLE	; b4
	DB		SWTX_S6  - SWTX_STATE_TABLE	; b5
	DB		SWTX_S7  - SWTX_STATE_TABLE	; b6
	DB		SWTX_S8  - SWTX_STATE_TABLE	; b7
	DB		SWTX_S9  - SWTX_STATE_TABLE	; STOP bit onset edge
	DB		SWTX_S10 - SWTX_STATE_TABLE	; STOP bit terminus

Timer1_ISR:
			push	PSW					; resource preservation
			push	acc

			mov	a, BCTHI				; if BCTHI is non-zero, we need to roll 
											;  through the timer again...
			jz		SWTX_PROCESS_STATE
			dec	BCTHI
			ajmp	Timer1_ISR_EXIT

SWTX_PROCESS_STATE:
			push	DPH					; resource preservation
			push	DPL

			mov	a, SUTXST			; read state offset from table
			mov	DPTR, #SWTX_STATE_TABLE
			movc	a, @A+DPTR			; acc now contains state offset
			jmp	@A+DPTR				; execute State x

Timer1_ISR_END:						; ALL TX states return here

			pop	DPL					; resource restoration
			pop	DPH
Timer1_ISR_EXIT:
			pop	acc					; resource restoration
			pop	PSW

			reti

;SWTX_S0: TX START bit state
; At this point, user code has placed the char to be transmitted in TDR and has 
; called the Timer1 interrupt handler explicitly by setting TF1.
;	- Clear STI
;	- Drop TX_GPIO (START bit onset edge)
;	- Configure TH0, BCTHI for next bit time, which will be the LSB
;	- Enable TH0
;	- Set next state to SWTX_S1
;
SWTX_S0:
			mov	BCTHI, #HIGH(TX_BT); load bit time value into 16-bit virtual 
											;  counter
			mov	TH0, #-LOW(TX_BT)
			clr	SW_TX_GPIO			; START bit onset edge
			clr	TF1					; clear any pending interrupts

			inc	SUTXST				; next state is SWTX_S1
SWTX_S0_END:
			ajmp	Timer1_ISR_END

;SWTX_S1 thru SWTX_S9: TX b0..b7 and STOP bit
; At this point, we start shifting the character in TDR out the TX_GPIO pin, bit 
; by bit, one bit per state transition.  We shift in an extra '1' at the MSB which 
; becomes the STOP bit.
;	- If BCTHI is non-zero, then we need to spin through the timer again
;		- DEC BCTHI
;		- let timer roll over on its own
;		- leave state as is
;	- If BCTHI is zero:
;		- stop timer
;		- set up timer for next bit
;		- right-shift TDR
;		- enable timer
;		- output bit
;		- advance state variable
;
SWTX_S1:
SWTX_S2:
SWTX_S3:
SWTX_S4:
SWTX_S5:
SWTX_S6:
SWTX_S7:
SWTX_S8:
SWTX_S9:
			mov	BCTHI, #HIGH(TX_BT); load bit time value into 16-bit virtual 
											;  counter
			mov	TH0, #-LOW(TX_BT)

			mov	a, TDR				; right shift next bit to transmit into Carry
			setb	C						; shift STOP bit into MSB
			rrc	a
			mov	TDR, a				; re-store value
			mov	SW_TX_GPIO, C		; output bit on GPIO pin
			clr	TF1					; clear any pending interrupts

			inc	SUTXST				; advance to next state

SWTX_S1_END:
			ajmp	Timer1_ISR_END

;SWTX_S10 STOP bit complete / reset to IDLE
; At this point, we've shifted the STOP bit out, and we're ready to reset the state 
; machine and indicate transmit complete, including initiating a user-level interrupt
; if it's enabled.
;	- If BCTHI is non-zero, then we need to spin through the timer again
;		- DEC BCTHI
;		- let timer roll over on its own
;		- leave state as is
;	- If BCTHI is zero:
;		- stop timer
;		- set STI
;		- clear STXBSY
;		- check for IE7 support, and activate if enabled
;		- set state variable to S0
;
SWTX_S10:
			clr	ET1					; Disable Timer1 interrupts
			setb	TF1					; Force a pending Timer1 interrupt.  This 
											;  allows the Enable Timer1 interrupt 
											;  operation to immediately trigger a 
											;  transmit operation

			mov	SUTXST, #00h		; reset state variable to IDLE state
			setb	STI					; Set STI to indicate transmit complete
			clr	STXBSY				; Clear TXBSY to indicate transmitter 
											;  available
			jnb	SES, SWTX_S10_END	; activate user-level interrupt IE7 if 
											;  enabled
			orl	EIE2, #00100000b	; enable IE7; leave priority alone
			orl	PRT1IF, #10000000b; activate IE7
SWTX_S10_END:
			ajmp	Timer1_ISR_END

;-----------------------------------------------------------------------------
; IE7_ISR
;
; This is the user-level interrupt handler for the SW UART.  Note:  this code
; MUST check both SRI and TRI, and if both are set, it must handle one case, and
; retrigger IE7 for the other case (or handle it in the same call) if that case
; is interrupt handled.  This is not required, for example, if the RX case is 
; handled in the interrupt and the TX case is polled.
;
; Note, if the TX case is polled, STI should not be cleared here.
;
; In this example, if SRI is set, indicating that a character was received by
; the SW UART, that received character is stored in a circular buffer (RX_BUF).
; If STI is set, indicating transmit complete, the character stored in TX_VAL
; is transmitted (and post incremented).
; 
;
IE7_ISR:
			push	PSW
			push	acc

			anl	PRT1IF, #NOT(10000000b); clear IE7
			jbc	SRI, SW_RX_HANDLE	; handle receive first, since 
											;  it's the most sensitive to 
											;  latency
			jbc	STI, SW_TX_HANDLE	; handle TX case
IE7_ISR_END:
			pop	acc
			pop	PSW
			reti							; all IE7_ISR routines return here...

SW_RX_HANDLE:
			push	ar0					; resource preservation

			mov	r0, RX_TAIL			; point r0 to location to store
			mov	@r0, RDR				; read value into buffer
			inc	RX_TAIL				; update the TAIL pointer
			mov	a, RX_TAIL			; wrap pointer if necessary
			add	a, #-(RX_BUF+RX_BUFSIZE)
			jnc	SW_RX_HANDLE_END
			mov	RX_TAIL, #RX_BUF	; wrap the pointer

SW_RX_HANDLE_END:
			jnb	STI, NO_TX_PENDING; if TX interrupt is pending,
			orl	PRT1IF, #10000000b;  activate it (IE7)

NO_TX_PENDING:
			pop	ar0
			ajmp	IE7_ISR_END

SW_TX_HANDLE:
         setb  STXBSY            ; Claim SW UART Transmitter
			mov	TDR, TX_VAL			; load byte to trasmit into TDR
			setb	ET1					; start SW UART transmitter
			inc	TX_VAL				; next byte to store

SW_TX_HANDLE_END:
			jnb	SRI, NO_RX_PENDING; if RX interrupt is pending,
			orl	PRT1IF, #10000000b;  activate it (IE7)
NO_RX_PENDING:
			ajmp	IE7_ISR_END			; exit

;-----------------------------------------------------------------------------
; SUBROUTINES
;-----------------------------------------------------------------------------

;-----------------------------------------------------------------------------
; SW UART SUBROUTINES (non-user code)
;-----------------------------------------------------------------------------

;-----------------------------------------------------------------------------
; SW_UART_Init
;
; Init:
;  - /INT0 is falling-edge triggered
;  - Timer0 in Mode 3, (2) 8-bit timers, interrupt handlers for TL0, TH0 (TF0, TF1)
;    timers initially disabled...
;  - RX/TX State machines and state variables
;  - SW UART TX state machine and RX state machine operate at HIGH priority
SW_UART_Init:
			; Init /INT0
			clr	EX0					; disable /INT0 interrupts
			setb	IT0					; /INT0 is falling-edge triggered
			clr	IE0					; forcibly clear /INT0 interrupt flag
			setb	PX0					; /INT0 is HIGH priority interrupt

			; Init Timer0
			clr	ET0					; disable Timer0 interrupts
			clr	ET1					; disable Timer1 interrupts
			clr	TR0					; Timer0 off
			clr	TR1					; Timer1 off
			clr	TF0					; forcibly clear interrupt flags
			clr	TF1
			orl	TMOD, #00000011b	; Timer0 in Mode 3 (2) 8-bit timers
			anl	TMOD, #NOT(00001100b); GATE0=0; C/T0 = 0
			orl	CKCON, #00001000b	; Timer0 uses system clock as time base
			setb	PT0					; Timer0 interrupt is HIGH priority
			setb	PT1					; Timer1 interrupt is HIGH priority

			; User-level interrupt (IE7) is initialized explicitly by the state
			;  machines
	
			; Init State Machines and Variables
			clr	a						; Init state machines
			mov	SURXST, a			;  RX state variable
			mov	SUTXST, a			;  TX state variable
			mov	BCRHI, a				;  RX bit timer MSB
			mov	BCTHI, a				;  TX bit timer MSB
			clr	SES					; Disable user-level interrupt support
			clr	SREN					; Disable SW UART receiver
			clr	TXBSY					; clear TXBSY indicator
			clr	SRI					; clear RX complete indicator
			clr	STI					; clear TX complete indicator

			ret

;-----------------------------------------------------------------------------
; SW_UART_Enable
;
; The SW_UART is enabled by enabling the interrupt handlers that move the transmit
; and receive state machines from their IDLE states to their corresponding next
; states.  /INT0 transitions the RX state machine from IDLE to START.  Timer1,
; which is called explicitly by the user code (setb TF1), transitions the
; transmit state machine from IDLE/START to TX_LSB.
;
; The user-level interrupt (IE7) is enabled in the state machines themselves
; after polling EIS (external interrupt support).
;
SW_UART_Enable:
			clr	IE0					; clear pending /INT0 interrupts
			setb	TF1					; Force a pending Timer1 interrupt
			setb	EX0					; enable /INT0 interrupts
			clr	ET1					; keep Timer1 interrupts disabled
			setb	TR1					; Enable Timer1

			ret
;-----------------------------------------------------------------------------
; SW_UART_Disable
;
; The SW UART is disabled by disabling all of its state machine interrupts, 
; including the user-level interrupt (IE7), if the status register indicates that
; it's enabled.
SW_UART_Disable:
			clr	EX0					; disable /INT0 interrupts
			clr	ET0					; disable Timer0 interrupts
			clr	ET1					; disable Timer1 interrupts
			jnb	SES, SW_UART_Dis_End; check to see if IE7 use is enabled
			anl	EIE2, #NOT(00100000b); disable IE7 interrupts
SW_UART_Dis_End:
			ret

;-----------------------------------------------------------------------------
; End of file.

END