//------------------------------------------------------------------------------------
//
// Copyright 2001	Cygnal Integrated Products, Inc.
//
// FILE NAME		: AN015_1.c
// TARGET DEVICE	: C8051F00x, C8051F01x
// CREATED ON		: 03/10/01
// CREATED BY		: JS
//
// Software UART program, using PCA as baud rate source.
// PCA module 0 is used as receive baud rate source and START detector.  For START
// detection, module 0 is configured in negative-edge capture mode.  For all other
// SW_UART operations, module 0 is configured as a software timer.  Module match
// interrupts are used to generate the baud rate.  Module 1 generates the transmit
// baud rate in software timer mode. 
// Code assumes an external crystal is attached between the XTAL1 and XTAL2 pins.
// The frequency of the external crystal should be defined in the SYSCLK constant.
// 
// INITIALIZATION PROCEDURE:
// 1) Define SYSCLK according to external crystal frequency.
// 2) Define desired BAUD_RATE.
// 3) Call SW_UART_INIT().
// 4) Set SREN to enable SW_UART receiver.
// 5) Set SES only if user-level interrupt support is desired.
// 6) Call SW_UART_ENABLE().
//
// TO TRANSMIT:
// 1) Poll STXBSY for zero.
// 2) Write data to TDR.
// 3) Set CCF1 to initiate transmit.
// 4) STI will be set upon transmit completion.  An IE7 interrupt is generated if 
//    user-level interrupts are enabled.
//
// TO RECEIVE:
// 1) If in polled mode, poll SRI.  If in interrupt mode, check SRI in IE7 Interrupt
//    Service Routine.
// 2) Read data from RDR.
// 
// Test code is included, for both polled and interrupt mode.  Test code assumes
// the HW_UART pins and SW_UART pins are connected externally:
// P0.0 (HW_UART TX) ->  P0.3 (SW_UART RX)
// P0.1 (HW_UART RX) ->  P0.2 (SW_UART TX)
// 
// To use the test code in polled mode, comment out the call to the INTERRUPT_TEST()
// at the bottom of the main routine, and uncomment the call to POLLED_TEST().  To 
// test the interrupt mode, comment out the POLLED_TEST() call and uncomment the
// INTERRUPT_TEST() call.
//
// The test routines configure the HW_UART to operate with Timer 1 as the baud rate
// source.  The Timer 1 preload values are auto-calculated from the SYSCLK and BAUD_RATE
// constants.
//
//-----------------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------------
#include <c8051f000.h>								// SFR declarations

//-----------------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------------

#define	BAUD_RATE		57600						// User-definable SW_UART baud rate
#define	SYSCLK			18432000					// System clock derived from 18.432MHz XTL

#define 	TIME_COUNT		SYSCLK/BAUD_RATE/4	// Number of PCA counts for one bit-time.
															// (PCA configured to count SYSCLK/4)

#define	TH_TIME_COUNT	TIME_COUNT*3/2			// 3/2 bit-time, for use after receiving
															// a START bit.  RX should be LOW for one
															// bit-time after the edge of the START,
															// and the first bit sample starts in the
															// middle of the next bit-time.

#define	HW_TIME_COUNT	SYSCLK/BAUD_RATE/16 	// Time count for HW_UART baud rate
															// generation.  Auto-calculated from the
															// SYSCLK and BAUD_RATE constants defined
															// above.

//-----------------------------------------------------------------------------------
//Global VARIABLES
//-----------------------------------------------------------------------------------

bit	SRI;												// SW_UART Receive Complete Indicator
bit	STI;												// SW_UART Transmit Complete Indicator
bit	STXBSY;											// SW_UART TX Busy flag
bit 	SREN;												// SW_UART RX Enable
bit	SES;												// SW_UART User-level Interrupt Support Enable

sbit	SW_RX = P0^2;  								// SW_UART Receive pin
sbit	SW_TX = P0^3;  								// SW_UART Transmit pin

char 	TDR;												// SW_UART TX Data Register
char 	RDR;												// SW_UART RX Data Register (latch)

// Test Variables
char k, m;												// Test indices.
char idata SW_BUF[20];								// SW_UART test receive buffer.

bit	HW_DONE;											// HW transfer complete flag (15 characters
															// transmitted.)
bit	SW_DONE;											// SW transfer complete flag (15 characters
															// transmitted.)

//------------------------------------------------------------------------------------
// Function PROTOTYPES
//------------------------------------------------------------------------------------

void	SW_UART_INIT();								// SW_UART initialization routine
void	SW_UART_ENABLE();								// SW_UART enable routine
void	PCA_ISR();										// SW_UART interrupt service routine
void 	INTERRUPT_TEST(void);						// SW_UART test routine using interrupt mode
void 	POLLED_TEST(void);							// SW_UART test routine using polled mode
void 	USER_ISR(void);								// SW_UART test interrupt service routine
void 	HWU_INIT(void);								// HW_UART initialization and setup routine
void	HW_UART_ISR(void);							// HW_UART interrupt service routine						
//------------------------------------------------------------------------------------
// MAIN Routine
//------------------------------------------------------------------------------------
// - Disables Watchdog Timer
// - Configures external crystal; switches SYSCLK to external crystal when stable.
// - Configures crossbar and ports.
// - Initializes and enables SW_UART.
// - Calls Test Routines.
//
void MAIN (void){
	int delay;											// Delay counter.

	OSCXCN = 0x66;										// Enable external crystal	
	WDTCN = 0xDE;										// disable watchdog timer
	WDTCN = 0xAD;

	// Port Setup
	XBR0 = 0x0C;										// HW_UART routed to pins P0.0 and P0.1;
															// CEX0 routed to pin P0.2.
	XBR2 = 0x40; 										// Enable crossbar, pull-ups enabled.

	PRT0CF = 0x09;										// P0.0 (HW TX), and P0.3 (SW TX)
															// configured for push-pull output.

	delay=256;											// Delay >1 ms  before polling XTLVLD.
	while(delay--);

	while (!(OSCXCN & 0x80));						// Wait until external crystal has
														   // started.
	OSCICN = 0x0C;										// Switch to external oscillator
	OSCICN = 0x88;										// Disable internal oscillator; enable 
														   // missing clock detector.

	while (!(OSCXCN & 0x80));						// Wait until external crystal has
															// started.
	OSCICN = 0x08;										// Switch to external oscillator

//	POLLED_TEST();										// Call Polled mode SW_UART test routine.
	INTERRUPT_TEST();									// Call Interrupt mode SW_UART test routine.

	while(1);											// Spin forever
}


//------------------------------------------------------------------------------------
// Functions
//------------------------------------------------------------------------------------

//------------------------------------------------------------------------------------
// INTERRUPT_TEST: SW_UART Interrupt Mode Test
// Test code to transmit and receive 15 characters to/from the HW_UART (in interrupt
// mode), with SW_UART in interrupt mode.
// - Initializes and enables the SW_UART & HW_UART
// - Clears all test variables & counters
// - Transfers 15 characters from HW to SW_UART, and 15 characters from SW to HW_UART,
//   simultaneously. 
//
void INTERRUPT_TEST(void){

	SW_UART_INIT();									// Initialize SW_UART
	SW_UART_ENABLE();									// Enable SW_UART
	SREN = 1;											// Enable SW_UART Receiver
	SES = 1;												// User-level interrupt support enabled.

	HWU_INIT();											// Configure HW_UART for testing routine.
	k=m=0;												// Clear user ISR counters.
	
	HW_DONE=0;											// Clear transfer complete indicators
	SW_DONE=0;											// 

	IE |= 0x10;											// Enable HW_UART interrupts
	STI=1;												// Indicate transmit complete to initiate
															// next transfer.
	EIE2 |= 0x20;										// Start SW_TX by enabling
	PRT1IF |= 0x80;									// and forcing an IE7 interrupt
	
   TI = 1;                                   // Initiate a HW_UART transmit
                                             // by forcing TX interrupt.

	while(!(HW_DONE&SW_DONE));						// Wait for transfers to finish.
}

//---------------------------------------------------------------------------------------
// POLLED_TEST: SW_UART Polled Mode Test
// Test code to transmit and receive 15 characters to/from the HW_UART, with SW_UART
// in polled mode.
// - Initializes and enables the SW_UART & HW_UART
// - Clears all test variables & counters
// - Sends 15 characters from the HW_UART to be received by SW_UART.
// - Sends 15 characters from the SW_UART to be received by the HW_UART.
//
void POLLED_TEST(void){

	SW_UART_INIT();									// Initialize SW_UART
	SW_UART_ENABLE();									// Enable SW_UART
	SREN = 1;											// Enable SW_UART Receiver
	SES=0;												// Disable user-level interrupt support.

	HWU_INIT();											// Configure HW_UART for testing routine.
	k=m=0;												// Clear test counter variables.
	HW_DONE=0;											// Clear transfer complete indicators
	SW_DONE=0;											//
	IE |= 0x10;											// Enable HW_UART interrupts.

   TI = 1;                                   // Initiate a HW_UART transmit
                                             // by forcing TX interrupt.

	// Receive 15 characters with SW_UART; transmit with HW_UART.
	while(SREN){                              // Run while SW_UART Receiver is
                                             // enabled.	
		if (SRI){										// If Receive Complete:
			SRI = 0;										// Clear receive flag.
			SW_BUF[k++] = RDR;						// Read receive buffer.
			if (k==15){									// If 15 characters have been received:
				SREN = 0;								// Disable SW_UART Receiver.
			}												// Indicate 15 characters received.
		}
	}

	// Transmit 15 characters with SW_UART; receive with HW_UART.
	while(STXBSY);										// Poll Busy flag.
   STXBSY = 1;                               // Claim SW_UART Transmitter
	TDR=m++;												// Load TX data.
	CCF1=1;												// Initiate first SW_UART TX
															// by forcing a PCA module 1 interrupt.
	while(!SW_DONE){									// SW_UART transmitting here; HW_UART receiving.

		if (STI){										// If Transmit Complete:
			STI = 0;										// Clear transmit flag.
			if (m<16){									// Transmit 15 characters.
				STXBSY = 1;                      // Claim SW_UART Transmitter.
            TDR = m++;								// Transmit, increment variable.
				CCF1 = 1;								// Force module 1 interrupt to initiate TX.
			}
         else											// If this is 15th character,
				SW_DONE=1;								// Indicate last character transmitted.
		}
	}
}

//------------------------------------------------------------------------------------------
// HWU_INIT: HW_UART Initialization Routine
// Sets up HW_UART for use in SW_UART testing.
// - HW_UART in mode 1
// - Timer 1 used as baud rate source, clocked by SYSCLK.
//
void HWU_INIT(void) {

	PCON |= 0x80;										// SMOD=1 (HW_UART uses Timer 1 overflow
															// with no divide down).

	TMOD = 0x20;										// Configure Timer 1 for use by HW_UART
	CKCON |= 0x10;										// Timer 1 derived from SYSCLK
	TH1 = -HW_TIME_COUNT;							// Timer 1 initial value
	TL1 = -HW_TIME_COUNT;							// Timer 1 reload value

	TR1 = 1;    										// Start Timer 1

	RI=0;													// Clear HW_UART receive and transmit
	TI=0;													// complete indicators.

	SCON = 0x50;										// Configure HW_UART for mode 1, receiver enabled.
}
//------------------------------------------------------------------------------------------
// SW_UART_INIT: SW_UART initialization routine
// Prepares SW_UART for use.
// - Configures PCA: Module 0 in negative capture mode; module 1 in software 
//   	timer mode; PCA time base = SYSCLK/4; PCA interrupt disabled; PCA counter
//		disabled.
// - Clears pending PCA module 0 and 1 interrupts
// - Resets TX and RX state variables
//
void	SW_UART_INIT(void){

	PCA0CPM0 = 0x10;									// Module 0 in negative capture mode; module
															// 0 interrupt disabled.

	PCA0CPM1 = 0x48; 									// Module 1 in software timer mode; module
															// 1 interrupt disabled.	

	PCA0CN = 0;											// Leave PCA disabled
	PCA0MD = 0x02;										// PCA timebase = SYSCLK/4; PCA counter
															// interrupt disabled.

	CCF0 = 0;											// Clear pending PCA module 0 and
	CCF1 = 0;											// module 1 capture/compare interrupts.

	SRI = 0;												// Clear Receive complete flag.
	STI = 0;												// Clear Transmit complete flag.

   SW_TX = 1;											// TX line initially high.	
	STXBSY = 0;											// Clear SW_UART Busy flag

}

//------------------------------------------------------------------------------------------
// SW_UART_ENABLE: SW_UART Enable Routine
// Enables SW_UART for use.
// - Enables PCA module 0 interrupts
// - Enables PCA module 1 interrupts
// - Starts PCA counter.
//
void	SW_UART_ENABLE(void){
	
	PCA0CPM0 |= 0x01;									// Enable module 0 (receive) interrupts.
	PCA0CPM1 |= 0x01;									// Enable module 1 (transmit) interrupts.

	CR = 1;												// Start PCA counter.
	EIE1 |= 0x08;										// Enable PCA interrupts
	EA = 1;												// Globally enable interrupts

}

//------------------------------------------------------------------------------------
// Interrupt Service Routines
//------------------------------------------------------------------------------------
//
// PCA_ISR: PCA Interrupt Service Routine.
// This ISR is triggered by both transmit and receive functions, for each bit that
// is transmitted or received. 
// - Checks module 0 interrupt flag (CCF0); if set, services receive state.
// - Checks module 1 interrupt flag (CCF1); if set, services transmit state.
//
void	PCA_ISR(void) interrupt 9 {

	static char 	SUTXST = 0;						// SW_UART TX state variable
	static char 	SURXST = 0;						// SW_UART RX state variable
	static unsigned char 	RXSHIFT;				// SW_UART RX Shift Register
	
	unsigned int	PCA_TEMP;						// Temporary storage variable for
															// manipulating PCA module high & low bytes.	

	// Check receive interrupt flag first; service if CCF0 is set.
	if (CCF0){
		CCF0 = 0;										// Clear interrupt flag.
		switch (SURXST){
			
			// State 0: START bit received.
			// In this state, a negative edge on SW_TX has caused the interrupt,
			// meaning a START has been detected and the PCA0CP0 registers have 
			// captured the value of PCA0.
			// - Check for receive enable and good START bit
			// - Switch PCA module 0 to software timer mode
			// - Add 3/2 bit time to module 0 capture registers to sample LSB.
			// - Increment RX state variable.
			case 0:
				if (SREN & ~SW_RX){					// Check for receive enable and a good
															// START bit.  
																
					PCA_TEMP = (PCA0CPH0 << 8);	// Read module 0 contents into
					PCA_TEMP |= PCA0CPL0;			// PCA_TEMP.

					PCA_TEMP += TH_TIME_COUNT;		// Add 3/2 bit times to PCA_TEMP

					PCA0CPL0 = PCA_TEMP;				// Restore PCA0CPL0 and PCA0CPH0
					PCA0CPH0 = (PCA_TEMP >> 8);	// with the updated value

					PCA0CPM0 = 0x49;					// Change module 0 to software
															// timer mode, interrupts enabled.

					SURXST++;							// Update RX state variable.
				}
				break;
			
			// States 1-8: Bit Received
			// - Sample SW_RX pin
			// - Shift new bit into RXSHIFT
			// - Add 1 bit time to module 0 capture registers
			// - Increment RX state variable
			case 1:
			case 2:
			case 3:
			case 4:
			case 5:
			case 6:
			case 7:
			case 8:
					
				RXSHIFT = RXSHIFT >> 1;				// Shift right 1 bit
				if (SW_RX)								// If SW_RX=1, 
					RXSHIFT |= 0x80;					// shift '1' into RXSHIFT msb
				
				PCA_TEMP = (PCA0CPH0 << 8);		// Read module 0 contents into
				PCA_TEMP |= PCA0CPL0;				// PCA_TEMP.

				PCA_TEMP += TIME_COUNT;				// Add 1 bit time to PCA_TEMP

				PCA0CPL0 = PCA_TEMP;					// Restore PCA0CPL0 and PCA0CPH0
				PCA0CPH0 = (PCA_TEMP >> 8);		// with the updated value
				
				SURXST++;								// Update RX state variable.
				break;

			// State 9: 8-bits received, Capture STOP bit.
			// - Move RXSHIFT into RDR.
			// - Set SRI (indicate receive complete).
			// - Prepare module 0 for next transfer.
			// - Reset RX state variable.
			// - Trigger IE7 if user-level interrupt support is enabled.
			case 9:

				RDR = RXSHIFT;							// Move received data to receive register.
				SRI = 1;									// Set receive complete indicator.

				PCA0CPM0 = 0x11;						// Switch module 0 to negative capture
															// mode; interrupt enabled for START
															// detection.

				SURXST = 0;								// Reset RX state variable.

				if (SES){								// If user-level interrupt support enabled
					EIE2 |= 0x20;						// Enable IE7.
					PRT1IF |= 0x80;					// Trigger IE7.
				}
				break;
				
			}
		}
		
		// Check Transmit interrupt; service if CCF1 is set.
		else if (CCF1){ 
			CCF1 = 0;									// Clear interrupt flag
			switch (SUTXST){
				
				// State 0: Transmit Initiated.
				// Here, the user has loaded a byte to transmit into TDR, and set the
				// module 1 interrupt to initiate the transfer.
				// - Transmit START bit (drop SW_TX)
				// - Read PCA0, add one bit time, & store in module 1 capture registers
				//   for first bit.
				// - Increment TX state variable.
				case 0:

					SW_TX = 0;							// Drop TX pin as START bit.
					
					PCA_TEMP = PCA0L;					// Read PCA counter value into
					PCA_TEMP |= (PCA0H << 8);		// PCA_TEMP.

					PCA_TEMP += TIME_COUNT;			// Add 1 bit time.

					PCA0CPL1 = PCA_TEMP;				// Store updated match value into
					PCA0CPH1 = (PCA_TEMP >> 8);	// module 1 capture/compare registers.

					PCA0CPM1 |= 0x48;					// Enable module 1 software timer.

					SUTXST++;							// Update TX state variable.				
					break;

				// States 1-9: Transmit Bit.
				// - Output LSB of TDR onto TX
				// - Shift TDR 1 bit right.
				// - Shift a '1' into MSB of TDR for STOP bit in State 9.
				// - Add 1 bit time to module 1 capture register
				case 1:
				case 2:
				case 3:
				case 4:
				case 5:
				case 6:
				case 7:
				case 8:
				case 9:
					
					SW_TX = (TDR & 0x01);			// Output LSB of TDR onto SW_TX pin.
					TDR >>= 1;							// Shift TDR right 1 bit.
					TDR |= 0x80;						// Shift '1' into MSB of TDR for
															// STOP bit in State 9.

					PCA_TEMP = (PCA0CPH1 << 8);	// Read module 1 contents into
					PCA_TEMP |= PCA0CPL1;			// PCA_TEMP.

					PCA_TEMP += TIME_COUNT;			// Add 1 bit time to PCA_TEMP
	
					PCA0CPL1 = PCA_TEMP;				// Restore PCA0CPL1 and PCA0CPH1
					PCA0CPH1 = (PCA_TEMP >> 8);	// with the updated value	

					SUTXST++;                     // Update TX state variable.
					break;
					
				// State 10: Last bit has been transmitted.  Transmit STOP bit
				// and end transfer.  
				// - Transmit STOP bit
				// - Set TX Complete indicator, clear Busy flag
				// - Reset TX state
				// - Prepare module 1 for next transfer.
				// - Trigger IE7 interrupt if user-level interrupts enabled.
				case 10:

					STI = 1;								// Indicate TX complete.
					SUTXST = 0;							// Reset TX state.
					SW_TX = 1;							// SW_TX should remain high.

					PCA0CPM1 = 0x01;					// Disable module 1 software timer; leave
															// interrupt enabled for next transmit.					
					
					if (SES){							// If user-level interrupt support enabled:
						EIE2 |= 0x20;					// Enable IE7.
						PRT1IF |= 0x80;				// Trigger IE7.
					}
					STXBSY = 0;							// SW_UART TX free.	
					break;
				}
		}
}

//----------------------------------------------------------------------------------------
// USER_ISR: User SW_UART Interrupt Service Routine (IE7 ISR)
// If interrupt-mode test code is enabled, this ISR
// transmits 15 characters and receives 15 characters. This routine is triggered each
// time a SW_UART transmit or receive is completed.
// - Checks receive complete indicator, and services.
// - Checks transmit complete indicator, and services.
// - Checks for transmits or receives that completed during the ISR; if so, triggers the
//   interrupt again.
//
void USER_ISR(void) interrupt 19 {				// IE7 Interrupt Service Routine

	PRT1IF &= ~(0x80);								// Clear IE7 interrupt flag
	
	if (SRI){											// If Receive Complete:
		SRI = 0;											// Clear receive flag.
		SW_BUF[k++] = RDR;							// Read receive buffer.
		if (k==16){										// If 15 characters have been received:
			SREN=0;										// Disable SW_UART Receiver.
			}												// Indicate 15 characters received.
		}

	else if (STI){										// If Transmit Complete:
		STI = 0;											// Clear transmit flag.

		if (m<15){                             // If less than 15 characters transmitted:
			STXBSY = 1;                         // Claim SW_UART Transmitter.
         TDR = m++;									// Increment variable, transmit.
         CCF1 = 1;									// Force module 1 interrupt to initiate TX
      }										         
      else 			
         SW_DONE=1;									// Indicate last character transmitted.
	}

	if (STI|SRI)										// If SRI or STI is set, re-trigger
		PRT1IF |= 0x80;								// interrupt to service.
}

//--------------------------------------------------------------------------------------
// HW_UART_ISR: Hardware UART Interrupt Service Routine
// Transmits characters from 1 to 15, and receives 15 characters.	
// - Checks receive interrupt, and services.
// - Checks transmit interrupt, and services.
//	
void	HW_UART_ISR(void) interrupt 4 {

	static char i=0;									// Transmit data variable.
	static char j=0;									// Receive data index.
	static idata char HW_BUF[20];					// Receive data buffer.
	
	if (RI){												// If Receive Complete:
		
		RI=0;												// Clear receive flag
		HW_BUF[j++] = SBUF;							// Read receive buffer		
		if (j==15)										// If 15 characters received:
			REN=0;										// Disable HW_UART receiver.
	}

	else if (TI){										// If Transmit Complete:

		TI = 0;											// Clear transmit flag
		if (i<15)										// If characters left to transmit:
			SBUF=i++;									// Increment variable, transmit.
		else												// If 15 characters transmitted,
			HW_DONE=1;									// Indicate HW TX finished.			

	}
}

// End of File