;----------------------------------------------------------------------------------
;	CYGNAL INTEGRATED PRODUCTS, INC.
;
;
; 	FILE NAME  			: RTC_1.asm 
; 	TARGET DEVICE		: C8051F0xx
; 	DESCRIPTION			: Software implementation of a real-time clock
;
;	AUTHOR				: JS
;
;	Software implementation of a real-time clock using a 32KHz crystal oscillator.
;	This program uses the crystal driver, XTAL2 to drive Comparator 0.  The positive 
;	comparator input is from XTAL2, and the negative input is an averaged version of
;	XTAL2.  The averaging is done by a low pass filter. The output of Comparator 0
;	is routed to the Timer 2 input (T2). 
;
; 	Timer 2 is configured in auto-reload mode, and is set to trigger on
;	the external input pin connected to the Comparator 0 output.
;
;	This code assumes the following:
;
;	(1)	An external oscillator is connected between XTAL1 and XTAL2
;	(2)	A low pass averaging filter is connected bewteen XTAL2 and CP0-
;	(3)	XTAL2 is routed to CP0+
;	(4)	CP0 output is routed to Timer 2 input through the port pins assigned
;			by the crossbar
;
;	For a 32KHz crystal, the low pass filter consists of a 0.022uF capacitor and a
;  1Mohm resistor.
;----------------------------------------------------------------------------------

;----------------------------------------------------------------------------------
; EQUATES
;----------------------------------------------------------------------------------

	$MOD8F000

; Count value: This value is used to define what is loaded into timer 2 after each 
; overflow.	The count value is 3200, meaning the timer will count 3200 ticks before an
; overflow.  Used with the 32KHz crystal, this means the timer will overflow every
; tenth of a second.

	COUNT				EQU		3200d		; count value

; Compensation factors for system clock switching used to update Timer 2 after a 
; system clock change
	
	EXT_COR			EQU		5d				
	INT_COR			EQU		3d

;----------------------------------------------------------------------------------
; VARIABLES
;----------------------------------------------------------------------------------

DSEG 

	org 30h

	TENTHS:			DS		1				; counts tenths of seconds
	SECONDS:			DS		1				; counts seconds
	MINUTES:			DS		1				; counts minutes
	HOURS:			DS		1				; counts hours
	DAYS:				DS		1				; counts days

	STORE_T:			DS		1				; storage byte for tenths, used by SAVE routine
	STORE_S:			DS		1				; storage byte for seconds
	STORE_M:			DS		1				; minutes
	STORE_H:			DS		1				; hours
	STORE_D:			DS		1				; days

BSEG

	org	20h

	SET_EXT_OSC:	DBIT	1				; flag to change system clock to external osc
	SET_INT_OSC:	DBIT 	1				; flag to change system clock to internal osc
								
;----------------------------------------------------------------------------------
; RESET and INTERRUPT VECTORS
;----------------------------------------------------------------------------------

CSEG

; Reset Vector

	org 	00h
	ljmp 	MAIN

; Timer 2 ISR Vector

	org 	2Bh
	ljmp	T2_ISR							; jump to Timer 2 ISR

;----------------------------------------------------------------------------------
; MAIN PROGRAM
;----------------------------------------------------------------------------------

	org 	0B3h
	
MAIN:

	mov	OSCXCN, #61h					; enable external oscillator
	
	mov 	WDTCN, #0DEh					; disable watchdog timer
	mov 	WDTCN, #0ADh							

	; Setup Crossbar
	mov	XBR0, #80h						; enable CP0 output
	mov	XBR1, #20h						; enable T2 input
	mov  	XBR2,	#40h						; enable crossbar

	; Setup Comparator 0
	mov	CPT0CN, #08h					; set positive hysteresis to 10mV
	orl	CPT0CN, #02h					; set negative hysteresis to 10mV
	orl	CPT0CN, #80h					; enable CP0
																	
	acall	RTC_INIT							; Initialize RTC and Timer 2

WAIT:

	mov	ACC, OSCXCN						; wait until the external oscillator is steady
	jnb	ACC.7, Wait						; by checking the XTLVLD bit in OSCXCN

	setb	TR2								; turn on Timer 2 (starts RTC)

	setb	EA									; enable global interrupts

	jmp	$									; spin forever

;-----------------------------------------------------------------------------------
;	Initialization Subroutine
;-----------------------------------------------------------------------------------

RTC_INIT:

	; Clear all counters
	mov	TENTHS,	#0
	mov	SECONDS,	#0
	mov	MINUTES, #0
	mov	HOURS, #0
	mov	DAYS, #0
	
	; Setup Timer2 in auto-reload mode to count falling edges on external T2

	mov	TH2, #HIGH(-COUNT)			; set initial value for timer 2
	mov	TL2, #LOW(-COUNT)

	mov 	RCAP2H, #HIGH(-COUNT)		; set reload value for timer 2
	mov	RCAP2L,	#LOW(-COUNT)

	mov  	T2CON, #02h						; configure Timer 2 to increment on a high/low 
												; transition from external input pin (T2)
	
	setb	ET2								; enable Timer 2 interrupt

	ret
;-----------------------------------------------------------------------------------
; Timer 2 ISR
;-----------------------------------------------------------------------------------

T2_ISR:
	

	clr	TF2								; clear overflow interrupt flag
	push	PSW								; preserve PSW (carry bit)
	push	ACC								; preserve ACC

	; Check for overflows
	

	mov	A, TENTHS						;
	cjne	A, #9d, INC_TEN				; if tenths less than 9, jump to increment
	mov	TENTHS, #0						; if tenths = 9, reset to zero, and check 
												; seconds
																	
	mov	A, SECONDS								
	cjne	A, #59d, INC_SEC				; if seconds less than 59, jump to increment
	mov	SECONDS, #0						; if seconds = 59, reset to zero, and check
												; minutes								
										
	mov	A, MINUTES						;
	cjne	A, #59d, INC_MIN				; if minutes less than 59, jump to increment
	mov	MINUTES, #0						; if minutes = 59, reset to zero, and check 
												; hours

	mov	A, HOURS							;
	cjne	A, #23d, INC_HOUR				; if hours less than 23, jump to increment
	mov	HOURS, #0						; if hours = 23, reset to zero, and check 
												; days
																	
	inc	DAYS								; DAYS will roll over after 255

	jmp	CHECK_OSC						; jump to check for oscillator change request

;Increment counters----------------------------------------------------------------
	
INC_TEN:
	
	inc	TENTHS							; increment tenths counter
	jmp	CHECK_OSC						; jump to check for oscillator change request
	
INC_SEC:

	inc	SECONDS							; increment seconds counter
	jmp	CHECK_OSC						; jump to check for oscillator change request

INC_MIN:

	inc	MINUTES							; increment minutes counter
	jmp	CHECK_OSC						; jump to check for oscillator change request

INC_HOUR:

	inc	HOURS								; increment hours counter
	jmp	CHECK_OSC						; jump to check for oscillator change request

;Oscillator changes----------------------------------------------------------------

CHECK_OSC:

	jbc	SET_EXT_OSC, EXT_OSC			; check for external oscillator select
	jbc	SET_INT_OSC, INT_OSC			; check for internal oscillator select
	jmp	END_ISR							; leave ISR

EXT_OSC:										; switch system clock to external oscillator
			
	mov	ACC, OSCICN						; check current system clock
	jb		ACC.3, END_ISR					; exit if already using external oscillator

	orl	CKCON, #20h						; select system clock (divide by 1) for 
												; Timer 2
	clr	T2CON.2							; disable Timer 2 during clock change
	clr	T2CON.1							; select SYSCLK as Timer 2 input

	mov	A, #LOW(EXT_COR)				; load correction value into accumulator
	add	A, TL2							; add correction value to Timer 2 register
	mov	TL2, A							; store updated Timer 2 value

	orl	OSCICN, #08h					; set external oscillator as system clock
	setb	T2CON.2							; enable Timer 2 after clock change

	jmp	END_ISR							; leave ISR

INT_OSC:										; switch system clock to internal oscillator
	
	mov	ACC, OSCICN						; check current system clock
	jnb	ACC.3, END_ISR					; exit if already using internal oscillator

	clr	T2CON.2							; disable Timer 2 during clock change
	anl	OSCICN, #0f7h					; select internal oscillator as system clock
	
	mov	A, #LOW(INT_COR)				; load correction value into accumulator
	add	A, TL2							; add correction value to Timer 2 register
	mov	TL2, A							; store updated Timer 2 value
	
	setb	T2CON.1							; select external Timer 2 input
	setb	T2CON.2							; enable Timer 2 after clock change

	jmp	END_ISR							; leave ISR

END_ISR:
	
	pop	ACC								; restore ACC
	pop	PSW								; restore PSW
	reti

;-----------------------------------------------------------------------------------
; Counter Save Routine
;-----------------------------------------------------------------------------------

SAVE:

	mov	C, ET2							; preserve ET2 in Carry	

	clr	ET2								; disable Timer 2 interrupt during store

	mov	STORE_T, TENTHS				; copy all counters
	mov	STORE_S, SECONDS				;
	mov	STORE_M, MINUTES				;
	mov	STORE_H, HOURS					;
	mov	STORE_D, DAYS					;

	mov	ET2, C							; restore ET2
	ret

;---------------------------------------------------------------------------------

END