//------------------------------------------------------------------------------------
// JTAG_FLASH.c
//------------------------------------------------------------------------------------
// This program contains some primitive routines which read, write, and erase the FLASH
// through the JTAG port on a C8051Fxxx device under test (DUT).  The JTAG pins on the
// DUT are connected to port pins on the C8051F000 master device.
//
// Target device: C8051F000, C8051F010
//
// Tool chain: KEIL Eval 'c'
//

//------------------------------------------------------------------------------------
// Includes
//------------------------------------------------------------------------------------
#include <c8051f000.h>                       // SFR declarations

//------------------------------------------------------------------------------------
// Global CONSTANTS
//------------------------------------------------------------------------------------
sbit   LED = P1^6;                           // green LED: '1' = ON; '0' = OFF

// GPIO pins connecting to JTAG pins on device to be programmed (DUT)
sbit   TCK = P3^7;                           // JTAG Test Clock
sbit   TMS = P3^6;                           // JTAG Mode Select
sbit   TDI = P3^5;                           // JTAG Data Input
sbit   TDO = P3^4;                           // JTAG Data Output

#define   TRUE 1
#define   FALSE 0
                        
// JTAG Instruction Register Addresses
#define   INST_LENGTH 16                     // number of bits in the 
                                             // Instruction Register
#define   BYPASS      0xffff
#define   EXTEST      0x0000
#define   SAMPLE      0x0002

#define   RESET       0x2fff                 // System RESET Instruction

#define   IDCODE      0x1004                 // IDCODE Instruction address/HALT
#define   IDCODE_LEN  32                     // number of bits in the ID code

#define   FLASHCON    0x4082                 // FLASH Control Instruction address
#define   FLCN_LEN    8                      // number of bits in FLASHCON

#define   FLASHDAT    0x4083                 // FLASH Data Instruction address
#define   FLD_RDLEN   10                     // number of bits in an FLASHDAT read
#define   FLD_WRLEN   8                      // number of bits in an FLASHDAT write

#define   FLASHADR    0x4084                 // FLASH Address Instruction address
#define   FLA_LEN     16                     // number of bits in FLASHADR

#define   FLASHSCL    0x4085                 // FLASH Scale Instruction address
#define   FLSC_LEN    8                      // number of bits in FLASHSCL

//------------------------------------------------------------------------------------
// Function PROTOTYPES
//------------------------------------------------------------------------------------

void init (void);
void JTAG_StrobeTCK (void);
void JTAG_Reset (void);
unsigned int JTAG_IR_Scan (unsigned int instruction, int num_bits);
unsigned long JTAG_DR_Scan (unsigned long dat, int num_bits);
void JTAG_IWrite (unsigned int ireg, unsigned long dat, int num_bits);
unsigned long JTAG_IRead (unsigned int ireg, int num_bits);
int FLASH_ByteRead (unsigned int addr, unsigned char *pdat);
int FLASH_ByteWrite (unsigned int addr, unsigned char dat);
int FLASH_PageErase (unsigned int addr);

//------------------------------------------------------------------------------------
// MAIN Routine

void main (void) {
   
   unsigned long id;
   unsigned char dest;
   int pass;

   id = 0x12345678L;
   
   init ();                                    // initialize ports

   JTAG_Reset ();                              // Reset the JTAG state machine on DUT
   
   JTAG_IR_Scan (RESET, INST_LENGTH);          // Reset the DUT

   JTAG_IR_Scan (IDCODE, INST_LENGTH);         // load IDCODE into IR and HALT the DUT
   id = JTAG_DR_Scan (0x0L, IDCODE_LEN);       // read the IDCODE
                                               // IDCODE should = 0x10000243 for
                                               // C8051F000 rev D device

   // here we erase the FLASH page 0x1000 - 0x11ff, read 0x1000 (it's an 0xff),
   // write a 0x66 to 0x1000, and read 0x1000 again (it's changed to an 0x66).
   while (1) {
      pass = FLASH_PageErase (0x7c00);         // erase page prior to writing...
      while (!pass);                           // handle Write Lock condition

      dest = 0x5a;                             // set test variable to non-0xff value

      pass = FLASH_ByteRead (0x7c00, &dest);   // dest should return 0xff
      while (!pass);                           // handle Read Lock condition

      dest = 0x66;
      pass = FLASH_ByteWrite (0x7c00, dest);   // store 0x66 at 0x1000
      while (!pass);                           // handle Read Lock condition

      pass = FLASH_ByteRead (0x7c00, &dest);   // dest should return 0x66
      while (!pass);                           // handle Read Lock condition

      pass = FLASH_PageErase (0x7c00);
      while (!pass);

      pass = FLASH_ByteRead (0x7c00, &dest);
      while (!pass);
   }
}

//------------------------------------------------------------------------------------
// Functions and Procedures
//------------------------------------------------------------------------------------

//------------------------------------------------------------------------------------
// init
//------------------------------------------------------------------------------------
// This routine disables the watchdog timer and initializes the GPIO pins
//
void init (void) {

   WDTCN = 0xde;                          // disable watchdog timer
   WDTCN = 0xad;

   XBR2 |= 0x40;                          // enable crossbar
   PRT1CF |= 0x40;                        // enable P1.6 (LED) as a push-pull output
   PRT3CF |= 0xe0;                        // make P3.7-5 push-pull outputs
   P3 &= 0x1f;                            // TCK, TMS, and TDI all low

}

//------------------------------------------------------------------------------------
// JTAG_StrobeTCK
//------------------------------------------------------------------------------------
// This routine strobes the TCK pin (brings high then back low again) 
// on the target system.
//
void JTAG_StrobeTCK (void) {

   TCK = 1;
   TCK = 0;
}

//------------------------------------------------------------------------------------
// JTAG_Reset
//------------------------------------------------------------------------------------
// This routine places the JTAG state machine on the target system in
// the Test Logic Reset state by strobing TCK 5 times while leaving
// TMS high.  Leaves the JTAG state machine in the Run_Test/Idle state.
//
void JTAG_Reset (void) {
   
   TMS = 1;

   JTAG_StrobeTCK ();                     // move to Test Logic Reset state
   JTAG_StrobeTCK ();
   JTAG_StrobeTCK ();
   JTAG_StrobeTCK ();
   JTAG_StrobeTCK ();

   TMS = 0;

   JTAG_StrobeTCK ();                     // move to Run_Test/Idle state
}

//------------------------------------------------------------------------------------
// JTAG_IR_Scan
//------------------------------------------------------------------------------------
// This routine loads the supplied <instruction> of <num_bits> length into the JTAG 
// Instruction Register on the target system.  Leaves in the Run_Test/Idle state.
// The return value is the n-bit value read from the IR.
// Assumes the JTAG state machine starts in the Run_Test/Idle state.
//
unsigned int JTAG_IR_Scan (unsigned int instruction, int num_bits) {

   unsigned int retval;                   // JTAG instruction read
   int   i;                               // JTAG IR bit counter

   retval = 0x0;
   
   TMS = 1;                     
   JTAG_StrobeTCK ();                     // move to SelectDR
   TMS = 1;
   JTAG_StrobeTCK ();                     // move to SelectIR
   TMS = 0;
   JTAG_StrobeTCK ();                     // move to Capture_IR
   TMS = 0;
   JTAG_StrobeTCK ();                     // move to Shift_IR state

   for (i=0; i < num_bits; i++) {

      TDI = (instruction & 0x01);         // shift IR, LSB-first
      instruction = instruction >> 1;

      retval = retval >> 1;
      if (TDO) {
         retval |= (0x01 << (num_bits - 1));
      }

      if (i == (num_bits - 1)) {
         TMS = 1;                         // move to Exit1_IR state
      }

      JTAG_StrobeTCK();
   }

   TMS = 1;
   JTAG_StrobeTCK ();                     // move to Update_IR
   TMS = 0;
   JTAG_StrobeTCK ();                     // move to RTI state

   return retval;
}

//------------------------------------------------------------------------------------
// JTAG_DR_Scan
//------------------------------------------------------------------------------------
// This routine shifts <num_bits> of <data> into the Data Register, and returns
// up to 32-bits of data read from the Data Register.
// Leaves in the Run_Test/Idle state.
// Assumes the JTAG state machine starts in the Run_Test/Idle state.
//
unsigned long JTAG_DR_Scan (unsigned long dat, int num_bits) {

   unsigned long   retval;                // JTAG return value
   int   i;                               // JTAG DR bit counter

   retval = 0x0L;

   TMS = 1;                     
   JTAG_StrobeTCK ();                     // move to SelectDR
   TMS = 0;
   JTAG_StrobeTCK ();                     // move to Capture_DR
   TMS = 0;
   JTAG_StrobeTCK ();                     // move to Shift_DR state

   for (i=0; i < num_bits; i++) {

      TDI = (dat & 0x01);                 // shift DR, LSB-first
      dat = dat >> 1;

      retval = retval >> 1;
      if (TDO) {
         retval |= (0x01L << (num_bits - 1));
      }

      if ( i == (num_bits - 1)) {
         TMS = 1;                         // move to Exit1_DR state
      }

      JTAG_StrobeTCK();
   }
   TMS = 1;
   JTAG_StrobeTCK ();                     // move to Update_DR
   TMS = 0;
   JTAG_StrobeTCK ();                     // move to RTI state

   return retval;
}

//------------------------------------------------------------------------------------
// JTAG_IWrite
//------------------------------------------------------------------------------------
// This routine performs an indirect write to register <ireg>, containing <dat>, of
// <num_bits> in length.  It follows the write operation with a polling operation, and
// returns when the operation is completed.  Note: the polling implemented here refers
// to the JTAG register write operation being completed, NOT the FLASH write operation.
// Polling for the FLASH write operation is handled at a higher level
// Examples of valid indirect registers are:
//  FLCN - FLASH Control
//  FLSC - FLASH Scale
//  FLA - FLASH Address
//  FLD - FLASH Data
// Leaves in the Run_Test/Idle state.
//
void JTAG_IWrite (unsigned int ireg, unsigned long dat, int num_bits) {
   
   int done;                              // TRUE = write complete; FALSE otherwise
   
   JTAG_IR_Scan (ireg, INST_LENGTH);      // load IR with <ireg>

   dat |= (0x03L << num_bits);            // append 'WRITE' opcode to data
   
   // load DR with <dat>
   JTAG_DR_Scan (dat, num_bits + 2);      // initiate the JTAG write

   // load DR with '0', and check for BUSY bit to go to '0'.
   do {
      done = !(JTAG_DR_Scan (0x0L, 1));   // poll for JTAG_BUSY bit
   } while (!done);
}

//------------------------------------------------------------------------------------
// JTAG_IRead
//------------------------------------------------------------------------------------
// This routine performs an indirect read of register <ireg>, of <num_bits> in length.
// It follows the read operation with a polling operation, and returns when the 
// operation is completed.  Note: the polling implemented here refers to the JTAG 
// register read operation being completed, NOT the FLASH read operation.
// Polling for the FLASH read operation is handled at a higher level.
// Examples of valid indirect registers are:
//  FLCN - FLASH Control
//  FLSC - FLASH Scale
//  FLA - FLASH Address
//  FLD - FLASH Data
// Leaves in the Run_Test/Idle state.
//
unsigned long JTAG_IRead (unsigned int ireg, int num_bits) {
   
   unsigned long retval;                  // value returned from READ operation
   int done;                              // TRUE = write complete; FALSE otherwise
   
   JTAG_IR_Scan (ireg, INST_LENGTH);      // load IR with <ireg>

   // load DR with read opcode (0x02)

   JTAG_DR_Scan (0x02L, 2);               // initiate the JTAG read

   do {
      done = !(JTAG_DR_Scan (0x0L, 1));   // poll for JTAG_BUSY bit
   } while (!done);
   
   retval = JTAG_DR_Scan (0x0L, num_bits + 1);  // allow poll operation to
                                                // read remainder of the bits
   retval = retval >> 1;                        // shift JTAG_BUSY bit off the end

   return retval;
}

//------------------------------------------------------------------------------------
// FLASH_ByteRead
//------------------------------------------------------------------------------------
// This routine reads the byte at <addr> and stores it at the address pointed to by
// <pdat>.
// Returns TRUE if the operation was successful; FALSE otherwise (page read-protected).
//
int FLASH_ByteRead (unsigned int addr, unsigned char *pdat)
{
   unsigned long testval;                      // holds result of FLASHDAT read
   int done;                                   // TRUE/FALSE flag
   int retval;                                 // TRUE if operation successful

   JTAG_IWrite (FLASHSCL, 0x86L, FLSC_LEN);    // set FLASHSCL based on SYSCLK 
                                               //  frequency (2MHz = 0x86)

   // set FLASHADR to address to read from
   JTAG_IWrite (FLASHADR, (unsigned long) addr, FLA_LEN);

   JTAG_IWrite (FLASHCON, 0x02L, FLCN_LEN);    // set FLASHCON for FLASH Read 
                                               //  operation (0x02)

   JTAG_IRead (FLASHDAT, FLD_RDLEN);           // initiate the read operation

   JTAG_IWrite (FLASHCON, 0x0L, FLCN_LEN);     // set FLASHCON for 'poll' operation

   do {
      done = !(JTAG_IRead (FLASHDAT, 1));      // poll for FLBUSY to de-assert
   } while (!done);

   testval = JTAG_IRead (FLASHDAT, FLD_RDLEN); // read the resulting data

   retval = (testval & 0x02) ? FALSE: TRUE;    // FLFail is next to LSB

   testval = testval >> 2;                     // shift data.0 into LSB position

   *pdat = (unsigned char) testval;            // place data in return location

   return retval;                              // return FLASH Pass/Fail
}

//------------------------------------------------------------------------------------
// FLASH_ByteWrite
//------------------------------------------------------------------------------------
// This routine writes the data <dat> to FLASH at the address <addr>.
// Returns TRUE if the operation was successful; FALSE otherwise (page 
// write-protected).
//
int FLASH_ByteWrite (unsigned int addr, unsigned char dat)
{
   unsigned long testval;                      // holds result of FLASHDAT read
   int done;                                   // TRUE/FALSE flag
   int retval;                                 // TRUE if operation successful

   JTAG_IWrite (FLASHSCL, 0x86L, FLSC_LEN);    // set FLASHSCL based on SYSCLK 
                                               //  frequency (2MHz = 0x86)

   // set FLASHADR to address to write to
   JTAG_IWrite (FLASHADR, (unsigned long) addr, FLA_LEN);

   JTAG_IWrite (FLASHCON, 0x10L, FLCN_LEN);    // set FLASHCON for FLASH Write 
                                               //  operation (0x10)

   // initiate the write operation
   JTAG_IWrite (FLASHDAT, (unsigned long) dat, FLD_WRLEN);

   JTAG_IWrite (FLASHCON, 0x0L, FLCN_LEN);     // set FLASHCON for 'poll' operation

   do {
      done = !(JTAG_IRead (FLASHDAT, 1));      // poll for FLBusy to de-assert
   } while (!done);

   testval = JTAG_IRead (FLASHDAT, 2);         // read FLBusy and FLFail

   retval = (testval & 0x02) ? FALSE: TRUE;    // FLFail is next to LSB

   return retval;                              // return FLASH Pass/Fail
}

//------------------------------------------------------------------------------------
// FLASH_PageErase
//------------------------------------------------------------------------------------
// This routine performs an erase of the page in which <addr> is contained.
// This routine assumes that no FLASH operations are currently in progress.
// This routine exits with no FLASH operations currently in progress.
// Returns TRUE if the operation was successful; FALSE otherwise (page protected).
//
int FLASH_PageErase (unsigned int addr)
{
   unsigned long testval;                      // holds result of FLASHDAT read
   int done;                                   // TRUE/FALSE flag
   int retval;                                 // TRUE if operation successful

   JTAG_IWrite (FLASHSCL, 0x86L, FLSC_LEN);    // set FLASHSCL based on SYSCLK 
                                               //  frequency (2MHz = 0x86)

   // set FLASHADR to address within page to erase
   JTAG_IWrite (FLASHADR, (unsigned long) addr, FLA_LEN);   

   JTAG_IWrite (FLASHCON, 0x20L, FLCN_LEN);    // set FLASHCON for FLASH Erase 
                                               //  operation (0x20)

   JTAG_IWrite (FLASHDAT, 0xa5L, FLD_WRLEN);   // set FLASHDAT to 0xa5 to initiate 
                                               //  erase procedure

   JTAG_IWrite (FLASHCON, 0x0L, FLCN_LEN);     // set FLASHCON for 'poll' operation

   do {
      done = !(JTAG_IRead (FLASHDAT, 1));      // poll for FLBusy to de-assert
   } while (!done);

   testval = JTAG_IRead (FLASHDAT, 2);         // read FLBusy and FLFail

   retval = (testval & 0x02) ? FALSE: TRUE;    // FLFail is next to LSB

   // set return value based on FLFail bit
   return retval;                              // return FLASH Pass/Fail
}
