//-----------------------------------------------------------------------------
// T63x_ADC_TemperatureSensor.c
//-----------------------------------------------------------------------------
// Copyright (C) 2008 Silicon Laboratories, Inc.
//
// This program is intended to run on either a C8051F336 or C8051T63x device,
// and is an example of developing code on an 'F336 for the 'T63x.
//
// #if statements in the code are used to define necessary differences in the
// setup and operation of the two devices.
//
// This software prints the device die temperature out the hardware 
// UART at 115200bps. The calibrated internal 24.5MHz oscillator is used as the 
// system clock source.
//
// The ADC is configured to look at the on-chip temp sensor.  The sampling
// rate of the ADC is determined by the constant <SAMPLE_RATE>, which is given
// in Hz.
// 
// The ADC0 End of Conversion Interrupt Handler retrieves the sample
// from the ADC and adds it to a running accumulator.  Every <INT_DEC> 
// samples, the ADC updates and stores its result in the global variable
// <result>, which contains the accumulated result. The sampling technique of 
// adding a set of values and decimating them (posting results every (n)th 
// sample) is called 'integrate and dump.'  It is easy to implement and 
// requires very few resources.
//
// For each power of 4 of <INT_DEC>, you gain 1 bit of noise resolution.
// For example, <INT_DEC> = 16 gains you 2 bits of resolution: 4^2 = 16.

// Target:         C8051F336, C8051T63x
// Tool chain:     Keil C51 7.50 / Keil EVAL C51
// Command Line:   None
//
// Release 1.0
//    -Initial Revision (BD)
//    -30 JAN 2008
//

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------

#include <compiler_defs.h>             // Compiler-specific declarations
#include <C8051T630_defs.h>            // SFR declarations
#include <stdio.h>                     // Standard UART I/O (printf)

//-----------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------

// Comment out one of the following two lines to indicate which device is used
//#define C8051F336
#define C8051T630

// ** Definitions for the C8051F336 family **
// Below are definitions to calculate basic temperature sensor parameters for
// the 'F336 device.
// When using the 'F336 family, VREF is set to VDD, and assumed to be 3.3V.
//
// The temperature sensor offset is typically 785 mV at 0 degrees C.
// The temperature sensor slope  is typically 2.25 mV / degree C.
//
// Based on the VREF voltage and the offset number, the temperature sensor 
// offset <TOFF_LSB10> is calculated in output codes from the ADC.
// Based on the VREF voltage and the slope number, the temperature sensor
// slope is calculated in ADC output codes per 100 degrees C (to gain better
// accuracy with fixed-point mathematics).

#ifdef C8051F336
SFR (AMX0N, 0xBA);                     // AMX0N is not included in the 'T63x

#define VREF_UV      3300000
#define TOFF_UV      785000
#define TSLOPE_UV    2250

#define TOFF_LSB10 (((TOFF_UV*1024L)/VREF_UV)<<6)
#define LSB10_P_100C ((100L*(TSLOPE_UV)*1024L)/(VREF_UV))
#endif
// ** End definitions for the C8051F336 family **


// ** Definitions for the C8051T63x family **
// Below are definitions to calculate basic temperature sensor parameters for
// the 'T63x device.
// When using the 'T63x family, VREF is set to the internal 1.8V regulator.
// The gain of the ADC in this example is set to 1x.
// The temperature sensor slope  is typically 3.48 mV / degree C.
//
// The temperature sensor offset in the 'T63x is pre-measured under the VREF
// and ADC conditions above, and stored in the registers TOFFH and TOFFL.
// The temperature sensor offset <TOFF_LSB10> is simply set to read TOFF.
// Based on the VREF voltage and the slope number, the temperature sensor
// slope is calculated in ADC output codes per 100 degrees C (to gain better
// accuracy with fixed-point mathematics).
#ifdef C8051T630
#define VREF_UV      1800000
#define TSLOPE_UV    3480

#define TOFF_LSB10   (TOFF)
#define LSB10_P_100C ((100L*(TSLOPE_UV)*1024L)/(VREF_UV))
#endif
// ** End definitions for the C8051T63x family **


#define SYSCLK       24500000          // SYSCLK frequency in Hz
#define BAUDRATE     115200            // Baud rate of UART in bps
#define SAMPLE_RATE  10000             // Sample frequency in Hz

#define INT_DEC      256               // integrate and decimate ratio
#define ADD_BITS     4                 // ADD_BITS is calculated from INT_DEC
                                       // it should be equal to the number of
                                       // additional bits of noise resolution
                                       // obtained by averaging INT_DEC samples
                                       // 1 = 0 bit
                                       // 4 = 1 bit
                                       // 16 = 2 bits
                                       // 64 = 3 bits
                                       // etc.

SBIT (LED, SFR_P1, 3);                 // LED='1' means ON
SBIT (SW1, SFR_P0, 7);                 // SW1='0' means switch pressed

//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------

void SYSCLK_Init (void);
void PORT_Init (void);
void UART0_Init (void);
void ADC0_Init (void);
void Timer2_Init (int counts);
INTERRUPT_PROTO_USING (ADC0_ISR, INTERRUPT_ADC0_EOC, 3);

//-----------------------------------------------------------------------------
// Global VARIABLES
//-----------------------------------------------------------------------------

long result;                           // ADC0 decimated value
bit new_temp_data = 0;

//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------

void main (void) {
   long temperature;                   // Temperature in hundredths of a
                                       // degree C
   int temp_int, temp_frac;            // Integer and fractional portions of
                                       // temperature

   // Disable Watchdog timer
   PCA0MD &= ~0x40;                    // WDTE = 0 (clear watchdog timer 
                                       // enable)

   SYSCLK_Init ();                     // Initialize oscillator
   PORT_Init ();                       // Initialize crossbar and GPIO
   UART0_Init ();                      // Initialize UART0
   Timer2_Init (SYSCLK/SAMPLE_RATE);   // Initialize Timer2 to overflow at
                                       // <SAMPLE_RATE>

   ADC0_Init ();                       // Init ADC0

	AD0EN = 1;                          // Enable ADC0

   EA = 1;                             // Enable global interrupts

	while (1) 
   {
      while (new_temp_data == 0);
      EA = 0;                          // Disable interrupts
      // read result, and calculate temperature x 10.
      temperature = (result * 1000L) / (LSB10_P_100C << ADD_BITS);
      EA = 1;                          // Re-enable interrupts

      new_temp_data = 0;

      // calculate temperature in tenths of a degree Celsius
      temp_int = temperature / 10;
      temp_frac = temperature % 10;
	   printf ("Temperature is %+02d.%01d\n", temp_int, temp_frac);

      LED = ~SW1;                      // LED reflects state of SW1
	}
}

//-----------------------------------------------------------------------------
// Initialization Subroutines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// SYSCLK_Init
//-----------------------------------------------------------------------------
//
// This routine initializes the system clock to use the internal 24.5MHz 
// oscillator as its clock source.  Also enables missing clock detector reset.
//
void SYSCLK_Init (void)
{
   OSCICN = 0x83;                      // Configure internal oscillator for
                                       // its maximum frequency}
   RSTSRC = 0x04;                      // Enable missing clock detector
}

//-----------------------------------------------------------------------------
// PORT_Init
//-----------------------------------------------------------------------------
//
// Configure the Crossbar and GPIO ports.
// P0.0 - 
// P0.1 -
// P0.2 - 
// P0.3 - 
// P0.4 - UART TX (push-pull)
// P0.5 - UART RX
// P0.6 -
// P0.7 - SW1
// P1.3 - LED (push-pull)
//
void PORT_Init (void)
{
   P0MDOUT = 0x10;                     // Set UART TX to push-pull output
   P1MDOUT = 0x08;                     // Set LED to push-pull output
   XBR0 = 0x01;                        // Route UART RX and TX to pins
   XBR1 = 0x40;                        // Enable crossbar, disable weak pull-up
}

//-----------------------------------------------------------------------------
// UART0_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// Configure the UART0 using Timer1, for <BAUDRATE> and 8-N-1.
//-----------------------------------------------------------------------------
void UART0_Init (void)
{
   SCON0 = 0x10;                       // SCON0: 8-bit variable bit rate
                                       //        level of STOP bit is ignored
                                       //        RX enabled
                                       //        ninth bits are zeros
                                       //        clear RI0 and TI0 bits
   if (SYSCLK/BAUDRATE/2/256 < 1) {
      TH1 = -(SYSCLK/BAUDRATE/2);
      CKCON &= ~0x0B;                  // T1M = 1; SCA1:0 = xx
      CKCON |=  0x08;
   } else if (SYSCLK/BAUDRATE/2/256 < 4) {
      TH1 = -(SYSCLK/BAUDRATE/2/4);
      CKCON &= ~0x0B;                  // T1M = 0; SCA1:0 = 01                  
      CKCON |=  0x01;
   } else if (SYSCLK/BAUDRATE/2/256 < 12) {
      TH1 = -(SYSCLK/BAUDRATE/2/12);
      CKCON &= ~0x0B;                  // T1M = 0; SCA1:0 = 00
   } else {
      TH1 = -(SYSCLK/BAUDRATE/2/48);
      CKCON &= ~0x0B;                  // T1M = 0; SCA1:0 = 10
      CKCON |=  0x02;
   }

   TL1 = TH1;                          // Init Timer1
   TMOD &= ~0xf0;                      // TMOD: timer 1 in 8-bit autoreload
   TMOD |=  0x20;                       
   TR1 = 1;                            // START Timer1
   TI0 = 1;                            // Indicate TX0 ready
}

//-----------------------------------------------------------------------------
// ADC0_Init
//-----------------------------------------------------------------------------
//
// Configure ADC0 to use Timer2 overflows as conversion source, to
// generate an interrupt on conversion complete, and to sense the output of
// the temp sensor.  Enables ADC end of conversion interrupt. Leaves ADC 
// disabled.
//
void ADC0_Init (void)
{
   ADC0CN = 0x02;                      // ADC0 disabled; normal tracking
                                       // mode; ADC0 conversions are initiated 
                                       // on overflow of Timer2;
   AMX0P = 0x10;                       // Select temp sensor as mux input

   ADC0CF = (SYSCLK/3125000) << 3;     // ADC conversion clock <= 3.125 MHz
   ADC0CF |= 0x04;                     // Left justify ADC outputs

#ifdef C8051F336
   AMX0N = 0x11;
   REF0CN = 0x0e;                      // Enable temp sensor, VREF = VDD, bias
                                       // generator is on
#endif

#ifdef C8051T630
   ADC0CF |= 0x01;                     // PGA gain = 1
   REF0CN = 0x14;                      // Enable temp sensor, VREF = Regulator
#endif

   EIE1 |= 0x08;                       // Enable ADC0 EOC interrupt
}

//-----------------------------------------------------------------------------
// Timer2_Init
//-----------------------------------------------------------------------------
//
// Configure Timer2 to auto-reload at interval specified by <counts> (no 
// interrupt generated) using SYSCLK as its time base.
//
void Timer2_Init (int counts)
{
   TMR2CN = 0x00;                      // STOP Timer2; Clear TF2H and TF2L;
                                       // disable low-byte interrupt; disable
                                       // Split mode; select internal timebase
   CKCON |= 0x20;                      // Timer2 uses SYSCLK as its timebase

   TMR2RL  = -counts;                  // Init reload values
   TMR2    = 0xffff;                   // Set to reload immediately
   ET2 = 0;                            // Disable Timer2 interrupts
   TR2 = 1;                            // Start Timer2
}

//-----------------------------------------------------------------------------
// Interrupt Service Routines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// ADC0_ISR
//-----------------------------------------------------------------------------
//
// ADC0 end-of-conversion ISR 
// Here we take the ADC0 sample, add it to a running total <accumulator>, and
// decrement our local decimation counter <int_dec>.  When <int_dec> reaches
// zero, we post the decimated result in the global variable <result>.
//
void ADC0_ISR (void) interrupt INTERRUPT_ADC0_EOC using 3
{
   static unsigned int_dec=INT_DEC;    // Integrate/decimate counter
                                       // we post a new result when
                                       // int_dec = 0
   static long accumulator=0L;         // Here's where we integrate the
                                       // ADC samples             

   AD0INT = 0;                         // Clear ADC conversion complete
                                       // indicator

   // Read ADC value and add to running total. Also subtract out offset now.
   accumulator += ((ADC0 - TOFF_LSB10)>>6);



   int_dec--;                          // Update decimation counter

   if (int_dec == 0) {                 // If zero, then post result
      int_dec = INT_DEC;               // reset counter
      result = accumulator >> ADD_BITS;
      accumulator = 0L;                // Reset accumulator
      new_temp_data = 1;
   }
}
