//-----------------------------------------------------------------------------
// F411_VR_SSTFlash.c
//-----------------------------------------------------------------------------
// Copyright 2006 Silicon Laboratories, Inc.
// http://www.silabs.com
//
// Program Description:
//
// This file contains the interfacing functions to the SST Flash, allowing the
// user to Read memory, Write memory, and Erase memory.
//
// How To Use:    See Readme.txt
//
// FID:            41X000010
// Target:         C8051F411
// Tool chain:     Keil C51 7.50 / Keil EVAL C51
//                 Silicon Laboratories IDE version 2.6
// Project Name:   F411_VR
//
// Release 1.3
//    -All changes by TP
//    -02 Feb 2006
//    -added Read_MEM_Init (duplicate of Read_MEM) to avoid
//       the compiler warning (multiple calls to segment)
//
// Release 1.2
//    -All changes by TP
//    -21 Nov 2005
//    -project version updated, no changes to this file.
//
// Release 1.1
//    -All changes by TP
//    -16 Aug 2004
//    -added SPIF polling while sending the WREN command
//
// Release 1.0
//    -Initial Revision (TP)
//    -15 AUG 2004
//

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <c8051f410.h>                 // SFR declarations

//-----------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------
// SST Instruction Opcodes (as shown in the datasheet)
#define EWSR 0x50                      // enable write status register
#define WRSR 0x01                      // write status register
#define RDSR 0x05                      // read status register
#define WREN 0x06                      // write enable
#define BPROG 0x02                     // byte program
#define READ 0x03                      // read
#define CERASE 0x60                    // chip erase
#define READID 0x90                    // chip ID

// Address definition
typedef union ADDRESS {                // access an address as a
   unsigned long ULong;                // unsigned long variable or
   unsigned char UByte[4];             // 4 unsigned byte variables
   // [0] = A31-24, [1] = A23-16, [2] = A15-8, [3] = A7-0
} ADDRESS;

//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------
void SSTFlash_Init (void);

void Write_MEM (unsigned long address, unsigned char data_byte);
unsigned char Read_MEM (unsigned long address);
void Erase_MEM (void);
char ReadID_MEM (void);

//-----------------------------------------------------------------------------
// SSTFlash_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// Unprotect the memory so that all of memory may be written and read.
// NOTE: The SPI must be initialized before this function is called.
//
void SSTFlash_Init (void)
{
   NSSMD0 = 0;                         // enable the flash

   // send the enable write status register command
   SPI0DAT = EWSR;                     // load the XMIT register
   while (TXBMT != 1)                  // wait until EWSR command is moved into
   {                                   // the XMIT buffer
   }
   SPIF = 0;
   while (SPIF != 1)                   // wait until the SPI finishes sending
   {                                   // the EWSR command to the flash
   }
   SPIF = 0;

   NSSMD0 = 1;                         // allow the command to execute

   NSSMD0 = 0;                         // enable the flash

   // send the write status register command and clear the BP bits
   SPI0DAT = WRSR;                     // load the XMIT register
   while (TXBMT != 1)                  // wait until the XMIT register can
   {                                   // accept more data
   }
   SPI0DAT = 0x00;                     // set the block protection bits to 0
   while (TXBMT != 1)                  // wait until the data is moved into
   {                                   // the XMIT buffer
   }
   SPIF = 0;
   while (SPIF != 1)                   // wait until the SPI finishes sending
   {                                   // the data to the flash
   }
   SPIF = 0;

   NSSMD0 = 1;                         // allow the command to execute
}

//-----------------------------------------------------------------------------
// Write_MEM
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   :
//   1)  long address - address in the 512 kB external SST Flash
//                      range is postive values up to 2^19: 0 to 524287,
//                           or, 0 to 0x7FFFF
//   2)  char data_byte - the data to be written to memory
//                      range is positive range of character: 0 to 255
//
// Write one byte of data to a 24-bit address in the SST Flash Memory using
// the SPI.
//
void Write_MEM (unsigned long address, unsigned char data_byte)
{
   ADDRESS temp_addr;
   temp_addr.ULong = address;

   NSSMD0 = 0;                         // enable the flash

   // send the write enable command
   SPI0DAT = WREN;                     // load the XMIT register
   while (TXBMT != 1)                  // wait until the command is moved into
   {                                   // the XMIT buffer
   }
   SPIF = 0;
   while (SPIF != 1)                   // wait until the command reaches the
   {                                   // flash
   }
   SPIF = 0;

   NSSMD0 = 1;                         // allow the WREN to execute

   NSSMD0 = 0;                         // enable the flash

   // send the byte-program command
   SPI0DAT = BPROG;                    // load the XMIT register
   while (TXBMT != 1)                  // wait until the command is moved into
   {                                   // the XMIT buffer
   }
   SPI0DAT = temp_addr.UByte[1];       // load the high byte of the address
   while (TXBMT != 1)                  // wait until the addr is moved into
   {                                   // the XMIT buffer
   }
   SPI0DAT = temp_addr.UByte[2];       // load the middle byte of the address
   while (TXBMT != 1)                  // wait until the addr is moved into
   {                                   // the XMIT buffer
   }
   SPI0DAT = temp_addr.UByte[3];       // load the low byte of the address
   while (TXBMT != 1)                  // wait until the addr is moved into
   {                                   // the XMIT buffer
   }
   SPI0DAT = data_byte;                // load the byte of data
   while (TXBMT != 1)                  // wait until the data is moved into
   {                                   // the XMIT buffer
   }
   SPIF = 0;
   while (SPIF != 1)                   // wait until the last byte of the
   {                                   // write instruction reaches the flash
   }
   SPIF = 0;

   NSSMD0 = 1;                         // allow the WR instruction to execute
}

//-----------------------------------------------------------------------------
// Read_MEM
//-----------------------------------------------------------------------------
//
// Return Value :
//   1)  char data_byte - the data byte read from memory
//                      range is positive range of character: 0 to 255
// Parameters   :
//   1)  long address - address in the 512 kB external SST Flash
//                      range is postive values up to 2^19: 0 to 524287,
//                           or, 0 to 0x7FFFF
//
// Read one byte of data from a 24-bit address in the SST Flash Memory using
// the SPI.
//
unsigned char Read_MEM (unsigned long address)
{
   ADDRESS temp_addr;
   temp_addr.ULong = address;

   NSSMD0 = 0;                         // enable the flash

   // send the read instruction
   SPI0DAT = READ;                     // load the XMIT register
   while (TXBMT != 1)                  // wait until the command is moved into
   {                                   // the XMIT buffer
   }
   SPI0DAT = temp_addr.UByte[1];       // load the high byte of the address
   while (TXBMT != 1)                  // wait until the data is moved into
   {                                   // the XMIT buffer
   }
   SPI0DAT = temp_addr.UByte[2];       // load the middle byte of the address
   while (TXBMT != 1)                  // wait until the data is moved into
   {                                   // the XMIT buffer
   }
   SPI0DAT = temp_addr.UByte[3];       // load the low byte of the address
   while (TXBMT != 1)                  // wait until the data is moved into
   {                                   // the XMIT buffer
   }
   SPI0DAT = 0xFF;                     // load junk data in order to receive
                                       // data from the flash
   while (TXBMT != 1)                  // wait until the junk data is moved
   {                                   // into the XMIT buffer
   }
   SPIF = 0;
   while (SPIF != 1)                   // wait until the read data is received
   {
   }
   SPIF = 0;

   NSSMD0 = 1;                         // disable the flash

   return SPI0DAT;
}

//-----------------------------------------------------------------------------
// Erase_MEM
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// Erase all data from the SST flash memory.
//
void Erase_MEM (void)
{
   unsigned char mem_status = 0x01;

   NSSMD0 = 0;                         // enable the flash

   // send the write enable command
   SPI0DAT = WREN;                     // load the XMIT register
   while (TXBMT != 1)                  // wait until the command is moved into
   {                                   // the XMIT buffer
   }
   SPIF = 0;
   while (SPIF != 1)                   // wait until the command reaches the
   {                                   // flash
   }
   SPIF = 0;

   NSSMD0 = 1;                         // allow the WREN to execute

   NSSMD0 = 0;                         // enable the flash

   // send the chip erase instruction
   SPI0DAT = CERASE;                   // load the XMIT register
   while (TXBMT != 1)                  // wait until the command is moved into
   {                                   // the XMIT buffer
   }
   SPIF = 0;
   while (SPIF != 1)                   // wait until the command reaches the
   {                                   // flash
   }
   SPIF = 0;

   NSSMD0 = 1;                         // allow the erase to execute

   // poll on the busy bit in the flash until the erase operation is complete
   NSSMD0 = 0;                         // enable the flash
   SPI0DAT = RDSR;                     // send the read status register command
   while (TXBMT != 1)                  // wait until the SPI can accept more
   {                                   // data
   }
   while (mem_status == 0x01)
   {
      SPI0DAT = 0xFF;                  // send junk in order to receive data
      while (TXBMT != 1)               // wait until the junk data is moved
      {                                // into the XMIT buffer
      }
      SPIF = 0;
      while (SPIF != 1)                // wait until the read data is received
      {
      }
      SPIF = 0;
      mem_status = SPI0DAT & 0x01;     // check the BUSY bit
   }

   NSSMD0 = 1;                         // disable the flash
}

//-----------------------------------------------------------------------------
// Read_MEM_Init
//-----------------------------------------------------------------------------
//
// Return Value :
//   1)  char data_byte - the data byte read from memory
//                      range is positive range of character: 0 to 255
// Parameters   :
//   1)  long address - address in the 512 kB external SST Flash
//                      range is postive values up to 2^19: 0 to 524287,
//                           or, 0 to 0x7FFFF
//
// Read one byte of data from a 24-bit address in the SST Flash Memory using
// the SPI.  This function is called by Recording_Search in F411_VR.c
// and is a duplicate of Read_MEM to avoid a warning by the compiler.
//
unsigned char Read_MEM_Init (unsigned long address)
{
   ADDRESS temp_addr;
   temp_addr.ULong = address;

   NSSMD0 = 0;                         // enable the flash

   // send the read instruction
   SPI0DAT = READ;                     // load the XMIT register
   while (TXBMT != 1)                  // wait until the command is moved into
   {                                   // the XMIT buffer
   }
   SPI0DAT = temp_addr.UByte[1];       // load the high byte of the address
   while (TXBMT != 1)                  // wait until the data is moved into
   {                                   // the XMIT buffer
   }
   SPI0DAT = temp_addr.UByte[2];       // load the middle byte of the address
   while (TXBMT != 1)                  // wait until the data is moved into
   {                                   // the XMIT buffer
   }
   SPI0DAT = temp_addr.UByte[3];       // load the low byte of the address
   while (TXBMT != 1)                  // wait until the data is moved into
   {                                   // the XMIT buffer
   }
   SPI0DAT = 0xFF;                     // load junk data in order to receive
                                       // data from the flash
   while (TXBMT != 1)                  // wait until the junk data is moved
   {                                   // into the XMIT buffer
   }
   SPIF = 0;
   while (SPIF != 1)                   // wait until the read data is received
   {
   }
   SPIF = 0;

   NSSMD0 = 1;                         // disable the flash

   return SPI0DAT;
}

//-----------------------------------------------------------------------------
// ReadID_MEM
//-----------------------------------------------------------------------------
//
// Return Value :
//   1)  char data_byte - the device ID read from memory at address 0x000001
//                      (this address is specified in the SST Flash datasheet)
//                      range is positive range of character: 0 to 255
// Parameters   : None
//
// Read the part ID from the flash memory (used for debugging).
//
char ReadID_MEM (void)
{
   NSSMD0 = 0;                         // enable the flash

   SPI0DAT = READID;                   // send the read ID instruction
   while (TXBMT != 1)                  // wait until the SPI can accept more
   {                                   // data
   }
   SPI0DAT = 0x00;                     // send the device ID address
   while (TXBMT != 1)                  // wait until the SPI can accept more
   {                                   // data
   }
   SPI0DAT = 0x00;                     // send the device ID address
   while (TXBMT != 1)                  // wait until the SPI can accept more
   {                                   // data
   }
   SPI0DAT = 0x01;                     // send the device ID address
   while (TXBMT != 1)                  // wait until the SPI can accept more
   {                                   // data
   }
   SPI0DAT = 0xA5;                     // send dummy data for shift register
   while (TXBMT != 1)                  // wait until the SPI can accept more
   {                                   // data
   }
   SPIF = 0;
   while (SPIF != 1)                   // wait until the read data is received
   {
   }
   SPIF = 0;

   NSSMD0 = 1;                         // disable the flash

   return SPI0DAT;
}

//-----------------------------------------------------------------------------
// End Of File
//-----------------------------------------------------------------------------