//-----------------------------------------------------------------------------
// F411_VR_LED.c
//-----------------------------------------------------------------------------
// Copyright 2006 Silicon Laboratories, Inc.
// http://www.silabs.com
//
// Program Description:
//
// This file contains the functions that use the PWM to brighten, dim, and
// flutter the LEDs.
//
// These functions work by using a pointer (LED_DCH) to an "LED byte," which
// is just a byte in memory associated with each LED.  In F411_VR.c,
// one of the Timer ISRs updates the PCA PWM registers with all the LED bytes
// every interrupt period, so the LEDs that don't change are still updated,
// but visually nothing changes.
//
// When the timer interrupts, PCA0CPH1 is reloaded with the current value
// LED0_DC, which is changed by the functions (Dim, Brighten, and Flutter)
// based on the desired LED behavior.  By decrementing the time the LED is
// on in steps (based on the ADJ variable), the LED appears to "dim" off,
// and by incrementing the time the LED is on in steps, the LED appears to
// "brighten" slowly.
//
// The LED_DCH pointer must be pointed to the correct LEDx_DC byte BEFORE
// each of these functions is called.
//
// LED_DCH -> LED0_DC -> PCA0CPH1, where CEX1 (output from the PCA) is tied
// to LED0
//
// For example, the resulting dim LED waveform might look something like
// this, since the LEDs are ON when CEX1 = 0:
//
//                                     _              __             ___
// CEX1  _____________________________| |____________|  |___________|   |
//
//      |   1st step    |    2nd step   |   3rd step    |    4th step   |
//
// (continued)
//                  ____           _____          ______         _______
// CEX1  __________|    |_________|     |________|      |_______|       |
//
//      |   5th step    |    6th step   |    7th step   |    8th step   |
//
// (continued)
//              ________       _________      __________     ___________
// CEX1  ______|        |_____|         |____|          |___|           |
//
//      |   9th period  |  10th period  |   11th period  | 12th  period |
//
// (continued)
//          ____________   _____________________________________________
// CEX1  __|            |_|
//
//      |  13th period  |  14th period  |   15th period  | 16th  period |
//
// The LED has appeared to "dim" slowly off.
//
//
//
// NOTE: The calling function must have the same register context as the LED
// functions, so it must either have the keyword "using 0" or all "using 0"
// keywords for the LED functions need to be removed.
//
// How To Use:    See Readme.txt
//
// FID:            41X000008
// Target:         C8051F411
// Tool chain:     Keil C51 7.50 / Keil EVAL C51
//                 Silicon Laboratories IDE version 2.6
// Project Name:   F411_VR
//
// Release 1.3
//    -All changes by TP
//    -02 Feb 2006
//    -project version updated, no changes to this file
//
// Release 1.2
//    -All changes by TP
//    -21 Nov 2005
//    -project version updated, no changes to this file
//
// Release 1.1
//    -All changes by TP
//    -16 Aug 2004
//    -project version updated, no changes to this file
//
// Release 1.0
//    -Initial Revision (TP)
//    -15 AUG 2004
//

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <c8051f410.h>                 // SFR declarations

//-----------------------------------------------------------------------------
// Global Variables
//-----------------------------------------------------------------------------
unsigned char ADJ = 15;
unsigned int LED_PWM = 65535;
int LED_PWM_CHANGE = 0x0000;
unsigned char *LED_DCH;
unsigned char LED0_DC = 0x00;
unsigned char LED1_DC = 0x00;
// add another LEDx_DC variable here, if desired, and point to it with *LED_DCH
// before calling the LED functions

//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------
void Dim_LED (void);
void Brighten_LED (void);
void Flutter_LED (void);


//-----------------------------------------------------------------------------
// Dim_LED
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// Dim the LED using the PCA in 8-bit PWM mode.  The Timer0 ISR in
// F411_VR.c updates the value LED_DCH is pointing to.
//
// NOTE: This function requires that the LED_DCH pointer be "pointing" to the
// appropriate LED byte, as explained above.
//
void Dim_LED (void) using 0
{
   // retrieve the previous value of the duty cycle
   unsigned char duty_cycle = *LED_DCH;

   ADJ = 0xF1;                         // set the ADJ such that the LED will
                                       // get dimmer
   LED_PWM = 65535;                    // reset the Timer 0 interval
   LED_PWM_CHANGE = 0;                 // do not change the Timer 0 interval
   TCON |= 0x10;                       // start Timer 0

   // wait until the LED is fully off
   while (duty_cycle != 0x00)
   {
      duty_cycle = *LED_DCH;
   }

   TCON &= ~0x10;                      // stop Timer 0 (no more updates to the
                                       // PCA duty cycle)
}

//-----------------------------------------------------------------------------
// Brighten_LED
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// Brighten the LED using the PCA in 8-bit PWM mode.  The Timer0 ISR in
// F411_VR.c updates the value LED_DCH is pointing to.
//
// NOTE: This function requires that the LED_DCH pointer be "pointing" to the
// appropriate LED byte, as explained above.
//
void Brighten_LED (void) using 0
{
   // retrieve the previous value of the duty cycle
   unsigned char duty_cycle = *LED_DCH;

   ADJ = 0x0F;                         // set the ADJ such that the LED will
                                       // brighten
   LED_PWM = 65535;                    // reset the Timer 0 interval
   LED_PWM_CHANGE = 0;                 // do not change the Timer 0 interval
   TCON |= 0x10;                       // start Timer 0

   // wait until the LED is fully on
   while (duty_cycle != 0xFF)
   {
      duty_cycle = *LED_DCH;
   }

   TCON &= ~0x10;                      // stop Timer 0 (no more updates to the
                                       // PCA duty cycle)
}

//-----------------------------------------------------------------------------
// Flutter_LED
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// Cause the LED to dim on and off.  The Timer0 ISR in F411_VR.c updates
// the value LED_DCH is pointing to.
//
// NOTE: This function requires that the LED_DCH pointer be "pointing" to the
// appropriate LED byte, as explained above.
//
void Flutter_LED (void) using 0
{
   // retrieve the previous value of the duty cycle
   unsigned char duty_cycle = *LED_DCH;

   // check if the LED is currently on or off
   if (duty_cycle == 0xFF)
   {
      ADJ = 0xF1;
   }
   else
   {
      ADJ = 0x0F;
   }
   LED_PWM = 65535;                    // reset the Timer 0 interval
   LED_PWM_CHANGE = -200;              // change the Timer 0 interval each
                                       // interrupt cycle so the LED has a
                                       // "fluttering" effect
   TCON |= 0x10;                       // start Timer 0

   // Wait for a flutter cycle to finish
   while (LED_PWM > 17000)
   {
   }

   TCON &= ~0x10;                      // stop Timer 0 (no more updates to the
                                       // PCA duty cycle)
}

//-----------------------------------------------------------------------------
// End Of File
//-----------------------------------------------------------------------------