//-----------------------------------------------------------------------------
// F411_VR_DPCM.c
//-----------------------------------------------------------------------------
// Copyright 2006 Silicon Laboratories, Inc.
// http://www.silabs.com
//
// Program Description:
//
// This file contains the DPCM encoding and decoding functions.
//
// NOTE: For another reference for DPCM, please see Chipcon's app note an026.
//
// NOTE: The calling function must have the same register context as the DCPM
// functions, so it must either have the keyword "using 2" or all "using 2"
// keywords for the DPCM functions need to be removed
//
// How To Use:    See Readme.txt
//
// FID:            41X000006
// Target:         C8051F411
// Tool chain:     Keil C51 7.50 / Keil EVAL C51
//                 Silicon Laboratories IDE version 2.6
// Project Name:   F411_VR
//
// Release 1.3
//    -All changes by TP
//    -02 Feb 2006
//    -project version updated, no changes to this file
//
// Release 1.2
//    -All changes by TP
//    -21 Nov 2005
//    -expanded the 4-bit codes to 6 bits for the 12-bit ADC
//
// Release 1.1
//    -All changes by TP
//    -16 Aug 2004
//    -project version updated, no changes to this file
//
// Release 1.0
//    -Initial Revision (TP)
//    -15 AUG 2004
//

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <c8051f410.h>                 // SFR declarations

//-----------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------

// 12-bit quantization codes (6 bits, so 64 codes total = 31 positive, 31
// negative, and 2 zeroes)
#define quant1       1
#define quant2       2
#define quant3       4
#define quant4       7
#define quant5       11
#define quant6       16
#define quant7       22
#define quant8       29
#define quant9       37
#define quant10      46
#define quant11      56
#define quant12      67
#define quant13      79
#define quant14      92
#define quant15      106
#define quant16      130
#define quant17      146
#define quant18      163
#define quant19      181
#define quant20      200
#define quant21      220
#define quant22      241
#define quant23      263
#define quant24      286
#define quant25      310
#define quant26      335
#define quant27      361
#define quant28      388
#define quant29      416
#define quant30      512
#define quant31      1024

// the mapping from quantization values to dpcm codes (array index)
xdata short Q_VALUES[64] = {0,               // 0
                           -quant31,         // 1
                           -quant30,         // 2
                           -quant29,         // 3
                           -quant28,         // 4
                           -quant27,         // 5
                           -quant26,         // 6
                           -quant25,         // 7
                           -quant24,         // 8
                           -quant23,         // 9
                           -quant22,         // 10
                           -quant21,         // 11
                           -quant20,         // 12
                           -quant19,         // 13
                           -quant18,         // 14
                           -quant17,         // 15
                           -quant16,         // 16 negative middle
                           -quant15,         // 17
                           -quant14,         // 18
                           -quant13,         // 19
                           -quant12,         // 20
                           -quant11,         // 21
                           -quant10,         // 22
                           -quant9,          // 23
                           -quant8,          // 24
                           -quant7,          // 25
                           -quant6,          // 26
                           -quant5,          // 27
                           -quant4,          // 28
                           -quant3,          // 29
                           -quant2,          // 30
                           -quant1,          // 31
                           0,                // 32
                           quant1,           // 33
                           quant2,           // 34
                           quant3,           // 35
                           quant4,           // 36
                           quant5,           // 37
                           quant6,           // 38
                           quant7,           // 39
                           quant8,           // 40
                           quant9,           // 41
                           quant10,          // 42
                           quant11,          // 43
                           quant12,          // 44
                           quant13,          // 45
                           quant14,          // 46
                           quant15,          // 47
                           quant16,          // 48 positive middle
                           quant17,          // 49
                           quant18,          // 50
                           quant19,          // 51
                           quant20,          // 52
                           quant21,          // 53
                           quant22,          // 54
                           quant23,          // 55
                           quant24,          // 56
                           quant25,          // 57
                           quant26,          // 58
                           quant27,          // 59
                           quant28,          // 60
                           quant29,          // 61
                           quant30,          // 62
                           quant31};         // 63

//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------
unsigned char DPCM_Encode (short sample_diff);
short DPCM_Decode (unsigned char dpcm_code);

//-----------------------------------------------------------------------------
// DPCM_Encode
//-----------------------------------------------------------------------------
//
// Return Value :
//   1)  char dpcm_code - the 6-bit quantized DPCM code
//                      range is positive range of 6-bit value: 0 to 63
// Parameters   :
//   1)  short sample_diff - the difference between the predicted value and
//                      the sample from the ADC
//                      range is: -4096 to 4095 (difference of 12-bit values)
//
// Encode the sample using DPCM compression.
//
// The coding uses the following scheme (0 is unused) for an 8-bit sample:
//
// code:      1   2   3   4   5   6   7   8   9   10  11  12  13  14  15
// q value: -64 -32 -16  -8  -4  -2  -1   0   1    2   4   8  16  32  64
//
// The difference will be rounded down if positive and rounded up if
// negative (i.e. 41 => 32, and -41 => -32).
//
// NOTE: the calling function must have the same register context, so it must
// either have the keyword "using 2" or all "using 2" keywords need to be
// removed
//
unsigned char DPCM_Encode (short sample_diff) using 2
{
   short sample_diff_us;
   unsigned char dpcm_code;

   // determine if the difference is positive or negative
   if (sample_diff < 0)
   {
      sample_diff_us = -sample_diff;   // use the absolute value
   }
   else
   {
      sample_diff_us = sample_diff;
   }

   // narrow down which bits need to be set to use the proper quantization code
   // using a binary search algorithm (divide in halves)
   // the sign of the difference no longer matters

   // first tier
   if (sample_diff_us >= quant16)
   {
      // second tier
      if (sample_diff_us >= quant24)
      {
         // third tier
         if (sample_diff_us >= quant28)
         {
            // fourth tier
            if (sample_diff_us >= quant30)
            {
               // fifth tier
               if (sample_diff_us >= quant31)
               {
                  dpcm_code = 63;
               }
               // fifth tier
               else
               {
                  dpcm_code = 62;
               }
            }
            // fourth tier
            else
            {
               // fifth tier
               if (sample_diff_us >= quant29)
               {
                  dpcm_code = 61;
               }
               // fifth tier
               else
               {
                  dpcm_code = 60;
               }
            }
         }
         // third tier
         else
         {
            // fourth tier
            if (sample_diff_us >= quant26)
            {
               // fifth tier
               if (sample_diff_us >= quant27)
               {
                  dpcm_code = 59;
               }
               // fifth tier
               else
               {
                  dpcm_code = 58;
               }
            }
            // fourth tier
            else
            {
               // fifth tier
               if (sample_diff_us >= quant25)
               {
                  dpcm_code = 57;
               }
               // fifth tier
               else
               {
                  dpcm_code = 56;
               }
            }
         }
      }
      // second tier
      else
      {
         // third tier
         if (sample_diff_us >= quant20)
         {
            // fourth tier
            if (sample_diff_us >= quant22)
            {
               // fifth tier
               if (sample_diff_us >= quant23)
               {
                  dpcm_code = 55;
               }
               // fifth tier
               else
               {
                  dpcm_code = 54;
               }
            }
            // fourth tier
            else
            {
               // fifth tier
               if (sample_diff_us >= quant21)
               {
                  dpcm_code = 53;
               }
               // fifth tier
               else
               {
                  dpcm_code = 52;
               }
            }
         }
         // third tier
         else
         {
            // fourth tier
            if (sample_diff_us >= quant18)
            {
               // fifth tier
               if (sample_diff_us >= quant19)
               {
                  dpcm_code = 51;
               }
               // fifth tier
               else
               {
                  dpcm_code = 50;
               }
            }
            // fourth tier
            else
            {
               // fifth tier
               if (sample_diff_us >= quant17)
               {
                  dpcm_code = 49;
               }
               // fifth tier
               else
               {
                  dpcm_code = 48;
               }
            }
         }
      }
   }
   // first tier
   else
   {
      // second tier
      if (sample_diff_us >= quant8)
      {
         // third tier
         if (sample_diff_us >= quant12)
         {
            // fourth tier
            if (sample_diff_us >= quant14)
            {
               // fifth tier
               if (sample_diff_us >= quant15)
               {
                  dpcm_code = 47;
               }
               // fifth tier
               else
               {
                  dpcm_code = 46;
               }
            }
            // fourth tier
            else
            {
               // fifth tier
               if (sample_diff_us >= quant13)
               {
                  dpcm_code = 45;
               }
               // fifth tier
               else
               {
                  dpcm_code = 44;
               }
            }
         }
         // third tier
         else
         {
            // fourth tier
            if (sample_diff_us >= quant10)
            {
               // fifth tier
               if (sample_diff_us >= quant11)
               {
                  dpcm_code = 43;
               }
               // fifth tier
               else
               {
                  dpcm_code = 42;
               }
            }
            // fourth tier
            else
            {
               // fifth tier
               if (sample_diff_us >= quant9)
               {
                  dpcm_code = 41;
               }
               // fifth tier
               else
               {
                  dpcm_code = 40;
               }
            }
         }
      }
      // second tier
      else
      {
         // third tier
         if (sample_diff_us >= quant4)
         {
            // fourth tier
            if (sample_diff_us >= quant6)
            {
               // fifth tier
               if (sample_diff_us >= quant7)
               {
                  dpcm_code = 39;
               }
               // fifth tier
               else
               {
                  dpcm_code = 38;
               }
            }
            // fourth tier
            else
            {
               // fifth tier
               if (sample_diff_us >= quant5)
               {
                  dpcm_code = 37;
               }
               // fifth tier
               else
               {
                  dpcm_code = 36;
               }
            }
         }
         // third tier
         else
         {
            // fourth tier
            if (sample_diff_us >= quant2)
            {
               // fifth tier
               if (sample_diff_us >= quant3)
               {
                  dpcm_code = 35;
               }
               // fifth tier
               else
               {
                  dpcm_code = 34;
               }
            }
            // fourth tier
            else
            {
               // fifth tier
               if (sample_diff_us >= quant1)
               {
                  dpcm_code = 33;
               }
               // fifth tier
               else
               {
                  dpcm_code = 32;
               }
            }
         }
      }
   }

   // convert the DPCM code to its 2's compliment if the original sample
   // difference was negative
   // For example, 41 (101001), which represents a difference of 60, 2's
   // complimented becomes 23 (010111), which represents a difference of -60
   if (sample_diff < 0)
   {
      dpcm_code = ~dpcm_code + 1;     // use the 2's compliment of the dpcm
                                      // code
      dpcm_code &= 0x3F;              // use only the 6 LSBs for the dpcm code
   }

   return dpcm_code;
}

//-----------------------------------------------------------------------------
// DPCM_Decode
//-----------------------------------------------------------------------------
//
// Return Value :
//   1)  short predicted_value - the signed and quantized difference between
//                      the predicted_value and the ADC sample, which is used
//                      create the predicted_value for the next DPCM cycle
//                      range is: -4096 to 4095 (difference of 12-bit values)
// Parameters   :
//   1)  char dpcm_code - the 6-bit code indicating the quantized difference
//                      between the old_prediction and the current sample value
//                      range is positive range of 6-bit value: 0 to 63
//
// Decode the DPCM code to a signed difference between the current predicted
// value and the next.
//
// NOTE: the calling function must have the same register context, so it must
// either have the keyword "using 2" or all "using 2" keywords need to be
// removed
short DPCM_Decode (unsigned char dpcm_code) using 2
{
   return Q_VALUES[dpcm_code];
}

//-----------------------------------------------------------------------------
// End Of File
//-----------------------------------------------------------------------------