//-----------------------------------------------------------------------------
// F411_VR.c
//-----------------------------------------------------------------------------
// Copyright 2006 Silicon Laboratories, Inc.
// http://www.silabs.com
//
// Program Description:
//
// This program uses the DPCM functions to encode voice samples and saves them
// to flash memory.  This program also interfaces with a speaker or headphones
// to play back the recorded voice.
//
// How To Use:    See Readme.txt
//
// FID:            41X000005
// Target:         C8051F411
// Tool chain:     Keil C51 7.50 / Keil EVAL C51
//                 Silicon Laboratories IDE version 2.6
// Project Name:   F411_VR
//
// Release 1.3
//    -All changes by TP
//    -02 Feb 2006
//    -minor changes in comments
//
// Release 1.2
//    -All changes by TP
//    -21 Nov 2005
//    -Revised for a 2-button version of the board with
//       volume wheel.
//
// Release 1.1
//    -All changes by TP
//    -16 Aug 2004
//    -project version updated, no changes to this file
//
// Release 1.0
//    -Initial Revision (TP)
//    -15 AUG 2004
//


//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <c8051f410.h>                 // SFR declarations
#include "F411_VR_DPCM.h"              // contains DPCM functions
#include "F411_VR_SSTFlash.h"          // contains functions to write to the
                                       // serial SST external 512 kb Flash
#include "F411_VR_LED.h"               // contains functions to control the
                                       // intensity of the LEDs

//-----------------------------------------------------------------------------
// 16-bit SFR Definitions for 'F411
//-----------------------------------------------------------------------------

// SFR16 Defintions (Timers, ADC, and DAC)
sfr16 TMR2RL = 0xCA;                   // Timer 2 Reload address
sfr16 TMR2 = 0xCC;                     // Timer 2 Counter address
sfr16 TMR3RL = 0x92;                   // Timer 3 Reload address
sfr16 TMR3 = 0x94;                     // Timer 3 Counter address
sfr16 ADC0DAT = 0xBD;                  // ADC 16-bit address
sfr16 IDA0DAT = 0x96;                  // IDAC 16-bit address

//-----------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------

// General Constants
#define SYSCLK 6125000                 // system clock in Hz (24.5 MHz / 4)

// T1 runs at SYSCLK / 48
#define POLLING 1992                   // poll the switches at 64 Hz
#define PRCHANGE 20000                 // wait approx. 150ms after a button is
                                       // pressed to "debounce" the switches

#define SAMP_FREQ 764                  // use 8kHz sampling for both the ADC
                                       // and DAC

#define MAX_MEM_ADDR 0x0007FFFF        // 512K = 2^19 address bits
#define NEAR_END_ADDR 0x00065F55       // for every 4 samples, 3 bytes are
                                       // written to memory, so 10.67 kHz
                                       // ((8 kHz * 4)/3) writing time = 106666
                                       // addresses every 10 seconds, so give
                                       // about 10 seconds of warning

#define mid_range 2048                 // middle value of 12-bit ADC and DAC

// System States
#define IDLE 0x00                      // indicates no current action
#define RECORDING 0x01                 // indicates the device is recording
#define PLAYING 0x02                   // indicates the device is playing
#define END_MEM 0x04                   // flag used if the end of memory is
                                       // reached
#define ERASED 0x08                    // flag used if memory is erased

// Port Pin Definitions
sbit REC_PLAY = P1^7;
sbit ERASE = P1^6;
sbit TRANS = P1^3;
sbit LED0 = P2^1;
sbit LED1 = P2^0;

sbit SCK = P0^4;
sbit MISO = P0^5;
sbit MOSI = P0^6;
sbit NSS = P0^7;

//-----------------------------------------------------------------------------
// Global VARIABLES
//-----------------------------------------------------------------------------

unsigned char system_state = IDLE;     // start in idle mode
                                       // bit 3 of the system_state indicates
                                       // if the end of memory has been
                                       // reached
                                       // bit 4 of the system_state indicates
                                       // if the memory has been erased since
                                       // the last action (1 = erased)


// Ending address of the recording in memory
unsigned long rec_end_addr = 0x00000000;

// flags to communicate between the T1 ISR and the ADC/DAC ISRs for various
// termination events
bit ADC_STOP_FLAG = 0;
bit MEM_END_NEAR_FLAG = 0;
bit MEM_END_FLAG = 0;
bit REC_END_FLAG = 0;
bit DAC_STOP_FLAG = 0;
bit ENTER_SUSPEND = 0;

//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------
// System and peripheral initialization functions
void System_Init (void);
void VDDMon_Init (void);
void Port_Init (void);
void ADC0_Init (void);
void DAC0_Init (void);
void PCA_Init (void);
void SPI0_Init (void);
void RTC_Init (void);
void Timer0_Init (int period);
void Timer1_Init (int period);
void Timer2_Init (int period);
void Timer3_Init (int period);
void Recording_Search (void);

// Interrupt service routines
void Timer0_ISR (void);                // LED updates
void Timer1_ISR (void);                // Switch polling
void ADC0_ISR (void);                  // Recording
void Timer3_ISR (void);                // Playback

//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------
void main (void)
{
   unsigned char i;

   // Watchdog timer disabled in VR_STARTUP.A51

   System_Init ();                     // Initialize system clock
   Port_Init ();                       // Initialize crossbar and GPIO
   ADC0_Init ();                       // Initialize ADC0 (microphone)
   DAC0_Init ();                       // Initialize DAC0 (speaker)
   PCA_Init ();                        // Initialize the PCA for 8-bit PWM
                                       // in modules 0, 1, and 2
   SPI0_Init();                        // Initialize the interface to the flash

   RTC_Init ();                        // Stop the RTC from causing a wake-up
                                       // from suspend

   Timer0_Init (LED_PWM);              // Initialize timer 0 to provide a
                                       // 76 Hz interrupt rate for the LEDs

   Timer1_Init (POLLING);              // Initialize timer 1 to provide a
                                       // 64 Hz interrupt rate for the switches

   Timer2_Init (SAMP_FREQ);            // Initialize the timer to provide an
                                       // 8KHz interrupt rate for sampling

   Timer3_Init (SAMP_FREQ);            // Initialize the timer to provide an
                                       // 8KHz interrupt rate for sampling

   EA = 1;                             // enable global interrupts

   SSTFlash_Init ();                   // disable the write protection in the
                                       // external SST flash memory
   Recording_Search ();                // search for a recording already
                                       // present in memory

   TR1 = 1;                            // start polling the switches

   // loop forever
   while (1)
   {
      if (ENTER_SUSPEND == 1)          // check if no interaction has occurred
      {                                // for some time
         // disable everything to save the most power and set everything to a
         // dormant state
         ENTER_SUSPEND = 0;

         TR1 = 0;                      // stop polling the switches

         EA = 0;                       // disable all interrupts

         XBR1 = 0x40;                  // disable the PCA
         XBR0 = 0x00;                  // disable the SPI

         SCK = 0;                      // drive the SPI pins low so the
         MISO = 0;                     // external Flash won't attempt to draw
         MOSI = 0;                     // current while unpowered

         IDA0CN &= ~0x80;              // disable DAC0
         REF0CN &= ~0x01;              // disable VREF

         LED0 = 1;                     // turn the LEDs off
         LED1 = 1;

         TRANS = 1;                    // turn off the external circuitry

         RSTSRC = 0x00;                // disable missing clock detector
         VDM0CN &= ~0x80;              // disable the VDD Monitor

         OSCICN |= 0x20;               // enter suspend mode and wait
                                       // until a port match event occurs

         // re-enable and reinitialize the system
         VDDMon_Init ();

         TRANS = 0;                    // turn on the external circuitry

         REF0CN |= 0x01;               // re-enable VREF
         IDA0CN |= 0x80;               // re-enable DAC0

         XBR0 = 0x02;                  // re-enable SPI
         XBR1 = 0x42;                  // re-enable PCA0_0 and PCA0_1

         // wait 10us until the Flash is ready to receive writes and reads
         for (i = 0; i < 64; i++);

         SSTFlash_Init ();             // re-initialize the SST flash

         EA = 1;                       // enable global interrupts

         // wait until the button that woke the system is released
         while ((REC_PLAY == 0) || (ERASE == 0));

         TR1 = 1;                      // begin polling the buttons again
      }
   }
}

/////////////////////////// INITIALIZATION ROUTINES ///////////////////////////

//-----------------------------------------------------------------------------
// System_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// This routine initializes the system clock to use the internal 24.5MHz / 4
// oscillator as its clock source and enables the missing clock detector reset.
// Additionally, this routine sets up VREF, the internal regulator, and the
// VDD monitor.
//
void System_Init (void)
{
   OSCICN = 0x85;                      // configure internal oscillator
   RSTSRC = 0x04;                      // enable missing clock detector

   REF0CN = 0x01;                      // set up and enable VREF pin

   REG0CN = 0x10;                      // set up and enable 2.5V VDD from the
                                       // internal regulator

   VDDMon_Init ();                     // initialize the VDD Monitor
}

//-----------------------------------------------------------------------------
// VDDMon_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// This routine initializes the VDD Monitor and enables it as a reset source.
//
void VDDMon_Init (void)
{
   char i;

   VDM0CN = 0x80;                      // enable the VDD monitor
   for (i = 0; i < 80; i++);           // wait for the monitor to stabilize
   RSTSRC = 0x06;                      // enable missing clock detector and
                                       // VDD monitor as reset sources
}

//-----------------------------------------------------------------------------
// PORT_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// P0.0 = DAC0 (analog, skip)
// P0.1-3 = unused (skip)
// P0.4-7 = SPI interface (digital, do not skip)
// P1.0-1 = unused (skip)
// P1.2 = VREF (analog, skip)
// P1.3 = analog power-on transistor (digital, skip)
// P1.4 = unused (skip)
// P1.5 = ADC0 (analog, skip)
// P1.6-7 = REC_PLAY and ERASE switches (digital, skip)
// P2.0-1 = LED PCA outputs (digital, do not skip)
//
void Port_Init (void)
{
   P0MDIN = 0xFE;                      // make switch and SPI pins digital
   P0MDOUT = 0xD0;                     // make SPI pins push-pull
   P1MDIN = 0xC8;                      // make trans and switches digital
   P1MDOUT = 0x08;                     // make trans pin push-pull
   P2MDIN = 0x03;                      // make PCA pins digital
   P2MDOUT = 0x03;                     // make PCA pins push-pull
   P0SKIP = 0x0F;                      // skip pins not belonging to SPI
   P1SKIP = 0xFF;                      // skip all P1 pins
   P2SKIP = 0xFC;                      // skip pins not belonging to LED PCA

   XBR0 = 0x02;                        // enable SPI
   XBR1 = 0x42;                        // enable PCA0_0 and PCA0_1

   TRANS = 0;                          // turn on the power to all analog
                                       // components

   P0MAT = 0x00;                       // the buttons will go low when pressed,
   P1MAT = 0xC0;                       // causing the port match event
   P0MASK = 0x00;                      // mask off all P0 and P1 pins except
   P1MASK = 0xC0;                      // the switches
   EIE2 = 0x00;                        // disable the port match interrupt
                                       // (not required to wake up the core)
}

//-----------------------------------------------------------------------------
// ADC0_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// Configure ADC0 to update with a Timer 2 overflow using P1.5 as its positive
// input in post-tracking mode, enable burst mode, and use a repeat factor of
// 16.
//
void ADC0_Init (void)
{
   ADC0CN = 0x43;                      // ADC in low-power burst mode, use T2
                                       // overflow, right justify
   ADC0MX = 0x0D;                      // use P1.5 as the positive reference
   // set the ADC conversion about 5 MHz and use a repeat factor
   // of 16
   ADC0CF = (4 << 3) | (3 << 1);
   ADC0TK = 0xF4;                      // use post-tracking mode
   EIE1 |= 0x08;                       // enable the ADC conversion complete
                                       // interrupt
   EIP1 |= 0x08;                       // set the ADC convertion complete
                                       // interrupt to high priority
}

//-----------------------------------------------------------------------------
// DAC0_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// Configure DAC0 to be right justified, update with a Timer 3 overflow, and
// use a full-scale 2 mA output current.
//
void DAC0_Init (void)
{
   IDA0CN = 0x70;                      // set the IDAC to update on a write
                                       // to IDA0DAT (initially only)
   IDA0CN |= 0x00;                     // set the IDAC to use a 0.25 mA current.
   IDA0CN |= 0x04;                     // set the IDAC to be right-justified
   IDA0CN |= 0x80;                     // enable the IDAC

   IDA0L = 0x00;                       // initialize the IDAC to be mid-scale
   IDA0H = 0x08;

   IDA0CN &= ~0x70;                    // set the IDAC to update on T3 overflow,
   IDA0CN |= 0x33;                     // and use a 2 mA full-scale current
}

//-----------------------------------------------------------------------------
// PCA_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// Configure PCA0 modules 0 and 1 to 8-bit PWM mode using the system clock.
//
void PCA_Init (void)
{
   PCA0MD = 0x88;                      // set PCA to use system clock, disable
                                       // idle mode

   // PCA0 (for LED1)
   PCA0CPM0 = 0x42;                    // set PCA0 for 8-bit PWM mode
   PCA0CPH0 = 0x00;                    // set LED to off originally

   // PCA1 (for LED0)
   PCA0CPM1 = 0x42;                    // set PCA1 for 8-bit PWM mode
   PCA0CPH1 = 0x00;                    // set LED to off originally

   // add another PCA module for another LED here, if desired

   PCA0CN = 0x40;                      // turn on the PCA timer/counter
}

//-----------------------------------------------------------------------------
// SPI0_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// Configure the SPI to run in 4-wire master mode at SYSCLK / 4 (1.53 MHz)
// using clock phase 0 and clock polarity 0 to interface with the SST Flash
// memory.
//
void SPI0_Init (void)
{
   SPI0CFG = 0x40;                     // set the master mode, polarity and
                                       // phase
   // set the SPI frequency to SYSCLK / 2*(1+1) = SYSCLK / 4
   SPI0CKR = 0x01;
   SPI0CN = 0x0C;                      // clear flags, turn off NSS
                                       // set the 4-wire mode
   SPIEN = 1;                          // enable the SPI
}

//-----------------------------------------------------------------------------
// Timer0_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   :
//   1)  int period - number of timer counts to generate the desired period
//                    range is postive range of integer: 0 to 32767
//
// Configure Timer0 to 16-bit mode.  Timer0 is used to control the load
// time of the PCA PCA0CPHn registers, which changes the PWM intensity of the
// LEDs.
//
// The input parameter can be calculated as follows:
//   (Oscillator (Hz) / 4) / Desired_Freq (Hz) = Timer Ticks
//
void Timer0_Init (int period)
{
   TMOD |= 0x01;                       // set Timer 0 to mode 1 (16 bit)
   CKCON |= 0x04;                      // use the system clock
   ET0 = 1;                            // enable Timer 0 interrupts
   PT0 = 1;                            // set Timer 0 interrupts to high
                                       // priority (has to interrupt T1)

   TL0 = (-period) & 0x00FF;           // set the desired period
   TH0 = ((-period) & 0xFF00) >> 8;

   TR0 = 0;                            // keep Timer 0 off (LED
                                       // functions will turn it on)
}

//-----------------------------------------------------------------------------
// Timer1_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   :
//   1)  int period - number of timer counts to generate the desired period
//                    range is postive range of integer: 0 to 32767
//
// Configure Timer1 to 16-bit mode.  Timer1 controls the switch polling.
//
// To calculate:
//   (Oscillator (Hz) / 4) / 48 / Desired_Freq (Hz) = Timer Ticks
//
// NOTE -  the extra 48 in this equation is present because of the settings
// in CKCON.
//
void Timer1_Init (int period)
{
   TMOD |= 0x10;                       // set Timer 1 to mode 1 (16 bit)
   CKCON |= 0x02;                      // use the system clock / 48
   ET1 = 1;                            // enable Timer 1 interrupts

   TL1 = (-period) & 0x00FF;           // set the desired period
   TH1 = ((-period) & 0xFF00) >> 8;

   TR1 = 0;                            // keep Timer 1 off until needed
}

//-----------------------------------------------------------------------------
// Timer2_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   :
//   1)  int period - number of timer counts to generate the desired period
//                    range is postive range of integer: 0 to 32767
//
// Configure Timer2 to 16-bit auto reload mode.  Timer2 controls the ADC0
// start-of-conversion rate.
//
// To calculate:
//   (Oscillator (Hz) / 4) / Desired_Freq (Hz) = Timer Ticks
//
void Timer2_Init (int period)
{
   CKCON |= 0x10;                      // use the system clock
   TMR2CN = 0x00;                      // 16-bit auto-reload mode
   ET2 = 0;                            // disable T2 interrupts (use ADC
                                       // conversion complete interrupt)

   TMR2RL = -period;                   // set the desired period

   TMR2 = -period;                     // initialize the timer

   TR2 = 0;                            // keep Timer 2 off until the RECORD
                                       // function is used
}

//-----------------------------------------------------------------------------
// Timer3_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   :
//   1)  int period - number of timer counts to generate the desired period
//                    range is postive range of integer: 0 to 32767
//
// Configure Timer3 to 16-bit auto reload mode.  Timer3 controls the DAC output
// rate.
//
// To calculate:
//   (Oscillator (Hz) / 4) / Desired_Freq (Hz) = Timer Ticks
//
void Timer3_Init (int period)
{
   CKCON |= 0x40;                      // use the system clock
   TMR3CN = 0x00;                      // 16-bit auto-reload mode
   EIE1 |= 0x80;                       // enable Timer 3 interrupts
   EIP1 |= 0x80;                       // set Timer 3 interrupts to high
                                       // priority

   TMR3RL = -period;                   // set the desired period

   TMR3 = -period;                     // initialize the timer

   TMR3CN = 0x00;                      // keep Timer 3 off until the PLAY
                                       // function is used
}

//-----------------------------------------------------------------------------
// RTC_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// Enable the RTC so it doesn't cause a wake-up from suspend mode.
//
void RTC_Init (void)
{
   RTC0KEY = 0xA5;                     // unlock the RTC interface
   RTC0KEY = 0xF1;
   RTC0ADR = 0x06;                     // point to RTC0CN
   RTC0DAT = 0x80;                     // enable the RTC
}

//-----------------------------------------------------------------------------
// Recording_Search
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// Search for a recording already residing in memory on power-up and set the
// rec_end_addr accordingly.
//
void Recording_Search(void)
{
   unsigned long address = 0x00000000;
   bit end_flag = 0;

   // indicate to the user that the microcontroller is not ready to record
   // or playback
   LED_DCH = &LED0_DC;
   Brighten_LED ();
   LED_DCH = &LED1_DC;
   Brighten_LED ();

   // search through the SST flash until a series of 0xFF is found, indicating
   // cleared memory
   while (end_flag != 1)
   {
      if (Read_MEM_Init (address) == 0xFF)
      {
         // double-check that the 0xFF found is not just a data byte of 0xFF
         if (Read_MEM_Init (address+10) == 0xFF)
         {
            if (Read_MEM_Init (address+40) == 0xFF)
            {
               end_flag = 1;
            }
         }
      }

      address++;

      if (address == MAX_MEM_ADDR)
      {
         end_flag = 1;
      }
   }

   rec_end_addr = address-1;           // set the recording ending address

   // turn off the LEDs so the user knows the recording search has ended
   LED_DCH = &LED0_DC;
   Dim_LED ();
   LED_DCH = &LED1_DC;
   Dim_LED ();
}

////////////////////////// INTERRUPT SERVICE ROUTINES /////////////////////////

//-----------------------------------------------------------------------------
// Timer0_ISR
//-----------------------------------------------------------------------------
//
// Handle the 76Hz (13ms) Timer 0 interrupt.
//
// Timer 0 controls the rate at which the microcontroller changes the duty
// cycle of the PCA controlling the LEDs
//
// The LEDs are updated periodically, even if the LED PWM hasn't changed.
// By using the pointer (which is set before calling the LED functions) and
// updating all LEDs in the ISR every time, the same functions can be used for
// any number of LEDs.  To add an LED, simply set-up another PCA channel,
// point to that LED before calling the LED functions, and update the LED in
// the ISR.
//
void Timer0_ISR (void) interrupt 1 using 1
{
   *LED_DCH += ADJ;                    // calculate the new duty cycle based
                                       // on the values set by the LED
                                       // functions

   PCA0CPH1 = LED0_DC;                 // load all LEDs with the possibly
   PCA0CPH0 = LED1_DC;                 // updated value
   // add another LED update here, if desired

   TL0 = (-LED_PWM) & 0x00FF;          // wait the time specified by the
   TH0 = ((-LED_PWM) & 0xFF00) >> 8;   // calling LED function

   LED_PWM += LED_PWM_CHANGE;          // change the interrupt rate, if
                                       // necessary
}

//-----------------------------------------------------------------------------
// Timer1_ISR
//-----------------------------------------------------------------------------
//
// Handle the 64 Hz (15.63 ms) Timer 1 interrupt.
//
// Timer 1 controls the rate at which the microcontroller checks the switches
// for activity while in full power mode.
//
// for RECORD - press and hold REC_PLAY button, release stops recording
// for PLAYBACK - press and release REC_PLAY button, press and release again
//                  to stop
//
void Timer1_ISR (void) interrupt 3 using 0
{
   // interrupt again in 15.63 ms, unless a switch is pressed
   unsigned short reload_value = POLLING;

   static unsigned char record_counter = 0;
   static unsigned short suspend_counter = 0;

   bit switch_pressed_flag = 0;

   // REC_PLAY button pressed
   if (REC_PLAY == 0)
   {
      switch_pressed_flag = 1;         // record the user interaction

      // check if the recording time ran out, and stop any interaction
      // from the switch until the switch is released and pressed again
      if ((system_state & END_MEM) != END_MEM)
      {
         // the REC_PLAY button must be pressed and held for a period of time
         // in order to start the RECORD function
         record_counter++;

         // check if the REC_PLAY button was held down long enough to begin
         // recording (7 x 150 ms = 1.5 seconds)
         // ignore the ERASED and END_MEM state bits, check if the system is
         // idle and can start recording
         if ((record_counter > 7) && ((system_state & 0x03) == IDLE))
         {
            TR2 = 1;                   // turn on the RECORD timer
            system_state |= RECORDING; // start recording

            LED_DCH = &LED0_DC;        // point to the record LED's duty cycle
                                       // address
            Brighten_LED();            // ramp on the record LED

            record_counter = 0;        // reset the counter

            reload_value = PRCHANGE*2; // give a longer time period to check
                                       // the button (effectively debouncing)
         }

         // check if the recording time is running out (button must be held
         // to continue recording)
         if (TR2 == 1)
         {
            if (MEM_END_NEAR_FLAG == 1)
            {
               LED_DCH = &LED0_DC;
               Flutter_LED ();         // indicate to the user that time is
                                       // almost out
            }
         }
         else
         {
            // check if end of the memory has been reached
            if (MEM_END_FLAG == 1)
            {
               // stop recording
               system_state = IDLE | END_MEM;   // indicate that the end of
                                                // memory was reached
               MEM_END_FLAG = 0;

               LED_DCH = &LED0_DC;     // point to the record LED's duty cycle
                                       // address
               Dim_LED ();             // dim off the record LED
            }
         }
      }
   }
   else
   {
      // check if the switch was pressed, but not long enough to start
      // recording
      if (record_counter > 0)
      {
         switch_pressed_flag = 1;      // record the user interaction

         // the system is currently playing - stop playing
         // ignore the ERASED and END_MEM state bits
         if ((system_state & 0x03) == PLAYING)
         {
            system_state &= ~PLAYING;  // clear the PLAYING state bit
            DAC_STOP_FLAG = 1;
            IDA0DAT = 0x0800;

            LED_DCH = &LED1_DC;        // point to the play LED's duty cycle
                                       // address
            Dim_LED ();                // dim off the play LED
         }
         else
         {
            // the system is idle - start playing
            // ignore the ERASED and END_MEM state bits
            if ((system_state & 0x03) == IDLE)
            {
               system_state |= PLAYING;
               TMR3CN = 0x04;          // start the timer controlling the DAC
               REC_END_FLAG = 0;       // reset the "end of recording" flag
               DAC_STOP_FLAG = 0;

               LED_DCH = &LED1_DC;     // point to the play LED's duty cycle
                                       // address
               Brighten_LED ();        // ramp on the play LED
            }
         }

         record_counter = 0;           // switch-press registered, reset

      }
      // the REC_PLAY switch was not pressed
      else
      {
         // clear the END_MEM recording flag after the ADC ISR has turned off
         // the ADC
         if ((system_state & END_MEM) == END_MEM)
         {
            system_state &= ~RECORDING;
         }

         // the system is currently recording - stop recording
         if (system_state == RECORDING)
         {
            system_state &= ~RECORDING;
            ADC_STOP_FLAG = 1;         // notify the ADC to stop recording

            MEM_END_NEAR_FLAG = 0;     // clear all flags
            MEM_END_FLAG = 0;

            LED_DCH = &LED0_DC;        // point to the record LED's duty cycle
                                       // address
            Dim_LED ();                // dim off the record LED
         }

         // check if the playback has reached the end of the recording
         if (REC_END_FLAG == 1)
         {
            // stop playing
            system_state &= ~PLAYING;
            REC_END_FLAG = 0;

            LED_DCH = &LED1_DC;        // point to the play LED's duty cycle
                                       // address
            Dim_LED ();                // dim off the play LED
         }
      }
   }

   // ERASE button pressed
   if (ERASE == 0)
   {
      // do nothing if the device is currently recording or playing
      // ignore the ERASED and END_MEM bits
      if ((system_state & 0x03) == IDLE)
      {
         // Indicate to the user that the microcontroller is busy
         LED_DCH = &LED1_DC;
         Brighten_LED ();
         LED_DCH = &LED0_DC;
         Brighten_LED ();

         rec_end_addr = 0x00000000;    // reset the counter
         system_state |= ERASED;       // set the erase bit
         Erase_MEM ();                 // erase the external SST Flash

         LED_DCH = &LED1_DC;
         Dim_LED ();
         LED_DCH = &LED0_DC;
         Dim_LED ();
      }

      switch_pressed_flag = 1;         // record the user interaction
   }

   if (switch_pressed_flag == 0)
   {
      // check if the recorder is sitting and idle
      // ignore the ERASED and END_MEM bits
      if ((system_state & 0x03) == IDLE)
      {
         suspend_counter++;

         // if no interaction occurs in 5 seconds, enter suspend mode
         if (suspend_counter == 320)
         {
            suspend_counter = 0;
            ENTER_SUSPEND = 1;
         }
      }
   }
   else
   {
      suspend_counter = 0;             // reset the SUSPEND mode counter
                                       // if the user is interacting with the
                                       // recorder

      reload_value = PRCHANGE;         // interrupt again in 150 ms
   }

   // reload the timer for the next interrupt
   TL1 = (-reload_value) & 0x00FF;
   TH1 = ((-reload_value) & 0xFF00) >> 8;
}

//-----------------------------------------------------------------------------
// ADC0_ISR
//-----------------------------------------------------------------------------
//
// Handle the 8kHz Timer 2 interrupt.
//
// Timer 2 controls the rate at which the ADC samples the input (RECORD).
//
void ADC0_ISR (void) interrupt 10 using 2
{
   // RECORD

   // DPCM variables
   static data unsigned short predicted_value = mid_range;
   static data unsigned char packed_code = 0x00;
   data unsigned char dpcm_code = 0x00;

   // indicates how the current dpcm_code should be packed to be sent to memory
   // sample 1 dpcm_code = A, sample 2 dpcm_code = B, sample 3 dpcm_code = C
   // sample 4 dpcm_code = D, sample 5 is the same as sample 1, etc
   // [A|A|A|A|A|A|B|B] = byte 1
   // [B|B|B|B|C|C|C|C] = byte 2
   // [C|C|D|D|D|D|D|D] = byte 3
   static unsigned char state = 0;

   static short sample = 0x0000;

   AD0INT = 0;                         // clear the interrupt flag

   // check if the memory was erased
   if ((system_state & ERASED) == ERASED)
   {
      system_state &= ~ERASED;         // clear the erased bit
      predicted_value = mid_range;     // reset the dpcm predictor
      state = 0;                       // reset the packing state machine
   }

   // check for the end of memory
   if (rec_end_addr == MAX_MEM_ADDR)
   {
      TR2 = 0;                         // turn off T2
      MEM_END_NEAR_FLAG = 0;
      MEM_END_FLAG = 1;                // tell the T1 ISR to turn off the LED
      predicted_value = mid_range;     // reset the dpcm predictor
      state = 0;                       // reset the state machine
   }
   else
   {
      // check if the REC_PLAY switch was released and the recording should
      // stop
      if (ADC_STOP_FLAG == 1)
      {
         TR2 = 0;                      // turn off T2
         ADC_STOP_FLAG = 0;            // reset the flag

         // do not reset the state or the predicted_value variables here
         // the playback ISR doesn't know when a recording starts or ends,
         // so it will also not reset the state and predicted_value
      }
      // take the sample, average it, compress it, and send it to memory
      else
      {
         // since 16 samples are automatically accumulated by the ADC,
         // average them by dividing by 16 (right shifting by 4)
         sample = (ADC0DAT >> 4) & 0x0FFF;



         // calculate the difference between the sample and the predictor
         // and compress the sample to a 6-bit DPCM code
         dpcm_code = DPCM_Encode ((sample - predicted_value));

         // pack the DPCM code into the bytes sent to the Flash memory
         switch (state)
         {
             // state machine: 0 -> 1 -> 2 -> 3
             //                ^______________|
             case 0:

               // move the DPCM code into the 6 high bits
               // [A|A|A|A|A|A| | ] = byte 1
               packed_code = (dpcm_code << 2) & 0xFC;

               state = 1;
               break;

             case 1:

               // move the DPCM code into the 2 low bits
               // of the previously packed byte
               // [-|-|-|-|-|-|B|B] = byte 1

               packed_code |= (dpcm_code >> 4) & 0x03;

               Write_MEM (rec_end_addr, packed_code);
               rec_end_addr++;

               // move the rest of the DPCM code into the
               // 4 high bits of the next packed byte
               // [B|B|B|B| | | | ] = byte 2

               packed_code = (dpcm_code << 4) & 0xF0;

               state = 2;
               break;

             case 2:

               // move the next DPCM code into the
               // 4 low bits of the previously packed byte
               // [-|-|-|-|C|C|C|C] = byte 2

               packed_code |= (dpcm_code >> 2) & 0x0F;

               Write_MEM (rec_end_addr, packed_code);
               rec_end_addr++;

               // move the rest of the DPCM code into the
               // 2 high bits of the next packed byte
               // [C|C| | | | | | ] = byte 3

               packed_code = (dpcm_code << 6) & 0xC0;

               state = 3;
               break;

             case 3:

               // move the next DPCM code into the
               // 6 low bits of the previously packed byte
               // [-|-|D|D|D|D|D|D] = byte 3

               packed_code |= dpcm_code & 0x3F;

               Write_MEM (rec_end_addr, packed_code);
               rec_end_addr++;

               state = 0;
               break;

             default:
               state = 0;
               break;
         }

         // indicate that the T1 ISR should flutter the LED, since the end of
         // memory is close
         if (rec_end_addr == NEAR_END_ADDR)
         {
            MEM_END_NEAR_FLAG = 1;
         }

         // update the predictor for the next sample
         predicted_value += DPCM_Decode (dpcm_code);
      }
   }
}

//-----------------------------------------------------------------------------
// Timer3_ISR
//-----------------------------------------------------------------------------
//
// Handle the 8kHz Timer 3 interrupt.
//
// Timer 3 controls the rate at which the DAC outputs decompressed samples
// (PLAY).
//
void Timer3_ISR(void) interrupt 14 using 2
{
   // PLAY

   // next unwritten address
   static unsigned long current_play_addr = 0x00000000;

   // DPCM variables
   static unsigned short predicted_value = mid_range;
   static unsigned char packed_code = 0x00;
   unsigned char dpcm_code = 0x00;

   // indicates how the current dpcm_code should be unpacked when retrieved
   // from memory
   // sample 1 dpcm_code = A, sample 2 dpcm_code = B, sample 3 dpcm_code = C
   // sample 4 dpcm_code = D, sample 5 is the same as sample 1, etc
   // [A|A|A|A|A|A|B|B] = byte 1
   // [B|B|B|B|C|C|C|C] = byte 2
   // [C|C|D|D|D|D|D|D] = byte 3
   static unsigned char state = 0;

   TMR3CN &= 0x7F;                     // clear the T3 interrupt flag

   // check if the PLAY switch was pressed and playing should stop
   if (DAC_STOP_FLAG == 1)
   {
      TMR3CN = 0x00;                   // turn off T3
      DAC_STOP_FLAG = 0;               // reset the flag
      current_play_addr = 0x00000000;  // start at the beginning address
      predicted_value = mid_range;     // reset the predictor
      state = 0;                       // reset the playback state machine
   }
   else
   {
      // check for the end of the recording
      if (current_play_addr >= rec_end_addr)
      {
         TMR3CN = 0x00;                // turn off the timer
         REC_END_FLAG = 1;             // tell the T1 ISR to turn off the LED
         current_play_addr = 0x00000000;
         predicted_value = mid_range;  // reset the predictor
         state = 0;                    // reset the playback state machine
      }
      else
      {
         // unpack the DPCM code bytes retrieved from memory
         switch (state)
         {
             // state machine: 0 -> 1 -> 2 -> 3
             //                ^______________|
             case 0:

               packed_code = Read_MEM (current_play_addr);
               current_play_addr++;

               // take the DPCM code from the 6 high bits
               // [A|A|A|A|A|A| | ] = byte 1
               dpcm_code = (packed_code >> 2) & 0x3F;

               state = 1;
               break;

             case 1:

               // take the next DPCM code from the 2 low bits
               // of the previously retrieved byte
               // [-|-|-|-|-|-|B|B] = byte 1
               dpcm_code = (packed_code << 4) & 0x30;

               packed_code = Read_MEM (current_play_addr);
               current_play_addr++;

               // take the rest of the DPCM code from the
               // 4 high bits of the next retrieved byte
               // [B|B|B|B| | | | ] = byte 2

               dpcm_code |= (packed_code >> 4) & 0x0F;

               state = 2;
               break;

             case 2:

               // take the next DPCM code from the
               // 4 low bits of the previously retrieved byte
               // [-|-|-|-|C|C|C|C] = byte 2
               dpcm_code = (packed_code << 2) & 0x3C;

               packed_code = Read_MEM (current_play_addr);
               current_play_addr++;

               // take the rest of the DPCM code from the
               // 2 high bits of the next retrieved byte
               // [C|C| | | | | | ] = byte 3
               dpcm_code |= (packed_code >> 6) & 0x03;

               state = 3;
               break;

             case 3:

               // take the next DPCM code from the
               // 6 low bits of the previously retrieved byte
               // [-|-|D|D|D|D|D|D] = byte 3
               dpcm_code = packed_code & 0x3F;

               state = 0;
               break;

             default:
               state = 0;
               break;
         }

         // calculate the new predicted value
         predicted_value += DPCM_Decode (dpcm_code);

         // output the new sample to the speaker
         IDA0DAT = predicted_value;

         // overwrite the very last sample so the output is at the mid-range
         // when stopped
         // the discontinuity causes a small "clicking" sound when playback
         // starts and stops
         if (current_play_addr >= rec_end_addr)
         {
            IDA0DAT = mid_range;
         }
      }
   }
}

//-----------------------------------------------------------------------------
// End Of File
//-----------------------------------------------------------------------------