/*
	Copyright 2009 Anatol Gregory Mayen
	
	Licensed under the Apache License, Version 2.0 (the "License");
	you may not use this file except in compliance with the License. 
	You may obtain a copy of the License at 
	
	http://www.apache.org/licenses/LICENSE-2.0 
	
	Unless required by applicable law or agreed to in writing, software 
	distributed under the License is distributed on an "AS IS" BASIS, 
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
	See the License for the specific language governing permissions and 
	limitations under the License. 
*/
package eu.maydu.gwt.validation.client.validators.strings;

import com.google.gwt.user.client.ui.SuggestBox;
import com.google.gwt.user.client.ui.TextBoxBase;

import eu.maydu.gwt.validation.client.ValidationAction;
import eu.maydu.gwt.validation.client.ValidationResult;
import eu.maydu.gwt.validation.client.Validator;
import eu.maydu.gwt.validation.client.i18n.StandardValidationMessages;
import eu.maydu.gwt.validation.client.i18n.ValidationMessages;


/**
 * Validates that a given input is in the given length range.
 *  
 * @author Anatol Mayen
 *
 */
public class StringLengthValidator extends Validator<StringLengthValidator> {

	protected TextBoxBase text;
	protected SuggestBox suggest;
	protected int min, max;
	
	public StringLengthValidator(TextBoxBase text) {
		this(text, false);
	}
	
	public StringLengthValidator(TextBoxBase text, int min, int max) {
		this(text, min, max, false);
	}
	
	
	public StringLengthValidator(TextBoxBase text, boolean preventsPropagationOfValidationChain) {
		this(text, preventsPropagationOfValidationChain, null);
	}
	
	public StringLengthValidator(TextBoxBase text, boolean preventsPropagationOfValidationChain, String customMsgKey) {
		super();
		this.setPreventsPropagationOfValidationChain(preventsPropagationOfValidationChain);
		if(text == null)
			throw new IllegalArgumentException("text must not be null");
		this.text = text;
		this.setCustomMsgKey(customMsgKey);
	}
	
	public StringLengthValidator(TextBoxBase text, int min, int max, boolean preventsPropagationOfValidationChain) {
		this(text, min, max, preventsPropagationOfValidationChain, null);
	}
	
	public StringLengthValidator(TextBoxBase text, int min, int max, boolean preventsPropagationOfValidationChain, String customMsgKey) {
		super();
		this.setPreventsPropagationOfValidationChain(preventsPropagationOfValidationChain);
		if(text == null)
			throw new IllegalArgumentException("text must not be null");
		this.text = text;
		setMin(min).setMax(max);
		this.setCustomMsgKey(customMsgKey);
	}
	
	public StringLengthValidator(SuggestBox suggest) {
		this(suggest, false);
	}
	
	public StringLengthValidator(SuggestBox suggest, int min, int max) {
		this(suggest, min, max, false);
	}
	
	
	public StringLengthValidator(SuggestBox suggest, boolean preventsPropagationOfValidationChain) {
		this(suggest, preventsPropagationOfValidationChain, null);
	}
	
	public StringLengthValidator(SuggestBox suggest, boolean preventsPropagationOfValidationChain, String customMsgKey) {
		super();
		this.setPreventsPropagationOfValidationChain(preventsPropagationOfValidationChain);
		if(suggest == null)
			throw new IllegalArgumentException("suggest must not be null");
		this.suggest = suggest;
		this.setCustomMsgKey(customMsgKey);
	}
	
	public StringLengthValidator(SuggestBox suggest, int min, int max, boolean preventsPropagationOfValidationChain) {
		this(suggest, min, max, preventsPropagationOfValidationChain, null);
	}
	
	public StringLengthValidator(SuggestBox suggest, int min, int max, boolean preventsPropagationOfValidationChain, String customMsgKey) {
		super();
		this.setPreventsPropagationOfValidationChain(preventsPropagationOfValidationChain);
		if(suggest == null)
			throw new IllegalArgumentException("suggest must not be null");
		this.suggest = suggest;
		setMin(min).setMax(max);
		this.setCustomMsgKey(customMsgKey);
	}
	
	@Override
	public ValidationResult validate(ValidationMessages allMessages) {
		StandardValidationMessages messages = allMessages.getStandardMessages();
		String str;
		if(text != null)
			str = text.getText();
		else str = suggest.getText();
		if(str == null)
			str = "";
		if(str.equals("") && !isRequired())
			return null;
		if(str.length() < this.min || str.length() > this.max)
			return new ValidationResult(getErrorMessage(allMessages, messages.length(this.min, this.max, str.length()), this.min, this.max, str.length()));

		return null;
	}
	
	public void invokeActions(ValidationResult result) {
		if(text != null) {
			for(ValidationAction<TextBoxBase> action : getFailureActions())
				action.invoke(result, text);
		}else {
			for(ValidationAction<SuggestBox> action : getFailureActions())
				action.invoke(result, suggest);
		}
	}
	
	public StringLengthValidator setMin(int min) {
		this.min = min;
		return this;
	}
	
	public StringLengthValidator setMax(int max) {
		this.max = max;
		return this;
	}

}
