/*
	Copyright 2009 Anatol Gregory Mayen
	
	Licensed under the Apache License, Version 2.0 (the "License");
	you may not use this file except in compliance with the License. 
	You may obtain a copy of the License at 
	
	http://www.apache.org/licenses/LICENSE-2.0 
	
	Unless required by applicable law or agreed to in writing, software 
	distributed under the License is distributed on an "AS IS" BASIS, 
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
	See the License for the specific language governing permissions and 
	limitations under the License. 
*/
package eu.maydu.gwt.validation.client.validators.strings;

import java.util.LinkedList;
import java.util.List;

import com.google.gwt.user.client.ui.SuggestBox;
import com.google.gwt.user.client.ui.TextBoxBase;

import eu.maydu.gwt.validation.client.ValidationAction;
import eu.maydu.gwt.validation.client.ValidationResult;
import eu.maydu.gwt.validation.client.Validator;
import eu.maydu.gwt.validation.client.i18n.StandardValidationMessages;
import eu.maydu.gwt.validation.client.i18n.ValidationMessages;
import eu.maydu.gwt.validation.client.validators.ValidatorAlgorithmResult;
import eu.maydu.gwt.validation.client.validators.strings.algorithms.StringEqualsValidatorAlgorithm;

/**
 * 
 * This validator checks that two inputs are having the same
 * value. This is for example useful for password editing functionality,
 * where there are often two individual input fields that must contain
 * the same value.
 * 
 * @author Anatol Mayen
 *
 */
public class StringEqualsValidator extends Validator<StringEqualsValidator> {

	protected TextBoxBase text1, text2;
	protected SuggestBox suggest1, suggest2;
	protected StringEqualsValidatorAlgorithm algorithm = new StringEqualsValidatorAlgorithm();
	
	public StringEqualsValidator(TextBoxBase text1, TextBoxBase text2) {
		this(text1, text2, false);
	}
	
	public StringEqualsValidator(TextBoxBase text1, TextBoxBase text2, boolean preventsPropagationOfValidationChain) {
		this(text1, text2, preventsPropagationOfValidationChain, null);
	}
	
	public StringEqualsValidator(TextBoxBase text1, TextBoxBase text2, boolean preventsPropagationOfValidationChain, String customMsgKey) {
		super();
		this.setPreventsPropagationOfValidationChain(preventsPropagationOfValidationChain);
		if(text1 == null || text2 == null)
			throw new IllegalArgumentException("Textboxes must not be null");
		this.text1 = text1;
		this.text2 = text2;
		this.setCustomMsgKey(customMsgKey);
	}
	
	
	public StringEqualsValidator(SuggestBox suggest1, SuggestBox suggest2) {
		this(suggest1, suggest2, false);
	}
	
	
	public StringEqualsValidator(SuggestBox suggest1, SuggestBox suggest2, boolean preventsPropagationOfValidationChain) {
		this(suggest1, suggest2, preventsPropagationOfValidationChain, null);
	}
	
	public StringEqualsValidator(SuggestBox suggest1, SuggestBox suggest2, boolean preventsPropagationOfValidationChain, String customMsgKey) {
		super();
		this.setPreventsPropagationOfValidationChain(preventsPropagationOfValidationChain);
		if(suggest1 == null || suggest2 == null)
			throw new IllegalArgumentException("Suggestboxes must not be null");
		this.suggest1 = suggest1;
		this.suggest2 = suggest2;
		this.setCustomMsgKey(customMsgKey);
	}
	

	
	@Override
	public ValidationResult validate(ValidationMessages allMessages) {
		StandardValidationMessages messages = allMessages.getStandardMessages();
		String str1;
		String str2;
		if(text1 != null) {
			str1 = text1.getText();
			str2 = text2.getText();
		}else {
			str1 = suggest1.getText();
			str2 = suggest2.getText();
		}
		
		if(str1.equals("") && str2.equals("") && isRequired())
			return new ValidationResult(getErrorMessage(allMessages, messages.notNull()));
		
		List<String> strings = new LinkedList<String>();
		strings.add(str1);
		strings.add(str2);
		ValidatorAlgorithmResult result = algorithm.validate(strings);
		if(result == null)
			return null;
		else
			return new ValidationResult(getErrorMessage(allMessages, messages.stringsNotEqual()));
		
	}
	
	public void invokeActions(ValidationResult result) {
		if(text1 != null) {
			for(ValidationAction<TextBoxBase> action : getFailureActions()) {
				action.invoke(result, text1);
				action.invoke(result, text2);
			}
		}else {
			for(ValidationAction<SuggestBox> action : getFailureActions()) {
				action.invoke(result, suggest1);
				action.invoke(result, suggest2);
			}
		}
	}
	
	public void resetActions() {
		for(ValidationAction<SuggestBox> action : getFailureActions())
			action.reset();
	}
	
}
