/*
	Copyright 2009 Anatol Gregory Mayen
	
	Licensed under the Apache License, Version 2.0 (the "License");
	you may not use this file except in compliance with the License. 
	You may obtain a copy of the License at 
	
	http://www.apache.org/licenses/LICENSE-2.0 
	
	Unless required by applicable law or agreed to in writing, software 
	distributed under the License is distributed on an "AS IS" BASIS, 
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
	See the License for the specific language governing permissions and 
	limitations under the License. 
*/

package eu.maydu.gwt.validation.client.validators.numeric;

import com.google.gwt.user.client.ui.SuggestBox;
import com.google.gwt.user.client.ui.TextBoxBase;

import eu.maydu.gwt.validation.client.ValidationAction;
import eu.maydu.gwt.validation.client.ValidationResult;
import eu.maydu.gwt.validation.client.Validator;
import eu.maydu.gwt.validation.client.i18n.StandardValidationMessages;
import eu.maydu.gwt.validation.client.i18n.ValidationMessages;

/**
 * 
 * @author Anatol Mayen
 *
 */
public class LongValidator extends Validator<LongValidator> {

	private TextBoxBase text;
	private SuggestBox suggest;
	private long min, max;
	private boolean noMinMax = false;
	
	public LongValidator(TextBoxBase text) {
		this(text, null);
	}
	
	public LongValidator(TextBoxBase text, String customMsgKey) {
		this(text, false);
		this.noMinMax = true;
		this.setCustomMsgKey(customMsgKey);
	}
	
	public LongValidator(TextBoxBase text, long min, long max) {
		this(text, min, max, false);
	}
	
	
	public LongValidator(TextBoxBase text, boolean preventsPropagationOfValidationChain) {
		this(text, preventsPropagationOfValidationChain, null);
	}
	
	public LongValidator(TextBoxBase text, boolean preventsPropagationOfValidationChain, String customMsgKey) {
		super();
		this.setPreventsPropagationOfValidationChain(preventsPropagationOfValidationChain);
		if(text == null)
			throw new RuntimeException("text must not be null");
		this.text = text;
		this.noMinMax = true;
		this.setCustomMsgKey(customMsgKey);
	}
	
	public LongValidator(TextBoxBase text, long min, long max, boolean preventsPropagationOfValidationChain) {
		this(text, min, max, preventsPropagationOfValidationChain, null);
	}
	
	public LongValidator(TextBoxBase text, long min, long max, boolean preventsPropagationOfValidationChain, String customMsgKey) {
		super();
		this.setPreventsPropagationOfValidationChain(preventsPropagationOfValidationChain);
		if(text == null)
			throw new RuntimeException("text must not be null");
		this.text = text;
		this.min = min;
		this.max = max;
		this.noMinMax = false;
		this.setCustomMsgKey(customMsgKey);
	}
	
	public LongValidator(SuggestBox suggest) {
		this(suggest, null);
	}
	
	public LongValidator(SuggestBox suggest, String customMsgKey) {
		this(suggest, false);
		this.noMinMax = true;
		this.setCustomMsgKey(customMsgKey);
	}
	
	public LongValidator(SuggestBox suggest, long min, long max) {
		this(suggest, min, max, false);
	}
	
	
	public LongValidator(SuggestBox suggest, boolean preventsPropagationOfValidationChain) {
		this(suggest, preventsPropagationOfValidationChain, null);
	}
	
	public LongValidator(SuggestBox suggest, boolean preventsPropagationOfValidationChain, String customMsgKey) {
		super();
		this.setPreventsPropagationOfValidationChain(preventsPropagationOfValidationChain);
		if(suggest == null)
			throw new RuntimeException("suggest must not be null");
		this.suggest = suggest;
		this.noMinMax = true;
		this.setCustomMsgKey(customMsgKey);
	}
	
	public LongValidator(SuggestBox suggest, long min, long max, boolean preventsPropagationOfValidationChain) {
		this(suggest, min, max, preventsPropagationOfValidationChain, null);
	}
	
	public LongValidator(SuggestBox suggest, long min, long max, boolean preventsPropagationOfValidationChain, String customMsgKey) {
		super();
		this.setPreventsPropagationOfValidationChain(preventsPropagationOfValidationChain);
		if(suggest == null)
			throw new RuntimeException("suggest must not be null");
		this.suggest = suggest;
		this.min = min;
		this.max = max;
		this.noMinMax = false;
		this.setCustomMsgKey(customMsgKey);
	}
	
	@Override
	public ValidationResult validate(ValidationMessages allMessages) {
		StandardValidationMessages messages = allMessages.getStandardMessages();
		String str;
		if(text != null)
			str = text.getText();
		else str = suggest.getText();
		if(!isRequired() && str.equals(""))
			return null;
		str = str.trim();
		if(str.equals(""))
			return new ValidationResult(getErrorMessage(allMessages, messages.notAnInteger()));
		if(!this.noMinMax) {
			//Integer in range
			try {
				long value = Long.parseLong(str);
				if(value < this.min || value > this.max) {
					return new ValidationResult(getErrorMessage(allMessages, messages.notInRange(this.min, this.max, value), this.min, this.max, value));
				}
			}catch(NumberFormatException ex) {
				return new ValidationResult(getErrorMessage(allMessages, messages.notALong()));
			}
		}else {
			//Any integer
			try {
				long value = Long.parseLong(str);
			}catch(NumberFormatException ex) {
				return new ValidationResult(getErrorMessage(allMessages, messages.notALong()));
			}
		}
		return null;
	}
	
	public void invokeActions(ValidationResult result) {
		if(text != null) {
			for(ValidationAction<TextBoxBase> action : getFailureActions())
				action.invoke(result, text);
		}else {
			for(ValidationAction<SuggestBox> action : getFailureActions())
				action.invoke(result, suggest);
		}
	}

}
