/*
	Copyright 2009 Anatol Gregory Mayen
	
	Licensed under the Apache License, Version 2.0 (the "License");
	you may not use this file except in compliance with the License. 
	You may obtain a copy of the License at 
	
	http://www.apache.org/licenses/LICENSE-2.0 
	
	Unless required by applicable law or agreed to in writing, software 
	distributed under the License is distributed on an "AS IS" BASIS, 
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
	See the License for the specific language governing permissions and 
	limitations under the License. 
*/
package eu.maydu.gwt.validation.client;

import java.util.LinkedList;
import java.util.List;

import com.google.gwt.user.client.Window;

import eu.maydu.gwt.validation.client.i18n.ValidationMessages;
import eu.maydu.gwt.validation.client.i18n.utils.MessageInterpolator;


/**
 * Base class for validators.
 * 
 * The only method necessary to implement by actual validators is
 * the <code>validate()</code> method.
 * 
 * A validation result indicates whether errors occured.
 * A <code>null</code> outcome indicates that no validation errors occured.
 *
 *  @param T The object to which the validator should cast itself to allow chaining method invocations of <code>addActionForFailure</code> method
 *  
 * @author Anatol Mayen
 *
 */

public abstract class Validator<T> {
	
	
	protected List<ValidationAction> failureActions = new LinkedList<ValidationAction>();
	private boolean preventsPropagationOfValidationChain = false;
	
	/**
	 * The default value is <code>true</code>
	 */
	private boolean required = true;
	private String msgKey = null;
	
	/**
	 * A validation result indicates whether warnings or errors occured.
	 * A <code>null</code> outcome indicates that no validation errors occured.
	 * @return <code>null</code> if no warnings and errors occured, otherwise the problems that occured are returned.
	 */
	public abstract <V extends ValidationMessages> ValidationResult validate(V messages);

	
	/**
	 * Must be overwritten by the concrete Validator implementation.
	 * Should invoke the actions added with the needed parameters.
	 * 
	 */
	public abstract void invokeActions(ValidationResult result);

	/**
	 * Adds actions to this validator object that will be invoked if the
	 * validation fails.
	 * 
	 * @param action
	 */
	public T addActionForFailure(ValidationAction action) {
		if(!failureActions.contains(action))
			failureActions.add(action);
		return (T) this;
	}
	
	public List<ValidationAction> getFailureActions() {
		return failureActions;
	}
	
	public boolean preventsPropagationOfValidationChain() {
		return this.preventsPropagationOfValidationChain;
	}
	
	public void setPreventsPropagationOfValidationChain(boolean preventsPropagationOfValidationChain) {
		this.preventsPropagationOfValidationChain = preventsPropagationOfValidationChain;
	}
	
	
	/**
	 * Resets all the associated actions
	 */
	public void resetActions() {
		for(ValidationAction action : failureActions)
			action.reset();
	}
	
	/**
	 * Whether a value must be given by the user
	 * @return <i>true</i> if a value must be given by the user
	 */
	public boolean isRequired() {
		return required;
	}
	

	public void setRequired(boolean required) {
		this.required = required;
	}
	
	
	/**
	 * The custom key that was set for this validator instance.
	 * If none was set, <code>null</code> will be returned, indicating
	 * that some standard message will be used.
	 * 
	 * If set, the key must be known by the {@link ValidationMessages#getCustomMessage(String,Object...)} method.
	 * 
	 * @return See above
	 */
	public String getCustomMsgKey() {
		return this.msgKey ;
	}
	
	public void setCustomMsgKey(String msgKey) {
		this.msgKey = msgKey;
	}

	/**
	 * This method is used by the validator instance to get the localized error
	 * message displayed to the user when an error occured.
	 * 
	 * The localized error message may choose to include a {0} place holder. If
	 * it chooses to, it will be replaced by the given input string on validation
	 * in the <code>TextBoxBase</code> field.
	 * 
	 * Clients of this class may choose to override this method to supply their
	 * own message. This will be primarily useful, if you choose to parameterize
	 * the message in some specific way.
	 * 
	 * 
	 * @param <V>
	 * @param messages
	 * @param text The text that the user entered before validation occured.
	 * @return
	 */
	protected <V extends ValidationMessages> String getMessage(V messages, Object... parameters) {
		return messages.getCustomMessage(msgKey, parameters);
		/*
		String msg = messages.getCustomMessage(msgKey);
		if(msg == null)
			throw new RuntimeException("The custom localization key given could not be found: "+msgKey);
		return MessageInterpolator.merge(msg, parameters);
		*/
	}
	
	/**
	 * Convenience method for validators to get the message to use,
	 * which is either the standard or the custom message depending
	 * on the setting of the customMsgKey.
	 * 
	 * 
	 * @param <V>
	 * @param messages
	 * @param stdMessage
	 * @param parameters
	 * @return
	 */
	protected <V extends ValidationMessages> String getErrorMessage(V messages, String stdMessage, Object... parameters) {
		
		if(this.getCustomMsgKey() == null)
			return stdMessage;
		else
			return getMessage(messages, parameters);
		
	}
	
}