/*
	Copyright 2009 Anatol Gregory Mayen
	
	Licensed under the Apache License, Version 2.0 (the "License");
	you may not use this file except in compliance with the License. 
	You may obtain a copy of the License at 
	
	http://www.apache.org/licenses/LICENSE-2.0 
	
	Unless required by applicable law or agreed to in writing, software 
	distributed under the License is distributed on an "AS IS" BASIS, 
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
	See the License for the specific language governing permissions and 
	limitations under the License. 
*/
package eu.maydu.gwt.validation.client.i18n;

import com.google.gwt.core.client.GWT;

/**
 * The validation messages used by the maydu GWT Validation Framework.
 * If you don't have any custom server side validation exceptions to throw
 * you can use this class directly.
 * 
 * If you are in the need to do some more checks on the server side for validating
 * data you should create your own class that extends this class and additionally
 * override the <code>getCustomMessage(String)</code> method in order to localize
 * your custom server side validation errors.
 * 
 * You should then initialize all your client side <code>ValidationProcessor</code>
 * instances with your custom implementation that extends this class.
 * 
 * You could do as follows:
 * <pre>
 * <code>
 * public class MyCustomValidations extends ValidationMessages {
 * 		@Override
 * 		public String getCustomMessage(String key) {
 * 			return "your localized string, depending on key";
 * 		}
 * }
 * </code>
 * </pre>
 * Then on the client UI side:
 * 
 * <pre>
 * <code>
 * ...
 * public void setupValidation() {
 * 
 * 		ValidationProcessor validator = new ValidationProcessor(new MyCustomValidations());
 * 		validator.addValidators(...
 * ...
 * }
 * </code>
 * </pre>
 * On the server side you would do:
 * <pre>
 * <code>
 * 		...
 * 		//Do some custom validation
 * 		//If custom validation failed, talk it back to the client
 * 		
 * 		ServerValidation.exception("key", "propertyName");
 * 		
 * 		...
 * </code>
 * </pre>
 * So the value of <code>key</code> will be given as parameter to the <code>getCustomMessage(String)</code>
 * method. So you need to make sure that it actually returns the required localized string (You could use
 * a ConstantsWithLookup approach or any other approach that you can think of)
 * 
 * @author Anatol Mayen
 *
 */
public class ValidationMessages {
	
	protected StandardValidationMessages standardMessages;
	
	
	/**
	 * Creates a new object which uses the build in standard messages
	 */
	public ValidationMessages() {
		standardMessages = GWT.create(StandardValidationMessagesImpl.class);
	}
	
	/**
	 * Creates a new object populated with a custom implementation of
	 * a standard messages.
	 * 
	 * Use this if you don't like the standard messages or if you need to
	 * support more languages. Just implement the <code>StandardValidationMessages</code>
	 * interface and pass your custom object to this constructor.
	 * 
	 * <em>NOTICE: If you are adding new languages you could also 'donate' them to the 
	 * project: http://sourceforge.net/projects/gwt-vl/</em>
	 * 
	 * @param messages The custom standard messages implementation
	 */
	public ValidationMessages(StandardValidationMessages messages) {
		if(messages == null)
			throw new RuntimeException("The custom standard messages object must not be null");
		this.standardMessages = messages;
	}
	
	/**
	 * Override if you need custom messages support.
	 * It provides a varargs argument to support arbitrary number
	 * of parameters that are used for a specific validation message.
	 * 
	 * See eu.maydu.gwt.validation.i18n.utils.MessageInterpolator for 
	 * help on interpolating your messages.
	 * 
	 * @param key The key that was sent from the server side for the message. This is totally under your control.
	 * @param parameters The parameters that might have been added to this validation message
	 * @return Should return the localized string for the given key. You need to take care of course that the keys you sent from the server side will be known inside this method so that the correct result can be returned.
	 */
	public String getCustomMessage(String key, Object... parameters ) {
		throw new RuntimeException("This method is not implemented by default. You need to create your own class, that extends ValidationMessages and override this method for custom validation message localization.");
	}
	
	
	/**
	 * Override if you are using <code>ValidationActions</code> that also
	 * need to print the name of the failed property.
	 * 
	 * Take care that the <code>propertyName</code> is automatically prefixed
	 * with the token specified by <code>ValidationProcessor.PROPERTY_NAME_LOCALIZATION_PREFIX</code>.
	 * This is done automatically, so you have an easy way of distinguishing your normal custom
	 * localizatino and the customization of the property names.
	 * 
	 * Because in your properties file you would have:
	 * 
	 * name = Name (for just a custom message)
	 * 
	 * and 
	 * 
	 * property_birthday (for a property localization
	 * 
	 * This behavior can be overwritten by setting the <code>ValidationProcessor.setPropertyLocalizationPrefix</code>
	 * to <code>false</code>.
	 * 
	 * @param key The name of the property for which a validation error occured
	 * @return If overwritten should return the localized name of the given propertyName.
	 */
	public String getPropertyName(String propertyName) {
		return "no property localization given for key: "+propertyName;
	}
	
	
	/**
	 * Override if you are using the description functionality for describing a field.
	 * 
	 * 
	 * @param msgKey
	 * @return The message associated with the key
	 */
	public String getDescriptionMessage(String msgKey) {
		return "no message for key: "+msgKey;
	}
	
	/**
	 * Returns the standard validation messages. 
	 * @return
	 */
	public StandardValidationMessages getStandardMessages() {
		return standardMessages;
	}
	
	

}
