package eu.maydu.gwt.validation.client.validators.strings.algorithms;

import eu.maydu.gwt.validation.client.validators.ValidatorAlgorithmResult;
import eu.maydu.gwt.validation.client.validators.ValidatorAlgorithm;

/**
 * Validates strings per character.
 * 
 * It can be parameterized via a blacklist of characters that are not allowed or
 * a whitelist of characters that are allowed within a validated string.
 * 
 * @author Anatol Mayen
 * 
 */
public class CharacterValidatorAlgorithm implements ValidatorAlgorithm<String> {

	private String list = "";
	private boolean whitelist;

	/**
	 * The only error message returned when a character is found that is not allowed.
	 * Parameter 1: The character that was wrong
	 */
	public final static int NOT_A_VALID_CHARACTER = 1;

	
	/**
	 * Creates a new character validator where the given list is assumed
	 * to be the whitelist.
	 * 
	 * @param charList The whitelist of characters that are allowed
	 */
	public CharacterValidatorAlgorithm(String charList) { 
		this(charList, true);
	}
	
	/**
	 * Creates a new character validator where the given list is either
	 * a white- or a blacklist depending on the second parameter.
	 * 
	 * @param charList
	 * @param whitelist 
	 */
	public CharacterValidatorAlgorithm(String charList, boolean whitelist) { 
		list = charList;
		this.whitelist = whitelist;
	}
	
	/**
	 * Validates the String input according to the configuration.
	 * If the input is <code>null</code> no validation occurs and no errors will
	 * be reported.
	 */
	public ValidatorAlgorithmResult validate(String input) {
		if (input == null)
			return null;
		if (whitelist)
			return checkWhitelist(input);
		else
			return checkBlacklist(input);
	}

	public ValidatorAlgorithmResult checkWhitelist(String input) {

		for (int i = 0; i < input.length(); i++) {
			char c = input.charAt(i);
			if (list.indexOf(c) == -1)
				return result(c);
		}

		return null;
	}

	public ValidatorAlgorithmResult checkBlacklist(String input) {

		for (int i = 0; i < list.length(); i++) {
			char c = list.charAt(i);
			if (input.indexOf(c) != -1)
				return result(c);
		}

		return null;
	}

	private ValidatorAlgorithmResult result(char c) {
		return new ValidatorAlgorithmResult(NOT_A_VALID_CHARACTER, c);
	}
}
