/*
	Copyright 2009 Anatol Gregory Mayen
	
	Licensed under the Apache License, Version 2.0 (the "License");
	you may not use this file except in compliance with the License. 
	You may obtain a copy of the License at 
	
	http://www.apache.org/licenses/LICENSE-2.0 
	
	Unless required by applicable law or agreed to in writing, software 
	distributed under the License is distributed on an "AS IS" BASIS, 
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
	See the License for the specific language governing permissions and 
	limitations under the License. 
*/
package eu.maydu.gwt.validation.client.validators.datetime;

import com.google.gwt.i18n.client.DateTimeFormat;
import com.google.gwt.user.client.ui.TextBoxBase;

import eu.maydu.gwt.validation.client.ValidationAction;
import eu.maydu.gwt.validation.client.ValidationResult;
import eu.maydu.gwt.validation.client.Validator;
import eu.maydu.gwt.validation.client.i18n.StandardValidationMessages;
import eu.maydu.gwt.validation.client.i18n.ValidationMessages;

/**
 * 
 * The LocalizedDateValidator will validate a given date after regional
 * date syntax
 * 
 * @author Anatol Mayen
 *
 */
public class LocalizedDateValidator extends Validator<LocalizedDateValidator> {

	private TextBoxBase text;
	private DateTimeFormat formatter;
	
	public LocalizedDateValidator(TextBoxBase text, DateTimeFormat formatter) {
		this(text, formatter, false, false);
	}
	
	public LocalizedDateValidator(TextBoxBase text, DateTimeFormat formatter, String customMsgKey) {
		this(text, formatter, false, false, customMsgKey);
	}
	
	/**
	 * Deprecated because of the use of the <code>allowNullInput</code> parameter that
	 * is now generalized into the setRequired method of the <code>Validator</code>
	 * interface.
	 * 
	 * @param text
	 * @param formatter
	 * @param allowNullInput
	 */
	public LocalizedDateValidator(TextBoxBase text, DateTimeFormat formatter, boolean allowNullInput) {
		this(text, formatter, allowNullInput, false);
		setRequired(!allowNullInput);
	}
	
	
	public LocalizedDateValidator(TextBoxBase text, DateTimeFormat formatter, boolean allowNullInput, boolean preventsPropagationOfValidationChain) {
		super();
		this.setPreventsPropagationOfValidationChain(preventsPropagationOfValidationChain);
		if(text == null)
			throw new IllegalArgumentException("The text box element must not be null");
		if(formatter == null)
			throw new IllegalArgumentException("DateTimeFormat must not be null");
		this.text = text;
		this.formatter = formatter;
		setRequired(!allowNullInput);
	}
	
	public LocalizedDateValidator(TextBoxBase text, DateTimeFormat formatter, boolean allowNullInput, boolean preventsPropagationOfValidationChain, String customMsgKey) {
		super();
		this.setPreventsPropagationOfValidationChain(preventsPropagationOfValidationChain);
		if(text == null)
			throw new IllegalArgumentException("The text box element must not be null");
		if(formatter == null)
			throw new IllegalArgumentException("DateTimeFormat must not be null");
		this.text = text;
		this.formatter = formatter;
		setRequired(!allowNullInput);
		this.setCustomMsgKey(customMsgKey);
	}
	

	@Override
	public ValidationResult validate(ValidationMessages allMessages) {
		StandardValidationMessages messages = allMessages.getStandardMessages();
		String str = text.getText();		
		if(!isRequired() && str.equals(""))
			return null;
		str = str.trim();
		if(str.equals(""))
			return new ValidationResult(getErrorMessage(allMessages, messages.noDateGiven()));
		
		
		try {
			formatter.parseStrict(str);
		}catch(Exception ex) {
			return new ValidationResult(getErrorMessage(allMessages, messages.unparsableDate()));
		}
		
		return null;
	}
	
	@Override
	public void invokeActions(ValidationResult result) {
		for(ValidationAction<TextBoxBase> action : getFailureActions())
			action.invoke(result, text);
	}

}
