/*
	Copyright 2009 Anatol Gregory Mayen
	
	Licensed under the Apache License, Version 2.0 (the "License");
	you may not use this file except in compliance with the License. 
	You may obtain a copy of the License at 
	
	http://www.apache.org/licenses/LICENSE-2.0 
	
	Unless required by applicable law or agreed to in writing, software 
	distributed under the License is distributed on an "AS IS" BASIS, 
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
	See the License for the specific language governing permissions and 
	limitations under the License. 
*/
package eu.maydu.gwt.validation.client.description;

import com.google.gwt.event.dom.client.BlurEvent;
import com.google.gwt.event.dom.client.BlurHandler;
import com.google.gwt.event.dom.client.FocusEvent;
import com.google.gwt.event.dom.client.FocusHandler;
import com.google.gwt.i18n.client.LocaleInfo;
import com.google.gwt.user.client.ui.HTML;
import com.google.gwt.user.client.ui.PopupPanel;
import com.google.gwt.user.client.ui.SuggestBox;
import com.google.gwt.user.client.ui.TextBoxBase;
import com.google.gwt.user.client.ui.PopupPanel.PositionCallback;

import eu.maydu.gwt.validation.client.i18n.ValidationMessages;

/**
 * Creates an popup that is shown whenever the associated widget
 * gets the focus. The popup will contain arbitrary HTML interpreted
 * text that can easily be i18n.
 * 
 * The popup is shown directly under the associated widget. It will
 * autohide if the user clicks somewhere or moves the focus to another
 * component.
 * 
 * @author Anatol Mayen
 *
 */
public class PopupDescription implements Description<TextBoxBase> {

	private ValidationMessages messages;
	private LocaleInfo localeInfo = LocaleInfo.getCurrentLocale();
	private Location showAtLocation = Location.BOTTOM;
	private int horizontalMargin = 5;
	private int verticalMargin = 5;
	
	public enum Location {
		TOP,
		RIGHT,
		BOTTOM,
		LEFT
	}
	
	/**
	 * Creates a new PopupDescription and configures it with the specified 
	 * <code>ValidationMessages</code>. Be sure to overwrite the 
	 * <code>getDescriptionMessage</code> method to i18n the description
	 * text. 
	 * 
	 * 
	 * @param messages The validation messages instance to use 
	 */
	public PopupDescription(ValidationMessages messages) {
		this.messages = messages;
	}
	
	/**
	 * 
	 * Creates a new PopupDescription and configures it with the specified 
	 * <code>ValidationMessages</code>. Be sure to overwrite the 
	 * <code>getDescriptionMessage</code> method to i18n the description
	 * text.
	 * 
	 * Additionally the place where the popup will be shown can be set.
	 *  
	 * @param messages
	 * @param showAtLocation Defines where the PopupDescription should be placed relative to the input widget (TOP, RIGHT, BOTTOM, LEFT)
	 */
	public PopupDescription(ValidationMessages messages, Location showAtLocation) {
		this(messages);
		this.showAtLocation = showAtLocation;
	}
	
	
	/**
	 * 
	 * Creates a new PopupDescription and configures it with the specified 
	 * <code>ValidationMessages</code>. Be sure to overwrite the 
	 * <code>getDescriptionMessage</code> method to i18n the description
	 * text.
	 * 
	 * Additionally the place where the popup will be shown can be set.
	 * 
	 * The vertical and horizontal margin between the popup and the input widget can be set.
	 * The vertical margin is used for TOP and BOTTOM locations, whereas the horizontal margin
	 * is used when the location is set to LEFT or RIGHT.
	 *  
	 * @param messages
	 * @param showAtLocation Defines where the PopupDescription should be placed relative to the input widget (TOP, RIGHT, BOTTOM, LEFT)
	 * @param verticalMargin margin between popup and input widget
	 * @param horizontalMargin margin between popup and input widget
	 */
	public PopupDescription(ValidationMessages messages, Location showAtLocation, int verticalMargin, int horizontalMargin) {
		this(messages, showAtLocation);
		this.verticalMargin = verticalMargin;
		this.horizontalMargin = horizontalMargin;
	}
	

	/**
	 * Adds a popup description to a widget.
	 * 
	 * @param key This key is used to get the i18n text that should be displayed. This key will be passed as argument to the <code>ValidationMessages.getDescriptionMessage</code> method.
	 * @param widget The widget that should show a popup description when it gets the focus.
	 */
	public void addDescription(String key, final TextBoxBase widget) {
		
		final PopupPanel p = new PopupPanel(true);
		
		String content = messages.getDescriptionMessage(key);
		
		if(localeInfo.isRTL())
			content = "<div align=\"right\">" + content + "</div>";
		
		HTML html = new HTML(content, false);
		p.setWidget(html);
		
		
		widget.addFocusHandler(new FocusHandler() {

			public void onFocus(FocusEvent event) {
				p.setPopupPositionAndShow(new PositionCallback() {

					public void setPosition(int offsetWidth, int offsetHeight) {
						
						int left = calculateLeft(p, widget);
						int top = calculateTop(p, widget);

						p.setPopupPosition(left, top);
					}
				});				
			}
		});
		
		widget.addBlurHandler(new BlurHandler() {

			public void onBlur(BlurEvent event) {
				p.hide();
			}
			
		});
	}
	
	/**
	 * Adds a popup description to a widget.
	 * 
	 * @param key This key is used to get the i18n text that should be displayed. This key will be passed as argument to the <code>ValidationMessages.getDescriptionMessage</code> method.
	 * @param widget The widget that should show a popup description when it gets the focus.
	 */
	public void addDescription(String key, final SuggestBox suggest) {
		addDescription(key, suggest.getTextBox());
	}
	
	/**
	 * Calculates the left point of the popup.
	 * 
	 * @param panel The PopupPanel instance
	 * @param widget The input widget the popup is associated with
	 * 
	 * @return
	 */
	private int calculateLeft(PopupPanel panel, TextBoxBase widget) {
		int left;
		
		if(showAtLocation == Location.LEFT) {
			//Aligned left to the widget			
			left = widget.getAbsoluteLeft();
			left -= panel.getOffsetWidth() + horizontalMargin;
			
		}else if(showAtLocation == Location.RIGHT){
			//Aligned right to the widget
			left = widget.getAbsoluteLeft() + widget.getOffsetWidth() + horizontalMargin;
		}else {
			//aligned at top or bottom of input widget.
			if(!localeInfo.isRTL()) {
				left = widget.getAbsoluteLeft();
			}else {
				left = widget.getAbsoluteLeft()+widget.getOffsetWidth();
				left -= panel.getOffsetWidth();
			}
		}
		return left;
	}
	
	
	/**
	 * Calculates the top point of the popup.
	 * 
	 * @param panel The PopupPanel instance
	 * @param widget The input widget the popup is associated with
	 * 
	 * @return
	 */
	private int calculateTop(PopupPanel panel, TextBoxBase widget) {
		int top = -1;
		
		if(showAtLocation == Location.BOTTOM) {
		
			top = widget.getAbsoluteTop();
			int height = widget.getOffsetHeight();
			top = top+height+verticalMargin;
		
		}else if(showAtLocation == Location.TOP) {
		
			top = widget.getAbsoluteTop() - panel.getOffsetHeight() - verticalMargin;
			
		}else if(showAtLocation == Location.LEFT || showAtLocation == Location.RIGHT) {
			top = widget.getAbsoluteTop();
		}
		
		
		
		
		return top;
	}

}
