/**********************************  io.c  **********************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Z-Wave, the wireless language.
 *
 *              Copyright (c) 2007
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 *
 * Description: See below.
 *
 * Author:   Oleg Zadorozhnyy
 *
 * 2007-07-11   OZA     Integrated with Z-Wave Used with iccAVR
 *
 * Last Changed By:  $Author: oza $
 * Revision:         $Revision: 1.1 $
 * Last Changed:     $Date: 2007/08/31 11:39:20 $
 * Ported to iccAVR
 ****************************************************************************/

/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/
#include	"TYPES.H"		// Standard types
#include	"RS232_IO.H"		// Serial communications
#include        "io.h"
#ifdef __GNUC__
#include <avr/io.h>
#include <avr/interrupt.h>
#define __disable_interrupt() cli();
#define __enable_interrupt() sei();
#else
#include        <inavr.h>
#include        <iom128.h>
#endif

/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/
#define UART_RX_BUFSIZE 200
#define UART_TX_BUFSIZE 5
#define UDRE 5
#define DATA_REGISTER_EMPTY (1<<UDRE)

// Flags used by the I/O functions
typedef struct __IOFLAGS
{
  unsigned rx:1;
  unsigned tx:1;
  unsigned err:1;
}IOFLAGS;

/****************************************************************************/
/*                              PRIVATE DATA                                */
/****************************************************************************/
// Locals to hold the current position and length of the string
IOFLAGS	ioflags;
u32 	uBitrate;

// Receive and transmit buffer
volatile BYTE rxQueue[ UART_RX_BUFSIZE ];

volatile BYTE rxInPtr = 0;
volatile BYTE rxOutPtr = 0;
BYTE rxCount = 0;

/****************************************************************************/
// This function is called when the UART has received one byte
// IRQ handler
/****************************************************************************/
#ifdef __GNUC__
ISR(USART1_RX_vect)
#else
#pragma vector=USART1_RXC_vect
__interrupt void irqHandler(void)
#endif
{
  auto UCHAR tempChar= 0;
  if (mIOIsGetReady())
  {
    tempChar = mIOGetByte(); // Get the byte
    if ( rxCount < UART_RX_BUFSIZE )
    {
      rxQueue[ rxInPtr++ ] = tempChar; // Get the byte
      rxCount++;
    }
    if ( rxInPtr >= UART_RX_BUFSIZE )
    {
      rxInPtr = 0;
    }
  }
}

BYTE bySerialCheck( void )
{
    return rxCount;
}

BYTE bySerialGetByte( void )
{
    BYTE retVal;
    retVal = rxQueue[ rxOutPtr ];
    __disable_interrupt();
    if ( rxCount > 0 )     // Make sure...
    {
        rxCount--;
    }
    __enable_interrupt();
    if ( ++rxOutPtr >= UART_RX_BUFSIZE )
    {
        rxOutPtr = 0;
    }
    return retVal;
}

// set the uart baud rate
void mIOInit(u32 baudrate)
{
	// calculate division factor for requested baud rate, and set it
	u16 bauddiv = ((F_CPU+(baudrate*8L))/(baudrate*16L)-1);
	UBRR1L = bauddiv;
	UBRR1H = bauddiv>>8;

        UBRR0L = bauddiv;
        UBRR0H = bauddiv>>8;

        UCSR1A=0x00;
        UCSR1B=0x18;
        UCSR1C=0x06;

        UCSR0A=0x00;
        UCSR0B=0x18;
        UCSR0C=0x06;

        UCSR1B |= 1 << RXCIE1;
}

void mPORTInit()
{
  // Input/Output Ports initialization
  // Port A initialization
  // Func7=Out Func6=Out Func5=Out Func4=Out Func3=Out Func2=Out Func1=In Func0=In
  // State7=0 State6=0 State5=0 State4=0 State3=0 State2=0 State1=T State0=T
  PORTA=0xFF;
  DDRA=0xFC;
  // Port B initialization
  // Func7=In Func6=In Func5=In Func4=In Func3=In Func2=In Func1=In Func0=In
  // State7=T State6=T State5=T State4=T State3=T State2=T State1=T State0=T
  PORTB=0x00;
  DDRB=0x00;
  // Port C initialization
  // Func7=In Func6=In Func5=In Func4=In Func3=Out Func2=Out Func1=Out Func0=Out
  // State7=T State6=T State5=T State4=T State3=0 State2=0 State1=0 State0=0
  PORTC=0x0F;
  DDRC=0x0F;
  // Port D initialization
  // Func7=In Func6=In Func5=In Func4=In Func3=In Func2=In Func1=In Func0=In
  // State7=T State6=T State5=T State4=T State3=T State2=T State1=T State0=T
  PORTD=0x00;
  DDRD=0x00;
  // Port E initialization
  // Func7=In Func6=In Func5=In Func4=In Func3=In Func2=In Func1=In Func0=In
  // State7=T State6=T State5=T State4=T State3=T State2=T State1=T State0=T
  PORTE=0x00;
  DDRE=0x00;
  // Port F initialization
  // Func7=In Func6=In Func5=In Func4=In Func3=In Func2=In Func1=In Func0=In
  // State7=T State6=T State5=T State4=T State3=T State2=T State1=T State0=T
  PORTF=0x00;
  DDRF=0x00;
  // Port G initialization
  // Func4=In Func3=In Func2=In Func1=In Func0=In
  // State4=T State3=T State2=T State1=T State0=T
  PORTG=0x00;
  DDRG=0x00;

  ACSR=0x80;
  SFIOR=0x00;
}

// Open an IO channel
unsigned char IOOpen(u32 bitrate)
{
  // Store the requested bitrate
  uBitrate = bitrate;
  // Reset the current transmit string pointers
  ioflags.tx = 0;
  // Setup async communications
  mIOInit(uBitrate);
  // Setup a receive buffer
  rxInPtr = 0;
  rxOutPtr = 0;
  rxCount = 0;
  return 1;
}

// Put a single char into the output stream
void vSerialPutByte( BYTE byTheByte )
{
  while ((UCSR1A & DATA_REGISTER_EMPTY)==0);
  UDR1=byTheByte;
  while ((UCSR1A & DATA_REGISTER_EMPTY)==0); 
}

void vSerial0PutByte( BYTE byTheByte )
{
  while ((UCSR0A & DATA_REGISTER_EMPTY)==0);
  UDR0=byTheByte;
  while ((UCSR0A & DATA_REGISTER_EMPTY)==0);
}

void vSerial0PutNum( BYTE byTheByte )
{
  BYTE qq = (byTheByte>>4)|0x30;
  if (qq>0x39) qq+=7;
  while ((UCSR0A & DATA_REGISTER_EMPTY)==0);
  UDR0=qq;
  while ((UCSR0A & DATA_REGISTER_EMPTY)==0);
  
  qq = (byTheByte&0x0f)|0x30;
  if (qq>0x39) qq+=7;  
  while ((UCSR0A & DATA_REGISTER_EMPTY)==0);
  UDR0=qq;
  while ((UCSR0A & DATA_REGISTER_EMPTY)==0);  
}



