/*******************************  ZW_TIMER.C  *******************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Z-Wave, the wireless language.
 *
 *              Copyright (c) 2001
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 *
 * Description: Timer service functions that handle delayed functions calls.
 *              The time base is around 10 msec.
 *
 * Author:   Oleg Zadorozhnyy
 *
 * 2007-08-11   OZA     Ported to iccAVR
 *
 * Last Changed By:  $Author: oza $
 * Revision:         $Revision: 1.1 $
 * Last Changed:     $Date: 2007/08/31 11:39:20 $
 * Ported to iccAVR
 ****************************************************************************/

/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/
#include "TYPES.H"			// Standard types
#include "ZW_basis.h"
#include "ZW_timer_api.h"
#include "AES_module.h"
/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/
#define TIMER_MAX 10  /* Max. number of simultaneously active timers. */

/* timer elements */
static struct timerElement
{
  WORD  tickCounts;
  WORD  timeoutValue;
  void (*func)(void);
  BYTE  repeats;
} timerArray[TIMER_MAX];

/****************************************************************************/
/*                              PRIVATE DATA                                */
/****************************************************************************/
volatile PBYTE timerCount; /* Incremented every 10msec for ZW010x */
                                  /* and every 2msec for ZW020x and this */
                                  /* done by the timer interrupt function, */
                                  /* and decremented by the timer action function */

/*============================   TimerSave   =========================
**    Save timer specifications and start the timer
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
static void                   /*RET Nothing             */
TimerSave(
  BYTE btimerHandle,          /* IN Index into the timer table */
  VOID_CALLBACKFUNC(func)(),  /* IN Timeout function adddress         */
  BYTE btimerTicks,           /* IN Timeout value (value * 10 msec.)  */
  BYTE brepeats)              /* IN Number of function calls, -1 = forever  */
{
  register BYTE tmp;

  tmp = btimerTicks;
  if (!tmp)
  {
    tmp++; /* min 10 msec. */
  }
  /* create new active timer element */
  timerArray[btimerHandle].tickCounts = timerArray[btimerHandle].timeoutValue = tmp;
  timerArray[btimerHandle].repeats = brepeats;
  timerArray[btimerHandle].func = func;
}

/****************************************************************************/
/*                           EXPORTED FUNCTIONS                             */
/****************************************************************************/

/*================================   TimerAction   ===========================
**    Walk through the timer elements and calls the timeout functions.
**
**--------------------------------------------------------------------------*/
void                      /*RET Nothing    */
TimerAction( void )       /* IN Nothing    */
{
  BYTE timerHandle;
  while (timerCount)
  {
    timerCount--;
    /* walk through the timer array */
    for (timerHandle = 0; timerHandle < TIMER_MAX; timerHandle++)
    {
      if (timerArray[timerHandle].tickCounts)
      {
        if (--timerArray[timerHandle].tickCounts == 0) /* count change from 1 to 0 */
        {
          if (timerArray[timerHandle].repeats != 0)
          { /* repeat timer action */
            if (timerArray[timerHandle].repeats != TIMER_FOREVER)
            { /* repeat limit */
              timerArray[timerHandle].repeats--;
            }
          }
          if (timerArray[timerHandle].repeats)
          {
            timerArray[timerHandle].tickCounts = timerArray[timerHandle].timeoutValue;
          }
          /* call the timeout function */
          if(&timerArray[timerHandle].func!= NULL) timerArray[timerHandle].func();
        }
      }
    }
  }
}

/*================================   TimerStart   ============================
**    Register a function that is called when the specified time has elapsed.
**    The function is called "repeats" times before the timer is then stopped.
**
**--------------------------------------------------------------------------*/
BYTE                          /*RET Timer handle                      */
TimerStart(
  VOID_CALLBACKFUNC(func)(),  /* IN Timeout function adddress         */
  BYTE btimerTicks,            /*IN  Timeout value (value * 10 msec.)  */
  BYTE brepeats)               /*IN  Number of function calls, -1 = forever  */
{
  BYTE retCode;
  BYTE timerHandle;

  retCode = (BYTE)-1;
  for (timerHandle = 0; timerHandle < TIMER_MAX; timerHandle++)
  {
    /* find first free timer element */
    if (timerArray[timerHandle].tickCounts == 0)
    {
      TimerSave(timerHandle, func, btimerTicks, brepeats);
      retCode = timerHandle + 1;
      return (retCode);
    }
  }
  return(retCode);
}

/*================================   TimerRestart  ===========================
**    Set the specified timer back to the initial value.
**
**--------------------------------------------------------------------------*/
BYTE                      /*RET TRUE if timer restarted   */
TimerRestart(
  BYTE btimerHandle)       /* IN Timer number to restart   */
{
  BYTE retCode;

  retCode = FALSE;
  btimerHandle--;  /* Index 0...n */
  if ((btimerHandle < TIMER_MAX) && (timerArray[btimerHandle].timeoutValue))
  { /* valid timer element number */
    timerArray[btimerHandle].tickCounts = timerArray[btimerHandle].timeoutValue;
    retCode = TRUE;
  }
  return(retCode);
}

/*================================   TimerCancel   ===========================
**    Stop the specified timer.
**      0 and 0xFF indicates no timer running.. This is acceptable
**--------------------------------------------------------------------------*/
BYTE                      /*RET TRUE if timer cancelled   */
TimerCancel(
  BYTE btimerHandle)       /* IN Timer number to stop      */
{
  BYTE retCode;

  retCode = FALSE;
  btimerHandle--;  /* Index 0...n */
  if (btimerHandle < TIMER_MAX)
  {
    /* valid timer element number */
  ZW_DEBUG_SEND_BYTE('t');
  ZW_DEBUG_SEND_NUM(btimerHandle);          
  ZW_DEBUG_SEND_BYTE(' ');
    timerArray[btimerHandle].tickCounts = timerArray[btimerHandle].timeoutValue = 0; /* stop the timer */
    retCode = TRUE;
  }
  return(retCode);
}

/*===============================   SafeTimerStop  ===========================
**    Stop the specified timer.
**    and set timerhandle to 0
**--------------------------------------------------------------------------*/
void
SafeTimerStop(
  BYTE *pbTimerHandle)
{
  TimerCancel(*pbTimerHandle);
  *pbTimerHandle=0xff;
}

/*================================   TimerInit   =============================
**    Initialize Timers.
**
**--------------------------------------------------------------------------*/
void
ZW_TimerInit(void)
{
  BYTE timerHandle;
  for (timerHandle = 0; timerHandle < TIMER_MAX; timerHandle++)
  {
    /* Free timer element */
    timerArray[timerHandle].tickCounts = 0;
  }
}
