/********************************  ZW_Portable.C  ********************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Z-Wave, the wireless language.
 *
 *              Copyright (c) 2001
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 * Copyright Zensys A/S, 2001
 *
 * Description: Application for evaluation kit Portable controller
 *
 * Author:   Oleg Zadorozhnyy
 *
 * 2007-08-11   OZA     Integrated with Z-Wave Used with iccAVR
 *
 * Last Changed By:  $Author: oza $
 * Revision:         $Revision: 1.17 $
 * Last Changed:     $Date: 2009/03/19 12:39:20 $
 * Ported to iccAVR
 ****************************************************************************/
#define ZW_CONTROLLER
#define ZW_CONTROLLER_STATIC
/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/
#include "TYPES.H"      // Standard types
#include "LED.h"
#include "ZW_classcmd.h"
#include "ZW_CONTROLLER.H"
#include "ZW_Portable.h"
#include "ZW_eeprom.h"
#include "p_button.h"
#include <string.h>
#include "ZW_Security_AES_module.h"
#include <AES_module.h>
/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/
#define TRANSMIT_OPTION_AUTO_ROUTE    0x04    /* request retransmission via repeater nodes */

/*mainState states*/
#define GROUP_CONTROL_STATE       0x00
#define INCLUDE_NODE_STATE        0x01
#define REMOVE_NODE_STATE         0x02
#define NEW_CONTROLLER_STATE      0x03
#define RESET_CONTROLLER_STATE    0x04
#define ASSIGN_ROUTE_STATE        0x05
#define CHECK_BUTTON_STATE        0x06
#define EXECUTE_STATE             0x07
#define DIMMING_STATE             0x08
#define LEARN_STATE               0x09
#define PWR_UP_STATE              0x0A
#define ERROR_STATE               0x0B
#define WAIT_FOR_PB_RELEASE_STATE 0x0C
#define WAIT_FOR_PB_DOWN_STATE    0x0D
#define WAIT_FOR_PB_PRESS_STATE   0x0E
#define WAIT_FOR_PB_DEPRESS_STATE 0x0F
#define RETURN_TO_IDLE_STATE      0x10
#define EXECUTE_STATE_RESET       0x11

#define GROUP_CTRL_ACTIVATION   IS_DOWN_PB0(glKey)
#define ADD_NODE_ACTIVATION     IS_DOWN_PB1(glKey)
#define RESET_NODE_ACTIVATION   IS_DOWN_PB2(glKey)
#define ASSIGN_ROUTE_ACTIVATION IS_DOWN_PB3(glKey)
#define ADD_CTRL_ACTIVATION     IS_DOWN_PB4(glKey)
#define RESET_ACTIVATION        IS_DOWN_PB0(glKey)&&IS_DOWN_PB4(glKey)

/* Poll function button state defines */
#define POLLIDLE          0
#define POLLDIMPROTECT    1 /* Wait some time will button pressed before dimming */
#define POLLCHILDHANDLING 2 /* We're in child protect mode */
#define POLLLEARNMODE     3 /* We're dimming and in learnmode */

/****************************************************************************/
/*                              PRIVATE DATA                                */
/****************************************************************************/
BYTE mainState;                /*State variable for the main loop*/
BYTE qq;
BYTE goON;
BYTE resetTimerHandle = 0;
BYTE led0TimerHandle = 0;
BYTE nextNode = 0;
BYTE  staticCtrlNodeID = 0x00;   /* hold the node id of the static update controller*/
/* Group replication */
BYTE bOtherController = 0;
BYTE replicationTimerHandle = 0;
BOOL receivingRemote =FALSE;
BOOL bStartUpdateTimer = FALSE;
BYTE lastLearnedNodeType = 0;  /*Store the last learned node type*/
/*Global used when getting nodeinformation from protocol*/
NODEINFO nodeInfo;

BYTE timerHandle = 0xFF;
/*Buffer used to hold commands*/
ZW_APPLICATION_TX_BUFFER_NEW txBuffer;
BYTE groupMask[MAX_NODES/8];
BYTE groupSecurityMask[MAX_NODES/8];


BYTE NodeIDinNet[MAX_NODES/8];
BYTE SendNode = 0;

static BYTE nodesInGroup;
BYTE groupState;
BYTE groupLevel;
BYTE receivedReport = FALSE;
BYTE groupNodeList[MAX_NODES];
BYTE dimEndTimerHandle = 0;

BYTE routingNode, routeToNode;
BYTE lastLearnedNodeID = 0;

BOOL txInProgress = FALSE;
BYTE OutKeyNewController = 0;
/* A list of the known command classes. Except the basic class which allways */
/* should be supported. Used when node info is send */
t_nodeInfo nodeInfoList = {COMMAND_CLASS_CONTROLLER_REPLICATION,COMMAND_CLASS_VERSION,COMMAND_CLASS_SECURITY};

BYTE  nodeInfoAfterIncluded[] = {COMMAND_CLASS_VERSION,COMMAND_CLASS_SECURITY};
ZW_APPLICATION_TX_BUFFER_NEW txBuf;

BYTE glKey;                    /*PushButton variable*/

BYTE networkKeyTMP[16];                    /* The master key TMP*/
BYTE inclusionKey[16];                    /* The master key TMP*/

BYTE lengthFrame;
VOID_CALLBACKFUNC(funcFrame)(auto BYTE bStatus);
BYTE bCommandFrame;

BYTE securityFrameCount = 0;

BYTE reportNode;
#define CLASS_REPORT_COUNT  2

BYTE *securityFramePos;

BYTE nodeSecure;
BYTE nodeSecureIncl;
BYTE nodeInWork;

BYTE notSleep=0;

BYTE addNodeToGroup = 0;

BYTE nodeSecInclude = 0;

BYTE isControllerIncluded = 0;    /// if Included = 1 if Including = 0;
BYTE isControllerSecure = 0;      /// if Secure = 1 if none Secure = 0;

BYTE nodeInfoFrame[CLASS_REPORT_COUNT] = { COMMAND_CLASS_CONTROLLER_REPLICATION,
                              COMMAND_CLASS_VERSION
                              };

BOOL statAddNodeToGroup = FALSE;

BYTE SECURITY_SCHEME;
BOOL addController = FALSE;



BYTE  BufTMPStatus = 0;
/****************************************************************************/
/*                              EXPORTED FUNCTIONS                          */
/****************************************************************************/
BOOL getIoflagsTx();
BYTE getTimerTr();
void setTimerTr();
void clrTimerTr();

/****************************************************************************/
/*                              EXPORTED DATA                               */
/****************************************************************************/
extern  BYTE bPressed7DEBOUNCE;
extern 	LEARN_INFO learnNodeInfo;
//extern
BYTE bufHomeID[4];
extern BYTE enNonce[8];    /* External nonce */
extern BYTE enNodeID;      /* Associated host id */
extern BYTE plaintext16ByteChunk[16];
extern BYTE networkKey[16];
//extern I
BYTE nodeID;                /* This nodes Node-ID */
/* Message processing */
extern BYTE noncePacket[10];                    /* Buffer for outgoing nonce packet */

extern LEARN_INFO learnNodeInfo;
extern void ZW_SoftReset();
extern BOOL ZW_RequestNetworkUpdate( void ( *complFunc ) ( auto BYTE ) );
extern void SerialAPI_ApplicationNodeInformation( BYTE listening, APPL_NODE_TYPE nodeType, BYTE *nodeParm, BYTE parmLength );
extern void StopSecuritySendTimeOut();
extern void StopSecurityTimeOut();
extern void StartSecurityTimeOut(BYTE timeOut);
extern BYTE                          /*RET  FALSE if transmitter busy      */
ZW_SendData(
  BYTE  destNodeID,               /*IN  Destination node ID (0xFF == broadcast) */
  BYTE *pData,                /*IN  Data buffer pointer           */
  BYTE  dataLength,           /*IN  Data buffer length            */
  BYTE  txOptions,            /*IN  Transmit option flags         */
  VOID_CALLBACKFUNC(completedFunc)(BYTE)); /*IN  Transmit completed call back function  */
/****************************************************************************/
/*                              PRIVATE FUNCTIONS                           */
/****************************************************************************/
void StopLED0Blink(void);
void ReplicationSendGroup(BYTE bStatus);
void StopUpdateTimer(void);
void NewControllerReceiveStarted(LEARN_INFO *learnNodeInfo/*BYTE bStatus,  BYTE nodeID*/);
void NewControllerSendStarted(LEARN_INFO *learnNodeInfo);
void SetControllerSecure( BYTE SecStatus );
void securityEnd(BYTE bStatus);


void NodeS()
{
  if (nodeSecure !=1)
    nodeSecure = 1;
}

/*
 * Function:    Set the node bit in a node bitmask
 *
 * Parameters:  pMask   Nodemask
 *              bNodeID Node ID that should be set in the mask
 *
 * Return:      void
 *
 * External:
 */
void
ZW_NodeMaskSetBit(
  BYTE_P pMask,
  BYTE bNodeID)
{
  bNodeID--;
  *(pMask+(bNodeID>>3)) |= (0x1 << (bNodeID & 7));
}

void
setCtrlSecure()
{
   isControllerSecure = 1;
   SetControllerSecure(isControllerSecure);
   securityEnd(NULL);
}

void
setCtrlNoneSecure()
{
   isControllerSecure = 0;
   SetControllerSecure(isControllerSecure);
   securityEnd(NULL);
}

BYTE
isCtrlSecure()
{
    return isControllerSecure;
}

void
setCtrlIncluded()
{
   isControllerIncluded = 1;
}

void
setCtrlIncluding()
{
   isControllerIncluded = 0;
}

BYTE
isCtrlIncluded()
{
   return isControllerIncluded;
}

/*
 * Function: Clear the node bit in a node bitmask
 *
 * Parameters:  pMask   Nodemask
 *              bNodeID Node ID that should be cleared in the mask
 *
 * Return:   void
 *
 * External:
 */
void
ZW_NodeMaskClearBit(
  BYTE_P pMask,
  BYTE bNodeID)
{
  bNodeID--;
  *(pMask+(bNodeID >> 3)) &= ~(0x1 << (bNodeID & 7));
}

/***************************   ZW_NodeMaskNodeIn   *************************/
/*
 * Function:    Check if a node is in a nodemask
 *
 * Parameters:  pMask   Nodemask
 *              bNodeID Node ID that should be checked
 *
 * Return:      0 - not in nodemask, !=0 - found in nodemask
 *
 * External:
 */
/****************************************************************************/
BYTE
ZW_NodeMaskNodeIn(
  BYTE_P pMask,
  BYTE bNode)
{
  bNode--;
  return ( ((*(pMask+(bNode>>3)) >> (bNode & 7)) & 0x01) );
}

BOOL statusNodeIn(BYTE bNodeID)
{
  if (ZW_NodeMaskNodeIn(groupSecurityMask,bNodeID)) return TRUE;
  return FALSE;
}

void AddSecuritySlave(BYTE bNodeID, BOOL slaveStatus)
{
  if(!nodeSecInclude)
  {
    nodeSecure = 1;
    return;
  }
  
  if(bNodeID)
  {
    if(slaveStatus)
    {
      ZW_NodeMaskSetBit(groupSecurityMask, bNodeID);
    }
    else
    {
      ZW_NodeMaskClearBit(groupSecurityMask, bNodeID);
    }
    
  if (ZW_MEM_GET_BYTE(addressof(groupTable, GROUPTABLE, magicValue)) != MAGIC_VALUE)
  {
    qq = 0;
  }
    
    /*We only save the relevant byte to extern eeprom*/
    ZW_MEM_PUT_BYTE(addressof(groupTable, GROUPTABLE, nodeSecurityMask)+((bNodeID-1) >> 3),groupSecurityMask[(bNodeID-1) >> 3]);
  
  if (ZW_MEM_GET_BYTE(addressof(groupTable, GROUPTABLE, magicValue)) != MAGIC_VALUE)
  {
    qq = 0;
  }
  }
}


/*============================   LED0Blinker   ==============================
**    Function description
**      Handles the blinking of LED0
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
LED0Blinker(void)
{
  if(goON)
  {
    LED0_ON;
    goON = FALSE;
  }
  else
  {
    goON = TRUE;
    LED0_OFF;
  }
}
/*============================   StartLED0Blink   ===========================
**    Function description
**      (Re)starts the timer that runs LED0 blinking..
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
StartLED0Blink(
  BYTE blinkRate)       /*IN blink rate in 10ms*/
{
  StopLED0Blink();
  goON = TRUE;    /*Allways start with LED on*/
  led0TimerHandle = ZW_TIMER_START(LED0Blinker, blinkRate, TIMER_FOREVER);
}

/*
 * Function:    Clear all bits in a nodemask
 *
 * Parameters:  pMask   Nodemask that should be cleared
 *
 * Return:      void
 *
 * External:
 */
void
ZW_NodeMaskClear(
  BYTE_P pMask,
  BYTE bLength)
{
  /* Clear entire node mask */
  if (bLength)
  {
    do
    {
      *pMask = 0;
      pMask++;
    } while (--bLength);
  }
}

/*============================   ClearGroupTable   ======================
**    Function description
**      Clears the group table and calls call back when done.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void ClearGroupTable(void (*func)(void))
{
  ZW_NODE_MASK_CLEAR(groupMask, sizeof(groupMask));
  ZW_MEM_PUT_BUFFER(addressof(groupTable, GROUPTABLE, nodeMask),groupMask, sizeof(groupMask), func);
}

/*============================   ZW_ResetDone   ======================
**    Function description
**      Restarts from PWR_UP_STATE
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
ZW_ResetDone(void)
{
  StopUpdateTimer();
  ZW_TIMER_START(StopLED0Blink,LED0_ONTIME,TIMER_ONE_TIME);
  LED1_OFF;

  setCtrlSecure();
  ZW_MEM_PUT_BUFFER(EEOFFSET_NETWORK_KEY_START, networkKey, 16,0);
}

/*============================   ApplicationResetDone   ======================
**    Function description
**        Starts Protocol reset
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
ApplicationResetDone(void)
{
  ZW_SET_DEFAULT(ZW_ResetDone);
}

/*============================   StoreGroupLevel   ======================
**    Function description
**      Stores the current dim level in External EEPROM
**    Side effects:
**
**--------------------------------------------------------------------------*/
void StoreGroupLevel(
BYTE level)           /*IN Current group dim level*/
{
  ZW_MEM_PUT_BYTE(addressof(groupTable, GROUPTABLE, currentDimLevel),level);
}

/*==================================   DoReset   ============================
**    Function description
**      Starts controller reset..
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
DoReset(void)
{
    StartLED0Blink(BUSY_BLINK_RATE);

    groupLevel = 0;
    StoreGroupLevel(groupLevel);
    LED1_ON;
    /*Make sure to exit any learnstate before reseting*/

    ZW_ADD_NODE_TO_NETWORK(ADD_NODE_STOP, NULL);
    ZW_SetLearnMode(FALSE, NULL);

    staticCtrlNodeID = DEFAULT_SUC_NODE_ID;

    SECURITY_SCHEME = SECURITY;   
    GetRNGData(networkKey,NETWORK_KEY_LENGTH);

    InitSecurity(0);

    ClearGroupTable(ApplicationResetDone);
    
    ApplicationResetDone();

    nodeID = 1;
    mainState = EXECUTE_STATE_RESET;
}
/*============================   StopResetTimer  ===========================
**    Function description
**      Stops the reset activate timeout
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
StopResetTimer(void)
{
  if(resetTimerHandle!=0xff)
  {
    ZW_TIMER_CANCEL(resetTimerHandle);
    resetTimerHandle = 0xff;
  }
}
/*============================   StartResetTimer   ==========================
**    Function description
**      Starts a timeout. When it runs out the controller will be reset.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
StartResetTimer(void)
{
  StopResetTimer();
  resetTimerHandle = ZW_TIMER_START(DoReset,RESET_TIMEOUT,TIMER_ONE_TIME);
}

/*============================   StopReplicationTimer   ======================
**    This function stops the replication timer
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  <completion code>       */
StopReplicationTimer(void)
{
  if (replicationTimerHandle!=0xff)
  {
     ZW_TIMER_CANCEL(replicationTimerHandle);
     replicationTimerHandle = 0xff;
  }
}

/*============================   StopUpdateTimer   ======================
**    Function description
**      Used to stop the request update timer.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
StopUpdateTimer(void)
{
  if(timerHandle!=0xff)
  {
    ZW_TIMER_CANCEL(timerHandle);
    timerHandle = 0xff;
  }
}
/*============================   AbortReplication   ======================
**    This callback function is used to abort replication when the
**		replication timer times out. Replication is aborted if no
**		TRANSMISSION_COMPLETE is received
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  <completion code>       */
AbortReplication(void)
{
    StopReplicationTimer();
    ReplicationSendGroup(TRANSMIT_COMPLETE_FAIL);
}
/*============================   StartReplicationTimer   ======================
**    This function starts the replication timer used when
**		detecting replication attempts that fails.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  <completion code>       */
StartReplicationTimer(void)
{
    replicationTimerHandle = ZW_TIMER_START(AbortReplication,REPL_TIMEOUT,TIMER_ONE_TIME);
    if (replicationTimerHandle == 0xFF)
    {
            // Should do something about this
    }
}

/*============================   IsNodeInGroup   ======================
**    Function description
**      Check if a node is in the group
**    Side effects:
**
**--------------------------------------------------------------------------*/
BOOL IsNodeInGroup(BYTE bNodeID)
{
  return ZW_NODE_MASK_NODE_IN(groupMask, bNodeID);
}


void
TxCompleteSend(
  BYTE bStatus)             /*IN Transmit status*/
{
}


/*================================   SetSUCDone  ============================
**    Function description
**      Called when a SUC NodeID have been set.
**    Side effects:
**
**--------------------------------------------------------------------------*/
static void
SetSUCDone(
  BYTE txStaus)
{
  int i;
  /*txStaus == ZW_SUC_SET_SUCCEEDED if successful, but we handle failed the same way*/
  ZW_TIMER_START(StopLED0Blink,LED0_ONTIME, TIMER_ONE_TIME);
  mainState = RETURN_TO_IDLE_STATE;
  if (staticCtrlNodeID = ZW_GET_SUC_NODEID())
  {
    bStartUpdateTimer = TRUE;
  }
  if ((learnNodeInfo).bLen !=0)
  {
    nodeSecInclude = 1;
    AddSecuritySlave(learnNodeInfo.bSource,FALSE);
    nodeSecInclude = 0;
    if (isCtrlSecure())
    {
      for (i = 0; i<learnNodeInfo.bLen; i++)
      {
          if (learnNodeInfo.pCmd[ i ] == COMMAND_CLASS_SECURITY)
          {
            nodeSecInclude = 1;
            AddSecuritySlave(learnNodeInfo.bSource,TRUE);
            txBuffer.ZW_SecuritySchemeGetFrame.cmdClass                       = COMMAND_CLASS_SECURITY;
            txBuffer.ZW_SecuritySchemeGetFrame.cmd                            = SECURITY_SCHEME_GET;
            txBuffer.ZW_SecuritySchemeGetFrame.supportedSecuritySchemes       = SECURITY_SCHEME;
            NodeS();
            nodeInWork = learnNodeInfo.bSource;
            StartSecurityTimeOut(TIME_SECURITY_LIFE);
            ZW_SendData(learnNodeInfo.bSource, (BYTE *)&txBuffer, sizeof(txBuffer.ZW_SecuritySchemeGetFrame), TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_EXPLORE, TxCompleteSend);
            break;
          }
      }
    }
  }
}
/*============================   StopLED0Blink   =============================
**    Function description
**      Stops the LED0Blinker timer and turns of LED0
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
StopLED0Blink(void)
{
  LED0_OFF;
  if(led0TimerHandle!=0xff)
  {
    ZW_TIMER_CANCEL(led0TimerHandle);
    led0TimerHandle = 0xff;
  }
}

/*============================   SetSucNodeDelay   =========================
**    Function description
**      Timer call back used to delay the setSUCNodeID function to give both
**       controllers (Tx and RX) to analyze the routing table.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
SetSucNodeDelay( void)
{
  if (!ZW_SET_SUC_NODEID(staticCtrlNodeID, TRUE, TRUE, ZW_SUC_FUNC_NODEID_SERVER, SetSUCDone))
  {
    SetSUCDone(ZW_SUC_SET_FAILED);
  }
}


/*============================   TxCompleteCheckAck   ======================
**    Function description
**      This function checks transmit status and indicates if any
**      error occoured
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
TxCompleteKeyAck(
  BYTE bStatus)             /*IN Transmit status*/
{
  txInProgress = FALSE;
  ZW_TIMER_START(StopLED0Blink,LED0_ONTIME,TIMER_ONE_TIME);
  if(bStatus != TRANSMIT_COMPLETE_OK)
    mainState = CHECK_BUTTON_STATE;
  else
    mainState = RETURN_TO_IDLE_STATE;

  memcpy(networkKey,networkKeyTMP,16);
  ZW_MEM_PUT_BUFFER(EEOFFSET_NETWORK_KEY_START, networkKey, 16,0);
  LoadKeys();
}

BYTE getNextSendNode()
{
  BYTE i;
  for (i = ++SendNode; i < MAX_NODES; i++)
  {
    if (ZW_NodeMaskNodeIn(NodeIDinNet,i))
    {
      if (i!=nodeID) return i;
    }
  }
  return 0x0;
}

/*============================   UpdateRAMGroupList   ======================
**    Function description
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
BYTE                      /*RETURN number of nodes in the list*/
UpdateRAMGroupList(void)
{
  BYTE count = 0, i, j, n;
  /*Get a fresh copy of the groupnodemask*/
  ZW_MEM_GET_BUFFER(addressof(groupTable, GROUPTABLE, nodeMask),groupMask, sizeof(groupMask));
  for (i = 0; i < (MAX_NODES >> 3) + (MAX_NODES & 0x07 ? 1 : 0); i++)
  {
    for (j = 1, n = 1; j <= 8; j++, n <<= 1)
    {
      if ((groupMask[i] & n) != 0)
      {
        groupNodeList[count++] = ((j + (i << 3)));
        if (count > (MAX_NODES-1))
        {
          break;
        }
      }
    }
  }
  return (count);
}

/*============================   NewControllerAllDone1   ======================
**    Function description
**        Called when replication is all done
**        BYTE bStatus,   IN  Status of learn process
**        BYTE bSource,  IN  Node id of the node that send node info
**        BYTE* pCmd,    IN  Pointer to Node information
**        BYTE bLen.     IN  Node info length
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
NewControllerAllDone(
  LEARN_INFO *learnNodeInfo)    /*IN Learn node info*/
{
  BYTE i,sourceNode;
  ZW_GET_NODE_STATE((*learnNodeInfo).bSource, &nodeInfo);
  staticCtrlNodeID = ZW_GET_SUC_NODEID();
  if (nodeInfo.nodeType.basic == BASIC_TYPE_STATIC_CONTROLLER)
  /*If learnNodeInfo is not zero and equal the static ctrlID then assign it as SUC*/
  {
    if ((!staticCtrlNodeID) || ( staticCtrlNodeID == (*learnNodeInfo).bSource))
    {
      staticCtrlNodeID = (*learnNodeInfo).bSource;
      mainState = EXECUTE_STATE; /*Do not allow user to interrupt now*/
      if (ZW_TIMER_START(SetSucNodeDelay, 200, TIMER_ONE_TIME) == 0xFF)
      {
        SetSUCDone(ZW_SUC_SET_FAILED);
      }
      return;
    }
  }


  if ((*learnNodeInfo).bStatus == ADD_NODE_STATUS_DONE)
  {
    if ((*learnNodeInfo).bLen !=0)
    {
      sourceNode = (*learnNodeInfo).bSource;
      nodeSecInclude = 1;
      AddSecuritySlave(sourceNode,FALSE);
      nodeSecInclude = 0;
      if (isCtrlSecure())
      {
        for (i = 0; i<(*learnNodeInfo).bLen; i++)
        {
          if ((*learnNodeInfo).pCmd[ i ] == COMMAND_CLASS_SECURITY)
          {
            nodeSecInclude = 1;
            AddSecuritySlave(sourceNode,TRUE);
            sourceNode = (*learnNodeInfo).bSource;
            txBuffer.ZW_SecuritySchemeGetFrame.cmdClass                  = COMMAND_CLASS_SECURITY;
            txBuffer.ZW_SecuritySchemeGetFrame.cmd                       = SECURITY_SCHEME_GET;
            txBuffer.ZW_SecuritySchemeGetFrame.supportedSecuritySchemes  = SECURITY_SCHEME;
            NodeS();
            nodeInWork = (*learnNodeInfo).bSource;
            StartSecurityTimeOut(TIME_SECURITY_LIFE);
            ZW_SendData(sourceNode, (BYTE *)&txBuffer, sizeof(txBuffer.ZW_SecuritySchemeGetFrame), TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_EXPLORE, TxCompleteSend);
            ZW_TIMER_START(StopLED0Blink, LED0_ONTIME, TIMER_ONE_TIME);
            mainState = WAIT_FOR_PB_RELEASE_STATE;
            return;
          }
        }
      }
    }
  ZW_TIMER_START(StopLED0Blink, LED0_ONTIME, TIMER_ONE_TIME);
  mainState = WAIT_FOR_PB_RELEASE_STATE;


  if (IS_PB_HELD(glKey))
  {
      /* User wants to include a new slave node to the network. Handle it */
      StartLED0Blink(BUSY_BLINK_RATE);
      mainState = INCLUDE_NODE_STATE;
      NodeS();
      setCtrlIncluding();
//!!!      ZW_ADD_NODE_TO_NETWORK((ADD_NODE_ANY | ADD_NODE_OPTION_HIGH_POWER | ADD_NODE_OPTION_NETWORK_WIDE), NewControllerSendStarted);
  }
    return;

  }
  ZW_TIMER_START(StopLED0Blink, LED0_ONTIME, TIMER_ONE_TIME);
  mainState = WAIT_FOR_PB_RELEASE_STATE;
}



/*======================== ReplicationSendGroup ===========================
**    This functions handles sending group replication information as part of the
**    Controller replication process
**
**
**--------------------------------------------------------------------------*/
void
ReplicationSendGroup(
  BYTE bStatus)
{
  static BYTE bReplicationSuccess = ADD_NODE_STOP;
  register BYTE bTempVar;

	StopReplicationTimer();
  /* Transmit failed stop replication */
  if (bStatus != TRANSMIT_COMPLETE_OK)
  {
    nextNode = MAX_NODES;
    bReplicationSuccess = ADD_NODE_STOP_FAILED;
  }

  /* Find the next node that belongs to the group */
  for (; nextNode < MAX_NODES; nextNode++)
  {
    if (IsNodeInGroup(nextNode))
    {
      break;
    }
  }

  /* Check if we are finished with transferring the group */
  if (nextNode == MAX_NODES)
  {
    /* Here we could transfer a group name, start the next group or start
       transferring scenes, but we dont have any of that in this controller so
       we will just stop */
    ZW_ADD_NODE_TO_NETWORK(bReplicationSuccess, NewControllerAllDone);
    return;
  }
  /* Send next node ID to other controller */
  txBuffer.ZW_CtrlReplicationTransferGroupFrame.cmdClass = COMMAND_CLASS_CONTROLLER_REPLICATION;
  txBuffer.ZW_CtrlReplicationTransferGroupFrame.cmd      = CTRL_REPLICATION_TRANSFER_GROUP;
  txBuffer.ZW_CtrlReplicationTransferGroupFrame.groupId  = 1; /* We only have one group in this controller */
  txBuffer.ZW_CtrlReplicationTransferGroupFrame.nodeId   = nextNode;

  /* Go to next node ID */
  nextNode++;

  /* Send the node id to the other controller */
  bTempVar = ZW_REPLICATION_SEND_DATA(bOtherController, (BYTE*)&txBuffer, sizeof(ZW_CTRL_REPLICATION_TRANSFER_GROUP_FRAME), TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_LOW_POWER, ReplicationSendGroup);
	StartReplicationTimer();

  /* Check if we could send replication frame */
  if (bTempVar == FALSE)
  {
    ReplicationSendGroup(TRANSMIT_COMPLETE_NO_ACK); /* Fail replication */
  }
}

/*============================   NewControllerSendStarted   ==================
**    Function description
**        Called by the protocol as the replication progresses.
**        When NEW_CONTROLLER_LEARNED has been received the user can not abort
**        BYTE bStatus,   IN  Status of learn process
**        BYTE bSource,  IN  Node id of the node that send node info
**        BYTE* pCmd,    IN  Pointer to Node information
**        BYTE bLen.     IN  Node info length

**    Side effects:
**
**--------------------------------------------------------------------------*/
void
NewControllerSendStarted(
  LEARN_INFO *learnNodeInfo)
{
  
   BufTMPStatus = (BYTE)learnNodeInfo->bStatus ;
  
  if (learnNodeInfo->bStatus == ADD_NODE_STATUS_LEARN_READY)
  {
    /*We are ready to add a node*/
    StopLED0Blink();
    LED0_ON;
    mainState = LEARN_STATE;
    /* Used for group replication */
    bOtherController = 0;
  }
  else if ((learnNodeInfo->bStatus == ADD_NODE_STATUS_NODE_FOUND)||
           (learnNodeInfo->bStatus == ADD_NODE_STATUS_ADDING_CONTROLLER)||
           (learnNodeInfo->bStatus == ADD_NODE_STATUS_ADDING_SLAVE))
  {
    /* Store controller node id for group replication */
    if (learnNodeInfo->bStatus == ADD_NODE_STATUS_ADDING_CONTROLLER)
    {
      bOtherController = learnNodeInfo->bSource;
      addController = TRUE;
    }
    
    if (learnNodeInfo->bStatus == ADD_NODE_STATUS_ADDING_SLAVE)
    {
      addController = FALSE;
    }  
    
    StartLED0Blink(BUSY_BLINK_RATE);
    mainState = EXECUTE_STATE;
  }
  else if (learnNodeInfo->bStatus == ADD_NODE_STATUS_PROTOCOL_DONE)
  {
    /* If it was a controller we added then start group replication */
    if (bOtherController)
    {
      nextNode = 1;
      ReplicationSendGroup(TRANSMIT_COMPLETE_OK);
    }
    else  /* If not a controller then just stop */
      ZW_ADD_NODE_TO_NETWORK(ADD_NODE_STOP, NewControllerAllDone);
  }
  else if (learnNodeInfo->bStatus == ADD_NODE_STATUS_FAILED)
  {
    /*We have added a new device to the controller*/
    ZW_ADD_NODE_TO_NETWORK(ADD_NODE_STOP, NULL);
    mainState = ERROR_STATE;
  }
  else if (learnNodeInfo->bStatus == ADD_NODE_STATUS_DONE)
  {
    NewControllerAllDone(learnNodeInfo);
  }
}


/*============================   TxCompleteIgnoreAck   ======================
**    Function description
**      Callback function that ignores Transmit status and returns to idle state
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
TxCompleteIgnoreAck(
  BYTE bStatus)             /*IN Transmit status*/
{
  nextNode++;
  for (; nextNode < MAX_NODES; nextNode++)
  {
    if (IsNodeInGroup(nextNode))
    {
      break;
    }
  }
  if(nextNode<MAX_NODES)
  {
      if(nodesInGroup)
      {
        enNodeID = 0xff;
        if (statusNodeIn(nextNode))
          ZW_SendDataSecure(nextNode, (BYTE *)&txBuffer, lengthFrame, bCommandFrame, funcFrame);
        else
        {
          ZW_SendData(nextNode, (BYTE *)&txBuffer, lengthFrame, bCommandFrame, funcFrame);
        }
      }
  }
  else
  {
    ZW_TIMER_START(StopLED0Blink,LED0_ONTIME,TIMER_ONE_TIME);
    mainState = RETURN_TO_IDLE_STATE;
  }
}

/*============================   AddNodeToGroup   ======================
**    Function description
**      Stores a node in the group.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void StoreNodeBitInGroup(BYTE bNodeID, BYTE value)
{
  if(bNodeID)
  {
    if(value)
    {
      ZW_NodeMaskSetBit(groupMask, bNodeID);
    }
    else
    {
      ZW_NodeMaskClearBit(groupMask, bNodeID);
    }
    /*We only save the relevant byte to extern eeprom*/
    ZW_MEM_PUT_BYTE(addressof(groupTable, GROUPTABLE, nodeMask)+((bNodeID-1) >> 3),groupMask[(bNodeID-1) >> 3]);
  if (ZW_MEM_GET_BYTE(addressof(groupTable, GROUPTABLE, magicValue)) != MAGIC_VALUE)
  {
qq = 0;
  }
  }
}
/*============================   RemoveNodeFromGroup   ======================
**    Function description
**      Removes a node from a group.
**    Side effects:
**      updates the groupArray
**--------------------------------------------------------------------------*/
void RemoveNodeFromGroup(
BYTE bNodeID)             /*IN NodeID to remove*/
{
  StoreNodeBitInGroup(bNodeID, 0);
}

void SoftResetDelay()
{
  StopLED0Blink();
  ZW_SoftReset();
  mainState = WAIT_FOR_PB_RELEASE_STATE;
}

/*=========================   RemoveNodeCompleted   ========================
**    Function description
**        Callback function, used when removing switches from the network.
**          bStatus BYTE,     IN  Status of learn process
**          bSource BYTE,     IN  Node id of the node that send node info
**          BYTE* pCmd,       IN  Pointer to Node information
**          BYTE bLen));      IN  Node info length
**
**    Side effects: Shuts down RF receive mode when completed and removes the
**                  node from the group.
**--------------------------------------------------------------------------*/
void
RemoveNodeCompleted(
  LEARN_INFO *learnNodeInfo)
{
  if ((*learnNodeInfo).bStatus == REMOVE_NODE_STATUS_NODE_FOUND)
  {
    StartLED0Blink(BUSY_BLINK_RATE);
    mainState = EXECUTE_STATE;
  }
  else if (((*learnNodeInfo).bStatus == REMOVE_NODE_STATUS_REMOVING_SLAVE)||
           ((*learnNodeInfo).bStatus == REMOVE_NODE_STATUS_REMOVING_CONTROLLER))
  {
  }
  else if ((learnNodeInfo->bStatus == REMOVE_NODE_STATUS_DONE) ||
           (learnNodeInfo->bStatus == REMOVE_NODE_STATUS_FAILED) )
  {

    ZW_REMOVE_NODE_FROM_NETWORK(REMOVE_NODE_STOP, NULL);
    mainState = WAIT_FOR_PB_RELEASE_STATE;
    ZW_TIMER_START(StopLED0Blink,LED0_ONTIME,TIMER_ONE_TIME);

    RemoveNodeFromGroup((*learnNodeInfo).bSource);

    if(((*learnNodeInfo).bStatus == ADD_NODE_STATUS_FAILED))
    {
      StartLED0Blink(BUSY_BLINK_RATE);
      mainState = ERROR_STATE;
    }
  }
}

/*============================   GroupSend  =================================
**    Function description
**      Sends out multicast message to the group. If no nodes are in the group
**      it will call the callback func with TRANSMIT_COMPLETE_NO_ACK.
**    Side effects:
**      updates RAMGroupList
**--------------------------------------------------------------------------*/
void
GroupSend(
  BYTE *pData,                /*IN  Data buffer pointer           */
  BYTE dataLength,            /*IN  Data buffer length            */
  BYTE txOptions,             /*IN  Transmit option flags         */
  BYTE bCommand,
  VOID_CALLBACKFUNC(func)(auto BYTE bStatus))    /*IN  Transmit completed call back function  */
{
}

/*============================   ControlGroup   ======================
**    Function description
**      Executes a group command. This can be one of the following:
**        TOGGLE
**        DIM_START
**        DIM_STOP
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
ControlGroup(                     /*RET Nothing*/
  BYTE bCommand,                  /* IN What to do*/
  VOID_CALLBACKFUNC(func)(auto BYTE bStatus))  /* IN Callback function.*/
{
  BYTE bRouting;
  BYTE length = sizeof(ZW_BASIC_SET_FRAME);
  nodesInGroup = UpdateRAMGroupList();
  nextNode = 0;
  switch(bCommand)
  {
    case TOGGLE:

    bRouting = TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_EXPLORE;
    for (nextNode=0; nextNode < MAX_NODES; nextNode++)
    {
      if (nextNode!=nodeID)
        if (IsNodeInGroup(nextNode))
        {
          break;
        }
    }
    txBuffer.ZW_BasicSetFrame.cmdClass = COMMAND_CLASS_BASIC;
    ZW_GET_NODE_STATE(nextNode,&nodeInfo);
    if(nodeInfo.nodeType.generic == GENERIC_TYPE_SWITCH_MULTILEVEL)
      txBuffer.ZW_BasicSetFrame.cmdClass = COMMAND_CLASS_BASIC;
    else if(nodeInfo.nodeType.generic == GENERIC_TYPE_ENTRY_CONTROL)
      txBuffer.ZW_BasicSetFrame.cmdClass = COMMAND_CLASS_LOCK;
    txBuffer.ZW_BasicSetFrame.cmd = BASIC_SET;
    
    groupState ^= GROUP_ON_BIT;  /* Toggle */
    if (groupState&GROUP_ON_BIT)
    {
        txBuffer.ZW_BasicSetFrame.value = SWITCHED_ON;   /* Switch ON */
    }
    else
    {
      txBuffer.ZW_BasicSetFrame.value = SWITCHED_OFF;  /* Switch OFF */
    }
    break; /*TOGGLE*/

    case DIM_START:
      txBuffer.ZW_SwitchMultilevelStartLevelChangeFrame.cmdClass = COMMAND_CLASS_SWITCH_MULTILEVEL;
      txBuffer.ZW_SwitchMultilevelStartLevelChangeFrame.cmd = SWITCH_MULTILEVEL_START_LEVEL_CHANGE;
      /* Set dim direction in command and toggle direction bit in state */
      txBuffer.ZW_SwitchMultilevelStartLevelChangeFrame.level/*reservedUpDownIgnoreStartLevelReserved*/ = groupState&GROUP_DIR_BIT;
      groupState ^= GROUP_DIR_BIT;
      txBuffer.ZW_SwitchMultilevelStartLevelChangeFrame.startLevel = groupLevel;
      bRouting = TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_EXPLORE;
      length = sizeof(ZW_SWITCH_MULTILEVEL_START_LEVEL_CHANGE_FRAME);
    break; /*DIM_START*/

    case DIM_STOP:
      receivedReport = FALSE;
      bRouting = TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_EXPLORE;
      txBuffer.ZW_SwitchMultilevelStopLevelChangeFrame.cmdClass = COMMAND_CLASS_SWITCH_MULTILEVEL;
      txBuffer.ZW_SwitchMultilevelStopLevelChangeFrame.cmd = SWITCH_MULTILEVEL_STOP_LEVEL_CHANGE;
      length = sizeof(ZW_SWITCH_MULTILEVEL_STOP_LEVEL_CHANGE_FRAME);
    break; /*DIM_STOP*/

    case DIM_SET:
      /* Do routing because we want all lamps to turn on/off */
      bRouting = TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_EXPLORE;
      txBuffer.ZW_SwitchMultilevelSetFrame.cmdClass = COMMAND_CLASS_SWITCH_MULTILEVEL;
      txBuffer.ZW_SwitchMultilevelSetFrame.cmd = SWITCH_MULTILEVEL_SET;
      txBuffer.ZW_SwitchMultilevelSetFrame.value = groupLevel;
      length = sizeof(ZW_SWITCH_MULTILEVEL_SET_FRAME);
    break;

    default:
    break;
  }
  lengthFrame = length;
  funcFrame = func;
  bCommandFrame = bRouting;

  
    for (nextNode=0; nextNode < MAX_NODES; nextNode++)
    {
      if (nextNode!=nodeID)
        if (IsNodeInGroup(nextNode))
        {
          break;
        }
    }
    
    if(nodesInGroup)
    {
      enNodeID = 0xff;
      if (statusNodeIn(nextNode))
        ZW_SendDataSecure(nextNode, (BYTE *)&txBuffer, lengthFrame, bCommandFrame, funcFrame);
      else
      {
        ZW_SendData(nextNode, (BYTE *)&txBuffer, lengthFrame, bCommandFrame, funcFrame);
      }
    }
    else
    {
      func(TRANSMIT_COMPLETE_NO_ACK);
    }
}

/*============================   TxCompleteCheckAck   ======================
**    Function description
**      This function checks transmit status and indicates if any
**      error occoured
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
TxCompleteCheckAck(
  BYTE bStatus)             /*IN Transmit status*/
{
  enNodeID = 0xff;

  txInProgress = FALSE;

  nextNode++;
  for (; nextNode < MAX_NODES; nextNode++)
  {
    if (IsNodeInGroup(nextNode))
    {
      txBuffer.ZW_BasicSetFrame.cmdClass = COMMAND_CLASS_BASIC;
      ZW_GET_NODE_STATE(nextNode,&nodeInfo);
      if(nodeInfo.nodeType.generic == GENERIC_TYPE_SWITCH_MULTILEVEL)
        txBuffer.ZW_BasicSetFrame.cmdClass = COMMAND_CLASS_BASIC;
      else if(nodeInfo.nodeType.generic == GENERIC_TYPE_ENTRY_CONTROL)
        txBuffer.ZW_BasicSetFrame.cmdClass = COMMAND_CLASS_LOCK;

      if (groupState&GROUP_ON_BIT)
      {
        if(nodeInfo.nodeType.generic == GENERIC_TYPE_SWITCH_MULTILEVEL)
          txBuffer.ZW_BasicSetFrame.value = SWITCHED_ON;   /* Switch ON */
        else if(nodeInfo.nodeType.generic == GENERIC_TYPE_ENTRY_CONTROL)
          txBuffer.ZW_BasicSetFrame.value = 0x1;   /* Switch ON */
      }
      break;
    }
  }
  if(nextNode<MAX_NODES)
  {
      if(nodesInGroup)
      {
        if (statusNodeIn(nextNode))
          ZW_SendDataSecure(nextNode, (BYTE *)&txBuffer, lengthFrame, bCommandFrame, funcFrame);
        else
        {
          ZW_SendData(nextNode, (BYTE *)&txBuffer, lengthFrame, bCommandFrame, funcFrame);
        }
      }
      else
      {
        clrTimerTr();
        ZW_TIMER_START(StopLED0Blink,LED0_ONTIME,TIMER_ONE_TIME);
        if(bStatus != TRANSMIT_COMPLETE_OK)
          mainState = CHECK_BUTTON_STATE;
        else
          mainState = RETURN_TO_IDLE_STATE;
      }
  }
  else
  {
    clrTimerTr();
    ZW_TIMER_START(StopLED0Blink,LED0_ONTIME,TIMER_ONE_TIME);
    if(bStatus != TRANSMIT_COMPLETE_OK)
      mainState = CHECK_BUTTON_STATE;
    else
      mainState = RETURN_TO_IDLE_STATE;
  }
}

/*============================   ToggleGroup   ==============================
**    Function description
**      Toggles the group on or off
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
ToggleGroup(void)
{
  /*Toggle state of the switches*/
  StartLED0Blink(BUSY_BLINK_RATE);
  ControlGroup(TOGGLE,TxCompleteCheckAck);
}

/*============================   DimStarted   ===============================
**    Function description
**        Call back called when Dim has been started. Ignores transmit status
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
DimStarted(
  BYTE bStatus)       /*IN Transmit status*/
{
  mainState = DIMMING_STATE;
  enNodeID = 0xff;
  nextNode++;
  for (; nextNode < MAX_NODES; nextNode++)
  {
    if (IsNodeInGroup(nextNode))
    {
      break;
    }
  }
  if(nextNode<MAX_NODES)
  {
      if(nodesInGroup)
      {
        if (statusNodeIn(nextNode))
          ZW_SendDataSecure(nextNode, (BYTE *)&txBuffer, lengthFrame, bCommandFrame, funcFrame);
        else
        {
          ZW_SendData(nextNode, (BYTE *)&txBuffer, lengthFrame, bCommandFrame, funcFrame);
        }
      }
  }
}
/*============================   GetGroupLevel   ======================
**    Function description
**      Gets the current dim level in External EEPROM
**    Side effects:
**
**--------------------------------------------------------------------------*/
BYTE GetGroupLevel(void)           /*RET Current group dim level*/
{
  return ZW_MEM_GET_BYTE(addressof(groupTable, GROUPTABLE, currentDimLevel));
}

/*============================   StopDimEndTimer   ======================
**    Function description
**      Stops the timeout function that is run when Dimming is ended..
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
StopDimEndTimer(void)
{
  if (dimEndTimerHandle!=0xff)
  {
    ZW_TIMER_CANCEL(dimEndTimerHandle);
    dimEndTimerHandle = 0xff;
  }
}

/*============================   SetGroupDimLevel   ==========================
**    Function description
**      Sets the dim level of the nodes in the group to the last groupDimLevel
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
SetGroupDimLevel(void)
{
  StopDimEndTimer();
  ControlGroup(DIM_SET,TxCompleteIgnoreAck);
}

/*============================   StartDimEndTimer   ======================
**    Function description
**      Starts the timeout that is running when dim is ended
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
StartDimEndTimer(void)
{
  StopDimEndTimer();
  dimEndTimerHandle = ZW_TIMER_START(SetGroupDimLevel, DIM_END_TIMEOUT, TIMER_ONE_TIME);
}

void
SetGroupDimLevelDimStop(BYTE bStatus)
{
  StartDimEndTimer();
}

void
DimStopped(
  BYTE bStatus)
{
  /*We do not care if the dim stop was succesfull or not. We want a level from a switch in the group*/
  BYTE temp;
  enNodeID = 0xff;
  temp = nextNode;
  nextNode++;
  for (; nextNode < MAX_NODES; nextNode++)
  {
    if (IsNodeInGroup(nextNode))
    {
      break;
    }
  }
  if(nextNode<MAX_NODES)
  {
      if(nodesInGroup)
      {
        if (statusNodeIn(nextNode))
          ZW_SendDataSecure(nextNode, (BYTE *)&txBuffer, lengthFrame, bCommandFrame, funcFrame);
        else
        {
          ZW_SendData(nextNode, (BYTE *)&txBuffer, lengthFrame, bCommandFrame, funcFrame);
        }
      }
  }
  else
  {
    /* Locate multilevel switches only */
    temp = 0;

    if (temp < nodesInGroup)
    {
      ZW_GET_NODE_STATE(groupNodeList[temp++],&nodeInfo);

      while ((nodeInfo.nodeType.generic != GENERIC_TYPE_SWITCH_MULTILEVEL) && (temp < nodesInGroup)&&(temp==nodeID))
      {
        ZW_GET_NODE_STATE(groupNodeList[temp++], &nodeInfo);
      }
      txBuffer.ZW_SwitchMultilevelGetFrame.cmdClass = COMMAND_CLASS_SWITCH_MULTILEVEL;
      txBuffer.ZW_SwitchMultilevelGetFrame.cmd = SWITCH_MULTILEVEL_GET;
      /* Any nodes at all? */
        if (!nodesInGroup)
        {
          mainState = ERROR_STATE;
          return;
        }
       enNodeID = 0xff;
       if (statusNodeIn(groupNodeList[temp-1]))
          ZW_SendDataSecure(groupNodeList[temp-1], (BYTE *)&txBuffer, sizeof(ZW_SWITCH_MULTILEVEL_GET_FRAME), TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_EXPLORE, SetGroupDimLevelDimStop);
       else
       {
          ZW_SendData(groupNodeList[temp-1], (BYTE *)&txBuffer, sizeof(ZW_SWITCH_MULTILEVEL_GET_FRAME), TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_EXPLORE, SetGroupDimLevelDimStop);
       }
         funcFrame = SetGroupDimLevelDimStop;
    }
  }
}

/*=======================   AssociationAddToGroupComplete   ==================
**    Function description
**      Called when Association add to group has completed
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
AssociationAddToGroupComplete(
  BYTE bStatus)             /*Transmit status*/
{
  ZW_TIMER_START(StopLED0Blink, LED0_ONTIME, TIMER_ONE_TIME);
  LED1_OFF;
  if (bStatus != TRANSMIT_COMPLETE_OK)
  {
    mainState = ERROR_STATE;
  }
  else
  {
    mainState = RETURN_TO_IDLE_STATE;
  }
}

void
AssignRouteNodeCompleted(
  LEARN_INFO *learnNodeInfo);  /*IN Status*/

/*============================   AssociationAddToGroup   =====================
**    Function description
**    Callback function called when Assign Return Route has finished. This
**    function sends an Association Set frame to the routing slave node
**    routingNode, which inserts the routeToNode node into association group 1.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                    /*RET Nothing */
AssociationAddToGroup(
  BYTE bStatus)         /* IN Status of assign return route */
{
  if (bStatus == TRANSMIT_COMPLETE_OK)
  {
    txBuffer.ZW_AssociationSet1byteFrame.cmdClass = COMMAND_CLASS_ASSOCIATION;
    txBuffer.ZW_AssociationSet1byteFrame.cmd = ASSOCIATION_SET;
    txBuffer.ZW_AssociationSet1byteFrame.groupingIdentifier = 1;
    txBuffer.ZW_AssociationSet1byteFrame.nodeId1 = routeToNode;
    enNodeID = 0xff;

    if (statusNodeIn(routingNode))
    {
      if (!ZW_SendDataSecure(routingNode, (BYTE *)&txBuffer, sizeof(ZW_ASSOCIATION_SET_1BYTE_FRAME),TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_EXPLORE, AssociationAddToGroupComplete))
      {
        AssociationAddToGroupComplete(TRANSMIT_COMPLETE_NO_ACK);
      }
    }
    else
    {
      if (!ZW_SendData(routingNode, (BYTE *)&txBuffer, sizeof(ZW_ASSOCIATION_SET_1BYTE_FRAME),TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_EXPLORE, AssociationAddToGroupComplete))
      {
        AssociationAddToGroupComplete(TRANSMIT_COMPLETE_NO_ACK);
      }
    }
  }
  else
  {
    AssociationAddToGroupComplete(TRANSMIT_COMPLETE_NO_ACK);
  }
}


BYTE AssociationAddToGroupHandle;

void AssociationAddToGroupTime()
{
  
    if(AssociationAddToGroupHandle!=0xff)
    {
      ZW_TIMER_CANCEL(AssociationAddToGroupHandle);
      AssociationAddToGroupHandle = 0xff;
    }
    ZW_ADD_NODE_TO_NETWORK(ADD_NODE_STOP, AssignRouteNodeCompleted);
}




/*========================   AssignRouteNodeCompleted   ======================
**    Function description
**    Callback function for Assign route. This function fetches the node ids.
**
**          bStatus BYTE,     IN  Status of learn process
**          bSource BYTE,     IN  Node id of the node that send node info
**          BYTE* pCmd,       IN  Pointer to Node information
**          BYTE bLen));      IN  Node info length
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
AssignRouteNodeCompleted(
  LEARN_INFO *learnNodeInfo)  /*IN Status*/
{
  if((*learnNodeInfo).bLen)
  {
    /*Only store ids and info when bLen is not Zero*/
    lastLearnedNodeType = ((NODE_TYPE *)(*learnNodeInfo).pCmd)->generic;
    lastLearnedNodeID = (*learnNodeInfo).bSource;
  }
  if ((*learnNodeInfo).bStatus == ADD_NODE_STATUS_LEARN_READY)
  {
    /*If Button is pressed exit learnmode*/
    mainState = WAIT_FOR_PB_DOWN_STATE;
  }
  else if ((*learnNodeInfo).bStatus == ADD_NODE_STATUS_ADDING_SLAVE)
  {
    mainState = EXECUTE_STATE;

    addController = FALSE;
    if(!routeToNode)
    {
      /* This node ID is the device that calculated routes should be to */
      routeToNode = lastLearnedNodeID;
    }
    else if(!routingNode)
    {
      /* This node ID is the device that routes should be sent to*/
      routingNode = lastLearnedNodeID;
    }
    StartLED0Blink(BUSY_BLINK_RATE);
  }
  else if(((*learnNodeInfo).bStatus == ADD_NODE_STATUS_PROTOCOL_DONE))
  {
    /*Stop learnmode. Handle completed in Done*/
    
    /*  TO#2606   */
    AssociationAddToGroupHandle = ZW_TIMER_START(AssociationAddToGroupTime,40, TIMER_FOREVER);
  }
  else if(((*learnNodeInfo).bStatus == ADD_NODE_STATUS_DONE))
  {
    /*If either is zero go back and look for new*/
    if (!routingNode)
    {
      StopLED0Blink();
      LED1_ON;
      ZW_ADD_NODE_TO_NETWORK(ADD_NODE_EXISTING,AssignRouteNodeCompleted);
    }
    else
    {
      /* This node is the node that should receive the routes */
      StartLED0Blink(BUSY_BLINK_RATE);

      ZW_ADD_NODE_TO_NETWORK(ADD_NODE_STOP, NULL);

      if(ZW_ASSIGN_RETURN_ROUTE(routingNode, routeToNode, AssociationAddToGroup))
      {
        mainState = EXECUTE_STATE;
      }
      else
      {
        StopLED0Blink();
        mainState = ERROR_STATE;
      }
    }
  }
  else if((*learnNodeInfo).bStatus == ADD_NODE_STATUS_FAILED)
  {
    StopLED0Blink();
    mainState = ERROR_STATE;
  }
}

void
funcRequestNetworkUpdate(auto BYTE byteStatus)
{
  LED0_OFF;
}

/*=============================  ApplicationPoll   =========================
**    Application poll function for LED dimmer
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                    /*RET  Nothing                  */
ApplicationPoll( void ) /* IN  Nothing                  */
{
  if(getIoflagsTx()) return;
  if(getTimerTr()) return;
  
  switch (mainState)
  {
    case PWR_UP_STATE:
      nodesInGroup = UpdateRAMGroupList();
      groupLevel = GetGroupLevel();
      if (!ButtonPressed())              /*Stay here until no key is pressed*/
      {
        mainState = RETURN_TO_IDLE_STATE;
      }
      LED0_OFF;
      LED1_OFF;
      break;

    case RETURN_TO_IDLE_STATE:
      mainState = CHECK_BUTTON_STATE;
      /*Drop trough to next state*/

    case CHECK_BUTTON_STATE:
      /* No RF needed when were in this state. */

      if (bPressed7DEBOUNCE)
      {
        bPressed7DEBOUNCE = 0;
        if (ZW_RequestNetworkUpdate(funcRequestNetworkUpdate))LED0_ON;
      }
      if (glKey = ButtonPressed())
      {
        LED1_OFF;
        if (GROUP_CTRL_ACTIVATION)
        {
          mainState = GROUP_CONTROL_STATE;    /*group Toggle, dim and add node to group*/
        }
        if (ADD_NODE_ACTIVATION)
        {
          mainState = INCLUDE_NODE_STATE;     /*Add a new node to network*/
        }
        if (RESET_NODE_ACTIVATION)
        {
          mainState = REMOVE_NODE_STATE;      /*Remove Node from network (and group)*/
        }
        if (ASSIGN_ROUTE_ACTIVATION)
        {
          mainState = ASSIGN_ROUTE_STATE;     /*Assign route from node to node*/
        }
        if (ADD_CTRL_ACTIVATION)
        {
          mainState = NEW_CONTROLLER_STATE;   /*Add a slave controller to the network*/
        }
        if (RESET_ACTIVATION)
        {
          StartResetTimer();
          mainState = RESET_CONTROLLER_STATE; /*Reset this controller*/
        }
      }
      else
      {
        if (bStartUpdateTimer)
        {
            bStartUpdateTimer = FALSE;
        }
      }
      break;

    case GROUP_CONTROL_STATE:
      /*User hit the Group control button, handle it*/
      glKey = ButtonPressed();
      ZW_ADD_NODE_TO_NETWORK(ADD_NODE_STOP, NULL);
      if (RESET_ACTIVATION)
      {
        StartResetTimer();
        mainState = RESET_CONTROLLER_STATE; /*PB4 is down too! We are in the wrong state*/
        break;
      }
      if (!glKey)
      {
        /*Key was released before we entered dimming state*/
        setTimerTr();
        LED0_ON;
        mainState = CHECK_BUTTON_STATE;
        ToggleGroup();
        txInProgress = TRUE;
      }
      else if(IS_PB_HELD(glKey))
      {
        /*If the key is being held. Start the dimming routine*/
        mainState = EXECUTE_STATE;
        LED0_ON;
        ZW_RequestNetworkUpdate(/*NULL*/funcRequestNetworkUpdate);
        ControlGroup(DIM_START, DimStarted);
        mainState = DIMMING_STATE;
      }
      break; /*GROUP_CONTROL_STATE*/

    case INCLUDE_NODE_STATE:
      glKey = ButtonPressed();
      if (!glKey)
      {
        mainState = EXECUTE_STATE;
        LED0_ON;
        /* User released before we entered Include state - send nodeinformation */
        nextNode = 0xf0;
        ZW_SEND_NODE_INFO(NODE_BROADCAST, 0, TxCompleteIgnoreAck);
      }
      else if (IS_PB_HELD(glKey))
      {
        /* User wants to include a new slave node to the network. Handle it */
        StartLED0Blink(BUSY_BLINK_RATE);
        mainState = LEARN_STATE;
        NodeS();
        setCtrlIncluding();
        ZW_ADD_NODE_TO_NETWORK((ADD_NODE_ANY | ADD_NODE_OPTION_HIGH_POWER | ADD_NODE_OPTION_NETWORK_WIDE), NewControllerSendStarted);
      }
      break; /*INCLUDE_NODE_STATE*/

    case LEARN_STATE:
      if (!(ButtonPressed() & PB_MASK))
      {
        ZW_TIMER_START(StopLED0Blink,LED0_ONTIME,TIMER_ONE_TIME);
        mainState = WAIT_FOR_PB_RELEASE_STATE;
      }
      break; /*LEARN_STATE*/

    case REMOVE_NODE_STATE:
      LED0_ON;
      ZW_REMOVE_NODE_FROM_NETWORK(REMOVE_NODE_ANY, RemoveNodeCompleted);
      mainState = LEARN_STATE;
      break; /*REMOVE_NODE_STATE*/

    case ASSIGN_ROUTE_STATE:
      /*Get nodeID of the device the routing device should receive a route to*/
      if (!ButtonPressed())
      {
        /*When Button released do it*/
        LED0_ON;
        mainState = EXECUTE_STATE;
        routeToNode = 0;
        routingNode = 0;
///////////////////////////////////////////////////////////////////////////////////////
        ZW_ADD_NODE_TO_NETWORK(ADD_NODE_EXISTING, AssignRouteNodeCompleted);
        mainState = WAIT_FOR_PB_DOWN_STATE;
      }
      break; /*ASSIGN_ROUTE_STATE*/

    case NEW_CONTROLLER_STATE:
      glKey = ButtonPressed();
      if (!glKey)
      {
        /*Key was released, then start replication transmit*/
          mainState = ERROR_STATE;
      }
      else if(IS_PB_HELD(glKey))
      {
        /* If the key is being held. Start the New controller receive */
        LED0_ON;
        NodeS();//nodeSecure = 1;
        StopUpdateTimer();
        receivingRemote = TRUE;
        ClearGroupTable(NULL);
        setCtrlIncluded();
        ZW_SetLearnMode(TRUE, NewControllerReceiveStarted);
        mainState = WAIT_FOR_PB_RELEASE_STATE;
      }
      break; /*NEW_CONTROLLER_STATE*/

    case RESET_CONTROLLER_STATE:
      if (!ButtonPressed())
      {
        /*If button is released do not reset!!*/
        StopResetTimer();
        mainState = RETURN_TO_IDLE_STATE;
      }
      break; /*RESET_CONTROLLER_STATE*/
    /*****************************************************************/
    /*Sub states. These are not executed directly from a button press*/
    /*****************************************************************/
    case DIMMING_STATE: /* Wait for key to be released*/
      if (!ButtonPressed())
      {
        StartLED0Blink(BUSY_BLINK_RATE);
        mainState = EXECUTE_STATE;
        if(!statusNodeIn(addNodeToGroup))
          ControlGroup(DIM_STOP,DimStopped);
        else
        {
          enNodeID = 0xff;
          statAddNodeToGroup = TRUE;
          txBuffer.ZW_SecurityCommandsSupportedGetFrame.cmdClass                       = COMMAND_CLASS_SECURITY;
          txBuffer.ZW_SecurityCommandsSupportedGetFrame.cmd                            = SECURITY_COMMANDS_SUPPORTED_GET;
          ZW_SendDataSecure(addNodeToGroup, (BYTE *)&txBuffer, sizeof(txBuffer.ZW_SecurityCommandsSupportedGetFrame), TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_EXPLORE, 0);
        }
        addNodeToGroup = 0x0;
      }
      if (RESET_ACTIVATION)
      {
        StartResetTimer();
        mainState = RESET_CONTROLLER_STATE; /*Reset this controller*/
      }
      break; /*DIMMING_STATE*/

    case ERROR_STATE: /*Last operation failed..*/
      mainState = WAIT_FOR_PB_RELEASE_STATE;
      ZW_TIMER_START(SoftResetDelay, 250, TIMER_ONE_TIME);
      break;

    case WAIT_FOR_PB_DOWN_STATE:
      if (ButtonPressed())
      {
        mainState = RETURN_TO_IDLE_STATE;
        StopLED0Blink();
        LED1_OFF;
      }
      break; /*WAIT_FOR_PB_DOWN_STATE*/

    case WAIT_FOR_PB_RELEASE_STATE:
      if (!ButtonPressed())
      {
        LED0_OFF;
        ZW_ADD_NODE_TO_NETWORK(ADD_NODE_STOP, NULL);
        ZW_SetLearnMode(FALSE, NULL);
        mainState = RETURN_TO_IDLE_STATE;
      }
      if (RESET_ACTIVATION)
      {
        StartResetTimer();
        mainState = RESET_CONTROLLER_STATE; /*Reset this controller*/
      }
      break;

    case WAIT_FOR_PB_PRESS_STATE:
      if (ButtonPressed())
      {
        ZW_ADD_NODE_TO_NETWORK(ADD_NODE_STOP, NULL);
        ZW_SetLearnMode(FALSE, NULL);
        mainState = WAIT_FOR_PB_DEPRESS_STATE;
      }
      break;

    case WAIT_FOR_PB_DEPRESS_STATE:
      if (!ButtonPressed())
      {
        LED0_OFF;
        mainState = RETURN_TO_IDLE_STATE;
      }
      break;

    case EXECUTE_STATE: /* We are executing a command. State changes should be done by callbacks*/

      if (!ButtonPressed())  mainState = RETURN_TO_IDLE_STATE;
      break;

    case EXECUTE_STATE_RESET:
      if (!ButtonPressed())  mainState = RETURN_TO_IDLE_STATE;
      break;

    default: /*Something went horribly wrong.. Return to idle*/
      mainState = RETURN_TO_IDLE_STATE;
      break;
  }
}

/*============================   AddNodeToGroup   ======================
**    Function description
**      Adds a node to a group.
**    Side effects:
**      updates the groupArray
**--------------------------------------------------------------------------*/
void AddNodeToGroup(BYTE bNodeID)
{
  StoreNodeBitInGroup(bNodeID, 1);
}

extern  BYTE myDeviceOptionsMask;       /* Bitmask with application options   */
extern  APPL_NODE_TYPE myNodeType;      /* Device type Generic and Specific   */
extern  BYTE *myNodeParm;               /* Device parameter buffer pointer    */
extern  BYTE myParmLength;              /* Number of Device parameter bytes   */

void
securityEnd
(BYTE bStatus)
{
  // Initialize Application Node information
  // before transmitting "Node Information" to the Zwave module.
  ApplicationNodeInformation(
    &myDeviceOptionsMask,         /*OUT Bitmask with application options    */
    &myNodeType,                  /*OUT  Device type Generic and Specific   */
    &myNodeParm,                  /*OUT  Device parameter buffer pointer    */
    &myParmLength);               /*OUT  Number of Device parameter bytes   */


  // Send Application Node information to the Zwave module
  // before transmitting a "Node Information" frame
  // with "ZW_SendNodeInformation()".
  SerialAPI_ApplicationNodeInformation(
    myDeviceOptionsMask,
    myNodeType,
    myNodeParm,
    myParmLength);  
}

void
securityCommandsSupported(BYTE bStatus)
{
  enNodeID = 0xff;
  txBuf.ZW_SecurityCommandsSupportedReportFrame.cmdClass = COMMAND_CLASS_SECURITY;
  txBuf.ZW_SecurityCommandsSupportedReportFrame.cmd = SECURITY_COMMANDS_SUPPORTED_REPORT;
  if (securityFrameCount)
  {
    securityFrameCount--;
    if (securityFrameCount)
    {
      memcpy( &txBuf.ZW_SecurityCommandsSupportedReportFrame.cmdClassSupported[1], securityFramePos, (sizeof(txBuf.ZW_SecurityCommandsSupportedReportFrame)-3));
      securityFramePos+=(sizeof(txBuf.ZW_SecurityCommandsSupportedReportFrame)-3);
      txBuf.ZW_SecurityCommandsSupportedReportFrame.cmdClassSupported[0] = securityFrameCount;
      ZW_SendDataSecure(reportNode, (BYTE *)&txBuf, 3 + (sizeof(txBuf.ZW_SecurityCommandsSupportedReportFrame)-2), TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_EXPLORE, securityCommandsSupported);
    }
    else
    {
      memcpy( &txBuf.ZW_SecurityCommandsSupportedReportFrame.cmdClassSupported[1], securityFramePos, CLASS_REPORT_COUNT%(sizeof(txBuf.ZW_SecurityCommandsSupportedReportFrame)-3));
      txBuf.ZW_SecurityCommandsSupportedReportFrame.cmdClassSupported[0] = securityFrameCount;
      ZW_SendDataSecure(reportNode, (BYTE *)&txBuf, (CLASS_REPORT_COUNT%(sizeof(txBuf.ZW_SecurityCommandsSupportedReportFrame)-2)+3),TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_EXPLORE, securityCommandsSupported);
    }
  }
}

void
securityCommandsTimer(BYTE bStatus)
{
    NodeS();
    StartSecurityTimeOut(TIME_SECURITY_LIFE);
}

/*========================   ApplicationCommandHandler   ====================
**    Handling of a received application commands and requests
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void                              /*RET Nothing                  */
ApplicationCommandHandler(
  BYTE  rxStatus,                 /* IN Frame header info */
  BYTE  sourceNode,               /* IN Command sender Node ID */
  ZW_APPLICATION_TX_BUFFER *pCmd,///////BYTE *pCmd, /* IN Payload from the received frame, the union */
                                  /*    should be used to access the fields */
  BYTE   cmdLength)               /* IN Number of command bytes including the command */
{
  BYTE *pBuf;
  BYTE *pCmdBuf;

  ZW_APPLICATION_TX_BUFFER_NEW *pCmdTMP;
  pCmdTMP = (ZW_APPLICATION_TX_BUFFER_NEW*)pCmd;
  pCmdBuf = (BYTE *)pCmd;
  
  
  
  switch(pCmdTMP->ZW_Common.cmdClass)
  {
    case COMMAND_CLASS_SWITCH_MULTILEVEL:
      if (pCmdTMP->ZW_Common.cmd == SWITCH_MULTILEVEL_REPORT)
      {
        groupLevel = pCmdTMP->ZW_SwitchMultilevelReportFrame.value;
        StoreGroupLevel(groupLevel);
        StopDimEndTimer();
        if (!receivedReport)
        {
          SetGroupDimLevel();
        }
        receivedReport = TRUE;
      }
      break;
    case COMMAND_CLASS_SWITCH_BINARY:
      break;

    case COMMAND_CLASS_BASIC:
      break;

    case COMMAND_CLASS_CONTROLLER_REPLICATION:
      /* Handle replication frames */
      if (receivingRemote)
      {
        /* We are only interested in group replication here */
        /* but all other replication can be handled in the same way. */
        /* In this application we ignore everything else and just */
        /* send the command complete reply */
        if (pCmdTMP->ZW_Common.cmd == CTRL_REPLICATION_TRANSFER_GROUP)
        {
          /* We are only interested in group 1 */
          if (pCmdTMP->ZW_CtrlReplicationTransferGroupFrame.groupId == 1)
          {
            /* Add the node ID to the group */
            AddNodeToGroup(pCmdTMP->ZW_CtrlReplicationTransferGroupFrame.nodeId);
          }
        }
        /* Send command complete to other controller */
        /* NOTE: This call should be done when ready to receive */
        /* the next replication frame */
        ZW_REPLICATION_COMMAND_COMPLETE();
      }
      break;
    case COMMAND_CLASS_VERSION:
      if (pCmdTMP->ZW_Common.cmd == VERSION_GET)
      {
        txBuffer.ZW_VersionReportFrame.cmdClass = COMMAND_CLASS_VERSION;
        txBuffer.ZW_VersionReportFrame.cmd = VERSION_REPORT;
        txBuffer.ZW_VersionReportFrame.zWaveLibraryType = ZW_TYPE_LIBRARY();
        txBuffer.ZW_VersionReportFrame.zWaveProtocolVersion = ZW_VERSION_MAJOR;
        txBuffer.ZW_VersionReportFrame.zWaveProtocolSubVersion = ZW_VERSION_MINOR;
        txBuffer.ZW_VersionReportFrame.applicationVersion = PORTABLE_CTRL_VERSION;
        txBuffer.ZW_VersionReportFrame.applicationSubVersion = PORTABLE_CTRL_REVISION;
        ZW_SendData(sourceNode, (BYTE *)&txBuffer, sizeof(txBuffer.ZW_VersionReportFrame), TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_EXPLORE, NULL);
      }
      break;
    case COMMAND_CLASS_SECURITY:
      
      
      if (statusNodeIn(sourceNode))
      {
        switch(pCmdTMP->ZW_Common.cmd)
        {
          case SECURITY_SCHEME_REPORT:
            StopSecurityTimeOut();
            ZW_DEBUG_SEND_BYTE('1');
            nodeSecureIncl = 1;
            
          if (nodeSecure)
          {
            if (SECURITY_SCHEME==SECURITY_SCHEME_0)
            {
              if (pCmdTMP->ZW_SecuritySchemeReportFrame.supportedSecuritySchemes == SECURITY_SCHEME_0)
              {
                StopSecurityTimeOut();
                ZW_DEBUG_SEND_BYTE('2');
                enNodeID = 0xff;
                txBuffer.ZW_NetworkKeySetFrame.cmdClass                       = COMMAND_CLASS_SECURITY;
                txBuffer.ZW_NetworkKeySetFrame.cmd                            = NETWORK_KEY_SET;
                memset(inclusionKey, 0x0, 16);
                OutKeyNewController = 1;
                memcpy((BYTE *)txBuffer.ZW_NetworkKeySetFrame.networkKey,networkKey,16);
                NodeS();
                nodeInWork = sourceNode;
                StartSecurityTimeOut(TIME_SECURITY_LIFE);
                ZW_SendDataSecure(sourceNode, (BYTE *)&txBuffer, sizeof(txBuffer.ZW_NetworkKeySetFrame), TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_EXPLORE, 0);
              }
              else OutKeyNewController = 0xff;
            }
          }
          break;
  
          case SECURITY_NONCE_REPORT:
            enNodeID = sourceNode;
            StopSecuritySendTimeOut();
            RegisterENOut(sourceNode,&pCmdBuf[2]);
          break;
  
        case SECURITY_SCHEME_GET:
          StopSecurityTimeOut();
  ZW_DEBUG_SEND_BYTE('3');
          if (nodeSecure)
          {
            if ((BYTE)pCmdBuf[2]!=SECURITY_SCHEME_0)
            {
              nodeSecure = 0;
              setCtrlNoneSecure();
              /////TO#02438 Secure dev. ctrl ignores invalid scheme
              return;
            }
            MemoryGetID(bufHomeID,&nodeID);
            if (SECURITY_SCHEME==SECURITY_SCHEME_0)
            {
              if (pCmdBuf[2]==SECURITY_SCHEME_0)
              {
                memset(networkKey,0x00,16);
                LoadKeys();
              }
            }
            setCtrlSecure();
            txBuf.ZW_SecuritySchemeReportFrame.cmdClass = COMMAND_CLASS_SECURITY; /* The command class */
            txBuf.ZW_SecuritySchemeReportFrame.cmd = SECURITY_SCHEME_REPORT; /* The command */
            txBuf.ZW_SecuritySchemeReportFrame.supportedSecuritySchemes = SECURITY_SCHEME;
            ZW_SendData(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_SecuritySchemeReportFrame), TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_EXPLORE, NULL);
            nodeSecureIncl = 1;
            NodeS();
            nodeInWork = sourceNode;
            StartSecurityTimeOut(TIME_SECURITY_LIFE);
          }
          break;
  
        default:
          pBuf = (BYTE *)pCmd + 1;
          
          if (!nodeSecure) break;
          
          cmdLength = ProcessIncomingSecure(sourceNode, pBuf, TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_EXPLORE,cmdLength - 1);
          enNodeID = 0xff;
          if (cmdLength && (cmdLength < 255))
          {
            pCmdTMP = (ZW_APPLICATION_TX_BUFFER_NEW *)(pBuf + 9 + 1);
            switch(pCmdTMP->ZW_Common.cmdClass)
            {
              case COMMAND_CLASS_SWITCH_MULTILEVEL:
                if (pCmdTMP->ZW_Common.cmd == SWITCH_MULTILEVEL_REPORT)
                {
                  groupLevel = pCmdTMP->ZW_SwitchMultilevelReportFrame.value;
                  StoreGroupLevel(groupLevel);
                  StopDimEndTimer();
                  if (!receivedReport)
                  {
                    SetGroupDimLevel();
                  }
                  receivedReport = TRUE;
                }
                break;
              case COMMAND_CLASS_VERSION:
                if (pCmdTMP->ZW_Common.cmd == VERSION_GET)
                {
                  txBuffer.ZW_VersionReportFrame.cmdClass = COMMAND_CLASS_VERSION;
                  txBuffer.ZW_VersionReportFrame.cmd = VERSION_REPORT;
                  txBuffer.ZW_VersionReportFrame.zWaveLibraryType = ZW_TYPE_LIBRARY();
                  txBuffer.ZW_VersionReportFrame.zWaveProtocolVersion = ZW_VERSION_MAJOR;
                  txBuffer.ZW_VersionReportFrame.zWaveProtocolSubVersion = ZW_VERSION_MINOR;
                  txBuffer.ZW_VersionReportFrame.applicationVersion = PORTABLE_CTRL_VERSION;
                  txBuffer.ZW_VersionReportFrame.applicationSubVersion = PORTABLE_CTRL_REVISION;
                  ZW_SendDataSecure(sourceNode, (BYTE *)&txBuffer, sizeof(txBuf.ZW_VersionReportFrame), TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_EXPLORE, securityCommandsTimer);
                }
                break;
              case COMMAND_CLASS_SECURITY:
                if (pCmdTMP->ZW_Common.cmd == NETWORK_KEY_VERIFY)
                {
                  nodeSecInclude = 0;
                  StopSecurityTimeOut();
                  ZW_DEBUG_SEND_BYTE('4');
                  if (nodeSecure)
                  {
                    AddSecuritySlave(sourceNode,TRUE);
                    if (addController)
                    {
                      enNodeID = 0xff;
                      txBuf.ZW_SecuritySchemeInheritFrame.cmdClass = COMMAND_CLASS_SECURITY; /* The command class */
                      txBuf.ZW_SecuritySchemeInheritFrame.cmd = SECURITY_SCHEME_INHERIT; /* The command */
                      txBuf.ZW_SecuritySchemeInheritFrame.supportedSecuritySchemes = SECURITY_SCHEME;

                      ZW_SendDataSecure(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_SecuritySchemeInheritFrame), TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_EXPLORE, securityCommandsTimer);
                      NodeS();
                      nodeInWork = sourceNode;
                      StartSecurityTimeOut(TIME_SECURITY_LIFE);
                    }
                  }
                  else if (IS_PB_HELD(glKey))
                  {
                      /* User wants to include a new slave node to the network. Handle it */
                    StartLED0Blink(BUSY_BLINK_RATE);
                    mainState = INCLUDE_NODE_STATE;
                    NodeS();
                    setCtrlIncluding();
//                    ZW_ADD_NODE_TO_NETWORK((ADD_NODE_ANY | ADD_NODE_OPTION_HIGH_POWER | ADD_NODE_OPTION_NETWORK_WIDE), NewControllerSendStarted);
                  }
                }
                else if (pCmdTMP->ZW_Common.cmd == SECURITY_SCHEME_REPORT)
                {
                  StopSecurityTimeOut();
                  ZW_DEBUG_SEND_BYTE('5');
                  if (IS_PB_HELD(glKey))
                  {
                      /* User wants to include a new slave node to the network. Handle it */
                      StartLED0Blink(BUSY_BLINK_RATE);
                      mainState = INCLUDE_NODE_STATE;
                      NodeS();
                      setCtrlIncluding();
                      ZW_ADD_NODE_TO_NETWORK((ADD_NODE_ANY | ADD_NODE_OPTION_HIGH_POWER | ADD_NODE_OPTION_NETWORK_WIDE), NewControllerSendStarted);
                  }
                }
                else if (pCmdTMP->ZW_Common.cmd ==  SECURITY_SCHEME_INHERIT)
                {
                  StopSecurityTimeOut();
                  ZW_DEBUG_SEND_BYTE('6');
                  if(nodeSecureIncl==2)
                  if (nodeSecure)
                  {
                    enNodeID = 0xff;
                    if (pCmdTMP->ZW_SecuritySchemeInheritFrame.supportedSecuritySchemes == SECURITY_SCHEME_0)
                    {
                      SECURITY_SCHEME = pCmdTMP->ZW_SecuritySchemeInheritFrame.supportedSecuritySchemes;
                    }
                    else
                    {
                      /////TO#02438 Secure dev. ctrl ignores invalid scheme 
                      nodeSecure = 0;
                      setCtrlNoneSecure();                    
                      return;                    
                    }
                    
                    txBuf.ZW_SecuritySchemeReportFrame.cmdClass = COMMAND_CLASS_SECURITY; /* The command class */
                    txBuf.ZW_SecuritySchemeReportFrame.cmd = SECURITY_SCHEME_REPORT; /* The command */
                    txBuf.ZW_SecuritySchemeReportFrame.supportedSecuritySchemes = SECURITY_SCHEME;
                    ZW_SendDataSecure(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_SecuritySchemeReportFrame), TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_EXPLORE, securityEnd);
                  }
                }
                else if (pCmdTMP->ZW_Common.cmd == SECURITY_COMMANDS_SUPPORTED_GET)
                {
                  enNodeID = 0xff;
                  txBuf.ZW_SecurityCommandsSupportedReportFrame.cmdClass = COMMAND_CLASS_SECURITY;
                  txBuf.ZW_SecurityCommandsSupportedReportFrame.cmd = SECURITY_COMMANDS_SUPPORTED_REPORT;
                  securityFrameCount = CLASS_REPORT_COUNT/(sizeof(txBuf.ZW_SecurityCommandsSupportedReportFrame)-3);
                  if (securityFrameCount)
                  {
                    memcpy( &txBuf.ZW_SecurityCommandsSupportedReportFrame.cmdClassSupported[1], &nodeInfoFrame, (sizeof(txBuf.ZW_SecurityCommandsSupportedReportFrame)-3));
                    securityFramePos = nodeInfoFrame;
                    securityFramePos+=(sizeof(txBuf.ZW_SecurityCommandsSupportedReportFrame)-3);
                    txBuf.ZW_SecurityCommandsSupportedReportFrame.cmdClassSupported[0] = securityFrameCount;
                    reportNode = sourceNode;
                    ZW_SendDataSecure(sourceNode, (BYTE *)&txBuf, 3 + (sizeof(txBuf.ZW_SecurityCommandsSupportedReportFrame)-2), TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_EXPLORE, securityCommandsSupported);
                  }
                  else
                  {
                    memcpy( &txBuf.ZW_SecurityCommandsSupportedReportFrame.cmdClassSupported[1], &nodeInfoFrame, CLASS_REPORT_COUNT);
                    securityFramePos = nodeInfoFrame;
                    securityFramePos+=CLASS_REPORT_COUNT;
                    txBuf.ZW_SecurityCommandsSupportedReportFrame.cmdClassSupported[0] = securityFrameCount;
                    reportNode = sourceNode;
                    ZW_SendDataSecure(sourceNode, (BYTE *)&txBuf, 3 + CLASS_REPORT_COUNT, TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_EXPLORE, securityCommandsSupported);
                  }
                }
                else if (pCmdTMP->ZW_Common.cmd == SECURITY_COMMANDS_SUPPORTED_REPORT)
                {
                  if(statAddNodeToGroup)
                  {
                    statAddNodeToGroup = FALSE;
                    AddSecuritySlave(sourceNode,TRUE);
                  }
                  enNodeID = 0xff;
                  ControlGroup(DIM_STOP,DimStopped);
                }
                else if (pCmdTMP->ZW_Common.cmd==NETWORK_KEY_SET)
                {
                  StopSecurityTimeOut();
                  ZW_DEBUG_SEND_BYTE('7');
                  if(nodeSecureIncl)
                  if(nodeSecure)
                  {
                    memcpy(networkKey,pCmdTMP->ZW_NetworkKeySetFrame.networkKey,16);
                    /* Now network Key should contain the new networkKey */
                    /* Save the new networkKey in NV RAM */
                    /* Go and make the Authentication and the encryption keys */
                    LoadKeys();
                    enNodeID = 0xff;
                    setCtrlSecure();
                    txBuf.ZW_NetworkKeyVerifyFrame.cmdClass = COMMAND_CLASS_SECURITY;
                    txBuf.ZW_NetworkKeyVerifyFrame.cmd = NETWORK_KEY_VERIFY;
                    ZW_SendDataSecure(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_NetworkKeyVerifyFrame), TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_EXPLORE, NULL);
                    NodeS();
                    nodeSecureIncl = 2;
                    nodeInWork = sourceNode;
                    StartSecurityTimeOut(TIME_SECURITY_LIFE);
                  }
                }
  
                break;
              default:
                break;
            }
          }
        break;
        }
      }
      break;
  }
}

/****************************************************************************/
/*                           EXPORTED FUNCTIONS                             */
/****************************************************************************/
#ifdef ZW030x
/*===========================   ApplicationRfNotify   ===========================
**    Notify the application when the radio switch state
**    Called from the Z-Wave PROTOCOL When radio switch from Rx to Tx or from Tx to Rx
**    or when the modulator PA (Power Amplifier) turn on/off
**---------------------------------------------------------------------------------*/
void          /*RET Nothing */
ApplicationRfNotify(
  BYTE rfState)         /* IN state of the RF, the available values is as follow:
                               ZW_RF_TX_MODE: The RF switch from the Rx to Tx mode, the modualtor is started and PA is on
                               ZW_RF_PA_ON: The RF in the Tx mode, the modualtor PA is turned on
                               ZW_RF_PA_OFF: the Rf in the Tx mode, the modulator PA is turned off
                               ZW_RF_RX_MODE: The RF switch from Tx to Rx mode, the demodulator is started.*/
{
}
#endif

#if 0
/*============================   ApplicationInitHW   ========================
**    Non Z-Wave hardware initialization
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
BYTE                       /*RET  TRUE        */
ApplicationInitHW( BYTE bStatus )  /* IN  Nothing     */
{
  /* hardware initialization */
  PIN_IN(Button, 1);
  PIN_IN(Button, 2);
  PIN_IN(Button, 3);
  PIN_IN(Button, 4);
  PIN_IN(Button, 5);
  PIN_IN(Button, 6);
  /* Setup specifics for current dim module */
  PIN_OUT(LED1);
  PIN_OUT(LED2);
  PIN_OUT(LED3);
  PIN_OUT(LED4);
  PIN_OUT(LED5);
  PIN_OUT(LED6);

  return(TRUE);
}
#endif
extern BYTE nrNodeID;


void                       /*RET  TRUE        */
SetControllerSecure( BYTE SecStatus )  /* IN  Nothing     */
{
  ZW_MEM_PUT_BYTE(EEOFFSET_NETWORK_SECURITY, SecStatus);
}

/*===========================   ApplicationInitSW   =========================
**    Initialization of the Application Software
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
BYTE                      /*RET  TRUE       */
ApplicationInitSW( void ) /* IN   Nothing   */
{
  ZW_SET_RX_MODE(TRUE); 
  if (ZW_MEM_GET_BYTE(addressof(groupTable, GROUPTABLE, magicValue)) != MAGIC_VALUE)
  {
    ClearGroupTable(NULL);
    ZW_MEM_PUT_BYTE(addressof(groupTable, GROUPTABLE, magicValue), MAGIC_VALUE);
    setCtrlSecure();
    groupLevel = 0;
    StoreGroupLevel(groupLevel);
    ZW_MEM_PUT_BUFFER(addressof(groupTable, GROUPTABLE, nodeMask),groupMask, sizeof(groupMask), NULL);
    ZW_MEM_PUT_BUFFER(addressof(groupTable, GROUPTABLE, nodeSecurityMask),groupSecurityMask, sizeof(groupMask), NULL);
  }
  staticCtrlNodeID = ZW_GET_SUC_NODEID();
  if (staticCtrlNodeID)
  {
    bStartUpdateTimer = TRUE;
  }
  NodeS();

  mainState = PWR_UP_STATE;
  isControllerSecure = ZW_MEM_GET_BYTE(EEOFFSET_NETWORK_SECURITY);
  ZW_MEM_GET_BUFFER(addressof(groupTable, GROUPTABLE, nodeMask),groupMask, sizeof(groupMask));
  ZW_MEM_GET_BUFFER(addressof(groupTable, GROUPTABLE, nodeSecurityMask),groupSecurityMask, sizeof(groupMask));

  if (ZW_RequestNetworkUpdate(funcRequestNetworkUpdate)) LED0_ON;
  
  nodeID = 1;
  return(TRUE);
}

/*======================   ApplicationNodeInformation   =====================
**    Request Node information and current status
**    Called by the the Z-Wave application layer before transmitting a
**    "Node Information" frame.
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
extern void                  /*RET  Nothing */
ApplicationNodeInformation(
  BYTE   *deviceOptionsMask,      /*OUT Bitmask with application options     */
  APPL_NODE_TYPE  *nodeType,  /*OUT  Device type Generic and Specific   */
  BYTE       **nodeParm,      /*OUT  Device parameter buffer pointer    */
  BYTE       *parmLength)     /*OUT  Number of Device parameter bytes   */
{
  /* this is a listening node and it supports optional CommandClasses*/
  *deviceOptionsMask = APPLICATION_NODEINFO_NOT_LISTENING|APPLICATION_NODEINFO_OPTIONAL_FUNCTIONALITY;
  nodeType->generic = GENERIC_TYPE_GENERIC_CONTROLLER; /* Generic device type */
  nodeType->specific = SPECIFIC_TYPE_PORTABLE_REMOTE_CONTROLLER; /* Specific class */
  *nodeParm = nodeInfoList.memberClass;       /* Send list of known command classes. */
  *parmLength = sizeof(nodeInfoList);       /* Set length*/

  MemoryGetID(bufHomeID,&nodeID);  
  if (nodeID !=1 )
	if (isCtrlSecure())
	{
  		*nodeParm = nodeInfoAfterIncluded;        /* Send list of known command classes. */
  		*parmLength = sizeof(nodeInfoAfterIncluded);       /* Set length*/
	}
}

/*=====================   NewControllerReceiveStarted  ======================
**    Function description
**        Called by the protocol as the replication progresses.
**        When NEW_CONTROLLER_LEARNED has been received the user can not abort
**        BYTE bStatus,   IN  Status of learn process
**        BYTE bSource,  IN  Node id of the node that send node info
**        BYTE* pCmd,    IN  Pointer to Node information
**        BYTE bLen.     IN  Node info length

**    Side effects:
**
**--------------------------------------------------------------------------*/
void
NewControllerReceiveStarted(
  /*BYTE bStatus,  BYTE nodeID*/
  LEARN_INFO *learnNodeInfo)
{
  nodeSecInclude = 1;
  AddSecuritySlave(nodeID,TRUE);
  
  if (learnNodeInfo->bStatus == LEARN_MODE_STARTED)
  {
    StartLED0Blink(BUSY_BLINK_RATE);
    mainState = EXECUTE_STATE; /*The user should no longer be able to exit learn mode*/
  }
  else if (learnNodeInfo->bStatus == LEARN_MODE_FAILED)
  {
    receivingRemote = FALSE;
    mainState = ERROR_STATE;
  }
  else if (learnNodeInfo->bStatus == LEARN_MODE_DONE)
  {
    /*All protocol data have been transmitted.
      Any application data should be sent before setting NEW_CTRL_STATE_STOP_OK*/
    receivingRemote = FALSE;
    staticCtrlNodeID = ZW_GET_SUC_NODEID();
    if ( staticCtrlNodeID )
    {
      bStartUpdateTimer = TRUE;
    }
    ZW_TIMER_START(StopLED0Blink,LED0_ONTIME, TIMER_ONE_TIME);
    
    NodeS();
    
    nodeInWork = nodeID;
    StartSecurityTimeOut(TIME_SECURITY_LIFE);
    /* Update group list in case a group replication was done */
    mainState = WAIT_FOR_PB_RELEASE_STATE;
    
    
    ClearGroupTable(NULL);
  }
}

void
ApplicationControllerUpdate(
  BYTE bStatus,             /*IN  Status of learn process*/
  BYTE bSource,             /*IN  Node id of the node that send node info*/
  BYTE* pCmd,               /*IN  Pointer to application node information*/
  BYTE bLen)                /*IN  Node info length*/
{
  int i;
  bStatus = learnNodeInfo.bStatus;
  bSource = learnNodeInfo.bSource;
  pCmd    = learnNodeInfo.pCmd;
  bLen    = learnNodeInfo.bLen;

  

  
  if(bLen!=0)
  {
    lastLearnedNodeType = ((NODE_TYPE *)pCmd)->generic;
  }
  if (bStatus == UPDATE_STATE_DELETE_DONE)
  {
    RemoveNodeFromGroup(bSource);
  }
  else if (bStatus == UPDATE_STATE_SUC_ID)
  {
    /* OK we got a SUC */
  }
  else if (bStatus == UPDATE_STATE_NODE_INFO_RECEIVED)
  {
    /* Only add to group if we are dimming */
    if (mainState == DIMMING_STATE)
    {
      /* Only store it in the group if it was a switch type we learned */
      if ((lastLearnedNodeType == GENERIC_TYPE_SWITCH_BINARY) ||
          (lastLearnedNodeType == GENERIC_TYPE_SENSOR_BINARY) ||
          (lastLearnedNodeType == GENERIC_TYPE_SWITCH_MULTILEVEL) ||
          (lastLearnedNodeType == GENERIC_TYPE_ENTRY_CONTROL))
      {
        /* If it is already in the group then remove it */
        if (IsNodeInGroup(bSource))
        {
          RemoveNodeFromGroup(bSource);
        }
        else
        {
          AddNodeToGroup(bSource);
          addNodeToGroup  = 0x0;
          addNodeToGroup  = bSource;
          nodeSecInclude = 1;
          AddSecuritySlave(bSource,FALSE);
          if(isCtrlSecure())
          {
            for (i = 3; i<bLen; i++)
            {
              if (pCmd[ i ] == COMMAND_CLASS_SECURITY)
              {
                AddSecuritySlave(bSource,TRUE);
                addNodeToGroup  = bSource;
                break;
              }
            }
          }
          
          
          
          StartLED0Blink(ADD_TO_GROUP_RATE);
        }
      }
    }
  }
}


