/********************************  DoorLock.c  ********************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Z-Wave, the wireless language.
 *
 *              Copyright (c) 2009
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 *
 * Description: Slave application for evaluation kit Door Lock
 *
 * Author:   Oleg Zadorozhnyy, Valeriy Vyshnyak
 *
 * Last Changed By:  $Author: psh $
 * Revision:         $Revision: 13461 $
 * Last Changed:     $Date: 2009-03-12 12:02:42 +0200 (Чт, 12 Бер 2009) $
 *
 ****************************************************************************/
/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/
#include <config_app.h>
#ifdef ZW_CONTROLLER
#include <ZW_controller_api.h>
#else
#include <ZW_slave_api.h>
#endif
#include <ZW_pindefs.h>
#include <ZW_evaldefs.h>
#include <ZW_debug_api.h>
#include <ZW_uart_api.h>

#include <slave_learn.h>
#include <one_button.h>
#include <ZW_TransportLayer.h>

#include "DoorLock.h"
#include "eeprom.h"


#ifdef FLIRS
#include <ZW_FLiRS.h>
#endif

/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/

#ifdef APPL_PROD_TEST
/* Productiontest pin definition */

#ifdef ZW020x
/* SSNJ pin now input */
#define SET_PRODUCTIONTEST_PIN  PIN_IN(SSN, 0)
#define IN_PRODUCTIONTEST       (!PIN_GET(SSN))
#endif  /* ZW020x */

#ifdef ZW030x
/* SSNJ pin now input */
#define SET_PRODUCTIONTEST_PIN  PIN_IN(SSN, 0)
#define IN_PRODUCTIONTEST       (!PIN_GET(SSN))
#endif  /* ZW030x */

#endif /* APPL_PROD_TEST */


/* Timer repeat define */
#define FOREVER             TIMER_FOREVER
#define ONE_SHOT            TIMER_ONE_TIME
/****************************************************************************/
/*                              PRIVATE DATA                                */
/****************************************************************************/

/* A list of the known command classes. Except the basic class which allways */
/* should be supported. Used when node info is send */
static code BYTE nodeInfo[] = {
#ifdef ZW_ADVANCED_DOOR_LOCK
                    COMMAND_CLASS_DOOR_LOCK,
#else
                    COMMAND_CLASS_LOCK,
#endif
                    COMMAND_CLASS_BASIC,
                    COMMAND_CLASS_MANUFACTURER_SPECIFIC,
                    COMMAND_CLASS_VERSION,
                    COMMAND_CLASS_POWERLEVEL,
#ifdef SECURITY
                    COMMAND_CLASS_SECURITY,
#endif
                    };

#ifdef SECURITY
extern	BYTE nodeSecure;
extern  BYTE secureCmdHandling;
#define TSEC_CMD_HANDLING_SECURE    1

static code BYTE nodeInfoAfterIncluded[] = {
                    COMMAND_CLASS_VERSION,
                    COMMAND_CLASS_MANUFACTURER_SPECIFIC,
                    COMMAND_CLASS_SECURITY
					};



static code BYTE nodeInfoForTransport[] = {
#ifdef ZW_ADVANCED_DOOR_LOCK
                    COMMAND_CLASS_DOOR_LOCK,
#else
                    COMMAND_CLASS_LOCK,
#endif
                    COMMAND_CLASS_BASIC,
                    COMMAND_CLASS_MANUFACTURER_SPECIFIC,
                    COMMAND_CLASS_VERSION,
                    COMMAND_CLASS_POWERLEVEL,
                    };
#else
#define nodeInfoForTransport nodeInfo
#endif


/* node status */
volatile BYTE lockNodeStatus =  DEFAULT_LOCK_STATUS;
BYTE lockUpdateHandle = 0xff;

BYTE pollState = POLLIDLE;
ZW_APPLICATION_TX_BUFFER txBuf;

BYTE txOption;

#if defined(ZW020x) || defined(ZW030x)
BYTE timerPowerLevel = 0;
BYTE timerPowerLevelHandle = 0;
BYTE timerPowerLevelSec = 0;
BYTE testNodeID = ZW_TEST_NOT_A_NODEID;
BYTE testSourceNodeID;
BYTE testPowerLevel;
WORD testFrameCount;
WORD testFrameSuccessCount;
BYTE testState = 0;
BYTE currentPower = normalPower;
#endif


BYTE bNWIStartup;
BYTE bMyNodeID;

BYTE wakeUpReason;

BYTE myNodeID = 0;
BYTE myHomeID[4];
//extern void ( *cbFuncZWSecure ) ( BYTE txStatus );

/****************************************************************************/
/*                              EXPORTED DATA                               */
/****************************************************************************/

/****************************************************************************/
/*                            PRIVATE FUNCTIONS                             */
/****************************************************************************/

/*===============================   SaveStatus   ============================
**    Function description
**      Stores the current status of the lock on/off
**      in the application part of the EEPROM.
**
**    Side effects :
**
**--------------------------------------------------------------------------*/
static void SaveStatus(void)
{
  ZW_MEM_PUT_BYTE(EEOFFSET_LOCK_STATUS, lockNodeStatus);
  ZW_MEM_PUT_BYTE(EEOFFSET_MAGIC, MAGIC_VALUE);
}

/*===============================   LoadStatus   ===========================
**    Function description
**      Fetches status of the lock on/off from the
**      application part of the EEPROM and uses it as the current status.
**
**    Side effects :
**
**--------------------------------------------------------------------------*/
static void LoadStatus(void)
{
  lockNodeStatus = ZW_MEM_GET_BYTE(EEOFFSET_LOCK_STATUS);

	timerPowerLevel = ZW_MEM_GET_BYTE(EEOFFSET_POWER_LEVEL);
	timerPowerLevelSec = ZW_MEM_GET_BYTE(EEOFFSET_TIMER_POWER_LEVEL);

	testNodeID = ZW_MEM_GET_BYTE(EEOFFSET_TEST_NODE_ID);
	testPowerLevel = ZW_MEM_GET_BYTE(EEOFFSET_TEST_POWER_LEVEL);
	testFrameCount = ZW_MEM_GET_BYTE(EEOFFSET_TEST_FRAME_COUNT);
	testSourceNodeID = ZW_MEM_GET_BYTE(EEOFFSET_TEST_SOURCE_NODE_ID);

}


/*===========================   UpdateLock   ================================
**    Function description
**      Update lock according to current state (lockNodeStatus)
**
**    Side effects :
**
**--------------------------------------------------------------------------*/
static void           /*RET  Nothing                  */
UpdateLock( void )  /* IN  Nothing                  */
{
  lockUpdateHandle = 0xff; /* Now done */
  if (lockNodeStatus & M_LOCK_ON_OFF)
  {
#ifdef LOW_FOR_ON
    PIN_OFF(TRIACpin);
#else
    PIN_ON(TRIACpin);
#endif
  }
  else
  {
#ifdef LOW_FOR_ON
    PIN_ON(TRIACpin);
#else
    PIN_OFF(TRIACpin);
#endif
  }
  ZW_DEBUG_SEND_BYTE('U');
  ZW_DEBUG_SEND_NUM(lockNodeStatus & M_LOCK_ON_OFF);

#ifdef FLIRS
   FLiRS_SleepTimeoutStart(FALSE);
#endif

}

/*===============================   SetLock   ================================
**    Function description
**      Set the door lock to:
**
**      CURRENT_STATE - The current lock state stored in eeprom
**      TOGGLE        - Toggle the lock on/off
**      ON            - ON
**      OFF           - OFF
**
**    Side effects :
**
**--------------------------------------------------------------------------*/
static void     /*RET Nothing */
SetLock(
  BYTE bAction) /* IN Action to be done */
{
  switch (bAction)
  {
    case TOGGLE:
      lockNodeStatus ^= M_LOCK_ON_OFF;
      break;

    case CURRENT_STATE:
      break;

    case ON:
      lockNodeStatus |= M_LOCK_ON_OFF;
      break;

    case OFF:
      lockNodeStatus &= ~(M_LOCK_ON_OFF);
      break;
  }

  ZW_DEBUG_SEND_BYTE('L');
  ZW_DEBUG_SEND_NUM(bAction);
  ZW_DEBUG_SEND_NUM(lockUpdateHandle);

  /* If this function called second time, before the UpdateLock is executed, i prefer to delete */
  /*  the old timer and start a new one. Because, just ignoring to starting new timer, in most cases */
  /*  lead to situation, when UpdateLosk timer handler is never called. */
  if (lockUpdateHandle != 0xff)
  {
    ZW_TIMER_CANCEL(lockUpdateHandle);
  }
  /* Postpone the lock relay update a bit */
  lockUpdateHandle = ZW_TIMER_START(UpdateLock, LOCK_UPDATETIME, TIMER_ONE_TIME); /* in 100ms we update lock */

  ZW_DEBUG_SEND_NUM(lockUpdateHandle);

  /* Save new status in EEPROM */
  SaveStatus();
}

/*=============================  ApplicationPoll   =========================
**    Application poll function for LED dimmer
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                    /*RET  Nothing                  */
ApplicationPoll( void ) /* IN  Nothing                  */
{
  BYTE lastAction = OneButtonLastAction();
  ZW_DEBUG_CMD_POLL();

  if (bNWIStartup)
  {
#ifdef FLIRS
		FLiRS_SleepTimeoutStop();
#endif

    StartLearnModeNow(ZW_SET_LEARN_MODE_NWI);
    bNWIStartup = FALSE;
    pollState = POLLLEARNMODE;
    ZW_DEBUG_SEND_BYTE('N');
    ZW_DEBUG_SEND_BYTE('B');
  }

  switch (pollState)
  {
    case POLLIDLE:
      if (lastAction==BUTTON_WAS_PRESSED)
      {
#ifdef FLIRS
		FLiRS_SleepTimeoutStop();
#endif
        SetLock(TOGGLE);
        ZW_DEBUG_SEND_BYTE('B');
        ZW_DEBUG_SEND_BYTE('T');
      }
      else if (lastAction==BUTTON_IS_HELD)
      {
        /* do nothing */
        ZW_DEBUG_SEND_BYTE('B');
        ZW_DEBUG_SEND_BYTE('H');
      }
      else if (lastAction==BUTTON_TRIPLE_PRESS)
      {
        StartLearnModeNow(ZW_SET_LEARN_MODE_CLASSIC);
        pollState = POLLLEARNMODE;
        ZW_DEBUG_SEND_BYTE('N');
        ZW_DEBUG_SEND_BYTE('C');
      }
      break;

    case POLLLEARNMODE:
      /* Just wait... */
      if (lastAction==BUTTON_WAS_PRESSED)
      {
        /* Stop learn mode */
        StartLearnModeNow(FALSE);
        pollState = POLLIDLE;
		ZW_MEM_PUT_BYTE(EEOFFSET_LERN_MODE,TRUE);
        ZW_DEBUG_SEND_BYTE('N');
        ZW_DEBUG_SEND_BYTE('E');
      }
      break;
  }
}

#if defined(ZW020x) || defined(ZW030x)
/*===========================   SendTestReport   ============================
**    Send current Powerlevel test results
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void
SendTestReport( void )
{
  txBuf.ZW_PowerlevelTestNodeReportFrame.cmdClass = COMMAND_CLASS_POWERLEVEL;
  txBuf.ZW_PowerlevelTestNodeReportFrame.cmd = POWERLEVEL_TEST_NODE_REPORT;
  txBuf.ZW_PowerlevelTestNodeReportFrame.testNodeid = testNodeID;
  txBuf.ZW_PowerlevelTestNodeReportFrame.statusOfOperation = testState;
  txBuf.ZW_PowerlevelTestNodeReportFrame.testFrameCount1 = (BYTE)(*((BYTE*)&testFrameSuccessCount));
  txBuf.ZW_PowerlevelTestNodeReportFrame.testFrameCount2 = (BYTE)*(&testFrameSuccessCount);
  /* Send Report - we do not care if it gets there or not - if needed report can be requested again */
  Transport_SendReport(testSourceNodeID, (BYTE *)&txBuf, sizeof(txBuf.ZW_PowerlevelTestNodeReportFrame), txOption, NULL, FALSE);
}

/*=============================   SendTestDone   ============================
**    Test frame has been transmitted to DUT and the result is noted for
**    later reporting. If not finished then another Test frame is transmitted.
**    If all test frames has been transmitted then the test is stopped
**    and the final result is reported to the PowerlevelTest initiator.
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void
SendTestDone(
  BYTE bStatus)
{
  if (bStatus == TRANSMIT_COMPLETE_OK)
  {
    testFrameSuccessCount++;
  }
  if (testFrameCount && (--testFrameCount))
  {
    ZW_SEND_TEST_FRAME(testNodeID, testPowerLevel, SendTestDone);
  }
  else
  {
    if (testFrameSuccessCount)
    {
      testState = ZW_TEST_SUCCES;
    }
    else
    {
      testState = ZW_TEST_FAILED;
    }
    ZW_RF_POWERLEVEL_SET(currentPower);  /* Back to previous setting */
    SendTestReport();
  }
}

/*=======================   PowerLevelTimerCancel   ==========================
**    Cancels PowerLevel timer
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void
PowerLevelTimerCancel( void )
{
  ZW_TIMER_CANCEL(timerPowerLevelHandle);
  timerPowerLevelHandle = 0;
}

/*===============================   StartTest   ==============================
**    Timer callback which maked sure that the RF transmit powerlevel is set
**    back to normalPower after the designated time period.
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void
PowerLevelTimeout( void )
{
  if (!--timerPowerLevelSec)
  {
    ZW_RF_POWERLEVEL_SET(normalPower);  /* Reset powerlevel to normalPower */
    PowerLevelTimerCancel();
  }
}

/*===============================   StartTest   ==============================
**    Start the powerlevel test run
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void
StartTest(void)
{
  testState = ZW_TEST_INPROGRESS;
  currentPower = ZW_RF_POWERLEVEL_GET();  /* Get current (normalPower) */
  ZW_RF_POWERLEVEL_SET(testPowerLevel);
  ZW_SEND_TEST_FRAME(testNodeID, testPowerLevel, SendTestDone);
}
#endif


/*========================   ApplicationCommandHandler   ====================
**    Handling of a received application commands and requests
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void                              /*RET Nothing                  */
Transport_ApplicationCommandHandler(
  BYTE  rxStatus,                 /* IN Frame header info */
  BYTE  sourceNode,               /* IN Command sender Node ID */
  ZW_APPLICATION_TX_BUFFER *pCmd, /* IN Payload from the received frame, the union */
                                  /*    should be used to access the fields */
  BYTE   cmdLength)               /* IN Number of command bytes including the command */
{
  BYTE tmp;

  ZW_DEBUG_SEND_BYTE('H');
  ZW_DEBUG_SEND_NUM(pCmd->ZW_Common.cmdClass);
  ZW_DEBUG_SEND_NUM(pCmd->ZW_Common.cmd);

 	txOption = ((rxStatus & RECEIVE_STATUS_LOW_POWER) ?
 	            TRANSMIT_OPTION_LOW_POWER : 0) | TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_EXPLORE;


 	switch(pCmd->ZW_Common.cmdClass)
 	{
	 	case COMMAND_CLASS_VERSION:
	  	{
		    switch (pCmd->ZW_VersionGetFrame.cmd)
		    {
			    case VERSION_GET:
			    {
			      txBuf.ZW_VersionReportFrame.cmdClass = COMMAND_CLASS_VERSION;
			      txBuf.ZW_VersionReportFrame.cmd = VERSION_REPORT;
			      txBuf.ZW_VersionReportFrame.zWaveLibraryType = ZW_TYPE_LIBRARY();
			      txBuf.ZW_VersionReportFrame.zWaveProtocolVersion = ZW_VERSION_MAJOR;
			      txBuf.ZW_VersionReportFrame.zWaveProtocolSubVersion = ZW_VERSION_MINOR;
			      txBuf.ZW_VersionReportFrame.applicationVersion = APP_VERSION;
			      txBuf.ZW_VersionReportFrame.applicationSubVersion = APP_REVISION;
			      Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_VersionReportFrame), txOption, NULL, FALSE);
			      break;
			    }

			    case VERSION_COMMAND_CLASS_GET:
			    {
			      txBuf.ZW_VersionCommandClassReportFrame.cmdClass = COMMAND_CLASS_VERSION;
			      txBuf.ZW_VersionCommandClassReportFrame.cmd = VERSION_COMMAND_CLASS_REPORT;
			      txBuf.ZW_VersionCommandClassReportFrame.requestedCommandClass = pCmd->ZW_VersionCommandClassGetFrame.requestedCommandClass;
			      switch (txBuf.ZW_VersionCommandClassReportFrame.requestedCommandClass)
			      {
				      case COMMAND_CLASS_BASIC:
				        txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = BASIC_VERSION;
				        break;

				      case COMMAND_CLASS_LOCK:
				        txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = LOCK_VERSION;
				        break;

				      case COMMAND_CLASS_MANUFACTURER_SPECIFIC:
				        txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = MANUFACTURER_SPECIFIC_VERSION;
				        break;

				      case COMMAND_CLASS_VERSION:
				        txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = VERSION_VERSION;
				        break;

				      case COMMAND_CLASS_POWERLEVEL:
				        txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = POWERLEVEL_VERSION;
				        break;
#ifdef SECURITY
				      case COMMAND_CLASS_SECURITY:
				        txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = SECURITY_VERSION;
				        break;
#endif
				      default:
				        txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = UNKNOWN_VERSION;
				        break;
			      }
			      Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_VersionCommandClassReportFrame), txOption, NULL, FALSE);
			      break;
			    }
		    }
	    	break;
	  	}

		case COMMAND_CLASS_MANUFACTURER_SPECIFIC:
		{
		    switch (pCmd->ZW_ManufacturerSpecificGetFrame.cmd)
		    {
			    case MANUFACTURER_SPECIFIC_GET:
			    {
			      txBuf.ZW_ManufacturerSpecificReportFrame.cmdClass = COMMAND_CLASS_MANUFACTURER_SPECIFIC;
			      txBuf.ZW_ManufacturerSpecificReportFrame.cmd = MANUFACTURER_SPECIFIC_REPORT;
			      txBuf.ZW_ManufacturerSpecificReportFrame.manufacturerId1 = 0x00; // Zensys manufacturer ID
			      txBuf.ZW_ManufacturerSpecificReportFrame.manufacturerId2 = 0x00;
			      txBuf.ZW_ManufacturerSpecificReportFrame.productTypeId1 = 0x00; // Assigned by manufacturer
			      txBuf.ZW_ManufacturerSpecificReportFrame.productTypeId2 = 0x00;
			      txBuf.ZW_ManufacturerSpecificReportFrame.productId1 = 0x00; // Assigned by manufacturer
			      txBuf.ZW_ManufacturerSpecificReportFrame.productId2 = 0x00;
			      Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_ManufacturerSpecificReportFrame), txOption, NULL, FALSE);
			      break;
			    }
		    }
		    break;
		}
	}


#ifdef SECURITY
  if(nodeSecure)
  if (secureCmdHandling != TSEC_CMD_HANDLING_SECURE)
  {
//	  if(currentState == STATE_COMMAND_HANDLER)
//	  {
//	    currentState = STATE_APPL_IDLE;
//	  }
  	  return;
  }
#endif

	ZW_DEBUG_SEND_BYTE(' ');
	ZW_DEBUG_SEND_BYTE('V');
	ZW_DEBUG_SEND_BYTE(' ');

 	switch(pCmd->ZW_Common.cmdClass)
 	{


#ifdef ZW_ADVANCED_DOOR_LOCK
 	case COMMAND_CLASS_DOOR_LOCK:
 	{
 	  switch(pCmd->ZW_Common.cmd)
 	  {
	 	  case DOOR_LOCK_OPERATION_SET:
		  {
			DoorLockMode = pCmd->ZW_DoorLockOperationSetFrame.doorLockMode;
			break;
		  }
	 	  case DOOR_LOCK_OPERATION_GET:
	 	  {
	 	    txBuf.ZW_DoorLockOperationReportFrame.cmdClass 				=	COMMAND_CLASS_DOOR_LOCK;
	 	    txBuf.ZW_DoorLockOperationReportFrame.cmd 					=	DOOR_LOCK_OPERATION_REPORT;
			txBuf.ZW_DoorLockOperationReportFrame.doorLockMode			=	DoorLockMode;
	 	    txBuf.ZW_DoorLockOperationReportFrame.properties1			=	ZW_MEM_GET_BYTE(EEOFFSET_HANDLES_STATE);
			txBuf.ZW_DoorLockOperationReportFrame.doorCondition			=	DoorCondition;
			txBuf.ZW_DoorLockOperationReportFrame.lockTimeoutMinutes	=	ZW_MEM_GET_BYTE(EEOFFSET_LOCK_MINUTES);
			txBuf.ZW_DoorLockOperationReportFrame.lockTimeoutSeconds	=	ZW_MEM_GET_BYTE(EEOFFSET_LOCK_SECONDS);

			Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_DoorLockOperationReportFrame), txOption | TRANSMIT_OPTION_EXPLORE, NULL, FALSE);
	 	    break;
	 	  }
	 	  case DOOR_LOCK_CONFIGURATION_SET:
	 	  {
			ZW_MEM_PUT_BYTE(EEOFFSET_OPERATION_TYPE,pCmd->ZW_DoorLockConfigurationSetFrame.operationType);
			ZW_MEM_PUT_BYTE(EEOFFSET_HANDLES_STATE,pCmd->ZW_DoorLockConfigurationSetFrame.properties1);
			ZW_MEM_PUT_BYTE(EEOFFSET_LOCK_MINUTES,pCmd->ZW_DoorLockConfigurationSetFrame.lockTimeoutMinutes);
			ZW_MEM_PUT_BYTE(EEOFFSET_LOCK_SECONDS,pCmd->ZW_DoorLockConfigurationSetFrame.lockTimeoutSeconds);
	 	    break;
	 	  }
	 	  case DOOR_LOCK_CONFIGURATION_GET:
	 	  {
	 	    txBuf.ZW_DoorLockConfigurationReportFrame.cmdClass 			=	COMMAND_CLASS_DOOR_LOCK;
	 	    txBuf.ZW_DoorLockConfigurationReportFrame.cmd 				=	DOOR_LOCK_CONFIGURATION_REPORT;
	 	    txBuf.ZW_DoorLockConfigurationReportFrame.operationType		=	ZW_MEM_GET_BYTE(EEOFFSET_OPERATION_TYPE);
			txBuf.ZW_DoorLockConfigurationReportFrame.properties1		=	ZW_MEM_GET_BYTE(EEOFFSET_HANDLES_STATE);
			txBuf.ZW_DoorLockConfigurationReportFrame.lockTimeoutMinutes=	ZW_MEM_GET_BYTE(EEOFFSET_LOCK_MINUTES);
			txBuf.ZW_DoorLockConfigurationReportFrame.lockTimeoutSeconds=	ZW_MEM_GET_BYTE(EEOFFSET_LOCK_SECONDS);

			Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_DoorLockConfigurationReportFrame), txOption | TRANSMIT_OPTION_EXPLORE, NULL, FALSE);
	 	    break;
	 	  }
 	  }
 	  break;
	}

 	case COMMAND_CLASS_BASIC:
 	{
 	  switch(pCmd->ZW_Common.cmd)
 	  {
		  //Must be ignored to avoid unintentional operation. Cannot be mapped to another command class.
	 	  case BASIC_SET:
			break;
	 	  case BASIC_GET:
	 	  {
	      /* Answer with the called class */
	 	    txBuf.ZW_LockReportFrame.cmdClass = pCmd->ZW_LockGetFrame.cmdClass;
	 	    txBuf.ZW_LockReportFrame.cmd = BASIC_REPORT;
	 	    txBuf.ZW_LockReportFrame.lockState = (lockNodeStatus & M_LOCK_ON_OFF) ?
	 	      DOOR_LOCK_OPERATION_REPORT_DOOR_SECURED : DOOR_LOCK_OPERATION_REPORT_DOOR_UNSECURED;
	      	Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_LockReportFrame), txOption | TRANSMIT_OPTION_EXPLORE, NULL, FALSE);
	 	    break;
	 	  }
 	  }
 	  break;
 	}


#else
 	case COMMAND_CLASS_BASIC:
 	case COMMAND_CLASS_LOCK:
 	{
 	  switch(pCmd->ZW_Common.cmd)
 	  {
 	  /*case BASIC_SET:*/
 	  case LOCK_SET:
 	  {
 	    if(cmdLength >= sizeof(pCmd->ZW_LockSetFrame))
 	    {
        ZW_DEBUG_SEND_BYTE('S');
        ZW_DEBUG_SEND_NUM(pCmd->ZW_LockSetFrame.lockState);
//        if(pCmd->ZW_LockSetFrame.cmdClass == COMMAND_CLASS_LOCK)
        {
          if (pCmd->ZW_LockSetFrame.lockState == DOOR_LOCK_OPERATION_REPORT_DOOR_UNSECURED)
          {
            SetLock(OFF);
          }
          else
          {
            SetLock(ON);
          }
        }
 	    }
 	    break;
 	  }
 	  /*case BASIC_GET:*/
 	  case LOCK_GET:
 	  {
      /* Answer with the called class */
 	    txBuf.ZW_LockReportFrame.cmdClass = pCmd->ZW_LockGetFrame.cmdClass;
 	    txBuf.ZW_LockReportFrame.cmd = LOCK_REPORT;
 	    txBuf.ZW_LockReportFrame.lockState = (lockNodeStatus & M_LOCK_ON_OFF) ?
 	      DOOR_LOCK_OPERATION_REPORT_DOOR_SECURED : DOOR_LOCK_OPERATION_REPORT_DOOR_UNSECURED;
      Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_LockReportFrame), txOption | TRANSMIT_OPTION_EXPLORE, NULL, FALSE);
 	    break;
 	  }
 	  }
 	  break;
 	}
#endif 









#ifdef ZW_CONTROLLER
 	case COMMAND_CLASS_CONTROLLER_REPLICATION:
 	{
    /* Handle replication frames */
    /* We are only interested in group replication here */
    /* but all other replication can be handled in the same way. */
    /* In this application we ignore everything else and just */
    /* send the command complete reply */
#if 0
    if (pCmd->ZW_Common.cmd == CTRL_REPLICATION_TRANSFER_GROUP)
    {
      /* We are only interested in group 1 */
      if (pCmd->ZW_TransferGroupFrame.groupID == 1)
      {
        /* Add the node ID to the group */
        AddNodeToGroup(pCmd->ZW_TransferGroupFrame.nodeID);
      }
    }
#endif
 	  break;
 	}
#endif

#if defined(ZW020x) || defined(ZW030x)
 	case COMMAND_CLASS_POWERLEVEL:
  {
    switch (pCmd->ZW_Common.cmd)
    {
    case POWERLEVEL_SET:
    {
      if (pCmd->ZW_PowerlevelSetFrame.timeout || (pCmd->ZW_PowerlevelSetFrame.powerLevel == normalPower))  /* Only if timeout defined or powerlevel is normalPower */
      {
        tmp = ZW_RF_POWERLEVEL_SET(pCmd->ZW_PowerlevelSetFrame.powerLevel);  /* Set powerlevel according to command */
        if (timerPowerLevelHandle)
        {
          PowerLevelTimerCancel();
          timerPowerLevelSec = 0;
        }
        if (tmp != normalPower)  /* If normalPower then no timer is started */
        {
          timerPowerLevelSec = pCmd->ZW_PowerlevelSetFrame.timeout;
          timerPowerLevelHandle = ZW_TIMER_START(PowerLevelTimeout, TIMER_ONE_SECOND, TIMER_FOREVER);
		  timerPowerLevel = pCmd->ZW_PowerlevelSetFrame.powerLevel;
        }
      }
	  ZW_MEM_PUT_BYTE(EEOFFSET_POWER_LEVEL,timerPowerLevel);
	  ZW_MEM_PUT_BYTE(EEOFFSET_TIMER_POWER_LEVEL,timerPowerLevelSec);
      break;
    }

    case POWERLEVEL_GET:
    {
      txBuf.ZW_PowerlevelReportFrame.cmdClass = COMMAND_CLASS_POWERLEVEL;
      txBuf.ZW_PowerlevelReportFrame.cmd = POWERLEVEL_REPORT;
      txBuf.ZW_PowerlevelReportFrame.powerLevel = ZW_MEM_GET_BYTE(EEOFFSET_POWER_LEVEL);////ZW_RF_POWERLEVEL_GET();
      txBuf.ZW_PowerlevelReportFrame.timeout = timerPowerLevelSec;
      Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_PowerlevelReportFrame), txOption, NULL, FALSE);
      break;
    }

    case POWERLEVEL_TEST_NODE_SET:
    {
      if (testState != ZW_TEST_INPROGRESS)
      {
        testSourceNodeID = sourceNode;
        testNodeID = pCmd->ZW_PowerlevelTestNodeSetFrame.testNodeid;
        testPowerLevel = pCmd->ZW_PowerlevelTestNodeSetFrame.powerLevel;
        testFrameCount = (WORD)((WORD)pCmd->ZW_PowerlevelTestNodeSetFrame.testFrameCount1 << 8) + pCmd->ZW_PowerlevelTestNodeSetFrame.testFrameCount1;
        testFrameSuccessCount = 0;

		ZW_MEM_PUT_BYTE(EEOFFSET_TEST_NODE_ID,testNodeID);
		ZW_MEM_PUT_BYTE(EEOFFSET_TEST_POWER_LEVEL,testPowerLevel);
		ZW_MEM_PUT_BYTE(EEOFFSET_TEST_FRAME_COUNT,testFrameCount);
		ZW_MEM_PUT_BYTE(EEOFFSET_TEST_SOURCE_NODE_ID,testSourceNodeID);

        if (testFrameCount)
        {
          StartTest();
        }
        else
        {
          testState = ZW_TEST_FAILED;
          SendTestReport();
        }
      }
      break;
    }

    case POWERLEVEL_TEST_NODE_GET:
    {
      SendTestReport();
      break;
    }
    }
    break;
  }
#endif
  }
}

/****************************************************************************/
/*                           EXPORTED FUNCTIONS                             */
/****************************************************************************/

/*============================   ApplicationInitHW   ========================
**    Non Z-Wave hardware initialization
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/

BYTE                       /*RET  TRUE        */
ApplicationInitHW( BYTE bStatus )  /* IN  Nothing     */
{
#ifdef APPL_PROD_TEST
  BYTE i;

  SET_PRODUCTIONTEST_PIN;
  for (i = 0; i < 10; i++) ;  /* Short delay... */
  if (IN_PRODUCTIONTEST) /* Anyone forcing it into productiontest mode ? */
  {
    return(FALSE);  /*Enter production test mode*/
  }
#endif /* APPL_PROD_TEST */


#ifdef FLIRS
   EX1 = 0;
#endif


  /* hardware initialization */
  PIN_IN(Button, 1);
  wakeUpReason = bStatus;
  /*triac pin init*/
#ifdef LOW_FOR_ON
  PIN_ON(TRIACpin); /* HIGH for Relay OFF */
#endif
  PIN_OUT(TRIACpin); /* Relay output */

  Transport_OnApplicationInitHW(bStatus);

  return(TRUE);
}

#ifdef SECURITY
static void FuncZWSecure(BYTE txStatus)
{
	if (txStatus == TRANSPORT_WORK_END)
	{
#ifdef FLIRS
     FLiRS_SleepTimeoutStart(FALSE);
#endif
	}
	else if (txStatus == TRANSPORT_WORK_START)
	{
#ifdef FLIRS
		FLiRS_SleepTimeoutStop();
#endif
	}
	else if (txStatus == TRANSPORT_WORK_ERROR)
	{
#ifdef FLIRS
     FLiRS_SleepTimeoutStart(FALSE);
#endif
	}
}
#endif

/*===========================   ApplicationInitSW   =========================
**    Initialization of the Application Software
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
BYTE                      /*RET  TRUE       */
ApplicationInitSW( void ) /* IN   Nothing   */
{
  /* get stored values */
  if (ZW_MEM_GET_BYTE(EEOFFSET_MAGIC) == MAGIC_VALUE)
  {
    LoadStatus();
  }
  else
  { /* Its alive */
    lockNodeStatus = DOOR_LOCK_OPERATION_REPORT_DOOR_UNSECURED;
    SaveStatus(); /* Now EEPROM should be OK */
	ZW_MEM_PUT_BYTE(EEOFFSET_LERN_MODE,FALSE);
	timerPowerLevel = 0;
	timerPowerLevelSec = 0;
	testNodeID = 0;
	testPowerLevel = 0;
	testFrameCount = 0;
	testSourceNodeID = 0;
	ZW_MEM_PUT_BYTE(EEOFFSET_POWER_LEVEL,timerPowerLevel);
	ZW_MEM_PUT_BYTE(EEOFFSET_TIMER_POWER_LEVEL,timerPowerLevelSec);
	ZW_MEM_PUT_BYTE(EEOFFSET_TEST_NODE_ID,testNodeID);
	ZW_MEM_PUT_BYTE(EEOFFSET_TEST_POWER_LEVEL,testPowerLevel);
	ZW_MEM_PUT_BYTE(EEOFFSET_TEST_FRAME_COUNT,testFrameCount);
	ZW_MEM_PUT_BYTE(EEOFFSET_TEST_SOURCE_NODE_ID,testSourceNodeID);

  }

  ZW_DEBUG_CMD_INIT(1152);
  ZW_DEBUG_INIT(1152);
  ZW_DEBUG_SEND_BYTE('V');

  OneButtonInit();

  /* Check if we have a node id */
  MemoryGetID(NULL, &bMyNodeID);

  /* Set Network wide inclusion active if we don't have a node ID */
  if (bMyNodeID)
    bNWIStartup = FALSE;
  else
    if (!ZW_MEM_GET_BYTE(EEOFFSET_LERN_MODE))
      bNWIStartup = TRUE;

  ZW_DEBUG_SEND_NUM(ZW_MEM_GET_BYTE(EEOFFSET_LERN_MODE));

  UpdateLock();

  Transport_OnApplicationInitSW(nodeInfoForTransport, sizeof(nodeInfoForTransport),
    FALSE, EEOFFSET_TRANSPORT_SETTINGS_START, EEOFFSET_TRANSPORT_SETTINGS_SIZE,
    FuncZWSecure);

  return(TRUE);
}

/*============================   ApplicationTestPoll   ======================
**    Function description
**      This function is called when the dimmer enters test mode.
**    Side effects:
**       Code will not exit until it is reset
**--------------------------------------------------------------------------*/
void ApplicationTestPoll(void)
{
#ifdef APPL_PROD_TEST
  /*Send constant out*/
  ZW_SEND_CONST();
  while(1);
#endif
}

/*======================   ApplicationNodeInformation   =====================
**    Request Node information and current status
**    Called by the the Z-Wave application layer before transmitting a
**    "Node Information" frame.
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
extern void                  /*RET  Nothing */
ApplicationNodeInformation(
  BYTE   *deviceOptionsMask,      /*OUT Bitmask with application options     */
  APPL_NODE_TYPE  *nodeType,  /*OUT  Device type Generic and Specific   */
  BYTE       **nodeParm,      /*OUT  Device parameter buffer pointer    */
  BYTE       *parmLength)     /*OUT  Number of Device parameter bytes   */
{

  /* this is a listening node and it supports optional CommandClasses */
  *deviceOptionsMask = APPLICATION_NODEINFO_LISTENING | APPLICATION_NODEINFO_OPTIONAL_FUNCTIONALITY;
#ifdef FLIRS
  FLIRS_DEVICE_OPTIONS_MASK_MODIFY(*deviceOptionsMask);
#endif

  nodeType->generic = GENERIC_TYPE_ENTRY_CONTROL; /* Generic device type */

#ifdef ZW_ADVANCED_DOOR_LOCK
  nodeType->specific = SPECIFIC_TYPE_ADVANCED_DOOR_LOCK; //SPECIFIC_TYPE_DOOR_LOCK;   /* Specific class */
#else
  nodeType->specific = SPECIFIC_TYPE_DOOR_LOCK;   /* Specific class */
#endif

  *nodeParm = nodeInfo;                           /* Send list of known command classes. */
  *parmLength = sizeof(nodeInfo);                 /* Set length*/

#ifdef SECURITY
  MemoryGetID(myHomeID, &myNodeID);

#ifndef FLIRS
  nodeSecure = GetNodeSecure(EEOFFSET_TRANSPORT_SETTINGS_START);
#endif

  if (myNodeID)
	if (nodeSecure)
	{
  		*nodeParm = nodeInfoAfterIncluded;        /* Send list of known command classes. */
  		*parmLength = sizeof(nodeInfoAfterIncluded);       /* Set length*/
	}
#endif
}

/*==========================   ApplictionSlaveUpdate   =======================
**   Inform a slave application that a node information is received.
**   Called from the slave command handler when a node information frame
**   is received and the Z-Wave protocol is not in a state where it is needed.
**
**--------------------------------------------------------------------------*/
void
ApplicationSlaveUpdate(
  BYTE bStatus,     /*IN  Status event */
  BYTE bNodeID,     /*IN  Node id of the node that send node info */
  BYTE* pCmd,       /*IN  Pointer to Application Node information */
  BYTE bLen)       /*IN  Node info length                        */
{
}


#ifdef ZW_CONTROLLER
/*=====================   ApplicationControllerUpdate   =====================
**   Inform the controller of node information update done through
**   the network managment.
**
**--------------------------------------------------------------------------*/
void
ApplicationControllerUpdate(
  BYTE bStatus,     /*IN  Status event */
  BYTE bNodeID,     /*IN  Node id of the node that send node info */
  BYTE* pCmd,       /*IN  Pointer to Application Node information */
  BYTE bLen)       /*IN  Node info length                        */
{
}
#endif


/*============================   LearnCompleted   ========================
**    Function description
**      Called from learnmode.c when learnmode completed
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void									/*RET	Nothing */
LearnCompleted(
BYTE nodeID)					/* IN resulting nodeID */
{
  if (nodeID == 0)		/* Was it reset? */
  {
	 ZW_MEM_PUT_BYTE(EEOFFSET_LERN_MODE,FALSE);

	 ZW_DEBUG_SEND_BYTE('x');
	 ZW_DEBUG_SEND_NUM(ZW_MEM_GET_BYTE(EEOFFSET_LERN_MODE));
  }


#ifdef FLIRS
  if (nodeID != 0)		/* Was it reset? */
  {
#ifdef SECURITY

      FLiRS_SleepTimeoutStop();
	  ZW_DEBUG_SEND_BYTE('z');
	  ZW_DEBUG_SEND_BYTE('z');

#else
     FLiRS_SleepTimeoutStart(FALSE);
#endif
  }
#endif


  Transport_OnLearnCompleted(nodeID);

 	  ZW_DEBUG_SEND_BYTE(' ');
 	  ZW_DEBUG_SEND_BYTE('E');
	  ZW_DEBUG_SEND_BYTE(' ');

}

