/*******************************  bell.C  ********************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Z-Wave, the wireless language.
 *
 *              Copyright (c) 2001
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 * Copyright Zensys A/S, 2001
 *
 * Description: Slave application for doorbell test application
 *
 * Author:   Peter Shorty
 *
 * Last Changed By:  $Author: efh $
 * Revision:         $Revision: 21401 $
 * Last Changed:     $Date: 2011-09-27 16:37:40 +0200 (Tue, 27 Sep 2011) $
 *
 ****************************************************************************/
/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/

#include <config_app.h>

#define ZW_BEAM_RX_WAKEUP

#include <ZW_slave_api.h>
#include <ZW_pindefs.h>
#include <ZW_evaldefs.h>
#include <bell.h>
#include <eeprom_slave.h>
#include <slave_learn.h>
#include <ZW_uart_api.h>
#include <ZW_appltimer.h>
#include <ZW_power_api.h>
#include <ZW_transport_api.h>
#include <one_button.h>

/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/

#define LONG_SLEEP_TIMEOUT  200

/****************************************************************************/
/*                              PRIVATE DATA                                */
/****************************************************************************/


/* A list of the known command classes. Except the basic class which allways */
/* should be supported. Used when node info is send */
t_nodeInfo nodeInfo = {COMMAND_CLASS_SWITCH_BINARY,
                       COMMAND_CLASS_VERSION};

ZW_APPLICATION_TX_BUFFER txBuf;
BYTE txOption;

BYTE  bMyNodeID;

#define STATE_IDLE                  0
#define STATE_WAITING_FOR_ADD       1
#define STATE_ADDING_DONE           2
#define STATE_POWER_UP              4
#define STATE_WAITING_FOR_TIMEOUT   5


BYTE bMainState = STATE_IDLE;

/* Buffer used to hold commands */
ZW_APPLICATION_TX_BUFFER txBuffer;

BYTE bSleepTimeout; /* Timer handler for sleep timer */
BYTE bSleepTimeoutCount;

BYTE bWakeupReason;  /* Reason for waking up */

BYTE bBellStatus;

BYTE bNWIStartup;

void GotoSleep();

void              /*RET Nothing */
SendReport(
  BYTE sourceNode,  /* IN Source node - to whom report is to be send */
  BYTE classcmd,    /* IN Command class, which the report belongs to */
  BYTE reportparam); /* IN paramter to include in report frame */


#ifdef TIMING_DEBUG
#define TXDEBUGPin      6
#define TXDEBUGPinPort  P1
#define TXDEBUGPinDDR   P1DIR
#endif

/****************************************************************************/
/*                              EXPORTED DATA                               */
/****************************************************************************/


/****************************************************************************/
/*                            PRIVATE FUNCTIONS                             */
/****************************************************************************/
void NodeInfoSendComplete(BYTE bStatus);

/*=============================  SleepTimeout   =========================
**    Timeout function for going back to sleep
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void SleepTimeout()
{
  ZW_DEBUG_SEND_BYTE('T');
  bSleepTimeout = 0;
  GotoSleep();
}

/*=============================  StopSleepTimeout   =========================
**    Stop the timeout timer for going back to sleep
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void StopSleepTimeout()
{
  if (bSleepTimeout)
    TimerCancel(bSleepTimeout);
  bSleepTimeout = 0;
}



/*=============================  StartSleepTimeout   =========================
**    Start the timeout timer for going back to sleep
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void StartSleepTimeout()
{
  ZW_DEBUG_SEND_BYTE('S');

  if (!bSleepTimeout)
    bSleepTimeout = TimerStart(SleepTimeout, LONG_SLEEP_TIMEOUT, 1);
  else
  {
    TimerRestart(bSleepTimeout);
  }
}


/*=============================  ApplicationPoll   =========================
**    Application poll function for LED dimmer
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                    /*RET  Nothing                  */
ApplicationPoll( void ) /* IN  Nothing                  */
{
  register BYTE bStatus;

  bStatus = OneButtonLastAction();

  if (bNWIStartup)
  {
    StartLearnModeNow(ZW_SET_LEARN_MODE_NWI);
    bNWIStartup = FALSE;
    bMainState = STATE_WAITING_FOR_ADD;
  }

  switch (bMainState)
  {
    case STATE_POWER_UP:
      bBellStatus = 0;
      bMainState = STATE_IDLE;
      /* Turn on the receiver */
      ZW_SetRFReceiveMode(TRUE);
#ifdef TIMING_DEBUG
      PIN_ON(TXDEBUGPin);

      ZW_DEBUG_SEND_BYTE('W');
      ZW_DEBUG_SEND_NUM(bWakeupReason);
#endif
      /* Start sleep timeout */
      StartSleepTimeout();

      if (bWakeupReason == ZW_WAKEUP_RESET)
        LED_ON(1);
      else
        LED_ON(2);

      break;

    case STATE_IDLE:   /* Idle state check the button state */

      /* Check for button state */
      if (bStatus == BUTTON_WAS_PRESSED)
      {
        ZW_DEBUG_SEND_BYTE('p');
        // ZW_SendNodeInformation(NODE_BROADCAST, TRANSMIT_OPTION_LOW_POWER, NodeInfoSendComplete);
        bMainState = STATE_WAITING_FOR_TIMEOUT;
      }
      else if (bStatus == BUTTON_TRIPLE_PRESS)
      {
        LED_ON(3);
        ZW_DEBUG_SEND_BYTE('h');
        StartLearnModeNow(ZW_SET_LEARN_MODE_CLASSIC);
        bMainState = STATE_WAITING_FOR_ADD;
        StartSleepTimeout();
      }
      break;

    case STATE_WAITING_FOR_ADD:
      if (learnState == FALSE)
      {
        bMainState = STATE_ADDING_DONE;
      }
      else /* Stop sleep timeiout while in learn mode */
      {
        StopSleepTimeout();

        if (bStatus == BUTTON_WAS_PRESSED)
        {
          /* Stop learn mode */
          StartLearnModeNow(FALSE);
          bMainState = STATE_ADDING_DONE;
        }
      }
      break;

    case STATE_ADDING_DONE:
      StartSleepTimeout();
      bMainState = STATE_IDLE;
      break;

    case STATE_WAITING_FOR_TIMEOUT:
      StartSleepTimeout();
      bMainState = STATE_IDLE;
      break;
  }
}


/*========================   ApplicationCommandHandler   ====================
**    Handling of a received application commands and requests
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void                              /*RET Nothing                  */
ApplicationCommandHandler(
  BYTE  rxStatus,                 /* IN Frame header info */
  BYTE  sourceNode,               /* IN Command sender Node ID */
  ZW_APPLICATION_TX_BUFFER *pCmd, /* IN Payload from the received frame, the union */
                                  /*    should be used to access the fields */
  BYTE   cmdLength)               /* IN Number of command bytes including the command */
{
  register BYTE bTmpCount;
  BYTE value;
  BYTE classcmd = pCmd->ZW_Common.cmdClass;
  BYTE cmd = pCmd->ZW_Common.cmd;

 	txOption = ((rxStatus & RECEIVE_STATUS_LOW_POWER) ?
 	            TRANSMIT_OPTION_LOW_POWER : 0) | TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_EXPLORE;
  if ((classcmd == COMMAND_CLASS_SWITCH_BINARY) ||
      (classcmd == COMMAND_CLASS_BASIC))
  {
    /* Single switch command */
    switch(cmd)
    {

      /* Controller wants the level */
      case SWITCH_BINARY_GET:
        StopSleepTimeout();
        SendReport(sourceNode, classcmd, bBellStatus);
        break;

      /* Jump to the specified level param1 */
      case SWITCH_BINARY_SET:
		value = pCmd->ZW_BasicSetFrame.value;
		if (value)
		{
	        LED_ON(1);
	        LED_ON(2);
	        LED_ON(3);
	        bBellStatus = 0xff;
		}
		else
		{
	        LED_OFF(1);
	        LED_OFF(2);
	        LED_OFF(3);
	        bBellStatus = 0x00;
		}
        break;
    } /* switch */
  } /* if */
   /* else if == COMMAND_CLASS_SWITCH_ALL */

  else if (classcmd == COMMAND_CLASS_VERSION)
  {
    switch (cmd)
    {
      case VERSION_GET:
        txBuf.ZW_VersionReportFrame.cmdClass = COMMAND_CLASS_VERSION;
        txBuf.ZW_VersionReportFrame.cmd = VERSION_REPORT;
        txBuf.ZW_VersionReportFrame.zWaveLibraryType = ZW_TYPE_LIBRARY();

        txBuf.ZW_VersionReportFrame.zWaveProtocolVersion = ZW_VERSION_MAJOR;

        txBuf.ZW_VersionReportFrame.zWaveProtocolSubVersion = ZW_VERSION_MINOR;
        txBuf.ZW_VersionReportFrame.applicationVersion = APP_VERSION;
        txBuf.ZW_VersionReportFrame.applicationSubVersion = APP_REVISION;
        ZW_SEND_DATA(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_VersionReportFrame), txOption, NULL);
        break;

      case VERSION_COMMAND_CLASS_GET:
        txBuf.ZW_VersionCommandClassReportFrame.cmdClass = COMMAND_CLASS_VERSION;
        txBuf.ZW_VersionCommandClassReportFrame.cmd = VERSION_COMMAND_CLASS_REPORT;
        txBuf.ZW_VersionCommandClassReportFrame.requestedCommandClass = pCmd->ZW_VersionCommandClassGetFrame.requestedCommandClass;
        switch (pCmd->ZW_VersionCommandClassGetFrame.requestedCommandClass)
        {
          case COMMAND_CLASS_BASIC:
            txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = BASIC_VERSION;
            break;

          case COMMAND_CLASS_SWITCH_BINARY:
            txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = SWITCH_BINARY_VERSION;
            break;

          case COMMAND_CLASS_VERSION:
            txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = VERSION_VERSION;
            break;

          default:
            txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = UNKNOWN_VERSION;
            break;
        }
        ZW_SendData(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_VersionCommandClassReportFrame), txOption, NULL);
        break;
    }
  }
}

/*==============================   TxDone   ==============================
**
**  Function:  Transmit of a frame done, start the powerdown timer
**
**    Sends a report frame to source node.
**
**  Side effects: None
**
**--------------------------------------------------------------------------*/
void TxDone(BYTE bStatus)
{
  StartSleepTimeout();
}

/*==============================   SendReport   ==============================
**
**  Function:  SendReport
**
**    Sends a report frame to source node.
**
**  Side effects: None
**
**--------------------------------------------------------------------------*/
void              /*RET Nothing */
SendReport(
BYTE sourceNode,  /* IN Source node - to whom report is to be send */
BYTE classcmd,    /* IN Command class, which the report belongs to */
BYTE reportparam) /* IN paramter to include in report frame */
{
  txBuf.ZW_BasicReportFrame.cmdClass = classcmd;
/* Report command definition - BASIC/ALL/PROTECTION/MULTI report use the same value ;-) */
  txBuf.ZW_BasicReportFrame.cmd      = BASIC_REPORT;
  txBuf.ZW_BasicReportFrame.value    = reportparam;
/* Size of Report (BASIC/ALL/PROTECTION/MULTI GET response) frame are equal for BASIC/ALL/PROTECTION/MULTI */
  ZW_SendData(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_BasicReportFrame), txOption, TxDone);
}


/*===========================   NodeInfoSendComplete   =======================
**    Start retrieving meta data
**
**
**
**--------------------------------------------------------------------------*/
void NodeInfoSendComplete(BYTE bStatus)
{

}

/*================================   GotoSleep   ============================
**    Power down the ZW0201 chip
**
**
**
**--------------------------------------------------------------------------*/
void GotoSleep()
{
  ZW_DEBUG_SEND_BYTE('G');

  LED_OFF(1);
  LED_OFF(2);
  LED_OFF(3);

  PIN_IN(LED1, 0);
  PIN_IN(LED2, 0);  /* Now tristate */
  PIN_IN(LED3, 0);  /* Now tristate */

  /* TO#1996 - Dont enable external int before powerdown */
  // EX1 = 1;  /* enable int1 (button) before power down */

  /* Goto sleep */
#ifdef TIMING_DEBUG
  PIN_OFF(TXDEBUGPin);
#endif

  /* ZW_SetSleepMode can fail now, handle it so application doesnt
     get confused */
  if (!ZW_SetSleepMode(ZW_FREQUENTLY_LISTENING_MODE, ZW_INT_MASK_EXT1, 0))
    StartSleepTimeout();
}

/****************************************************************************/
/*                           EXPORTED FUNCTIONS                             */
/****************************************************************************/

/*============================   ApplicationInitHW   ========================
**    Non Z-Wave hardware initialization
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
BYTE                       /*RET  TRUE        */
ApplicationInitHW(BYTE bReason)  /* IN  Nothing     */
{
  /* Disable external int 1 */
  EX1 = 0;

  /* hardware initialization */
  PIN_IN(Button, 1);

  /* Setup specifics for current dim module */
  PIN_OUT(LED1);
  PIN_OUT(LED2);
  PIN_OUT(LED3);

#ifdef  TIMING_DEBUG
  PIN_OFF(TXDEBUGPin);
  PIN_OUT(TXDEBUGPin);
#endif

  bWakeupReason = bReason;

  return(TRUE);
}


/*===========================   ApplicationInitSW   =========================
**    Initialization of the Application Software
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
BYTE                      /*RET  TRUE       */
ApplicationInitSW( void ) /* IN   Nothing   */
{
  ZW_DEBUG_INIT(1152);

  ZW_DEBUG_SEND_NL();
  ZW_DEBUG_SEND_BYTE('B');
  ZW_DEBUG_SEND_BYTE('e');
  ZW_DEBUG_SEND_BYTE('l');
  ZW_DEBUG_SEND_BYTE('l');
  ZW_DEBUG_SEND_BYTE(' ');

  /* Initialize button detection */
  OneButtonInit();

  /* Turn off all LED's */
  LED_OFF(1);
  LED_OFF(2);
  LED_OFF(3);

  /* Initialize flash if necessary */
  // if (MemoryGetByte(NVM_ASSOCIATION) == 0xFF)
  //{
  //  ZW_DEBUG_SEND_BYTE('X');
  //  MemoryPutBuffer(NVM_ASSOCIATION, NULL, sizeof(groups), NULL);
  //}
  //else
  //  AssociationGetAll();

  /* Get our node ID */
  MemoryGetID(NULL, &bMyNodeID);

  /* Set Nwtwork wide inclusion active if we dont have aa node ID */
  if (bMyNodeID)
    bNWIStartup = FALSE;
  else
    bNWIStartup = TRUE;

  bMainState = STATE_POWER_UP;

  return(TRUE);
}


/*============================   ApplicationTestPoll   ======================
**    Function description
**      This function is called when the dimmer enters test mode.
**    Side effects:
**       Code will not exit until it is reset
**--------------------------------------------------------------------------*/
void ApplicationTestPoll(void)
{

}


/*======================   ApplicationNodeInformation   =====================
**    Request Node information and current status
**    Called by the the Z-Wave application layer before transmitting a
**    "Node Information" frame.
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
extern void                  /*RET  Nothing */
ApplicationNodeInformation(
  BYTE       *listening,      /*OUT  TRUE if this node is always on air */
  APPL_NODE_TYPE  *nodeType,  /*OUT  Device type Generic and Specific   */
  BYTE       **nodeParm,      /*OUT  Device parameter buffer pointer    */
  BYTE       *parmLength)     /*OUT  Number of Device parameter bytes   */
{
#ifdef ZW030x
  *listening = APPLICATION_NODEINFO_NOT_LISTENING | APPLICATION_FREQ_LISTENING_MODE_250ms;
#else
  *listening = APPLICATION_NODEINFO_NOT_LISTENING | APPLICATION_FREQ_LISTENING_MODE_1000ms;
#endif
  nodeType->generic = GENERIC_TYPE_SWITCH_BINARY; /* Generic device type */
  nodeType->specific = SPECIFIC_TYPE_NOT_USED; /* Specific class */
  *nodeParm = (BYTE *)&nodeInfo;        /* Send list of known command classes. */
  *parmLength = sizeof(nodeInfo);       /* Set length*/
}


/*==========================   ApplictionSlaveUpdate   =======================
**   Inform a slave application that a node information is received.
**   Called from the slave command handler when a node information frame
**   is received and the Z-Wave protocol is not in a state where it is needed.
**
**--------------------------------------------------------------------------*/
void
ApplicationSlaveUpdate(
  BYTE bStatus,     /*IN  Status event */
  BYTE bNodeID,     /*IN  Node id of the node that send node info */
  BYTE* pCmd,       /*IN  Pointer to Application Node information */
  BYTE bLen)       /*IN  Node info length                        */
{
}

/*============================   LearnCompleted   ========================
**    Function description
**      Called from learnmode.c when learnmode completed
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                  /*RET	Nothing */
LearnCompleted(
BYTE nodeID)          /* IN resulting nodeID */
{
  ZW_DEBUG_SEND_BYTE('C');
  bMainState = STATE_ADDING_DONE;
  bMyNodeID = nodeID;
}
