using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Collections;
using Zensys.Framework;
using Zensys.ZWave.Logging;

namespace Zensys.ZWave.UPnPBridge.Controls
{
    /// <summary>
    /// ListDataView class. Implements the data bound ListView control.
    /// </summary>
    public class ListDataView : ListView
    {
        private LogDataSource dataSource;
        private void BuildList()
        {
            lock (this)
            {
                BeginUpdate();
                this.Items.Clear();
                if (dataSource.SelectedSession != null)
                {
                    foreach (LogPacket topRow in dataSource.SelectedSession.Packets)
                    {
                        ListViewItem topNode = new ListViewItem();
                        topNode.Tag = topRow;
                        topNode.Text = Tools.ToHexString(topRow.Payload, " ");
                        if (topRow.IsPut)
                        {
                            topNode.ImageIndex = 2;
                            topNode.StateImageIndex = 2;
                        }
                        else
                        {
                            topNode.ImageIndex = 3;
                            topNode.StateImageIndex = 3;
                        }
                        topNode.SubItems.Add(new ListViewItem.ListViewSubItem(topNode, "DateTime"));
                        Items.Add(topNode);
                    }
                }
                EndUpdate();
            }
        }
        /// <summary>
        /// Gets the row.
        /// </summary>
        /// <param name="treeNode">The tree node.</param>
        /// <returns></returns>
        public LogSession GetRow(ListViewItem treeNode)
        {
            return treeNode.Tag as LogSession;
        }
        /// <summary>
        /// Selects the node.
        /// </summary>
        /// <param name="UID">The UID.</param>
        public void SelectNode(Guid UID)
        {
            foreach (ListViewItem treeNode in Items)
            {
                if (GetRow(treeNode).Id == UID)
                {
                    if (SelectedItems.Count == 0 || treeNode != SelectedItems[0])
                    {
                        treeNode.Selected = true;
                        break;
                    }
                }
            }
        }
        /// <summary>
        /// Gets or sets the data source.
        /// </summary>
        /// <value>The data source.</value>
        [Category("Data")]
        public LogDataSource DataSource
        {
            get
            {
                return dataSource;
            }
            set
            {
                if (value != null)
                {
                    dataSource = value;
                    BuildList();
                    dataSource.PacketDataChanged += new EventHandler<Zensys.ZWave.Logging.Events.PacketDataChangedEventArgs>(dataSource_PacketDataChanged);
                    dataSource.SelectedSessionChanged += new EventHandler<Zensys.ZWave.Logging.Events.SelectedSessionChangedEventArgs>(dataSource_SelectedSessionChanged);
                    dataSource.SessionDataCleared += new EventHandler<EventArgs>(dataSource_SessionDataCleared);
                }
            }
        }
        protected override void Dispose(bool disposing)
        {
            base.Dispose(disposing);
            if (dataSource != null)
            {
                dataSource.PacketDataChanged -= new EventHandler<Zensys.ZWave.Logging.Events.PacketDataChangedEventArgs>(dataSource_PacketDataChanged);
                dataSource.SelectedSessionChanged -= new EventHandler<Zensys.ZWave.Logging.Events.SelectedSessionChangedEventArgs>(dataSource_SelectedSessionChanged);
                dataSource.SessionDataCleared -= new EventHandler<EventArgs>(dataSource_SessionDataCleared);
            }
        }
        void dataSource_SelectedSessionChanged(object sender, Zensys.ZWave.Logging.Events.SelectedSessionChangedEventArgs e)
        {
            ReBuildList();
        }
        void dataSource_SessionDataCleared(object sender, EventArgs e)
        {
            ReBuildList();
        }
        void dataSource_PacketDataChanged(object sender, Zensys.ZWave.Logging.Events.PacketDataChangedEventArgs e)
        {
            ReBuildList();
        }
        private delegate void ReBuildListDelegate();
        private void ReBuildList()
        {
            if (InvokeRequired)
            {
                Invoke(new ReBuildListDelegate(ReBuildList));
            }
            else
            {
                BuildList();
            }
        }
    }
}
