using System;
using System.Collections.Generic;
using System.Text;
using Zensys.Framework.Application;
using Zensys.ZWave.UPnPBridge.Models;
using Zensys.ZWave.UPnPBridge.Classes;
using Zensys.ZWave.UPnPBridge.Interfaces;
using System.Collections;
using Zensys.ZWave.UPnPBridge.UI;
using Zensys.ZWave.UPnPBridge.Dispatchers;
using System.Windows.Forms;
using Zensys.ZWave.UPnPBridge.Properties;
using Zensys.ZWave.Enums;
using Zensys.ZWave.Devices;
using Zensys.ZWave.UPnPBridge.UPnPClasses;
using System.Threading;
using Zensys.Framework;
using Zensys.ZWave.ZWaveHAL;
using Zensys.ZWave.Exceptions;
using Zensys.ZWave.Application;
using System.IO;
using Zensys.ZWave.ZWaveHAL.Actions;

namespace Zensys.ZWave.UPnPBridge.Controllers
{
    public class ControllerManager: IControllerManager
    {

        private string mXmlZWaveDefinitionFile = @"ZWave_custom_cmd_classes.xml";
        private XmlDataManager mXmlDataManager;
        public XmlDataManager XmlDataManager
        {
            get { return mXmlDataManager; }
            set { mXmlDataManager = value; }
        }

        public void ReloadXml()
        {
            DefinitionConverter dconv = new DefinitionConverter(mXmlZWaveDefinitionFile, null);
            dconv.UpgradeConvert(false);
            XmlDataManager.ZWaveDefinition = dconv.ZWaveDefinition;
        }

        /// <summary>
        /// Initializes the specified application mode.
        /// </summary>
        /// <param name="applicationMode">The application mode.</param>
        public void Initialize(ApplicationMode applicationMode)
        {
            mApplicationMode = applicationMode;
            //Initialization of the Document Model
            switch (applicationMode)
            {
                case ApplicationMode.Console:
                    mDocumentModel = new DocumentModel(null);
                    break;
                case ApplicationMode.GUI:
                    mDocumentModel = new DocumentModel(MainForm.BindingContext);
                    break;
                default:
                    break;
            }
            Tools.StartHighPrecisionTimer();
            //Initialization of the ZWave Dll Manager
            XmlDataManager = new XmlDataManager();
            if (!string.IsNullOrEmpty(mXmlZWaveDefinitionFile) && File.Exists(mXmlZWaveDefinitionFile))
            {
                DefinitionConverter dconv = new DefinitionConverter(mXmlZWaveDefinitionFile, null);
                dconv.UpgradeConvert(false);
                XmlDataManager.ZWaveDefinition = dconv.ZWaveDefinition;
            }
            mZWaveManager = new ZWaveManager();
            mZWaveManager.Init(LibraryModes.Bridge, true);
            mApplicationSettings = new SettingsWrapper();
            LogManager = new LogManager(WriteMessage, WriteOperationFailure);
            ExceptionManager = new ExceptionManager(WriteException);
        }

        private void WriteMessage(string message)
        {
            Actions.LogFormActions.WriteMessage(message);
        }

        private void WriteOperationFailure(string message)
        {
            Actions.LogFormActions.WriteMessage(string.Format("{0}: {1}", Tools.CurrentDateTime.ToString("hh:mm:ss.fff"), "Request failed: " + message));
        }

        public void WriteException(Exception exception)
        {
            if (exception is RequestTimeoutException)
            {
                Actions.LogFormActions.WriteMessage(string.Format("{0}: {1}", Tools.CurrentDateTime.ToString("hh:mm:ss.fff"), "Request time-out period exceeded"));
            }
            else
            {
                Actions.LogFormActions.WriteMessage(string.Format("{0}: {1}", Tools.CurrentDateTime.ToString("hh:mm:ss.fff"), "Exception: " + exception.Message));
            }
        }

        private LogManager mLogManager = new LogManager(null, null);
        public LogManager LogManager
        {
            get { return mLogManager; }
            set { mLogManager = value; }
        }

        private ExceptionManager mExceptionManager = new ExceptionManager(null);
        public ExceptionManager ExceptionManager
        {
            get { return mExceptionManager; }
            set { mExceptionManager = value; }
        }
        private SettingsWrapper mApplicationSettings;
        public SettingsWrapper ApplicationSettings
        {
            get { return mApplicationSettings; }
        }
        private ApplicationMode mApplicationMode;
        public ApplicationMode ApplicationMode
        {
            get { return mApplicationMode; }
        }
	
        private DocumentModel mDocumentModel;
        public DocumentModel DocumentModel
        {
            get { return mDocumentModel; }
        }

        private ActionCollection mActions;
        /// <summary>
        /// Gets the actions. <see cref="ActionCollection"/>
        /// </summary>
        /// <value>The actions.</value>
        public ActionCollection Actions
        {
            get
            {
                if (mActions == null)
                {
                    mActions = new ActionCollection(this);
                }
                return mActions;
            }
        }
        private MainForm mMainForm;
        /// <summary>
        /// Gets or sets the main form.
        /// </summary>
        /// <value>The main form.</value>
        public MainForm MainForm
        {
            get
            {
                return mMainForm;
            }
            set
            {
                UnregisterView(mMainForm, new MainFormDispatcher());
                mMainForm = value;
                if (value != null)
                {
                    RegisterView(value, new MainFormDispatcher());
                }

            }
        }
        private CommandClassForm mCommandClassForm;
        /// <summary>
        /// Gets or sets the command class form.
        /// </summary>
        /// <value>The command class form.</value>
        public CommandClassForm CommandClassForm
        {
            get { return mCommandClassForm; }
            set
            {
                UnregisterView(mCommandClassForm, new CommandClassFormDispatcher());
                mCommandClassForm = value;
                if (value != null)
                {
                    RegisterView(value, new CommandClassFormDispatcher());
                    value.DockStateChanged += new EventHandler(delegate
                    {
                        if (value.FloatPane != null && value.FloatPane.Parent != null)
                            value.FloatPane.Parent.MinimumSize = value.MinimumSize;
                    });
                }
            }
        }
        private ControllerForm mControllerForm;
        /// <summary>
        /// Gets or sets the controller form.
        /// </summary>
        /// <value>The controller form.</value>
        public ControllerForm ControllerForm
        {
            get { return mControllerForm; }
            set
            {
                UnregisterView(mControllerForm, new ControllerFormDispatcher());
                mControllerForm = value;
                if (value != null)
                {
                    RegisterView(value, new ControllerFormDispatcher());
                    mControllerForm.BindingContext = mMainForm.BindingContext;
                    value.DockStateChanged += new EventHandler(delegate
                    {
                        if (value.FloatPane != null && value.FloatPane.Parent != null)
                            value.FloatPane.Parent.MinimumSize = value.MinimumSize;
                    });
                }
            }
        }
        private LogForm mLogForm;
        /// <summary>
        /// Gets or sets the log form.
        /// </summary>
        /// <value>The log form.</value>
        public LogForm LogForm
        {
            get { return mLogForm; }
            set
            {
                UnregisterView(mLogForm, new LogFormDispatcher());
                mLogForm = value;
                mLogForm.BindingContext = mMainForm.BindingContext;
                if (value != null)
                {
                    RegisterView(value, new LogFormDispatcher());
                    value.DockStateChanged += new EventHandler(delegate
                    {
                        if (value.FloatPane != null && value.FloatPane.Parent != null)
                            value.FloatPane.Parent.MinimumSize = value.MinimumSize;
                    });
                }
            }
        }
        private NodeForm mNodeForm;
        /// <summary>
        /// Gets or sets the node form.
        /// </summary>
        /// <value>The node form.</value>
        public NodeForm NodeForm
        {
            get { return mNodeForm; }
            set
            {
                UnregisterView(mNodeForm, new NodeFormDispatcher());
                mNodeForm = value;
                mNodeForm.BindingContext = mMainForm.BindingContext; 
                if (value != null)
                {
                    RegisterView(value, new NodeFormDispatcher());
                    value.DockStateChanged += new EventHandler(delegate
                    {
                        if (value.FloatPane != null && value.FloatPane.Parent != null)
                            value.FloatPane.Parent.MinimumSize = value.MinimumSize;
                    });
                }
            }
        }

        private AssociationsForm mAssociationsForm;
        /// <summary>
        /// Gets or sets the node form.
        /// </summary>
        /// <value>The node form.</value>
        public AssociationsForm AssociationsForm
        {
            get { return mAssociationsForm; }
            set
            {
                UnregisterView(mAssociationsForm, new AssociationsFormDispatcher());
                mAssociationsForm = value;
                mAssociationsForm.BindingContext = mMainForm.BindingContext;
                if (value != null)
                {
                    RegisterView(value, new AssociationsFormDispatcher());
                    value.DockStateChanged += new EventHandler(delegate
                    {
                        if (value.FloatPane != null && value.FloatPane.Parent != null)
                            value.FloatPane.Parent.MinimumSize = value.MinimumSize;
                    });
                }
            }
        }

        private UPnPBinaryLightCPForm mUPnPBinaryLightCPForm;
        public UPnPBinaryLightCPForm UPnPBinaryLightCPForm
        {
            get { return mUPnPBinaryLightCPForm; }
            set
            {
                UnregisterView(mUPnPBinaryLightCPForm, new UPnPBinaryLightCPDispatcher());
                mUPnPBinaryLightCPForm = value;
                mUPnPBinaryLightCPForm.BindingContext = mMainForm.BindingContext; 
                if (value != null)
                {
                    RegisterView(value, new UPnPBinaryLightCPDispatcher());
                    value.DockStateChanged += new EventHandler(delegate
                    {
                        if (value.FloatPane != null && value.FloatPane.Parent != null)
                            value.FloatPane.Parent.MinimumSize = value.MinimumSize;
                    });
                }
            }
        }

        private BridgedUPnPDevicesForm mBridgedUPnPDevicesForm;
        public BridgedUPnPDevicesForm BridgedUPnPDevicesForm
        {
            get { return mBridgedUPnPDevicesForm; }
            set
            {
                UnregisterView(mBridgedUPnPDevicesForm, new BridgedUPnPDevicesDispatcher());
                mBridgedUPnPDevicesForm = value;
                mBridgedUPnPDevicesForm.BindingContext = mMainForm.BindingContext; 
                if (value != null)
                {
                    RegisterView(value, new BridgedUPnPDevicesDispatcher());
                    value.DockStateChanged += new EventHandler(delegate
                    {
                        if (value.FloatPane != null && value.FloatPane.Parent != null)
                            value.FloatPane.Parent.MinimumSize = value.MinimumSize;
                    });
                }
            }
        }

        private UPnPDeviceScannerForm mUPnPDeviceScannerForm;
        public UPnPDeviceScannerForm UPnPDeviceScannerForm
        {
            get { return mUPnPDeviceScannerForm; }
            set
            {
                UnregisterView(mUPnPDeviceScannerForm, new UPnPDeviceScannerDispatcher());
                mUPnPDeviceScannerForm = value;
                mUPnPDeviceScannerForm.BindingContext = mMainForm.BindingContext; 
                if (value != null)
                {
                    RegisterView(value, new UPnPDeviceScannerDispatcher());
                    value.DockStateChanged += new EventHandler(delegate
                    {
                        if (value.FloatPane != null && value.FloatPane.Parent != null)
                            value.FloatPane.Parent.MinimumSize = value.MinimumSize;
                    });
                }
            }
        }

        private UPnPRendererCPForm mUPnPRendererCPForm;
        public UPnPRendererCPForm UPnPRendererCPForm
        {
            get { return mUPnPRendererCPForm; }
            set
            {
                UnregisterView(mUPnPRendererCPForm, new UPnPRendererCPDispatcher());
                mUPnPRendererCPForm = value;
                mUPnPRendererCPForm.BindingContext = mMainForm.BindingContext; 
                if (value != null)
                {
                    RegisterView(value, new UPnPRendererCPDispatcher());
                    value.DockStateChanged += new EventHandler(delegate
                    {
                        if (value.FloatPane != null && value.FloatPane.Parent != null)
                            value.FloatPane.Parent.MinimumSize = value.MinimumSize;
                    });
                }
            }
        }

        private ZWaveManager mZWaveManager;
        /// <summary>
        /// Gets or sets the ZWave manager.
        /// </summary>
        /// <value>The Z wave manager.</value>
        public ZWaveManager ZWaveManager
        {
            get { return mZWaveManager; }
            set { mZWaveManager = value; }
        }
        /// <summary>
        /// Registers the view.
        /// </summary>
        /// <param name="view">The view.</param>
        /// <param name="dispatcher">The dispatcher.</param>
        public void RegisterView(IView view, IViewDispatcher dispatcher)
        {
            dispatcher.Bind(view, Actions, DocumentModel);
        }
        /// <summary>
        /// Unregisters the view.
        /// </summary>
        /// <param name="view">The view.</param>
        /// <param name="dispatcher">The dispatcher.</param>
        public void UnregisterView(IView view, IViewDispatcher dispatcher)
        {
            if (view != null)
                dispatcher.Drop(view, Actions, DocumentModel);
        }
       
        #region IDisposable Members

        public void Dispose()
        {
            UnregisterView(this.MainForm, new MainFormDispatcher());
            try
            {
                //ClosePort();
                //ZWaveSaveToFile();
                //foreach (ZensysBridgedDevice zenDev in UPnPdownDeviceList)
                //{
                //    ZWaveUPnPDeviceStop(zenDev);
                //    ZWaveNodes.Remove(zenDev.NodeId);
                //    UPnPdownDeviceList.Remove(zenDev);
                //}
            }
            catch (Exception)
            {
            }
        }

        #endregion

        private bool isBusy = false;
        private ProgressForm progressForm;
        internal object DoAction(QueueAction action)
        {
            return DoAction(action.ActionHandler, action.ActionHandlerParameters, action.ProgressInfo, action.CanCancel, action.CommandType);
        }
        internal object DoAction(Delegate eventHandler, object[] parameters, string progressInfo, bool canCancel, byte commandType)
        {
            if (isBusy) return null;
            isBusy = true;
            try
            {
                System.Windows.Forms.Application.DoEvents();
            }
            catch (Exception)
            {
            }
            if (progressForm == null)
            {
                progressForm = new ProgressForm();
                progressForm.OnCancel += new EventHandler(ProgressFormOnCancel);
            }
            progressForm.ProgressDelegate = eventHandler;
            progressForm.ProgressDelegateParameters = parameters;
            progressForm.ProgressInfo = progressInfo;
            progressForm.CanCancel = canCancel;
            progressForm.CommandType = commandType;
            progressForm.ShowDialog();
            if (progressForm.ProgressException != null)
            {
                if (progressForm.ProgressException is RequestTimeoutException)
                {
                    this.ShowMessage(string.Format("{0} request timeout exceeded", eventHandler.Method.Name), true);
                }
                else
                {
                    this.ShowMessage(progressForm.ProgressException.Message, true);
                }
            }
            isBusy = false;
            return progressForm.Result;
        }

        private bool mNetworkWideInclusionCanceled;
        public bool NetworkWideInclusionCanceled
        {
            get { return mNetworkWideInclusionCanceled; }
            set { mNetworkWideInclusionCanceled = value; }
        }
	
        private void ProgressFormOnCancel(object sender, EventArgs e)
        {
            ProgressForm form = (ProgressForm)sender;
            if (DocumentModel.Controller != null)
            {
                IController device = DocumentModel.Controller;
                try
                {
                    if (device.ConnectionStatus == Zensys.ZWave.Enums.ConnectionStatuses.Closed)
                    {
                        device.Open(device.SerialPort);
                    }
                    device.StopRequest(form.CommandType);
                    NetworkWideInclusionCanceled = true;
                }
                catch (Exception ex)
                {
                    throw ex;
                }
            }
            isBusy = false;
        }

        public void ShowView(Type type, bool show)
        {
            if (type == typeof(ControllerForm))
            {
                if (ControllerForm == null || ControllerForm.IsDisposed)
                {
                    ControllerForm = new ControllerForm();
                }
                if (show)
                {
                    ControllerForm.Show(MainForm.DockPanelMain);
                }
                else
                {
                    ControllerForm.Hide();
                }

            }
            else if (type == typeof(NodeForm))
            {
                if (NodeForm == null || NodeForm.IsDisposed)
                {
                    NodeForm = new NodeForm();
                }
                if (show)
                {
                    NodeForm.Show(MainForm.DockPanelMain);
                }
                else
                {
                    NodeForm.Hide();
                }
            }
            else if (type == typeof(CommandClassForm))
            {
                if (CommandClassForm == null || CommandClassForm.IsDisposed)
                {
                    CommandClassForm = new CommandClassForm();
                }
                if (show)
                {
                    CommandClassForm.Show(MainForm.DockPanelMain);
                }
                else
                {
                    CommandClassForm.Hide();
                }
            }
            else if (type == typeof(AssociationsForm))
            {
                if (AssociationsForm == null || AssociationsForm.IsDisposed)
                {
                    AssociationsForm = new AssociationsForm();
                }
                if (show)
                {
                    AssociationsForm.Show(MainForm.DockPanelMain);
                }
                else
                {
                    AssociationsForm.Hide();
                }
            }
            else if (type == typeof(UPnPBinaryLightCPForm))
            {
                if (UPnPBinaryLightCPForm == null || UPnPBinaryLightCPForm.IsDisposed)
                {
                    UPnPBinaryLightCPForm = new UPnPBinaryLightCPForm();
                }
                if (show)
                {
                    UPnPBinaryLightCPForm.Show(MainForm.DockPanelMain);
                }
                else
                {
                    UPnPBinaryLightCPForm.Hide();
                }
            }
            else if (type == typeof(BridgedUPnPDevicesForm))
            {
                if (BridgedUPnPDevicesForm == null || BridgedUPnPDevicesForm.IsDisposed)
                {
                    BridgedUPnPDevicesForm = new BridgedUPnPDevicesForm();
                }
                if (show)
                {
                    BridgedUPnPDevicesForm.Show(MainForm.DockPanelMain);
                }
                else
                {
                    BridgedUPnPDevicesForm.Hide();
                }
            }
            else if (type == typeof(UPnPDeviceScannerForm))
            {
                if (UPnPDeviceScannerForm == null || UPnPDeviceScannerForm.IsDisposed)
                {
                    UPnPDeviceScannerForm = new UPnPDeviceScannerForm();
                }
                if (show)
                {
                    UPnPDeviceScannerForm.Show(MainForm.DockPanelMain);
                }
                else
                {
                    UPnPDeviceScannerForm.Hide();
                }
            }
            else if (type == typeof(UPnPRendererCPForm))
            {
                if (UPnPRendererCPForm == null || UPnPRendererCPForm.IsDisposed)
                {
                    UPnPRendererCPForm = new UPnPRendererCPForm();
                }
                if (show)
                {
                    UPnPRendererCPForm.Show(MainForm.DockPanelMain);
                }
                else
                {
                    UPnPRendererCPForm.Hide();
                }
            }
            else if (type == typeof(LogForm))
            {
                if (LogForm == null || LogForm.IsDisposed)
                {
                    LogForm = new LogForm();
                }
                if (show)
                {
                    LogForm.Show(MainForm.DockPanelMain);
                }
                else
                {
                    LogForm.Hide();
                }
            }
        }

        internal void ShowErrorMessage(string message)
        {
            MessageBox.Show(message, Resources.CaptionZWaveController, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
        }
        internal void ShowMessage(string message, bool isError)
        {
            this.MainForm.BeginInvoke(new EventHandler(delegate
            {
                MessageBox.Show(this.MainForm, message, AboutBox.AssemblyProduct, MessageBoxButtons.OK, (isError == true) ? MessageBoxIcon.Exclamation : MessageBoxIcon.Information);
            }));
        }
        private Dictionary<byte, Queue<QueueAction>> mFramesToSend = new Dictionary<byte, Queue<QueueAction>>();
        /// <summary>
        /// The frames queue
        /// </summary>
        /// <value>The frames to send.</value>
        public Dictionary<byte, Queue<QueueAction>> FramesToSend
        {
            get { return mFramesToSend; }
        }

        public void AddQueueCommand(byte nodeId, QueueAction action)
        {
            Queue<QueueAction> queue = null;
            if (FramesToSend.ContainsKey(nodeId))
            {
                queue = FramesToSend[nodeId];
            }
            else
            {
                queue = new Queue<QueueAction>();
            }
            queue.Enqueue(action);
            FramesToSend[nodeId] = queue;
            this.MainForm.LabelCommandQueue = queue.Count.ToString();
            //System.Diagnostics.Debug.WriteLine("Queue changed.");
        }
        public void RemoveQueueCommand(byte nodeId)
        {
            if (FramesToSend.ContainsKey(nodeId))
            {
                Queue<QueueAction> queue = FramesToSend[nodeId];
                queue.Dequeue();
                this.MainForm.LabelCommandQueue = queue.Count.ToString();
            }
        }

        public void ClearQueueCommands()
        {
            FramesToSend.Clear();
        }

        AutoResetEvent queueGo = new AutoResetEvent(false);
        private object awakeNodesListLocker = new object();
        private SortedList<byte, bool> awakeNodesList = new SortedList<byte, bool>();

        public void ExecuteQueueCommand(byte sourceNodeId, bool isWakeUpNoMoreInformationFrameNeeded)
        {
            lock (awakeNodesListLocker)
            {
                if (!awakeNodesList.Keys.Contains(sourceNodeId))
                    awakeNodesList.Add(sourceNodeId, isWakeUpNoMoreInformationFrameNeeded);
                else if (isWakeUpNoMoreInformationFrameNeeded)
                {
                    awakeNodesList[sourceNodeId] = isWakeUpNoMoreInformationFrameNeeded;
                }
            }
            queueGo.Set();
        }

        private void ProcessQueuedCommand()
        {
            Queue<QueueAction> queue;
            while (true)
            {
                queueGo.WaitOne();
                bool someoneAwake = true;
                while (someoneAwake)
                {
                    someoneAwake = false;
                    Queue<QueueAction> innerqueue = new Queue<QueueAction>();
                    lock (awakeNodesListLocker)
                    {
                        for (int i = awakeNodesList.Count - 1; i >= 0; i--)
                        {
                            queue = null;
                            if (FramesToSend.ContainsKey(awakeNodesList.Keys[i]))
                            {
                                queue = FramesToSend[awakeNodesList.Keys[i]];
                                if (queue.Count > 0)
                                {
                                    innerqueue.Enqueue(queue.Dequeue());
                                    if (queue.Count > 0)
                                    {
                                        someoneAwake = true;
                                    }
                                }
                            }
                            if (queue == null || queue.Count == 0)
                            {
                                if (awakeNodesList.Values[i])
                                {
                                    innerqueue.Enqueue(CreateWakeUpNoMoreQueueAction(awakeNodesList.Keys[i]));
                                }
                                awakeNodesList.RemoveAt(i);
                            }
                        }
                    }
                    while (innerqueue.Count > 0)
                    {
                        if (!CustomLoader.GetEntryPoint().IsBusy)
                        {
                            QueueAction action = innerqueue.Dequeue();
                            if (action != null && action.ActionHandler != null)
                            {
                                this.DoAction(action);
                                if (action.WaitResponseTimeout > 0)
                                {
                                    Thread.Sleep(action.WaitResponseTimeout);
                                }
                            }
                            this.MainForm.LabelCommandQueue = innerqueue.Count.ToString();
                        }
                    }
                }
                Thread.Sleep(10);
            }
        }

        private QueueAction CreateWakeUpNoMoreQueueAction(byte sourceNodeId)
        {
            QueueAction ret = null;
            IDeviceInfo device = null;
            foreach (IDeviceInfo var in DocumentModel.Devices)
            {
                if (var.Id == sourceNodeId)
                {
                    device = var;
                    break;
                }
            }
            if (device != null)
            {
                ret = new QueueAction(
                    new ProcedureCaller<CommandClass,
                        Command,
                        byte,
                        IController,
                        List<byte>>(Actions.CommandClassFormActions.CommandClassHAL.SendCommand),
                    new object[] 
                    {
                        XmlDataManager.FindCommandClass("COMMAND_CLASS_WAKE_UP",1),
                        XmlDataManager.FindCommand("COMMAND_CLASS_WAKE_UP",1,"WAKE_UP_NO_MORE_INFORMATION"),
                        device.Id,
                        DocumentModel.Controller,
                        new List<byte>()
                    },
                Resources.MsgSendCommand, false, (byte)CommandTypes.CmdZWaveSendData, "Wake Up No More", 0);
            }
            return ret;
        }
    }
}
