using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.Devices;
using Zensys.ZWave;
using Zensys.PCController.Controllers;
using System.Collections;
using System.Timers;
using System.Windows.Forms;
using Zensys.ZWave.Logging;
using Zensys.ZWave.Events;
using Zensys.ZWave.ZWaveHAL;
using Zensys.Framework;

namespace Zensys.PCController.Models
{
    public class DocumentModel
    {
        public event EventHandler<EventArgs> DocumentModelStateChanged;
        public event EventHandler<EventArgs> ControllerChanged;
        private BindingContext mContext;
        public DocumentModel(BindingContext context)
        {
            mDevices = new ThreadedBindingList<IDeviceInfo>();
            mAssociativeDevices = new ThreadedBindingList<IDeviceInfo>();
            mAssociativeGroups = new ThreadedBindingList<AssociativeGroup>();

            mRoutesCollection = new RoutesCollection();
            mRoutesCollection.RouteCollectionChanged += new EventHandler<EventArgs>(OnDocumentModelStateChanged);
            mRoutesCollection.SelectedDestinationRouteChanged += new EventHandler<EventArgs>(OnDocumentModelStateChanged);
            mRoutesCollection.SelectedSourceRouteChanged += new EventHandler<EventArgs>(OnDocumentModelStateChanged);

            mTopologyMapItemCollection = new TopologyMapItemCollection();
            mTopologyMapItemCollection.CollectionChanged += new EventHandler<NotifyCollectionChangedEventArgs>(OnDocumentModelStateChanged);

            if (context != null)
            {
                mContext = context;
                mDevicesCurrencyManager = (CurrencyManager)context[mDevices];
                mAssociativeDevicesCurrencyManager = (CurrencyManager)context[mAssociativeDevices];
                mAssociativeGroupsCurrencyManager = (CurrencyManager)context[mAssociativeGroups];
            }
        }

        #region Devices (nodes)
        private ThreadedBindingList<IDeviceInfo> mDevices;
        public ThreadedBindingList<IDeviceInfo> Devices
        {
            get { return mDevices; }
            set { mDevices = value; }
        }

        private CurrencyManager mDevicesCurrencyManager;
        public CurrencyManager DevicesCurrencyManager
        {
            get { return mDevicesCurrencyManager; }
        }

        public IDeviceInfo CurrentDevice
        {
            get
            {
                if (mDevicesCurrencyManager.Position != -1)
                {
                    return (IDeviceInfo)mDevicesCurrencyManager.Current;
                }
                else
                {
                    return null;
                }
            }
        }
        #endregion

        #region AssociativeDevices
        private ThreadedBindingList<IDeviceInfo> mAssociativeDevices;
        public ThreadedBindingList<IDeviceInfo> AssociativeDevices
        {
            get { return mAssociativeDevices; }
            set { mAssociativeDevices = value; }
        }

        private CurrencyManager mAssociativeDevicesCurrencyManager;
        public CurrencyManager AssociativeDevicesCurrencyManager
        {
            get { return mAssociativeDevicesCurrencyManager; }
        }

        public IDeviceInfo CurrentAssociativeDevice
        {
            get
            {
                if (mAssociativeDevicesCurrencyManager.Position != -1)
                {
                    return (IDeviceInfo)mAssociativeDevicesCurrencyManager.Current;
                }
                else
                {
                    return null;
                }
            }
        }
        #endregion

        #region AssociativeGroups
        private ThreadedBindingList<AssociativeGroup> mAssociativeGroups;
        public ThreadedBindingList<AssociativeGroup> AssociativeGroups
        {
            get { return mAssociativeGroups; }
            set { mAssociativeGroups = value; }
        }

        private CurrencyManager mAssociativeGroupsCurrencyManager;
        public CurrencyManager AssociativeGroupsCurrencyManager
        {
            get { return mAssociativeGroupsCurrencyManager; }
        }

        public AssociativeGroup CurrentAssociativeGroup
        {
            get
            {
                if (mAssociativeGroupsCurrencyManager.Position != -1)
                {
                    return (AssociativeGroup)mAssociativeGroupsCurrencyManager.Current;
                }
                else
                {
                    return null;
                }
            }
        }
        #endregion

        void OnDocumentModelStateChanged(object sender, EventArgs e)
        {
            SetDocumentModelStateChanged();
        }

        public void SetDocumentModelStateChanged()
        {
            if (DocumentModelStateChanged != null)
                DocumentModelStateChanged(this, EventArgs.Empty);
        }

        private DataSourceStates mState;
        public DataSourceStates State
        {
            get { return mState; }
        }

        private IController mController = null;
        public IController Controller
        {
            get { return mController; }
            set { mController = value; }
        }

        internal void SetUpController(IController value)
        {
            mController = value;
            if (mController != null)
            {
                mController.ApplicationCommandHandlerEvent -= new DeviceAppCommandHandlerEventHandler(ControllerManager.Instance.Actions.CommonActions.ControllerApplicationCommandHandlerEvent);
                mController.ApplicationCommandHandlerEvent += new DeviceAppCommandHandlerEventHandler(ControllerManager.Instance.Actions.CommonActions.ControllerApplicationCommandHandlerEvent);

                mController.ControllerUpdated -= new ControllerUpdatedEventHandler(ControllerManager.Instance.Actions.ControllerFormActions.ControllerUpdated);
                mController.ControllerUpdated += new ControllerUpdatedEventHandler(ControllerManager.Instance.Actions.ControllerFormActions.ControllerUpdated);

                mController.LearnModeStatusChanged -= new LearnModeStatusChangedEventHandler(ControllerManager.Instance.Actions.ControllerFormActions.LearnModeStatusChanged);
                mController.LearnModeStatusChanged += new LearnModeStatusChangedEventHandler(ControllerManager.Instance.Actions.ControllerFormActions.LearnModeStatusChanged);

                mController.NodeStatusChanged -= new NodeStatusChangedEventHandler(ControllerManager.Instance.Actions.NodeFormActions.OnControllerNodeStatusChanged);
                mController.NodeStatusChanged += new NodeStatusChangedEventHandler(ControllerManager.Instance.Actions.NodeFormActions.OnControllerNodeStatusChanged);

                mController.CommandClassesStore.GetExternalStorageBuffer -= new GetExternalStorageBufferEventHandler(CommandClassesStore_GetExternalStorageBuffer);
                mController.CommandClassesStore.GetExternalStorageBuffer += new GetExternalStorageBufferEventHandler(CommandClassesStore_GetExternalStorageBuffer);

                mController.CommandClassesStore.PutExternalStorageBuffer -= new PutExternalStorageBufferEventHandler(CommandClassesStore_PutExternalStorageBuffer);
                mController.CommandClassesStore.PutExternalStorageBuffer += new PutExternalStorageBufferEventHandler(CommandClassesStore_PutExternalStorageBuffer);
            }
            SetDocumentModelStateChanged();
        }

        public byte[] ExternalCommandClassesStorage = new byte[2000];

        private void CommandClassesStore_PutExternalStorageBuffer(uint offset, byte[] buffer, uint length)
        {
            for (uint i = offset; i < offset + length; i++)
            {
                ExternalCommandClassesStorage[i] = buffer[i - offset];
            }
        }

        private byte[] CommandClassesStore_GetExternalStorageBuffer(uint offset, byte length)
        {
            byte[] ret = new byte[length];
            for (uint i = offset; i < offset + length; i++)
            {
                ret[i - offset] = ExternalCommandClassesStorage[i];
            }
            return ret;
        }

        private RoutesCollection mRoutesCollection;
        public RoutesCollection RoutesCollection
        {
            get { return mRoutesCollection; }
        }

        internal void RemoveDevice(byte deviceId)
        {
            lock (mDevices)
            {
                IDeviceInfo deviceForDelete = null;
                foreach (IDeviceInfo device in mDevices)
                {
                    if (device.Id == deviceId)
                    {
                        deviceForDelete = device;
                    }
                }
                if (deviceForDelete != null)
                {
                    mDevices.Remove(deviceForDelete);
                }
            }
        }
        internal void RemoveAssociativeDevice(byte deviceId)
        {
            lock (mDevices)
            {
                IDeviceInfo deviceForDelete = null;
                foreach (IDeviceInfo device in mAssociativeDevices)
                {
                    if (device.Id == deviceId)
                    {
                        deviceForDelete = device;
                    }
                }
                if (deviceForDelete != null)
                {
                    mAssociativeDevices.Remove(deviceForDelete);
                }
            }
        }

        public AssociativeGroup CreateAssociativeGroup()
        {
            AssociativeGroup ret = new AssociativeGroup(mContext);
            return ret;
        }

        private TopologyMapItemCollection mTopologyMapItemCollection;
        public TopologyMapItemCollection TopologyMapItemCollection
        {
            get { return mTopologyMapItemCollection; }
        }
    }
}
