using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.InstallerTool.Models;
using Zensys.ZWave.InstallerTool.Classes;
using Zensys.ZWave.InstallerTool.UI;
using Zensys.ZWave.Enums;
using System.Windows.Forms;
using Zensys.ZWave.InstallerTool.Interfaces;
using Zensys.ZWave.InstallerTool.Dispatchers;
using Zensys.ZWave.Devices;
using Zensys.ZWave.InstallerTool.Properties;
using System.Collections;
using Zensys.ZWave.ZWaveHAL;
using Zensys.Framework.Application;
using Zensys.Framework;
using Zensys.ZWave.Exceptions;
using System.Threading;
using Zensys.ZWave.Application;
using System.IO;

namespace Zensys.ZWave.InstallerTool.Controllers
{
    public class ControllerManager : IDisposable
    {
        private string mXmlZWaveDefinitionFile = @"ZWave_custom_cmd_classes.xml";
        private XmlDataManager mXmlDataManager;
        public XmlDataManager XmlDataManager
        {
            get { return mXmlDataManager; }
            set { mXmlDataManager = value; }
        }
        private ControllerManager()
        {
        }

        AutoResetEvent queueGo = new AutoResetEvent(false);
        public void Initialize(ApplicationMode applicationMode)
        {
            mApplicationMode = applicationMode;
            //Initialization of the Document Model
            switch (applicationMode)
            {
                case ApplicationMode.Console:
                    mDocumentModel = new DocumentModel(null);
                    break;
                case ApplicationMode.GUI:
                    mDocumentModel = new DocumentModel(MainForm.BindingContext);
                    break;
                default:
                    break;
            }
            Tools.StartHighPrecisionTimer();
            //Initialization of the Xml Data Manager
            XmlDataManager = new XmlDataManager();
            if (!string.IsNullOrEmpty(mXmlZWaveDefinitionFile) && File.Exists(mXmlZWaveDefinitionFile))
            {
                DefinitionConverter dconv = new DefinitionConverter(mXmlZWaveDefinitionFile, null);
                dconv.UpgradeConvert(false);
                XmlDataManager.ZWaveDefinition = dconv.ZWaveDefinition;
            }
            //Initialization of the ZWave Dll Manager
            mZWaveManager = new ZWaveManager();
            mZWaveManager.Init(LibraryModes.PcController, true);
            mApplicationSettings = new SettingsWrapper();
            LogManager = new LogManager(WriteMessage, WriteOperationFailure);
            ExceptionManager = new ExceptionManager(WriteException);
        }

        private ActionCollection mActions;
        private MainForm mMainForm;
        private LogForm mLogForm;
        private DockableView mActiveView;


        private LogManager mLogManager = new LogManager(null, null);
        public LogManager LogManager
        {
            get { return mLogManager; }
            set { mLogManager = value; }
        }

        private ExceptionManager mExceptionManager = new ExceptionManager(null);
        public ExceptionManager ExceptionManager
        {
            get { return mExceptionManager; }
            set { mExceptionManager = value; }
        }

        private void WriteMessage(string message)
        {
            Actions.LogFormActions.WriteMessage(message);
        }

        private void WriteOperationFailure(string message)
        {
            Actions.LogFormActions.WriteMessage(string.Format("{0}: {1}", Tools.CurrentDateTime.ToString("hh:mm:ss.fff"), "Request failed: " + message));
        }

        public void WriteException(Exception exception)
        {
            if (exception is RequestTimeoutException)
            {
                Actions.LogFormActions.WriteMessage(string.Format("{0}: {1}", Tools.CurrentDateTime.ToString("hh:mm:ss.fff"), "Request time-out period exceeded"));
            }
            else
            {
                Actions.LogFormActions.WriteMessage(string.Format("{0}: {1}", Tools.CurrentDateTime.ToString("hh:mm:ss.fff"), "Exception: " + exception.Message));
            }
        }
        private SettingsWrapper mApplicationSettings;
        public SettingsWrapper ApplicationSettings
        {
            get { return mApplicationSettings; }
        }
        private ApplicationMode mApplicationMode;
        /// <summary>
        /// Gets the application mode.
        /// </summary>
        /// <value>The application mode.</value>
        public ApplicationMode ApplicationMode
        {
            get { return mApplicationMode; }
        }

        private DocumentModel mDocumentModel;
        /// <summary>
        /// Gets the document model.
        /// </summary>
        /// <value>The document model.</value>
        public DocumentModel DocumentModel
        {
            get { return mDocumentModel; }
        }
        /// <summary>
        /// Gets the actions. <see cref="ActionCollection"/>
        /// </summary>
        /// <value>The actions.</value>
        public ActionCollection Actions
        {
            get
            {
                if (mActions == null)
                {
                    mActions = new ActionCollection(this);
                }
                return mActions;
            }
        }
        /// <summary>
        /// Gets or sets the Main Form.
        /// </summary>
        /// <value>The main form.</value>
        public MainForm MainForm
        {
            get
            {
                return mMainForm;
            }
            set
            {
                Instance.UnregisterView(mMainForm, new MainFormDispatcher());
                mMainForm = value;
                if (value != null)
                {
                    Instance.RegisterView(value, new MainFormDispatcher());
                }

            }
        }
        /// <summary>
        /// Gets or sets the Log form.
        /// </summary>
        /// <value>The Log form.</value>
        public LogForm LogForm
        {
            get { return mLogForm; }
            set
            {
                Instance.UnregisterView(mLogForm, new LogFormDispatcher());
                mLogForm = value;
                if (value != null)
                {
                    Instance.RegisterView(value, new LogFormDispatcher());
                }
            }
        }
        private RoutingTableForm mRoutingTableForm;
        public RoutingTableForm RoutingTableForm
        {
            get { return mRoutingTableForm; }
            set
            {
                Instance.UnregisterView(mRoutingTableForm, new RoutingTableFormDispatcher());
                mRoutingTableForm = value;
                if (value != null)
                {
                    Instance.RegisterView(value, new RoutingTableFormDispatcher());
                }
            }
        }
        private ControllerForm mControllerForm;

        public ControllerForm ControllerForm
        {
            get { return mControllerForm; }
            set
            {
                Instance.UnregisterView(mControllerForm, new ControllerFormDispatcher());
                mControllerForm = value;
                if (value != null)
                {
                    Instance.RegisterView(value, new ControllerFormDispatcher());
                }
            }
        }
        private NodeForm mNodeForm;

        public NodeForm NodeForm
        {
            get { return mNodeForm; }
            set
            {
                Instance.UnregisterView(mNodeForm, new NodeFormDispatcher());
                mNodeForm = value;
                if (value != null)
                {
                    Instance.RegisterView(value, new NodeFormDispatcher());
                }
            }
        }
        private static ControllerManager mInstance;
        /// <summary>
        /// Gets the instance.
        /// </summary>
        /// <value>The instance.</value>
        public static ControllerManager Instance
        {
            get
            {
                if (mInstance == null)
                {
                    mInstance = new ControllerManager();
                }
                return mInstance;
            }
        }

        /// <summary>
        /// Gets or sets the Active view.
        /// </summary>
        /// <value>The Active view.</value>
        public DockableView ActiveView
        {
            get { return mActiveView; }
            set { mActiveView = value; }
        }

        /// <summary>
        /// Registers the view.
        /// </summary>
        /// <param name="view">The view.</param>
        /// <param name="dispatcher">The dispatcher.</param>
        public void RegisterView(IView view, IViewDispatcher dispatcher)
        {
            dispatcher.Bind(view, Instance.Actions, this.DocumentModel);
        }
        /// <summary>
        /// Unregisters the view.
        /// </summary>
        /// <param name="view">The view.</param>
        /// <param name="dispatcher">The dispatcher.</param>
        public void UnregisterView(IView view, IViewDispatcher dispatcher)
        {
            if (view != null)
                dispatcher.Drop(view, Instance.Actions, this.DocumentModel);
        }

        private ZWaveManager mZWaveManager;
        /// <summary>
        /// Gets or sets the ZWave manager.
        /// </summary>
        /// <value>The ZWave manager.</value>
        public ZWaveManager ZWaveManager
        {
            get { return mZWaveManager; }
            set { mZWaveManager = value; }
        }

        /// <summary>
        /// Starts the application.
        /// </summary>
        /// <param name="args">The args.</param>
        public static void Start(string[] args)
        {
            System.Windows.Forms.Application.EnableVisualStyles();
            System.Windows.Forms.Application.SetCompatibleTextRenderingDefault(false);
            AppDomain.CurrentDomain.ProcessExit += new EventHandler(OnProcessExit);
            AppDomain.CurrentDomain.UnhandledException += new UnhandledExceptionEventHandler(OnUnhandledException);
            Instance.MainForm = new MainForm();
            Instance.Initialize(ApplicationMode.GUI);
            System.Windows.Forms.Application.Run(Instance.MainForm);
        }

        /// <summary>
        /// Called when [process exit].
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        static void OnProcessExit(object sender, EventArgs e)
        {
            if (mInstance != null)
                mInstance.Dispose();
        }
        /// <summary>
        /// Called when [unhandled exception].
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.UnhandledExceptionEventArgs"/> instance containing the event data.</param>
        static void OnUnhandledException(object sender, UnhandledExceptionEventArgs e)
        {
            //LogManager.Instance.WriteLog(string.Format("Exception {0} occured", e.ExceptionObject.GetType().FullName), LogDetailLevel.Short);
            //NotificationForm nf = new NotificationForm(e.ExceptionObject.GetType().FullName, e.ExceptionObject.ToString());
            //nf.Text = "Error";
            ////((Exception)e.ExceptionObject).TargetSite.GetMethodBody()
            //switch (nf.ShowDialog())
            //{
            //    case DialogResult.Abort:
            //        Application.Exit();
            //        break;
            //    case DialogResult.Ignore:
            //        return;
            //    case DialogResult.OK:
            //        break;
            //    default:
            //        break;
            //}
        }


        #region IDisposable Members

        /// <summary>
        /// Performs application-defined tasks associated with freeing, releasing, or resetting unmanaged resources.
        /// </summary>
        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (disposing)
            {
                //this.progressForm.Dispose();
                //UnregisterView(this.MainForm, new MainFormDispatcher());
            }
        }
        #endregion
        private bool isBusy = false;
        private ProgressForm progressForm;
        internal object DoAction(QueueAction action)
        {
            return DoAction(action.ActionHandler, action.ActionHandlerParameters, action.ProgressInfo, action.CanCancel, action.CommandType);
        }

        internal object DoAction(Delegate eventHandler, object[] parameters, string progressInfo, bool canCancel, byte commandType)
        {
            if (isBusy) return null;
            isBusy = true;
            try
            {
                System.Windows.Forms.Application.DoEvents();
            }
            catch (Exception)
            {
            }
            if (progressForm == null)
            {
                progressForm = new ProgressForm();
                progressForm.OnCancel += new EventHandler(ProgressFormOnCancel);
            }
            progressForm.ProgressDelegate = eventHandler;
            progressForm.ProgressDelegateParameters = parameters;
            progressForm.ProgressInfo = progressInfo;
            progressForm.CanCancel = canCancel;
            progressForm.CommandType = commandType;
            progressForm.ShowDialog();
            if (progressForm.ProgressException != null)
            {
                if (progressForm.ProgressException is RequestTimeoutException)
                {
                    this.ShowMessage(string.Format("{0} request timeout exceeded", eventHandler.Method.Name), true);
                }
                else
                {
                    this.ShowMessage(progressForm.ProgressException.Message, true);
                }
            }
            isBusy = false;
            return progressForm.Result;
        }

        private bool mNetworkWideInclusionCanceled;
        public bool NetworkWideInclusionCanceled
        {
            get { return mNetworkWideInclusionCanceled; }
            set { mNetworkWideInclusionCanceled = value; }
        }

        void ProgressFormOnCancel(object sender, EventArgs e)
        {
            ProgressForm form = (ProgressForm)sender;
            if (DocumentModel.Controller != null)
            {
                IController device = DocumentModel.Controller;
                try
                {
                    if (device.ConnectionStatus == Zensys.ZWave.Enums.ConnectionStatuses.Closed)
                    {
                        device.Open(device.SerialPort);
                    }
                    device.StopRequest(form.CommandType);
                    NetworkWideInclusionCanceled = true;
                }
                catch (Exception ex)
                {
                    throw ex;
                }
            }
            isBusy = false;
        }


        private Dictionary<byte, Queue<QueueAction>> mFramesToSend = new Dictionary<byte, Queue<QueueAction>>();
        /// <summary>
        /// The frames queue
        /// </summary>
        /// <value>The frames to send.</value>
        public Dictionary<byte, Queue<QueueAction>> FramesToSend
        {
            get { return mFramesToSend; }
        }

        internal void ShowMessage(string message, bool isError)
        {
            this.MainForm.BeginInvoke(new EventHandler(delegate
            {
                MessageBox.Show(this.MainForm, message, AssemblyAttributeAccessor.AssemblyProduct, MessageBoxButtons.OK, (isError == true) ? MessageBoxIcon.Exclamation : MessageBoxIcon.Information);
            }));

        }

        internal DialogResult ShowConfirmMessage(string message, bool isError)
        {
            DialogResult result = DialogResult.None;
            this.MainForm.BeginInvoke(new EventHandler(delegate
            {
                result = MessageBox.Show(this.MainForm, message, AssemblyAttributeAccessor.AssemblyProduct, MessageBoxButtons.YesNo, (isError == true) ? MessageBoxIcon.Exclamation : MessageBoxIcon.Information);
            }));

            return result;
        }


        public void ShowView(Type type, bool show)
        {
            if (type == typeof(ControllerForm))
            {
                if (ControllerForm == null || ControllerForm.IsDisposed)
                {
                    ControllerForm = new ControllerForm();
                }
                if (show)
                {
                    ControllerForm.Show(MainForm.MainDockPanel);
                }
                else
                {
                    ControllerForm.Hide();
                }

            }
            else if (type == typeof(NodeForm))
            {
                if (NodeForm == null || NodeForm.IsDisposed)
                {
                    NodeForm = new NodeForm();
                }
                if (show)
                {
                    NodeForm.Show(MainForm.MainDockPanel);
                }
                else
                {
                    NodeForm.Hide();
                }
            }
            else if (type == typeof(LogForm))
            {
                if (LogForm == null || LogForm.IsDisposed)
                {
                    LogForm = new LogForm();
                }
                if (show)
                {
                    LogForm.Show(MainForm.MainDockPanel);
                }
                else
                {
                    LogForm.Hide();
                }
            }
            else if (type == typeof(RoutingTableForm))
            {
                if (RoutingTableForm == null || RoutingTableForm.IsDisposed)
                {
                    RoutingTableForm = new RoutingTableForm();
                }
                if (show)
                {
                    RoutingTableForm.Show(MainForm.MainDockPanel);
                }
                else
                {
                    RoutingTableForm.Hide();
                }
            }
        }

        public void AddQueueCommand(byte nodeId, QueueAction action)
        {
            Queue<QueueAction> queue = null;
            if (FramesToSend.ContainsKey(nodeId))
            {
                queue = FramesToSend[nodeId];
            }
            else
            {
                queue = new Queue<QueueAction>();
            }
            queue.Enqueue(action);
            FramesToSend[nodeId] = queue;
            //this.MainForm.LabelCommandQueue = queue.Count.ToString();
            //System.Diagnostics.Debug.WriteLine("Queue changed.");
        }
        public void RemoveQueueCommand(byte nodeId)
        {
            if (FramesToSend.ContainsKey(nodeId))
            {
                Queue<QueueAction> queue = FramesToSend[nodeId];
                queue.Dequeue();
               // this.MainForm.LabelCommandQueue = queue.Count.ToString();
            }
        }

        public void ClearQueueCommands()
        {
            FramesToSend.Clear();
        }

        private object awakeNodesListLocker = new object();
        private SortedList<byte, bool> awakeNodesList = new SortedList<byte, bool>();
        public void ExecuteQueueCommand(byte sourceNodeId, bool isWakeUpNoMoreInformationFrameNeeded)
        {
            lock (awakeNodesListLocker)
            {
                if (!awakeNodesList.Keys.Contains(sourceNodeId))
                    awakeNodesList.Add(sourceNodeId, isWakeUpNoMoreInformationFrameNeeded);
                else if (isWakeUpNoMoreInformationFrameNeeded)
                {
                    awakeNodesList[sourceNodeId] = isWakeUpNoMoreInformationFrameNeeded;
                }
            }
            queueGo.Set();
        }
    }
}
