using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.ComponentModel;
using System.Drawing;
using System.Threading;

namespace Zensys.Framework.UI.Controls
{

    public class ListDataView : ListView
    {
        public delegate void BuildListDelegate();
        public delegate void AddItemDelegate(IBindingList sender, ListChangedEventArgs e);
        private IBindingList mDataSource;
        private CurrencyManager mCurrencyManager = null;
        private List<ItemColumn> mItemColumns = new List<ItemColumn>();
        public List<ItemColumn> ItemColumns
        {
            get { return mItemColumns; }
        }


        [Category("Data")]
        public IBindingList DataSource
        {
            get
            {
                return mDataSource;
            }
            set
            {
                if (value != null)
                {
                    mDataSource = value;
                    mDataSource.ListChanged += new ListChangedEventHandler(mDataSource_ListChanged);
                    mCurrencyManager = (CurrencyManager)this.BindingContext[value];
                    InitColumns();
                    InitBgWorker();
                    BuildList();
                }
                else
                {
                    this.mCurrencyManager = null;
                    this.Items.Clear();
                }
            }
        }

        private void InitBgWorker()
        {
            afterAddItemWorker.DoWork += new DoWorkEventHandler(worker_DoWork);
        }

        void mDataSource_ListChanged(object sender, ListChangedEventArgs e)
        {
            switch (e.ListChangedType)
            {
                case ListChangedType.ItemAdded:
                    AddItem((IBindingList)sender, e);
                    break;
                case ListChangedType.ItemChanged:
                    //
                    break;
                case ListChangedType.ItemDeleted:
                    // RemoveItem() removes both item and databounded object
                    break;
                case ListChangedType.ItemMoved:
                    //
                    break;
                case ListChangedType.PropertyDescriptorAdded:
                    break;
                case ListChangedType.PropertyDescriptorChanged:
                    break;
                case ListChangedType.PropertyDescriptorDeleted:
                    break;
                case ListChangedType.Reset:
                    BuildList();
                    break;
                default:
                    break;
            }
        }

        private void AddItem(IBindingList sender, ListChangedEventArgs e)
        {
            if (this.InvokeRequired)
            {
                this.Invoke(new AddItemDelegate(AddItem), sender, e);
            }
            else
            {
                lock (this)
                {
                    if (e.NewIndex == sender.Count-1)
                    {
                        ListViewItem item = new ListViewItem();
                        item.Tag = sender[e.NewIndex];
                        item.Text = GetColumnText(ItemColumns[0], sender[e.NewIndex]);
                        item.ImageIndex = GetColumnImageIndex(ItemColumns[0], sender[e.NewIndex]);
                        for (int i = 1; i < ItemColumns.Count; i++)
                        {
                            ListViewItem.ListViewSubItem subItem = new ListViewItem.ListViewSubItem();
                            subItem.Text = GetColumnText(ItemColumns[i], sender[e.NewIndex]);
                            item.SubItems.Add(subItem);
                        }
                        if (AlternativeBkColor != BkColor)
                        {
                            item.BackColor = e.NewIndex % 2 == 0 ? AlternativeBkColor : BkColor;
                        }
                        if (AutoScrollItems)
                        {
                            this.BeginUpdate();
                        }
                        Items.Add(item);
                        if (AutoScrollItems)
                        {
                            addedItemsStack.Clear();
                            addedItemsStack.Push(item);
                            if (!afterAddItemWorker.IsBusy)
                            {
                                afterAddItemWorker.RunWorkerAsync(item);
                                afterAddItemWorker.RunWorkerCompleted += new RunWorkerCompletedEventHandler(afterAddItemWorker_RunWorkerCompleted);
                            }
                        }
                    }
                }
            }
        }

        void afterAddItemWorker_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
           this.EndUpdate();
        }

        Stack<ListViewItem> addedItemsStack = new Stack<ListViewItem>();
        BackgroundWorker afterAddItemWorker = new BackgroundWorker();

        private void worker_DoWork(object sender, DoWorkEventArgs e)
        {
            ListViewItem lvi = null;
            while (addedItemsStack.Count > 0)
            {
                lvi = addedItemsStack.Pop();
                this.Invoke(new EventHandler(delegate
                {
                    lvi.EnsureVisible();
                }));
            }
        }

        private bool mAutoScrollItems;
        public bool AutoScrollItems
        {
            get { return mAutoScrollItems; }
            set { mAutoScrollItems = value; }
        }

        private Color mAlternativeBkColor = Color.White;
        public Color AlternativeBkColor
        {
            get { return mAlternativeBkColor; }
            set { mAlternativeBkColor = value; }
        }

        private Color mBkColor = Color.White;
        public Color BkColor
        {
            get { return mBkColor; }
            set { mBkColor = value; }
        }

        private void BuildList()
        {
            if (this.InvokeRequired)
            {
                this.Invoke(new BuildListDelegate(BuildList));
            }
            else
            {
                lock (this)
                {
                    BeginUpdate();
                    this.Items.Clear();
                    int position = 0;
                    foreach (object obj in mCurrencyManager.List)
                    {
                        ListViewItem item = new ListViewItem();
                        item.Tag = obj;
                        item.Text = GetColumnText(ItemColumns[0], obj);
                        item.ImageIndex = GetColumnImageIndex(ItemColumns[0], obj);
                        for (int i = 1; i < ItemColumns.Count; i++)
                        {
                            ListViewItem.ListViewSubItem subItem = new ListViewItem.ListViewSubItem();
                            subItem.Text = GetColumnText(ItemColumns[i], obj);
                            item.SubItems.Add(subItem);
                        }
                        if (AlternativeBkColor != BkColor)
                        {
                            item.BackColor = position % 2 == 0 ? AlternativeBkColor : BkColor;
                        }
                        Items.Add(item);
                        if (position == mCurrencyManager.Position)
                        {
                            item.Selected = true;
                        }
                        position++;
                    }
                    EndUpdate();
                }
            }
        }

        private void InitColumns()
        {
            this.Columns.Clear();
            foreach (ItemColumn icol in ItemColumns)
            {
                ColumnHeader col = new ColumnHeader();
                col.Text = icol.Caption;
                col.Width = icol.Width;
                col.Tag = icol;
                this.Columns.Add(col);
            }
        }

        private string GetLeafText(string[] displayMembers, string displayFormat, object obj)
        {
            object[] displayValues = new object[displayMembers.Length];
            for (int i = 0; i < displayMembers.Length; i++)
            {
                displayValues[i] = mCurrencyManager.GetItemProperties()[displayMembers[i]].GetValue(obj);
            }
            return string.Format(displayFormat, displayValues);
        }
        
        private string GetColumnText(ItemColumn iCol, object obj)
        {
            List<object> dataObjects = new List<object>();
            for (int i = 0; i < iCol.DataItems.Length; i++)
            {
                object ob = mCurrencyManager.GetItemProperties()[iCol.DataItems[i]].GetValue(obj);
                if (ob != null)
                {
                    dataObjects.Add(ob);
                }
                else
                {
                    dataObjects.Add("");
                }
            }
            return iCol.GetText(dataObjects);
        }

        private int GetColumnImageIndex(ItemColumn iCol, object obj)
        {
            List<object> dataObjects = new List<object>();
            for (int i = 0; i < iCol.DataItems.Length; i++)
            {
                object ob = mCurrencyManager.GetItemProperties()[iCol.DataItems[i]].GetValue(obj);
                if (ob != null)
                {
                    dataObjects.Add(ob);
                }
                else
                {
                    dataObjects.Add("");
                }
            }
            return iCol.GetImageIndex(dataObjects);
        }

        protected override void OnItemSelectionChanged(ListViewItemSelectionChangedEventArgs e)
        {
            base.OnItemSelectionChanged(e);
            if (e.IsSelected && mCurrencyManager!=null)
            {
                mCurrencyManager.Position = e.ItemIndex;
            }
        }

        public void RemoveItem(ListViewItem item)
        {
            if (DataSource != null)
            {
                DataSource.Remove(item.Tag);
            }
            Items.Remove(item);
        }

        public void ClearItems()
        {
            if (DataSource != null)
            {
                DataSource.Clear();
            }
            Items.Clear();
        }
    }
}
