using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.Enums;
using System.Diagnostics;
using Zensys.ZWave.Runtime;
using Zensys.ZWave.Devices;
using Zensys.ZWave.Logging;
using System.ComponentModel;

namespace Zensys.ZWave
{
    /// <summary>
    /// Represents a <b>ZWaveManager</b> that manage all layers.
    /// </summary>
    public class ZWaveManager
    {
        /// <summary>
        /// Represents the method that will handle the status changed event.
        /// </summary>
        public event EventHandler StatusChanged;
        /// <summary>
        /// Initializes a new instance of the <see cref="ZWaveManager"/> class.
        /// </summary>
        public ZWaveManager()
        {
        }
        /// <summary>
        /// Get or sets Library Mode.
        /// </summary>
        public static LibraryModes LibraryMode = LibraryModes.None;
        private IApplicationLayer mApplicationLayer;
        /// <summary>
        /// Gets or sets the application layer.
        /// </summary>
        /// <value>The application layer.</value>
        public IApplicationLayer ApplicationLayer
        {
            get
            {
                return mApplicationLayer;
            }
            set
            {
                mApplicationLayer = value;
            }
        }

        private IFrameLayer mFrameLayer;
        /// <summary>
        /// Gets or sets the frame layer.
        /// </summary>
        /// <value>The frame layer.</value>
        public IFrameLayer FrameLayer
        {
            get
            {
                return mFrameLayer;
            }
            set
            {
                mFrameLayer = value;
            }
        }

        private ISessionLayer mSessionLayer;
        /// <summary>
        /// Gets or sets the session layer.
        /// </summary>
        /// <value>The session layer.</value>
        public ISessionLayer SessionLayer
        {
            get
            {
                return mSessionLayer;
            }
            set
            {
                mSessionLayer = value;
            }
        }

        private ITransportLayer mTransportLayer;
        /// <summary>
        /// Gets or sets the transport layer.
        /// </summary>
        /// <value>The transport layer.</value>
        public ITransportLayer TransportLayer
        {
            get
            {
                return mTransportLayer;
            }
            set
            {
                mTransportLayer = value;
            }
        }

        private ManagerStatus mStatus = ManagerStatus.None;
        /// <summary>
        /// Gets or sets the status.
        /// </summary>
        /// <value>The status.</value>
        public ManagerStatus Status
        {
            get
            {
                return mStatus;
            }
            set
            {
                mStatus = value;
                if (StatusChanged != null)
                {
                    StatusChanged(this, EventArgs.Empty);
                }
            }
        }

        private ThreadedBindingList<LogPacket> mLogDataSource = new ThreadedBindingList<LogPacket>();
        /// <summary>
        /// Gets the log data source.
        /// </summary>
        /// <value>The log data source.</value>
        public ThreadedBindingList<LogPacket> LogDataSource
        {
            get { return mLogDataSource; }
        }

        //private ThreadedBindingList<LogPacket> mFrameDataSource = new ThreadedBindingList<LogPacket>();
        ///// <summary>
        ///// Gets the frame data source.
        ///// </summary>
        ///// <value>The frame data source.</value>
        //public ThreadedBindingList<LogPacket> FrameDataSource
        //{
        //    get { return mFrameDataSource; }
        //}


        /// <summary>
        /// Inits this instance.
        /// </summary>
        /// <param name="mode">The library mode.</param>
        public void Init(LibraryModes mode, bool useExternalCommandClassesStorage)
        {
            RuntimeManager.LoadAssemblies(this);
            FrameLayer.Init(TransportLayer, LogDataSource);
            SessionLayer.Init(FrameLayer, LogDataSource);
            ApplicationLayer.Init(SessionLayer, useExternalCommandClassesStorage);
            Status = ManagerStatus.Initialized;
            LibraryMode = mode;
        }

        /// <summary>
        /// Inits the specified application layer DLL name.
        /// </summary>
        /// <param name="applicationLayerDllName">Name of the application layer DLL.</param>
        /// <param name="applicationLayerInstanceName">Name of the application layer instance.</param>
        /// <param name="transportLayerDllName">Name of the transport layer DLL.</param>
        /// <param name="transportLayerInstanceName">Name of the transport layer instance.</param>
        /// <param name="sessionLayerDllName">Name of the session layer DLL.</param>
        /// <param name="sessionLayerInstanceName">Name of the session layer instance.</param>
        /// <param name="frameLayerDllName">Name of the frame layer DLL.</param>
        /// <param name="frameLayerInstanceName">Name of the frame layer instance.</param>
        /// <param name="mode">The library mode. See <see cref="LibraryModes"/></param>
        public void Init(string applicationLayerDllName,
            string applicationLayerInstanceName,
            string transportLayerDllName,
            string transportLayerInstanceName,
            string sessionLayerDllName,
            string sessionLayerInstanceName,
            string frameLayerDllName,
            string frameLayerInstanceName,
            LibraryModes mode, bool useExternalCommandClassesStorage)
        {
            RuntimeManager.LoadAssemblies(this, applicationLayerDllName,
             applicationLayerInstanceName,
             transportLayerDllName,
             transportLayerInstanceName,
             sessionLayerDllName,
             sessionLayerInstanceName,
             frameLayerDllName,
             frameLayerInstanceName);
            FrameLayer.Init(TransportLayer, LogDataSource);
            SessionLayer.Init(FrameLayer, LogDataSource);
            ApplicationLayer.Init(SessionLayer, useExternalCommandClassesStorage);
            Status = ManagerStatus.Initialized;
            LibraryMode = mode;
        }

        /// <summary>
        /// Changes the layer.
        /// </summary>
        /// <param name="layer">The layer.</param>
        /// <param name="layerDllName">Name of the layer DLL.</param>
        /// <param name="layerInstanceName">Name of the layer instance.</param>
        public void ChangeLayer(RuntimeManager.Layers layer, string layerDllName,
            string layerInstanceName)
        {
            bool useExternalCommandClassesStorage = false;
            if (ApplicationLayer != null)
            {
                useExternalCommandClassesStorage = ApplicationLayer.UseExternalCommandClassesStorage;
            }
            RuntimeManager.LoadAssembly(this, layer, layerDllName, layerInstanceName);
            FrameLayer.Init(TransportLayer, LogDataSource);
            SessionLayer.Init(FrameLayer, LogDataSource);
            ApplicationLayer.Init(SessionLayer, useExternalCommandClassesStorage);
        }
    }
}
