using System;
using System.Collections.Generic;
using System.Text;
using System.Threading;
using System.ComponentModel;

namespace Zensys.ZWave.Logging
{
    /// <summary>
    /// Provides a generic collection that supports data binding. Used on UI thread
    /// </summary>
    /// <typeparam name="T"></typeparam>
    public class ThreadedBindingList<T> : BindingList<T>
    {
        private object lockObject = new object();

        private SynchronizationContext mSyncContext = null;
        private SendOrPostCallback mUpdateCallback = null;
        private Queue<DataTransaction> innerQueue = new Queue<DataTransaction>();
        private BackgroundWorker bgWorker = new BackgroundWorker(); 
        
        
        private bool mIsAsyncModel = false;
        /// <summary>
        /// Gets or sets a value indicating whether this instance is async model.
        /// </summary>
        /// <value>
        /// 	<c>true</c> if this instance is async model; otherwise, <c>false</c>.
        /// </value>
        public bool IsAsyncModel
        {
            get { return mIsAsyncModel; }
            set { mIsAsyncModel = value; }
        }

        private bool mIsQueuedModel = false;

        /// <summary>
        /// Gets or sets a value indicating whether this instance is queued model.
        /// </summary>
        /// <value>
        /// 	<c>true</c> if this instance is queued model; otherwise, <c>false</c>.
        /// </value>
        public bool IsQueuedModel
        {
            get { return mIsQueuedModel; }
            set 
            {
                mIsQueuedModel = value;
                if (mIsQueuedModel)
                {
                    bgWorker = new BackgroundWorker();
                    bgWorker.WorkerSupportsCancellation = true;
                    bgWorker.DoWork += new DoWorkEventHandler(bgWorker_DoWork);
                    bgWorker.RunWorkerCompleted += new RunWorkerCompletedEventHandler(bgWorker_RunWorkerCompleted);
                }
                else
                {
                   
                }
            }
        }

        void bgWorker_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            ////lock (lockObject)
            //{
            //    if (innerQueue.Count > 0)
            //    {

            //        if (!bgWorker.IsBusy)
            //        {
            //            bgWorker.RunWorkerAsync();
            //        }
            //    }
            //}
        }

        void bgWorker_DoWork(object sender, DoWorkEventArgs e)
        {
            DataTransaction dataTransaction = null;
            while (innerQueue.Count > 0)
            {
                //lock (lockObject)
                {
                    dataTransaction = innerQueue.Dequeue();
                }
                this.mSyncContext.Send(this.mUpdateCallback, dataTransaction);
            }
        }
	
        /// <summary>
        /// Sets the SynchronizationContext.
        /// </summary>
        /// <value>The SynchronizationContext.</value>
        public SynchronizationContext Ctx
        {
            set
            {
                Initialize(value);
            }
        }

        private void Initialize(SynchronizationContext synchronizationContext)
        {
            this.mSyncContext = synchronizationContext;
            this.mUpdateCallback = new SendOrPostCallback(this.UpdateBindingList);
            IsQueuedModel = false;
            IsAsyncModel = false;
        }

        #region Overrides from the base class. Executed on the Worker thread or on the UI thread.

        /// <summary>
        /// Gets the number of elements actually contained in the <see cref="T:System.Collections.ObjectModel.Collection`1"/>.
        /// </summary>
        /// <value></value>
        /// <returns>
        /// The number of elements actually contained in the <see cref="T:System.Collections.ObjectModel.Collection`1"/>.
        /// </returns>
        public new int Count
        {
            get
            {
                if (mSyncContext == null)
                {
                    return base.Count;
                }
                else
                {
                    DataTransaction tr = new DataTransaction(DataOperationKind.GetCount, null, -1);
                    SyncSend(tr);
                    return (int)tr.TransactionResult;
                }
            }
        }

        /// <summary>
        /// Adds an object to the end of the <see cref="T:System.Collections.ObjectModel.Collection`1"/>.
        /// </summary>
        /// <param name="item">The object to be added to the end of the <see cref="T:System.Collections.ObjectModel.Collection`1"/>. The value can be null for reference types.</param>
        public new void Add(T item)
        {
            if (mSyncContext == null)
            {
                base.Add(item);
            }
            else
            {
                if (mIsAsyncModel)
                {
                    this.mSyncContext.Post(this.mUpdateCallback, new DataTransaction(DataOperationKind.Add, item, -1));
                }
                else
                {
                    SyncSend(new DataTransaction(DataOperationKind.Add, item, -1));
                }
            }
        }

       
        private void SyncSend(DataTransaction dataTransaction)
        {
            if (IsQueuedModel && dataTransaction.DataOperationKind != DataOperationKind.GetCount)
            {
                if (dataTransaction.DataOperationKind == DataOperationKind.Clear)
                {
                    //lock (lockObject) 
                    { innerQueue.Clear(); }
                }

                //lock (lockObject) 
                {
                    innerQueue.Enqueue(dataTransaction);
                    if (!bgWorker.IsBusy)
                    {
                        bgWorker.RunWorkerAsync();
                    }
                    else
                    {
                        System.Diagnostics.Debug.WriteLine("isBusy=true");
                    }
                }
            }
            else
            {
                this.mSyncContext.Send(this.mUpdateCallback, dataTransaction);
            }
        }

        /// <summary>
        /// Adds a new item to the end of the collection.
        /// </summary>
        /// <returns>
        /// The item that was added to the collection.
        /// </returns>
        /// <exception cref="T:System.InvalidCastException">
        /// The new item is not the same type as the objects contained in the <see cref="T:System.ComponentModel.BindingList`1"/>.
        /// </exception>
        protected override object AddNewCore()
        {
            if (mSyncContext == null)
            {
                return base.AddNewCore();
            }
            else
            {
                // we need the result for AddNewCore...
                // So we block until AddNewCore finishes on the UI thread.
                DataTransaction transaction = new DataTransaction(DataOperationKind.AddNewItem, null, -1);
                SyncSend(transaction);

                return transaction.TransactionResult;
            }
        }

        /// <summary>
        /// Removes all elements from the collection.
        /// </summary>
        protected override void ClearItems()
        {
            if (mSyncContext == null)
            {
                base.ClearItems();
            }
            else
            {
                if (mIsAsyncModel)
                {
                    this.mSyncContext.Post(this.mUpdateCallback, new DataTransaction(DataOperationKind.Clear, null, -1));
                }
                else
                {
                    SyncSend(new DataTransaction(DataOperationKind.Clear, null, -1));
                }
            }
        }

        /// <summary>
        /// Inserts the specified item in the list at the specified index.
        /// </summary>
        /// <param name="index">The zero-based index where the item is to be inserted.</param>
        /// <param name="item">The item to insert in the list.</param>
        protected override void InsertItem(int index, T item)
        {
            if (mSyncContext == null)
            {
                base.InsertItem(index, item);
            }
            else
            {
                DataTransaction tr =  new DataTransaction(DataOperationKind.InsertItem, item, index);
                if (mIsAsyncModel)
                {
                    this.mSyncContext.Post(this.mUpdateCallback, tr);
                }
                else
                {
                    SyncSend(tr);
                }
            }
        }

        /// <summary>
        /// Removes the item at the specified index.
        /// </summary>
        /// <param name="index">The zero-based index of the item to remove.</param>
        /// <exception cref="T:System.NotSupportedException">
        /// You are removing a newly added item and <see cref="P:System.ComponentModel.IBindingList.AllowRemove"/> is set to false.
        /// </exception>
        protected override void RemoveItem(int index)
        {
            if (mSyncContext == null)
            {
                base.RemoveItem(index);
            }
            else
            {
                if (mIsAsyncModel)
                {
                    this.mSyncContext.Post(this.mUpdateCallback, new DataTransaction(DataOperationKind.RemoveItem, null, index));
                }
                else
                {
                    SyncSend(new DataTransaction(DataOperationKind.RemoveItem, null, index));
                }
            }
        }

        /// <summary>
        /// Replaces the item at the specified index with the specified item.
        /// </summary>
        /// <param name="index">The zero-based index of the item to replace.</param>
        /// <param name="item">The new value for the item at the specified index. The value can be null for reference types.</param>
        /// <exception cref="T:System.ArgumentOutOfRangeException">
        /// 	<paramref name="index"/> is less than zero.
        /// -or-
        /// <paramref name="index"/> is greater than <see cref="P:System.Collections.ObjectModel.Collection`1.Count"/>.
        /// </exception>
        protected override void SetItem(int index, T item)
        {
            if (mSyncContext == null)
            {
                base.SetItem(index, item);
            }
            else
            {
                if (mIsAsyncModel)
                {
                    this.mSyncContext.Post(this.mUpdateCallback, new DataTransaction(DataOperationKind.SetItem, item, index));
                }
                else
                {
                    SyncSend(new DataTransaction(DataOperationKind.SetItem, item, index));
                }
            }
        }

        /// <summary>
        /// Discards a pending new item.
        /// </summary>
        /// <param name="itemIndex">The index of the of the new item to be added</param>
        public override void CancelNew(int itemIndex)
        {
            if (mSyncContext == null)
            {
                base.CancelNew(itemIndex);
            }
            else
            {
                if (mIsAsyncModel)
                {
                    this.mSyncContext.Post(this.mUpdateCallback, new DataTransaction(DataOperationKind.CancelNew, null, itemIndex));
                }
                else
                {
                    SyncSend(new DataTransaction(DataOperationKind.CancelNew, null, itemIndex));
                }
            }
        }

        /// <summary>
        /// Commits a pending new item to the collection.
        /// </summary>
        /// <param name="itemIndex">The index of the new item to be added.</param>
        public override void EndNew(int itemIndex)
        {
            if (mSyncContext == null)
            {
                base.EndNew(itemIndex);
            }
            else
            {
                if (mIsAsyncModel)
                {
                    this.mSyncContext.Post(this.mUpdateCallback, new DataTransaction(DataOperationKind.EndNew, null, itemIndex));
                }
                else
                {
                    SyncSend(new DataTransaction(DataOperationKind.EndNew, null, itemIndex));
                }
            }
        }

        /// <summary>
        /// Sorts the items if overridden in a derived class; otherwise, throws a <see cref="T:System.NotSupportedException"/>.
        /// </summary>
        /// <param name="prop">A <see cref="T:System.ComponentModel.PropertyDescriptor"/> that specifies the property to sort on.</param>
        /// <param name="direction">One of the <see cref="T:System.ComponentModel.ListSortDirection"/>  values.</param>
        /// <exception cref="T:System.NotSupportedException">
        /// Method is not overridden in a derived class.
        /// </exception>
        protected override void ApplySortCore(PropertyDescriptor prop, ListSortDirection direction)
        {
            // not implemented.
        }

        /// <summary>
        /// Removes any sort applied with <see cref="M:System.ComponentModel.BindingList`1.ApplySortCore(System.ComponentModel.PropertyDescriptor,System.ComponentModel.ListSortDirection)"/> if sorting is implemented in a derived class; otherwise, raises <see cref="T:System.NotSupportedException"/>.
        /// </summary>
        /// <exception cref="T:System.NotSupportedException">
        /// Method is not overridden in a derived class.
        /// </exception>
        protected override void RemoveSortCore()
        {
            // not implemented.
        }

        /// <summary>
        /// Searches for the index of the item that has the specified property descriptor with the specified value, if searching is implemented in a derived class; otherwise, a <see cref="T:System.NotSupportedException"/>.
        /// </summary>
        /// <param name="prop">The <see cref="T:System.ComponentModel.PropertyDescriptor"/> to search for.</param>
        /// <param name="key">The value of <paramref name="property"/> to match.</param>
        /// <returns>
        /// The zero-based index of the item that matches the property descriptor and contains the specified value.
        /// </returns>
        /// <exception cref="T:System.NotSupportedException">
        /// 	<see cref="M:System.ComponentModel.BindingList`1.FindCore(System.ComponentModel.PropertyDescriptor,System.Object)"/> is not overridden in a derived class.
        /// </exception>
        protected override int FindCore(PropertyDescriptor prop, object key)
        {
            // not implemented.
            return -1;
        }
        #endregion

        #region Process Posted messages. Executed on the UI thread.
        private void UpdateBindingList(object arg)
        {
            DataTransaction transaction = arg as DataTransaction;
            if (transaction == null)
            {
                return;
            }

            ExecuteTransaction(transaction);
        }

        private void ExecuteTransaction(DataTransaction transaction)
        {
            switch (transaction.DataOperationKind)
            {
                case DataOperationKind.Clear:
                    base.ClearItems();
                    break;
                case DataOperationKind.InsertItem:
                    base.InsertItem(transaction.Index, (T)transaction.DataObject);
                    break;
                case DataOperationKind.RemoveItem:
                    base.RemoveItem(transaction.Index);
                    break;
                case DataOperationKind.SetItem:
                    base.SetItem(transaction.Index, (T)transaction.DataObject);
                    break;
                case DataOperationKind.AddNewItem:
                    object res = base.AddNewCore();
                    transaction.TransactionResult = res;
                    break;
                case DataOperationKind.Add:
                    base.Add((T)transaction.DataObject);
                    break;
                case DataOperationKind.EndNew:
                    base.EndNew(transaction.Index);
                    break;
                case DataOperationKind.CancelNew:
                    base.CancelNew(transaction.Index);
                    break;
                case DataOperationKind.GetCount:
                    transaction.TransactionResult = base.Count;
                    break;
            }
        }
        #endregion

        #region Enums and classes for Posting.
        private enum DataOperationKind
        {
            Clear,
            InsertItem,
            RemoveItem,
            SetItem,
            AddNewItem,
            Add,
            EndNew,
            CancelNew,
            GetCount,
            ApplySort,          // use this once Sorting is implemented
            RemoveSort,         // use this once Sorting is implemented
            Find                // use this once Searching is implemented
        }

        private class DataTransaction
        {
            DataOperationKind kind;
            object obj;
            object result;
            int index;

            public DataTransaction(DataOperationKind kind, object obj, int index)
            {
                this.kind = kind;
                this.obj = obj;
                this.index = index;
            }

            public DataOperationKind DataOperationKind
            {
                get
                {
                    return this.kind;
                }
            }

            public Object DataObject
            {
                get
                {
                    return this.obj;
                }
            }

            public int Index
            {
                get
                {
                    return this.index;
                }
            }

            public object TransactionResult
            {
                get
                {
                    return this.result;
                }
                set
                {
                    this.result = value;
                }
            }
        }
        #endregion
    }
}

