using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.Enums;
using Zensys.ZWave.Events;

namespace Zensys.ZWave.Devices
{
    /// <summary>
    /// This interface provides properties and methods that allow the client program using Controller specific Serial API.
    /// </summary>
    public interface IController : IDevice
    {
        /// <summary>
        /// Occurs when Unsolitited frame received.
        /// </summary>
        event UnsolititedFrameReceivedEventHandler UnsolititedFrameReceived;
        /// <summary>
        /// Represents the method that will handle the controller updated event.
        /// </summary>
        event ControllerUpdatedEventHandler ControllerUpdated;
        /// <summary>
        /// Represents the method that will handle the node status changed event.
        /// </summary>
        event NodeStatusChangedEventHandler NodeStatusChanged;
        /// <summary>
        /// Represents the method that will handle the failed node status changed event.
        /// </summary>
        event FailedNodeStatusChangedEventHandler FailedNodeStatusChanged;
        /// <summary>
        /// Represents the method that will handle the learn mode status changed event.
        /// </summary>
        event LearnModeStatusChangedEventHandler LearnModeStatusChanged;
        /// <summary>
        /// Gets or sets the SUC node id.
        /// </summary>
        /// <value>The suc node id.</value>
        byte SucNodeId { get; set; }
        /// <summary>
        /// Gets or sets the collection of included nodes.
        /// </summary>
        /// <value>The included nodes.</value>
        List<IDeviceInfo> IncludedNodes { get; set; }
        ///// <summary>
        ///// Gets or sets the associations.
        ///// </summary>
        ///// <value>The associations.</value>
        //IAssociation Associations { get; set; }
        /// <summary>
        /// Gets or sets the <see cref="ICommandClassesStore"></see>.
        /// </summary>
        /// <value>The command classes store.</value>
        CommandClassesStore CommandClassesStore { get; set; }
        /// <summary>
        /// Gets the network role.
        /// </summary>
        /// <value>The network role.</value>
        ControllerRoles NetworkRole { get; }
        /// <summary>
        /// Adds the node to network.
        /// </summary>
        /// <returns></returns>
        IDeviceInfo AddNodeToNetwork();
        /// <summary>
        /// 
        /// </summary>
        void NodeAddedProtocolDone();
        /// <summary>
        /// Adds the node to network.
        /// </summary>
        /// <param name="mode">The mode.</param>
        /// <returns></returns>
        IDeviceInfo AddNodeToNetwork(Mode mode);
        /// <summary>
        /// Adds the node to network (wide inclusion).
        /// </summary>
        /// <returns></returns>
        IDeviceInfo AddNodeToNetworkWide();
        /// <summary>
        /// Removes the node from network.
        /// </summary>
        /// <returns></returns>
        IDeviceInfo RemoveNodeFromNetwork();
        /// <summary>
        /// Requests the node info.
        /// </summary>
        /// <param name="nodeId">The node id.</param>
        /// <returns></returns>
        IDeviceInfo RequestNodeInfo(byte nodeId);
        /// <summary>
        /// Changes the controller.
        /// </summary>
        /// <param name="controllerChangeMode">The controller change mode.</param>
        void ControllerChange(ControllerChangeMode controllerChangeMode);
        /// <summary>
        /// Stops the request.
        /// </summary>
        /// <param name="commandType">Type of the command.</param>
        void StopRequest(byte commandType);
        /// <summary>
        /// Gets the controller capabilities.
        /// </summary>
        void GetControllerCapabilities();
        /// <summary>
        /// Sends the Replication command.
        /// </summary>
        /// <param name="destNodeID">The destination node Id.</param>
        /// <param name="pData">The data.</param>
        /// <param name="txOptions">The Tx options.</param>
        /// <returns></returns>
        TransmitStatuses ReplicationSend(byte destNodeID, byte[] pData, TransmitOptions txOptions);
        /// <summary>
        /// Creates the new primary Controller.
        /// </summary>
        /// <param name="mode">The mode.</param>
        void CreateNewPrimaryCtrl(CreateNewPrimaryControllerMode mode);
        /// <summary>
        /// Sends the replication receive complete command.
        /// </summary>
        void ReplicationReceiveComplete();
        /// <summary>
        /// Assigns the return route.
        /// </summary>
        /// <param name="srcNodeId">The SRC node id.</param>
        /// <param name="destNodeId">The dest node id.</param>
        TransmitStatuses AssignReturnRoute(byte srcNodeId, byte destNodeId);
        /// <summary>
        /// Deletes the return route.
        /// </summary>
        /// <param name="srcNodeId">The SRC node id.</param>
        TransmitStatuses DeleteReturnRoute(byte srcNodeId);
        /// <summary>
        /// Removes the failed node Id.
        /// </summary>
        /// <param name="nodeId">The node id.</param>
        /// <returns></returns>
        FailedNodeStatus RemoveFailedNodeID(byte nodeId);
        /// <summary>
        /// Determines whether the specified node is failed.
        /// </summary>
        /// <param name="nodeId">The node id.</param>
        /// <returns>
        /// 	<c>true</c> if the specified node is failed; otherwise, <c>false</c>.
        /// </returns>
        bool IsFailedNode(byte nodeId);
        /// <summary>
        /// Determines whether Controller is primary.
        /// </summary>
        void IsPrimaryController();
        /// <summary>
        /// Replaces the failed node.
        /// </summary>
        /// <param name="nodeId">The node id.</param>
        /// <returns></returns>
        IDeviceInfo ReplaceFailedNode(byte nodeId);
        /// <summary>
        /// Gets the neighbor count.
        /// </summary>
        void GetNeighborCount();
        /// <summary>
        /// Determines whether the nodes are neighbours.
        /// </summary>
        void AreNodesNeighbours();
        /// <summary>
        /// Requests the node neighbor update.
        /// </summary>
        /// <param name="nodeId">The node id.</param>
        /// <returns></returns>
        RequestNeighborUpdateStatuses RequestNodeNeighborUpdate(byte nodeId);
        /// <summary>
        /// Requests the node neighbor update MR.
        /// </summary>
        /// <param name="nodeId">The node id.</param>
        /// <param name="nodeMask">The node mask.</param>
        /// <returns></returns>
        byte[] RequestNodeNeighborUpdateMR(byte nodeId, byte[] nodeMask);
        /// <summary>
        /// Gets the routing info.
        /// </summary>
        /// <param name="nodeId">The node id.</param>
        /// <param name="removeBad">The remove bad.</param>
        /// <param name="removeNonReps">The remove NonReps.</param>
        /// <returns></returns>
        byte[] GetRoutingInfo(byte nodeId, byte removeBad, byte removeNonReps);
        /// <summary>
        /// Gets the SUC node Id.
        /// </summary>
        void GetSUCNodeID();
        /// <summary>
        /// Sets the SUC node ID.
        /// </summary>
        /// <param name="nodeId">The node id.</param>
        /// <param name="sucState">if set to <c>true</c> SUC state.</param>
        /// <param name="txOptions">The Tx options.</param>
        /// <param name="capabilities">The capabilities.</param>
        /// <returns></returns>
        SetSucReturnValue SetSUCNodeID(byte nodeId, bool sucState, TransmitOptions txOptions, byte capabilities);
        /// <summary>
        /// Sends the SUC node Id.
        /// </summary>
        /// <param name="nodeId">The node id.</param>
        /// <param name="txOptions">The tx options.</param>
        /// <returns></returns>
        TransmitStatuses SendSUCID(byte nodeId, TransmitOptions txOptions);
        /// <summary>
        /// Assigns the SUC return route.
        /// </summary>
        /// <param name="srcNodeId">The source node id.</param>
        /// <returns></returns>
        TransmitStatuses AssignSUCReturnRoute(byte srcNodeId);
        /// <summary>
        /// Deletes the SUC return route.
        /// </summary>
        /// <param name="nodeId">The node id.</param>
        /// <returns></returns>
        TransmitStatuses DeleteSUCReturnRoute(byte nodeId);
        /// <summary>
        /// Gets the protocol info.
        /// </summary>
        /// <param name="nodeId">The node id.</param>
        /// <param name="checkIfVirtual">if set to <c>true</c> check if specified Node is virtual.</param>
        /// <returns></returns>
        IDeviceInfo GetProtocolInfo(byte nodeId, bool checkIfVirtual);
        /// <summary>
        /// Determines whether the specified node is virtual.
        /// </summary>
        /// <param name="nodeId">The node id.</param>
        /// <returns>
        /// 	<c>true</c> if the specified node is virtual; otherwise, <c>false</c>.
        /// </returns>
        bool IsVirtualNode(byte nodeId);
        /// <summary>
        /// Gets the nodes collection.
        /// </summary>
        /// <returns></returns>
        List<IDeviceInfo> GetNodes();
        /// <summary>
        /// Sets self as SUC.
        /// </summary>
        /// <param name="sucState">if set to <c>true</c> SUC state.</param>
        /// <param name="capabilities">The capabilities.</param>
        /// <returns></returns>
        bool SetSelfAsSUC(bool sucState, byte capabilities);
        /// <summary>
        /// Enables the SUC functionality.
        /// </summary>
        /// <param name="enable">if set to <c>true</c> enable.</param>
        /// <param name="capabilities">The capabilities.</param>
        /// <returns></returns>
        bool EnableSUC(bool enable, byte capabilities);
        /// <summary>
        /// Verifies the node info.
        /// </summary>
        /// <returns></returns>
        //bool VerifyNodeInfo();
        /// <summary>
        /// Sets the routing info.
        /// </summary>
        /// <param name="nodeId">The node id.</param>
        /// <param name="nodeMask">The node mask.</param>
        /// <returns></returns>
        bool SetRoutingInfo(byte nodeId, byte[] nodeMask);
    }
}
