using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave;
using Zensys.ZWave.Events;
using Zensys.ZWave.Enums;
using System.Threading;
using Zensys.ZWave.Exceptions;
using Zensys.ZWave.Logging;
using Zensys.Framework;
using System.ComponentModel;

namespace Zensys.ZWave.SerialPortSession
{
    public class SessionLayer : ISessionLayer
    {
        private const byte MIN_SEQUENCE_NUMBER = 1;
        private const byte MAX_SEQUENCE_NUMBER = 127;

        private Thread invoker;

        private BindingList<LogPacket> mLogDataSource;
        public BindingList<LogPacket> LogDataSource
        {
            get { return mLogDataSource; }
        }

        IFrameLayer mFrameLayer = null;

        private byte mSequenceNumber = MIN_SEQUENCE_NUMBER;
        public byte SequenceNumber
        {
            get
            {
                if (IsAutoIncrementSequenceNumber)
                {
                    mSequenceNumber++;
                }
                if (mSequenceNumber == MAX_SEQUENCE_NUMBER)
                {
                    mSequenceNumber = MIN_SEQUENCE_NUMBER;
                }
                return mSequenceNumber;
            }
        }
        public byte CurrentSequenceNumber
        {
            get
            {
                return mSequenceNumber;
            }
        }

        #region ISessionLayerEx Members

        private bool mIsReady = false;
        /// <summary>
        /// Gets a value indicating whether this instance is ready.
        /// </summary>
        /// <value><c>true</c> if this instance is ready; otherwise, <c>false</c>.</value>
        public bool IsReady
        {
            get
            {
                return mIsReady;
            }
            set
            {
                mIsReady = value;
            }
        }

        private bool mIsAutoIncrementSequenceNumber = true;
        /// <summary>
        /// Gets or sets a value indicating whether this instance is auto increment sequence number.
        /// </summary>
        /// <value>
        /// 	<c>true</c> if this instance is auto increment sequence number; otherwise, <c>false</c>.
        /// </value>
        public bool IsAutoIncrementSequenceNumber
        {
            get { return mIsAutoIncrementSequenceNumber; }
            set
            {
                if (mIsAutoIncrementSequenceNumber && mIsAutoIncrementSequenceNumber != value)
                {
                    mSequenceNumber++;
                }
                mIsAutoIncrementSequenceNumber = value;
            }
        }

        private TimeSpan mRequestTimeout = new TimeSpan(0, 0, 3);
        /// <summary>
        /// Gets or sets the request timeout.
        /// </summary>
        /// <value>The request timeout.</value>
        public TimeSpan RequestTimeout
        {
            get
            {
                return mRequestTimeout;
            }
            set
            {
                mRequestTimeout = value;
            }
        }


        /// <summary>
        /// Closes this instance.
        /// </summary>
        public void Close()
        {
            mFrameLayer.FrameReceived -= new FrameReceivedEventHandler(mFrameLayer_FrameReceived);
            mFrameLayer.AcknowledgeReceived -= new FrameReceivedEventHandler(mFrameLayer_AcknowledgeReceived);
            mFrameLayer.Close();
        }

        private Action<ResponseReceivedEventArgs> responseReceived;
        public Action<ResponseReceivedEventArgs> ResponseReceived
        {
            get { return responseReceived; }
            set { responseReceived = value; }
        }

        private Action<object> acknowledgeReceived;
        public Action<object> AcknowledgeReceived
        {
            get { return acknowledgeReceived; }
            set { acknowledgeReceived = value; }
        }


        /// <summary>
        /// Opens the specified frame layer.
        /// </summary>
        /// <param name="frameLayer">The frame layer.</param>
        public void Init(IFrameLayer frameLayer, BindingList<LogPacket> logDataSource)
        {
            mLogDataSource = logDataSource;
            mFrameLayer = frameLayer;
        }
        string pName = "";
        /// <summary>
        /// Opens the specified port name.
        /// </summary>
        /// <param name="portName">Name of the port.</param>
        public void Open(string portName)
        {
            pName = portName;
            mFrameLayer.Open(portName);
            mFrameLayer.FrameReceived += new FrameReceivedEventHandler(mFrameLayer_FrameReceived);
            mFrameLayer.AcknowledgeReceived += new FrameReceivedEventHandler(mFrameLayer_AcknowledgeReceived);
            mFrameLayer.ExceptionReceived += new ExceptionReceivedEventHandler(mFrameLayer_ExceptionReceived);
        }

        void mFrameLayer_ExceptionReceived(ExceptionReceivedEventArgs args)
        {
            innerException = args.InnerException;
            //asyncExceptionReceived.Set();
        }

        void mFrameLayer_AcknowledgeReceived(FrameReceivedEventArgs args)
        {
            response = args.Data;
            asyncAcknowledgeReceived.Set();
            if (acknowledgeReceived != null)
            {
                acknowledgeReceived(null);
            }
            Tools._writeDebugDiagnosticMessage("Ack got!", true, true);
        }

        void mFrameLayer_FrameReceived(FrameReceivedEventArgs args)
        {
            if (waitResponses)
            {
                byte[] responseData = null;
                if (responseCommand != 0)
                {
                    if (args.CommandType == responseCommand)
                    {
                        responseData = args.Data;
                        responses.Add(responseData);
                        if ((args.CommandType == (byte)CommandTypes.CmdZWaveSendData ||
                           args.CommandType == (byte)CommandTypes.CmdZWaveSendDataMeta ||
                           args.CommandType == (byte)CommandTypes.CmdZWaveSendDataMetaMR ||
                           args.CommandType == (byte)CommandTypes.CmdZWaveSendDataMR ||
                           args.CommandType == (byte)CommandTypes.CmdZWaveSendDataMulti) &&
                           responses.Count == 1)
                        {
                            if (responses[0].Length > 0 && responses[0][0] == 0)
                            {
                                asyncFrameReceived.Set();//abort SendData if response is FALSE
                            }
                        }
                    }
                    else
                    {
                        Tools._writeDebugDiagnosticMessage("responses:" + responses.Count + " waitResponsesCount: " + waitResponsesCount.ToString(), true, true);
                        Tools._writeDebugDiagnosticMessage("mFrameLayer_FrameReceived unsolicited frame received - " + pName + "responseCommand: " + responseCommand.ToString(), true, true);
                        if (args.CommandType == (byte)CommandTypes.CmdApplicationCommandHandler ||
                            args.CommandType == (byte)CommandTypes.CmdApplicationCommandHandler_Bridge ||
                            args.CommandType == (byte)CommandTypes.CmdApplicationSlaveCommandHandler)
                        {
                            asyncFrameReceived.Set();
                            RaiseResponseReceived(args);
                        }
                    }
                }
                else
                {
                    responseData = args.Data;
                    responses.Add(responseData);
                }

                if (responses.Count == waitResponsesCount)
                {
                    Tools._writeDebugDiagnosticMessage("asyncFrameReceived.Set responses:" + responses.Count + " waitResponsesCount: " + waitResponsesCount.ToString(), true, true);
                    asyncFrameReceived.Set();
                }
            }
            else
            {
                if (responseCommand != 0)
                {
                    if (args.CommandType == responseCommand && waitingForResponse == true)
                    {
                        waitingForResponse = false;
                        response = args.Data;
                        responseCommand = 0;
                        asyncFrameReceived.Set();

                    }
                    else
                    {
                        Tools._writeDebugDiagnosticMessage("unsolicited frame received - " + pName, true, true);
                    }
                }
                else
                {
                    response = args.Data;
                    asyncFrameReceived.Set();
                }
                RaiseResponseReceived(args);
            }
        }

        private void RaiseResponseReceived(FrameReceivedEventArgs args)
        {
            //ThreadPool.QueueUserWorkItem(new WaitCallback(RaiseResponseReceivedInternal), args);
            RaiseResponseReceivedInternal(args);
        }

        private void RaiseResponseReceivedInternal(object args)
        {
            if (ResponseReceived != null && args is FrameReceivedEventArgs)
            {
                System.Diagnostics.Debug.WriteLine("Session Layer. Event RaiseResponseReceivedInternal Raised - " + pName);
                FrameReceivedEventArgs _args = args as FrameReceivedEventArgs;
                try
                {
                    ResponseReceived(new ResponseReceivedEventArgs(_args.Data, _args.CommandType, _args.FrameBuffer));
                }
                catch (RequestTimeoutException rex)
                {
                    System.Diagnostics.Debug.WriteLine("RequestTimeoutException");
                }
            }
        }

        private byte[] response;
        private bool waitingForResponse = false;
        private byte responseCommand = 0;
        private Exception innerException = null;
        ManualResetEvent asyncFrameReceived = new ManualResetEvent(false);
        ManualResetEvent asyncAcknowledgeReceived = new ManualResetEvent(false);
        private void ResetRequest()
        {
            asyncFrameReceived.Reset();
            innerException = null;
        }

        private void EndRequest(int timeout)
        {
            Exception ex = null;
            if (!asyncFrameReceived.WaitOne(timeout, true))
            {
                waitResponses = false;
                responses.Clear();
                ex = new RequestTimeoutException();
            }

            if (innerException != null)
            {
                ex = innerException;
            }

            if (ex != null)
            {
                throw ex;
            }
        }
        private void EndRequest()
        {
            EndRequest((int)mRequestTimeout.TotalMilliseconds);
        }

        /// <summary>
        /// Executes the request.
        /// </summary>
        /// <param name="request">The request.</param>
        /// <returns></returns>
        public byte[] ExecuteRequest(byte[] request)
        {
            waitingForResponse = true;
            responseCommand = request[0];
            ResetRequest();
            mFrameLayer.Write(request);
            EndRequest();
            return response;
        }

        /// <summary>
        /// Sends the request.
        /// </summary>
        /// <param name="commandType">Type of the command.</param>
        /// <param name="parameters">The parameters.</param>
        /// <returns></returns>
        public byte[] ExecuteRequest(byte commandType, params byte[] parameters)
        {
            waitingForResponse = true;
            responseCommand = commandType;
            IDataFrame frame = mFrameLayer.CreateDataFrame(FrameTypes.Request, commandType, parameters);
            ResetRequest();
            mFrameLayer.Write(frame);
            EndRequest();
            return response;
        }

        private List<byte[]> responses = new List<byte[]>();
        private bool waitResponses = false;
        private int waitResponsesCount;
        /// <summary>
        /// Executes the request.
        /// </summary>
        /// <param name="commandType">Type of the command.</param>
        /// <param name="responseCount">The responses count.</param>
        /// <param name="parameters">The parameters.</param>
        /// <returns></returns>
        public List<byte[]> ExecuteRequest(byte commandType, int responseCount, int timeout, params byte[] parameters)
        {
            responses.Clear();
            waitResponses = true;
            waitResponsesCount = responseCount;
            responseCommand = commandType;
            IDataFrame frame = mFrameLayer.CreateDataFrame(FrameTypes.Request, commandType, parameters);
            ResetRequest();
            mFrameLayer.Write(frame);
            EndRequest(timeout);
            waitResponses = false;
            return responses;
        }
        /// <summary>
        /// Executes the request.
        /// </summary>
        /// <param name="request">The request.</param>
        /// <param name="responseCount">The responses count.</param>
        /// <returns></returns>
        public List<byte[]> ExecuteRequest(byte[] request, int responseCount, int timeout)
        {
            responses.Clear();
            waitResponses = true;
            waitResponsesCount = responseCount;
            responseCommand = request[0];
            ResetRequest();
            mFrameLayer.Write(request);
            EndRequest(timeout);
            waitResponses = false;
            return responses;
        }


        /// <summary>
        /// Executes the non request.
        /// </summary>
        /// <param name="commandType">Type of the command.</param>
        /// <param name="parameters">The parameters.</param>
        public void ExecuteNonRequest(byte commandType, params byte[] parameters)
        {
            responseCommand = 0;
            IDataFrame frame = mFrameLayer.CreateDataFrame(FrameTypes.Request, commandType, parameters);
            mFrameLayer.Write(frame);
            if (!asyncAcknowledgeReceived.WaitOne(2000))
            {
                Tools._writeDebugDiagnosticMessage("Ack not signaled", true, true);
            }
            asyncAcknowledgeReceived.Reset();
        }
        /// <summary>
        /// Sends the request.
        /// </summary>
        /// <param name="request">The request.</param>
        public void ExecuteNonRequest(byte[] request)
        {
            responseCommand = 0;
            mFrameLayer.Write(request);
            if (!asyncAcknowledgeReceived.WaitOne(2000))
            {
                Tools._writeDebugDiagnosticMessage("Ack not signaled", true, true);
            }
            asyncAcknowledgeReceived.Reset();
        }

        #endregion
    }
}
