using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.Devices;
using Zensys.ZWave.Logging;
using Zensys.ZWave.Enums;
using Zensys.Framework;

namespace Zensys.ZWave.SerialPortApplication.Devices
{
    public class BridgeController : Controller, IBridgeController
    {
        internal BridgeController(ISessionLayer sessionLayer, bool useExternalCommandClassesStorage)
            : base(sessionLayer, useExternalCommandClassesStorage)
        {
        }
        #region IBridgeController Members

        /// <summary>
        /// Sends the slave data.
        /// </summary>
        /// <param name="sourceId">The source id.</param>
        /// <param name="destinationId">The destination id.</param>
        /// <param name="data">The data.</param>
        /// <param name="txOptions">The tx options.</param>
        /// <returns></returns>
        public TransmitStatuses SendSlaveData(byte sourceId, byte destinationId, byte[] data, TransmitOptions txOptions)
        {
            LogTopSession();
            IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveSendSlaveData, true);

            byte[] request = new byte[data.Length + 6];
            request[0] = sourceId;
            request[1] = destinationId;
            request[3] = (byte)data.Length;
            for (int i = 0; i < data.Length; i++)
            {
                request[i + 4] = data[i];
            }
            request[request.Length - 2] = (byte)txOptions;
            request[request.Length - 1] = SessionLayer.SequenceNumber;
            List<byte[]> responses = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveSendSlaveData, 2, (int)SessionLayer.RequestTimeout.TotalMilliseconds, request);

            if (responses.Count == 2 && responses[1].Length > 1)
            {
                return (TransmitStatuses)responses[1][1];
            }
            else
            {
                return TransmitStatuses.ResMissing;
            }
        }

        /// <summary>
        /// Sends the slave node information.
        /// </summary>
        /// <param name="sourceId">The source id.</param>
        /// <param name="destinationId">The destination id.</param>
        /// <param name="txOptions">The tx options.</param>
        /// <returns></returns>
        public TransmitStatuses SendSlaveNodeInformation(byte sourceId, byte destinationId, TransmitOptions txOptions)
        {
            LogTopSession();
            IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveSendSlaveNodeInfo, true);
            byte[] request = new byte[4];
            request[0] = sourceId;
            request[1] = destinationId;
            request[request.Length - 2] = (byte)txOptions;
            request[request.Length - 1] = SessionLayer.SequenceNumber;
            List<byte[]> responses = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveSendSlaveNodeInfo, 2, (int)SessionLayer.RequestTimeout.TotalMilliseconds, request);

            if (responses.Count == 2 && responses[1].Length > 1)
            {
                return (TransmitStatuses)responses[1][1];
            }
            else
            {
                return TransmitStatuses.ResMissing;
            }
        }

        /// <summary>
        /// Sends the slave node information.
        /// </summary>
        /// <param name="nodeId">The node id.</param>
        /// <param name="listening">if set to <c>true</c> [listening].</param>
        /// <param name="generic">The generic.</param>
        /// <param name="specific">The specific.</param>
        /// <param name="nodeParameter">The node parameter.</param>
        public void SetSlaveNodeInformation(byte nodeId, bool listening, byte generic, byte specific, byte[] nodeParameter)
        {
            LogTopSession();
            if (nodeParameter == null)
            {
                throw new ArgumentNullException("nodeParm");
            }
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdSerialApiSlaveNodeInfo, true))
            {
                byte[] request = new byte[nodeParameter.Length + 6];
                request[0] = nodeId;
                request[1] = listening ? (byte)0x01 : (byte)0x00;
                request[2] = generic;
                request[3] = specific;
                request[4] = (byte)nodeParameter.Length;
                for (int i = 0; i < nodeParameter.Length; i++)
                {
                    request[i + 5] = nodeParameter[i];
                }
                request[request.Length - 1] = SessionLayer.SequenceNumber;
                SessionLayer.ExecuteNonRequest((byte)CommandTypes.CmdSerialApiSlaveNodeInfo, request);
            }
        }

        /// <summary>
        /// Sets the slave learn mode.
        /// </summary>
        /// <param name="nodeId">The node id.</param>
        /// <param name="learnMode">if set to <c>true</c> [learn mode].</param>
        /// <returns></returns>
        public byte[] SetSlaveLearnMode(byte nodeId, SlaveLearnMode learnMode)
        {
            LogTopSession();
            waitForEvent = true;
            slaveLearnModeResult = null;
            ResetSignalResponseReceived();
            switch (learnMode)
            {
                case SlaveLearnMode.VirtualSlaveLearnModeAdd:
                case SlaveLearnMode.VirtualSlaveLearnModeRemove:
                    if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveSetSlaveLearnMode, true))
                    {
                        SessionLayer.ExecuteNonRequest((byte)CommandTypes.CmdZWaveSetSlaveLearnMode, new byte[] { nodeId, Convert.ToByte(learnMode), SessionLayer.SequenceNumber });
                    }
                    WaitOneSignalResponseReceived(1000, true);
                    SessionLayer.IsAutoIncrementSequenceNumber = true;
                    break;
                case SlaveLearnMode.VirtualSlaveLearnModeDisable:
                case SlaveLearnMode.VirtualSlaveLearnModeEnable:
                    if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveSetSlaveLearnMode, true))
                    {
                        slaveLearnModeResult = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveSetSlaveLearnMode, new byte[] { nodeId, Convert.ToByte(learnMode), SessionLayer.SequenceNumber });
                    }
                    WaitOneSignalResponseReceived(2000, true);
                    break;
                default:
                    break;
            }
            waitForEvent = false;
            return slaveLearnModeResult;
        }


        /// <summary>
        /// Sets the slave learn mode enable and transmit node info.
        /// </summary>
        /// <param name="nodeId">The node id.</param>
        /// <returns>new node id</returns>
        public byte SetSlaveLearnModeEnableAndTx(byte nodeId)
        {
            byte newNodeId = 0xFF;
            LogTopSession();
            ResetSignalResponseReceived();
            IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveSetSlaveLearnMode, true);
            IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveSendSlaveNodeInfo, true);
            byte funcId = SessionLayer.SequenceNumber;
            byte[] response = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveSetSlaveLearnMode,
                new byte[] { nodeId, (byte)SlaveLearnMode.VirtualSlaveLearnModeEnable, funcId });

            byte[] request = new byte[4];
            request[0] = nodeId;
            request[1] = 0xFF;
            request[request.Length - 2] = (byte)TransmitOptions.TransmitOptionNone;
            request[request.Length - 1] = SessionLayer.SequenceNumber;
            response = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveSendSlaveNodeInfo, request);

            bool ret = WaitOneSignalResponseReceived(5000, true);
            if (ret && slaveLearnModeResult != null && slaveLearnModeResult.Length > 3)
            {
                newNodeId = slaveLearnModeResult[3];
            }
            if (newNodeId != 0)
            {
                response = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveSetSlaveLearnMode,
                   new byte[] { nodeId, (byte)SlaveLearnMode.VirtualSlaveLearnModeDisable, funcId });
            }
            return newNodeId;
        }
        /// <summary>
        /// Gets the virtual nodes.
        /// </summary>
        /// <returns></returns>
        public NodeMask GetVirtualNodes()
        {
            LogTopSession();
            NodeMask nodeMask = null;
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveGetVirtualNodes, true))
            {
                byte[] res = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveGetVirtualNodes);
                nodeMask = new NodeMask(res);
            }
            return nodeMask;
        }

        #endregion
        protected override void OnSessionLayerResponseReceivedCustomActions(Zensys.ZWave.Events.ResponseReceivedEventArgs args)
        {
            base.OnSessionLayerResponseReceivedCustomActions(args);
            switch (args.CommandType)
            {
                #region CmdZWaveSetSlaveLearnMode
                case ((byte)CommandTypes.CmdZWaveSetSlaveLearnMode):
                    {
                        if (args.Data.Length == 4)
                        {
                            slaveLearnModeResult = new byte[4];
                            Array.Copy(args.Data, slaveLearnModeResult, 4);
                            if (slaveLearnModeResult[1] == 0) //ASSIGN_COMPLETE
                            {
                                SetSignalResponseReceived();
                            }
                            else if (slaveLearnModeResult[3] == 0)//newId = 0 delete node operation
                            {
                                SetSignalResponseReceived();
                            }
                        }
                    }
                    break;
                #endregion
            }
        }

        public override TransmitStatuses SendData(byte nodeId, byte[] data, TransmitOptions txOptions)
        {
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveSendData_Bridge, false))
            {
                List<byte> request = new List<byte>();
                request.Add(Id);
                request.Add(nodeId);
                request.Add((byte)data.Length);
                request.AddRange(data);
                request.Add((byte)txOptions);
                request.AddRange(new byte[] { 0, 0, 0, 0 });
                request.Add(SessionLayer.SequenceNumber);
                List<byte[]> responses = SessionLayer.ExecuteRequest(
                    (byte)CommandTypes.CmdZWaveSendData_Bridge,
                    2,
                    SEND_DATA_TIMEOUT,
                    request.ToArray());
                if (responses.Count == 2 && responses[1].Length > 1)
                {
                    return (TransmitStatuses)responses[1][1];
                }
                else
                {
                    return TransmitStatuses.ResMissing;
                }
            }
            else
                return base.SendData(nodeId, data, txOptions);
        }

        public override TransmitStatuses SendDataMeta(byte nodeId, byte[] data, TransmitOptions txOptions)
        {
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveSendDataMeta_Bridge, false))
            {
                List<byte> request = new List<byte>();
                request.Add(Id);
                request.Add(nodeId);
                request.Add((byte)data.Length);
                request.AddRange(data);
                request.Add((byte)txOptions);
                request.AddRange(new byte[] { 0, 0, 0, 0 });
                request.Add(SessionLayer.SequenceNumber);
                List<byte[]> responses = SessionLayer.ExecuteRequest(
                    (byte)CommandTypes.CmdZWaveSendDataMeta_Bridge,
                    2,
                    SEND_DATA_TIMEOUT,
                    request.ToArray());
                if (responses.Count == 2 && responses[1].Length > 1)
                {
                    return (TransmitStatuses)responses[1][1];
                }
                else
                {
                    return TransmitStatuses.ResMissing;
                }
            }
            else
                return base.SendDataMeta(nodeId, data, txOptions);
        }

        public override TransmitStatuses SendDataMulti(List<byte> nodeIdList, byte[] data, TransmitOptions txOptions)
        {
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveSendDataMulti_Bridge, false))
            {
                LogTopSession();
                List<byte> request = new List<byte>();
                request.Add(Id);
                request.Add((byte)nodeIdList.Count);
                request.AddRange(nodeIdList);
                request.Add((byte)data.Length);
                request.AddRange(data);
                request.Add((byte)txOptions);
                request.Add(SessionLayer.SequenceNumber);

                List<byte[]> responses = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveSendDataMulti_Bridge, 2, SEND_DATA_TIMEOUT, request.ToArray());
                if (responses.Count == 2 && responses[1].Length > 1)
                {
                    return (TransmitStatuses)responses[1][1];
                }
                else
                {
                    return TransmitStatuses.ResMissing;
                }
            }
            else
                return base.SendDataMulti(nodeIdList, data, txOptions);
        }
    }
}
