using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using Zensys.Framework;

namespace Zensys.ZWave.Programmer.Controls
{
    /// <summary>
    /// HomeIdTextBox class. Implements HomeIdTextBox control.
    /// </summary>
    public class HomeIdTextBox : TextBox
    {
        /// <summary>
        /// MAX_TEXT_LENGTH constant.
        /// </summary>
        public const int MAX_TEXT_LENGTH = 8;
        /// <summary>
        /// MAX_ARRAY_LENGTH constant.
        /// </summary>
        public const int MAX_ARRAY_LENGTH = 4;
        /// <summary>
        /// Gets or sets the maximum number of characters the user can type or paste into the text box control.
        /// </summary>
        /// <value></value>
        /// <returns>The number of characters that can be entered into the control. The default is 32767.</returns>
        /// <exception cref="T:System.ArgumentOutOfRangeException">The value assigned to the property is less than 0. </exception>
        /// <PermissionSet>
        /// 	<IPermission class="System.Security.Permissions.EnvironmentPermission, mscorlib, Version=2.0.3600.0, Culture=neutral, PublicKeyToken=b77a5c561934e089" version="1" Unrestricted="true"/>
        /// 	<IPermission class="System.Security.Permissions.FileIOPermission, mscorlib, Version=2.0.3600.0, Culture=neutral, PublicKeyToken=b77a5c561934e089" version="1" Unrestricted="true"/>
        /// 	<IPermission class="System.Security.Permissions.SecurityPermission, mscorlib, Version=2.0.3600.0, Culture=neutral, PublicKeyToken=b77a5c561934e089" version="1" Flags="UnmanagedCode, ControlEvidence"/>
        /// 	<IPermission class="System.Diagnostics.PerformanceCounterPermission, System, Version=2.0.3600.0, Culture=neutral, PublicKeyToken=b77a5c561934e089" version="1" Unrestricted="true"/>
        /// </PermissionSet>
        public override int MaxLength
        {
            get
            {
                return MAX_TEXT_LENGTH;
            }
            set
            {
                base.MaxLength = MAX_TEXT_LENGTH;
            }
        }
        /// <summary>
        /// Raises the <see cref="E:System.Windows.Forms.Control.Leave"/> event.
        /// </summary>
        /// <param name="e">An <see cref="T:System.EventArgs"/> that contains the event data.</param>
        protected override void OnLeave(EventArgs e)
        {
            base.OnLeave(e);
            ConvertToHex();
        }
        /// <summary>
        /// Raises the <see cref="E:System.Windows.Forms.Control.LostFocus"/> event.
        /// </summary>
        /// <param name="e">An <see cref="T:System.EventArgs"/> that contains the event data.</param>
        protected override void OnLostFocus(EventArgs e)
        {
            base.OnLostFocus(e);
            ConvertToHex();
        }

        private void ConvertToHex()
        {
            byte[] result = new byte[MAX_ARRAY_LENGTH];
            string value = this.Text;
            if ((value.Length % 8) > 0)
            {
                value = value.PadLeft(8, '0');
            }
            try
            {
                byte[] tmpArr = Tools.FromHexString(value);
                if (tmpArr.Length < MAX_ARRAY_LENGTH)
                {
                    for (int i = 0; i < tmpArr.Length; i++)
                    {
                        result[MAX_ARRAY_LENGTH - (i + 1)] = tmpArr[i];
                    }
                }
                else if (tmpArr.Length >= MAX_ARRAY_LENGTH)
                {
                    for (int i = 0; i < MAX_ARRAY_LENGTH; i++)
                    {
                        result[i] = tmpArr[i];
                    }
                }
            }
            catch
            {
                result = new byte[MAX_ARRAY_LENGTH];
            }
            this.Text = Tools.ToHexString(result);
        }
    }
}
