using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.Programmer.UI;
using Zensys.ZWave.Programmer.Classes;
using Zensys.ZWave.Programmer.Interfaces;
using Zensys.ZWave.Programmer.Dispatchers;
using System.Windows.Forms;
using Zensys.ZWave.Enums;
using Zensys.ZWave.Programmer.Models;
using Zensys.ZWave.Programmer.Properties;
using Zensys.ZWave.Devices;
using System.Threading;

namespace Zensys.ZWave.Programmer.Controllers
{
	/// <summary>
	/// ControllerManager class.
	/// </summary>
	public class ControllerManager : IDisposable
	{
		private ControllerManager()
		{
			ApplicationExitCode = GetErrorCodeByMessage("");
			mDocumentModel = new DocumentModel();
		}
		private static ControllerManager mInstance;

		private ActionCollection mActions;
		private DocumentModel mDocumentModel;
		private MainForm mMainForm;
		private ZW010xForm mZW010xForm;
		private ZW020xForm mZW020xForm;
		private ZW030xForm mZW030xForm;
		private ZW040xForm mZW040xForm;
		private LogForm mLogForm;
		private ConsoleForm mConsoleForm;
		private DockableView mActiveView;
		private ProgressForm progressForm;

		public static int ApplicationExitCode = 255;

		/// <summary>
		/// Gets the document model.
		/// </summary>
		/// <value>The document model.</value>
		public DocumentModel DocumentModel
		{
			get { return mDocumentModel; }
		}
		/// <summary>
		/// Gets the instance.
		/// </summary>
		/// <value>The instance.</value>
		public static ControllerManager Instance
		{
			get
			{
				if (mInstance == null)
				{
					mInstance = new ControllerManager();
					mInstance.Init();
				}
				return mInstance;
			}
		}
		/// <summary>
		/// Gets the actions. <see cref="ActionCollection"/>
		/// </summary>
		/// <value>The actions.</value>
		public ActionCollection Actions
		{
			get
			{
				if (mActions == null)
				{
					mActions = new ActionCollection(this);
				}
				return mActions;
			}
		}
		/// <summary>
		/// Gets or sets the Main Form.
		/// </summary>
		/// <value>The main form.</value>
		public MainForm MainForm
		{
			get
			{
				return mMainForm;
			}
			set
			{
				Instance.UnregisterView(mMainForm, new MainFormDispatcher());
				mMainForm = value;
				if (value != null)
				{
					Instance.RegisterView(value, new MainFormDispatcher());
				}

			}
		}
		/// <summary>
		/// Gets or sets the ZW010X form.
		/// </summary>
		/// <value>The ZW010X form.</value>
		public ZW010xForm ZW010xForm
		{
			get { return mZW010xForm; }
			set
			{
				Instance.UnregisterView(mZW010xForm, new ZW010xFormDispatcher());
				mZW010xForm = value;
				if (value != null)
				{
					Instance.RegisterView(value, new ZW010xFormDispatcher());
				}
			}
		}
		/// <summary>
		/// Gets or sets the ZW020X form.
		/// </summary>
		/// <value>The ZW020X form.</value>
		public ZW020xForm ZW020xForm
		{
			get { return mZW020xForm; }
			set
			{
				Instance.UnregisterView(mZW020xForm, new ZW020xFormDispatcher());
				mZW020xForm = value;
				if (value != null)
				{
					Instance.RegisterView(value, new ZW020xFormDispatcher());
				}
			}
		}
		/// <summary>
		/// Gets or sets the ZW030X form.
		/// </summary>
		/// <value>The ZW030X form.</value>
		public ZW030xForm ZW030xForm
		{
			get { return mZW030xForm; }
			set
			{
				Instance.UnregisterView(mZW030xForm, new ZW030xFormDispatcher());
				mZW030xForm = value;
				if (value != null)
				{
					Instance.RegisterView(value, new ZW030xFormDispatcher());
				}
			}
		}
		/// <summary>
		/// Gets or sets the ZW040X form.
		/// </summary>
		/// <value>The ZW040X form.</value>
		public ZW040xForm ZW040xForm
		{
			get { return mZW040xForm; }
			set
			{
				Instance.UnregisterView(mZW040xForm, new ZW040xFormDispatcher());
				mZW040xForm = value;
				if (value != null)
				{
					Instance.RegisterView(value, new ZW040xFormDispatcher());
				}
			}
		}
		/// <summary>
		/// Gets or sets the Log form.
		/// </summary>
		/// <value>The Log form.</value>
		public LogForm LogForm
		{
			get { return mLogForm; }
			set
			{
				Instance.UnregisterView(mLogForm, new LogFormDispatcher());
				mLogForm = value;
				if (value != null)
				{
					Instance.RegisterView(value, new LogFormDispatcher());
				}
			}
		}
		/// <summary>
		/// Gets or sets the Console form.
		/// </summary>
		/// <value>The Console form.</value>
		public ConsoleForm ConsoleForm
		{
			get { return mConsoleForm; }
			set
			{
				Instance.UnregisterView(mConsoleForm, new ConsoleFormDispatcher());
				mConsoleForm = value;
				if (value != null)
				{
					Instance.RegisterView(value, new ConsoleFormDispatcher());
				}
			}
		}
		/// <summary>
		/// Gets or sets the Active view.
		/// </summary>
		/// <value>The Active view.</value>
		public DockableView ActiveView
		{
			get { return mActiveView; }
			set { mActiveView = value; }
		}

		/// <summary>
		/// Registers the view.
		/// </summary>
		/// <param name="view">The view.</param>
		/// <param name="dispatcher">The dispatcher.</param>
		public void RegisterView(IView view, IViewDispatcher dispatcher)
		{
			dispatcher.Bind(view, Instance.Actions, this.DocumentModel);
		}
		/// <summary>
		/// Unregisters the view.
		/// </summary>
		/// <param name="view">The view.</param>
		/// <param name="dispatcher">The dispatcher.</param>
		public void UnregisterView(IView view, IViewDispatcher dispatcher)
		{
			if (view != null)
				dispatcher.Drop(view, Instance.Actions, this.DocumentModel);
		}
		private ZWaveManager mZWaveManager;
		/// <summary>
		/// Gets or sets the ZWave manager.
		/// </summary>
		/// <value>The Z wave manager.</value>
		public ZWaveManager ZWaveManager
		{
			get { return mZWaveManager; }
			set { mZWaveManager = value; }
		}
		/// <summary>
		/// Inits this instance.
		/// </summary>
		private void Init()
		{
			ZWaveManager = new ZWaveManager();
            string startupPath = System.Windows.Forms.Application.StartupPath + @"\";

			ZWaveManager.Init(startupPath + "Zensys.ZWave.SerialPortApplication.dll",
				"Zensys.ZWave.SerialPortApplication.ApplicationLayer",
				startupPath + "Zensys.ZWave.SerialPortTransport.dll",
				"Zensys.ZWave.SerialPortTransport.TransportLayer",
				startupPath + "Zensys.ZWave.SerialPortSession.dll",
				"Zensys.ZWave.SerialPortSession.SessionLayer",
				startupPath + "Zensys.ZWave.ProgrammerFrame.dll",
				"Zensys.ZWave.ProgrammerFrame.FrameLayer",
				LibraryModes.Programmer, false);
		}
		/// <summary>
		/// Starts the application.
		/// </summary>
		/// <param name="args">The args.</param>
		public static void Start(string[] args)
		{
			if (args != null && args.Length > 0)
			{
				ProgrammerConsole programmerConsole = new ProgrammerConsole(Instance);
				programmerConsole.ProcessCommandLine(args);
			}
			else
			{
				RunUI();
			}
		}

		internal static void RunUI()
		{
			NativeMethods.FreeConsole();
            System.Windows.Forms.Application.EnableVisualStyles();
            System.Windows.Forms.Application.SetCompatibleTextRenderingDefault(false);
			AppDomain.CurrentDomain.ProcessExit += new EventHandler(OnProcessExit);
			AppDomain.CurrentDomain.UnhandledException += new UnhandledExceptionEventHandler(OnUnhandledException);
			Instance.MainForm = new MainForm();
            System.Windows.Forms.Application.Run(Instance.MainForm);
		}

		/// <summary>
		/// Called when [process exit].
		/// </summary>
		/// <param name="sender">The sender.</param>
		/// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
		static void OnProcessExit(object sender, EventArgs e)
		{
			if (mInstance != null)
				mInstance.Dispose();
		}
		/// <summary>
		/// Called when [unhandled exception].
		/// </summary>
		/// <param name="sender">The sender.</param>
		/// <param name="e">The <see cref="System.UnhandledExceptionEventArgs"/> instance containing the event data.</param>
		static void OnUnhandledException(object sender, UnhandledExceptionEventArgs e)
		{
			//LogManager.Instance.WriteLog(string.Format("Exception {0} occured", e.ExceptionObject.GetType().FullName), LogDetailLevel.Short);
			//NotificationForm nf = new NotificationForm(e.ExceptionObject.GetType().FullName, e.ExceptionObject.ToString());
			//nf.Text = "Error";
			////((Exception)e.ExceptionObject).TargetSite.GetMethodBody()
			//switch (nf.ShowDialog())
			//{
			//    case DialogResult.Abort:
			//        Application.Exit();
			//        break;
			//    case DialogResult.Ignore:
			//        return;
			//    case DialogResult.OK:
			//        break;
			//    default:
			//        break;
			//}
		}
		#region IDisposable Members

		/// <summary>
		/// Performs application-defined tasks associated with freeing, releasing, or resetting unmanaged resources.
		/// </summary>
		public void Dispose()
		{
			Dispose(true);
			GC.SuppressFinalize(this);
		}

		/// <summary>
		/// Releases unmanaged and - optionally - managed resources
		/// </summary>
		/// <param name="disposing"><c>true</c> to release both managed and unmanaged resources; <c>false</c> to release only unmanaged resources.</param>
		protected virtual void Dispose(bool disposing)
		{
			if (disposing)
			{
				if (this.progressForm != null && !this.progressForm.IsDisposed)
				{
					this.progressForm.Dispose();
				}
				UnregisterView(this.MainForm, new MainFormDispatcher());
			}
		}

		#endregion

		private bool isBusy = false;
		internal void DoAction(EventHandler eventHandler, string progressInfo, bool canCancel, byte commandType)
		{
			if (isBusy) return;
			isBusy = true;
            System.Windows.Forms.Application.DoEvents();
			if (progressForm == null)
			{
				progressForm = new ProgressForm();
				progressForm.OnCancel += new EventHandler(ProgressFormOnCancel);
			}
			progressForm.ProgressDelegate = eventHandler;
			progressForm.ProgressInfo = progressInfo;
			progressForm.SetInfo("");
			ShowMessage(progressInfo, false, false, true);
			progressForm.CanCancel = canCancel;
			progressForm.CommandType = commandType;
			if (this.MainForm.InvokeRequired)
			{
				progressForm.ShowDialog();
			}
			else
			{
				progressForm.ShowDialog(this.MainForm);
			}
			if (progressForm.ProgressException != null)
			{
				this.ShowMessage(
					progressForm.ProgressException.Message +
					Environment.NewLine +
					Environment.NewLine +
					progressForm.ProgressException.StackTrace, true);
			}
			isBusy = false;
		}

		void ProgressFormOnCancel(object sender, EventArgs e)
		{
			ProgressForm form = (ProgressForm)sender;
			isBusy = false;
		}
		internal void ShowConsoleInfo(string data)
		{
			if (this.ConsoleForm != null && !this.ConsoleForm.IsDisposed)
			{
				this.ConsoleForm.BeginInvoke(new EventHandler(delegate
				{
					this.ConsoleForm.ContentTextBox.Text = data;
				}));

			}
		}

		internal void ShowProgrammerStatus(string message)
		{
			string status = Resources.MsgStatusLabel + message;
			status = status.Replace("\r", "");
			status = status.Replace("\n", " ");
			this.MainForm.ProgrammerStatusLabel.Text = status;
		}

		internal void ShowMessage(string message, bool isError, bool isCommandPrompt, bool updateStatusOnly)
		{
			if (!isCommandPrompt && updateStatusOnly)
			{
				ShowProgrammerStatus(message);
			}
			else
			{
				ShowMessage(message, isError, isCommandPrompt);
			}
		}

		internal void ShowMessage(string message, bool isError, bool isCommandPrompt)
		{
			if (isCommandPrompt)
			{
				if (isError)
				{
					ApplicationExitCode = GetErrorCodeByMessage(message);
				}
				else
				{
					ApplicationExitCode = 0;
				}
				Console.WriteLine(message);
			}
			else
			{
				ShowProgrammerStatus(message);
				this.MainForm.BeginInvoke(new EventHandler(delegate
				{
					MessageBox.Show(this.MainForm, message, AboutBox.AssemblyProduct, MessageBoxButtons.OK, (isError == true) ? MessageBoxIcon.Exclamation : MessageBoxIcon.Information);
				}));
			}
		}

		private int GetErrorCodeByMessage(string message)
		{
			if (message == Resources.ErrorCantCompareEEPROM) return 1;
			else if (message == Resources.ErrorCantCompareFlash) return 2;
			else if (message == Resources.ErrorCantCompareSRAM) return 3;
			else if (message == Resources.ErrorCantDetectDevice) return 4;
			else if (message == Resources.ErrorCantEraseEEPROM) return 5;
			else if (message == Resources.ErrorCantEraseFlash) return 6;
			else if (message == Resources.ErrorCantGetFirmwareVersion) return 7;
			else if (message == Resources.ErrorCantInitEEPROM) return 8;
			else if (message == Resources.ErrorCantLockBitsRead) return 9;
			else if (message == Resources.ErrorCantLockBitsSet) return 10;
			else if (message == Resources.ErrorCantReadAppRfSettings) return 11;
			else if (message == Resources.ErrorCantReadEEPROM) return 12;
			else if (message == Resources.ErrorCantReadEEPROMOptions) return 13;
			else if (message == Resources.ErrorCantReadFlash) return 14;
			else if (message == Resources.ErrorCantReadFlashOptions) return 15;
			else if (message == Resources.ErrorCantReadGeneralRfSettings) return 16;
			else if (message == Resources.ErrorCantReadHomeId) return 17;
			else if (message == Resources.ErrorCantReadSram) return 18;
			else if (message == Resources.ErrorCantResetZWaveModule) return 19;
			else if (message == Resources.ErrorCantSetBootLoaderMode) return 20;
			else if (message == Resources.ErrorCantSetChipWorkingMode) return 21;
			else if (message == Resources.ErrorCantSetProgrammingMode) return 22;
			else if (message == Resources.ErrorCantUpgradeFirmware) return 23;
			else if (message == Resources.ErrorCantWriteAppRfSettings) return 24;
			else if (message == Resources.ErrorCantWriteEEPROM) return 25;
			else if (message == Resources.ErrorCantWriteFlash) return 26;
			else if (message == Resources.ErrorCantWriteFlashOptions) return 27;
			else if (message == Resources.ErrorCantWriteGeneralRfSettings) return 28;
			else if (message == Resources.ErrorCantWriteHomeId) return 29;
			else if (message == Resources.ErrorCantWriteSram) return 30;
			else if (message == Resources.ErrorCompareEepromFailed) return 31;
			else if (message == Resources.ErrorCompareSRAMFailed) return 32;
			else if (message == Resources.ErrorEEPROMHexFileNotSpecified) return 33;
			else if (message == Resources.ErrorFlashHexFileNotSpecified) return 34;
			else if (message == Resources.ErrorHexFileDataOutOfRange) return 35;
			else if (message == Resources.ErrorHexFileNotValid) return 36;
			else if (message == Resources.ErrorInvalidEndHomeId) return 37;
			else if (message == Resources.ErrorRfFrequencyNotSelected) return 38;
			else if (message == Resources.ErrorStartEndHomeIdEmpty) return 39;
			else if (message == Resources.ErrorUndefinedRfSettings) return 40;
			else if (message == Resources.ErrorCantInitMtp) return 41;
			else if (message == Resources.ErrorCantReadMtp) return 42;
			else if (message == Resources.ErrorCantEraseMtp) return 43;
			else if (message == Resources.ErrorCantProgramMtp) return 44;
			else if (message == Resources.ErrorCantCompareMtp) return 45;
			else if (message == Resources.ErrorProgramMtpFailed) return 46;
			else if (message == Resources.ErrorCompareMtpFailed) return 47;
			else if (message == Resources.ErrorMtpHexFileNotSpecified) return 48;
			else if (message == Resources.ErrorHexFileReadingFailed) return 49;
			else return 251;
		}
		internal void ShowMessage(string message, bool isError)
		{
			ShowProgrammerStatus(message);
			this.MainForm.BeginInvoke(new EventHandler(delegate
			{
				MessageBox.Show(this.MainForm, message, AboutBox.AssemblyProduct, MessageBoxButtons.OK, (isError == true) ? MessageBoxIcon.Exclamation : MessageBoxIcon.Information);
			}));

		}
		internal DialogResult ShowConfirmMessage(string message, bool isError)
		{
			DialogResult result = DialogResult.None;
			this.MainForm.Invoke(new EventHandler(delegate
			{
				result = MessageBox.Show(this.MainForm, message, AboutBox.AssemblyProduct, MessageBoxButtons.YesNo, (isError == true) ? MessageBoxIcon.Exclamation : MessageBoxIcon.Information);
			}));

			return result;
		}



		internal void SetActiveViewByChipType(ChipTypes chipType)
		{
			switch (chipType)
			{
				case ChipTypes.ZW010x:
					{
						this.MainForm.ZW010xToolStripMenuItem.Checked = true;
						this.Actions.MainFormActions.OnZW010xClick(this.MainForm.ZW010xToolStripMenuItem, new EventArgs());
					} break;
				case ChipTypes.ZW020x:
					{
						this.MainForm.ZW020xToolStripMenuItem.Checked = true;
						this.Actions.MainFormActions.OnZW020xClick(this.MainForm.ZW020xToolStripMenuItem, new EventArgs());
					} break;
				case ChipTypes.ZW030x:
					{
						this.MainForm.ZW030xToolStripMenuItem.Checked = true;
						this.Actions.MainFormActions.OnZW030xClick(this.MainForm.ZW030xToolStripMenuItem, new EventArgs());
					} break;
				case ChipTypes.ZW040x:
					{
						this.MainForm.ZW040xToolStripMenuItem.Checked = true;
						this.Actions.MainFormActions.OnZW040xClick(this.MainForm.ZW040xToolStripMenuItem, new EventArgs());
					} break;
			}
		}
        internal void SetProgressInfo(string info, bool isCommandPrompt)
        {
            if (!isCommandPrompt)
            {
                if (progressForm != null && !progressForm.IsDisposed)
                {
                    progressForm.SetProgressInfo(info);
                }
            }
            else
            {
                Console.WriteLine(info);
            }
        }
		internal void ChangeProgressInfo(ProgressStatuses progressStatus, int current, int total, bool isCommandPrompt)
		{
			if (!isCommandPrompt)
			{

				if (progressForm != null && !progressForm.IsDisposed)
				{
					string info;
					switch (progressStatus)
					{
						case ProgressStatuses.Read:
							{
								info = String.Format("{0} / {1} bytes read...", current, total);
							}
							break;
						case ProgressStatuses.Write:
							{
								info = String.Format("{0} / {1} bytes written...", current, total);
							}
							break;
						default:
							{
								info = "";
							}
							break;
					}
					progressForm.SetInfo(info);
					if (String.IsNullOrEmpty(info))
					{
						ShowMessage(progressForm.ProgressInfo, false, false, true);
					}
					else
					{
						ShowMessage(progressForm.ProgressInfo + " " + info, false, false, true);
					}
				}
			}
			else
			{
				switch (progressStatus)
				{
					case ProgressStatuses.Read:
						{
							Console.Write(String.Format("\b {0} / {1} bytes read...\r", current, total));
						}
						break;
					case ProgressStatuses.Write:
						{
							Console.Write(String.Format("\b {0} / {1} bytes written...\r", current, total));
						}
						break;
					default:
						{
							Console.WriteLine("");
						}
						break;
				}

			}
		}

		internal void UpdateFrequencies()
		{
			if (Instance.ZW020xForm != null && !Instance.ZW020xForm.IsDisposed)
			{
				UpdateFrequenciesLow(Instance.ZW020xForm.FrequencyComboBox);
				/*Instance.ZW020xForm.RadioButton870_42.Visible = Instance.DocumentModel.Settings.UseTestFrequencies;
				Instance.ZW020xForm.RadioButton910_42.Visible = Instance.DocumentModel.Settings.UseTestFrequencies;
				Instance.ZW020xForm.UsTfRadioButton.Visible = Instance.DocumentModel.Settings.UseTestFrequencies;
				Instance.ZW020xForm.EuTfRadioButton.Visible = Instance.DocumentModel.Settings.UseTestFrequencies;

				Instance.ZW020xForm.UsRadioButton.Visible = !Instance.DocumentModel.Settings.UseTestFrequencies;
				Instance.ZW020xForm.EuRadioButton.Visible = !Instance.DocumentModel.Settings.UseTestFrequencies;
				Instance.ZW020xForm.HkRadioButton.Visible = !Instance.DocumentModel.Settings.UseTestFrequencies;
				Instance.ZW020xForm.AnzRadioButton.Visible = !Instance.DocumentModel.Settings.UseTestFrequencies;
				Instance.ZW020xForm.MyRadioButton.Visible = !Instance.DocumentModel.Settings.UseTestFrequencies;*/
			}
			if (Instance.ZW030xForm != null && !Instance.ZW030xForm.IsDisposed)
			{
				UpdateFrequenciesLow(Instance.ZW030xForm.FrequencyComboBox);
				/*Instance.ZW030xForm.RadioButton870_42.Visible = Instance.DocumentModel.Settings.UseTestFrequencies;
				Instance.ZW030xForm.RadioButton910_42.Visible = Instance.DocumentModel.Settings.UseTestFrequencies;
				Instance.ZW030xForm.UsTfRadioButton.Visible = Instance.DocumentModel.Settings.UseTestFrequencies;
				Instance.ZW030xForm.EuTfRadioButton.Visible = Instance.DocumentModel.Settings.UseTestFrequencies;

				Instance.ZW030xForm.UsRadioButton.Visible = !Instance.DocumentModel.Settings.UseTestFrequencies;
				Instance.ZW030xForm.EuRadioButton.Visible = !Instance.DocumentModel.Settings.UseTestFrequencies;
				Instance.ZW030xForm.HkRadioButton.Visible = !Instance.DocumentModel.Settings.UseTestFrequencies;
				Instance.ZW030xForm.AnzRadioButton.Visible = !Instance.DocumentModel.Settings.UseTestFrequencies;
				Instance.ZW030xForm.MyRadioButton.Visible = !Instance.DocumentModel.Settings.UseTestFrequencies;*/
			}
		}

		private static void UpdateFrequenciesLow(ComboBox frequencyComboBox)
		{
			FrequencyWrapper selectedItem = (FrequencyWrapper)frequencyComboBox.SelectedItem;
			frequencyComboBox.Items.Clear();
			if (Instance.DocumentModel.Settings.UseTestFrequencies)
			{
				frequencyComboBox.Items.Add(new FrequencyWrapper(" ", 0xFF));	//actually this item means
					// "Frequency which is set in input HEX file" or "Do not change the frequency"
					// or "Frerquency unknown" or "Frequency is not set" depend from which operation is
					//  performed ("Program", "Set Option", etc.)
				frequencyComboBox.Items.Add(new FrequencyWrapper("EU_tf", MemoryLayout.RF_866));
				frequencyComboBox.Items.Add(new FrequencyWrapper("870.42MHz", MemoryLayout.RF_870));
				frequencyComboBox.Items.Add(new FrequencyWrapper("US_tf", MemoryLayout.RF_906));
				frequencyComboBox.Items.Add(new FrequencyWrapper("910.42MHz", MemoryLayout.RF_910));
			}
			else
			{
				frequencyComboBox.Items.Add(new FrequencyWrapper(" ", 0xFF)); //... same as above ^^^
				frequencyComboBox.Items.Add(new FrequencyWrapper("US", MemoryLayout.RF_US));
				frequencyComboBox.Items.Add(new FrequencyWrapper("EU", MemoryLayout.RF_EU));
				frequencyComboBox.Items.Add(new FrequencyWrapper("ANZ", MemoryLayout.RF_ANZ));
				frequencyComboBox.Items.Add(new FrequencyWrapper("HK", MemoryLayout.RF_HK));
				frequencyComboBox.Items.Add(new FrequencyWrapper("MY", MemoryLayout.RF_MY));
				frequencyComboBox.Items.Add(new FrequencyWrapper("IN", MemoryLayout.RF_IN));
				frequencyComboBox.Items.Add(new FrequencyWrapper("RU", MemoryLayout.RF_RU));
			}
			if (selectedItem != null)
			{
				foreach (FrequencyWrapper freqWrapper in frequencyComboBox.Items)
				{
					if (freqWrapper.Frequency == selectedItem.Frequency)
					{
						frequencyComboBox.SelectedItem = freqWrapper;
					}
				}

			}
			else
			{
				frequencyComboBox.SelectedIndex = 0;
			}
		}

		bool deviceFirmwareVersionDetectShowInConsole = true;
		internal bool DeviceFirmwareVersionDetect()
		{
			bool result = false;
			if (this.DocumentModel.Device != null)
			{
				int version, revision;
				string firmware = Resources.MsgFirmwareVersionLabel;
				try
				{
					if (DocumentModel.Device.GetCurrentFirmwareVersion(out version, out revision))
					{
						result = true;
						firmware += String.Format("{0}.{1:D2}", version, revision);
					}
				}
				catch
				{
				}
				if (!result)
				{
					firmware += "N/A";
				}
				if (this.MainForm != null)
				{
					this.MainForm.FirmwareStatusLabel.Text = firmware;
				}
				else if (deviceFirmwareVersionDetectShowInConsole)
				{
					Console.WriteLine(Resources.CaptionZWaveProgrammer + " " + firmware);
					deviceFirmwareVersionDetectShowInConsole = false;
				}
			}
			return result;
		}

		// Called on application UI startup to open and init the device
		internal void DeviceInit()
		{
			if (!String.IsNullOrEmpty(Settings.Default.LastUsedDevice))
			{
				if (DocumentModel.Settings.WriteOnPcbButton)
				{
					DocumentModel.Device = ZWaveManager.ApplicationLayer.CreateDevice();
					if (DocumentModel.Device != null)
					{
						try
						{
							//assign event hadler before opening the device, because it can be called during open:
							DocumentModel.Device.ApplicationCommandHandlerEvent += new Zensys.ZWave.Events.DeviceAppCommandHandlerEventHandler(Device_ApplicationCommandHandlerEvent);
							DocumentModel.Device.Open(DocumentModel.PortInfo.DeviceID);
						}
						catch
						{
						}
					}
				}
				this.DoAction(new EventHandler(delegate
					{
						this.Actions.ProgrammerActions.UpgradeLatestFirmware(false);
					}), Resources.MsgUpgradeFirmwareCheckProgress, false, 0x00);
			}
		}

		void Device_ApplicationCommandHandlerEvent(Zensys.ZWave.Events.DeviceAppCommandHandlerEventArgs args)
		{
			if (args.CommandClassKey == 0x00 && args.SourceNodeId == 0x00 && args.CommandBuffer!= null)
			{
                if (args.CommandBuffer.Length >= (int)ProgrammerFrameInfo.ProgrammerFrameSizeMin)
				{
					switch ((ProgrammerCommandTypes)args.CommandKey)
					{
						case ProgrammerCommandTypes.FUNC_ID_BUTTON_PRESSED:
							if (DocumentModel.Settings.WriteOnPcbButton)
							{
                                if (args.CommandBuffer.Length >= (int)ProgrammerFrameInfo.ProgrammerFrameButtonPressedSize)
								{
                                    if ((args.CommandBuffer[(int)ProgrammerFrameInfo.ProgrammerFrameButtonStateOffset]
										& (byte)ProgrammerFrameInfo.ProgrammerFrameButtonStateS1Pressed) != 0)
									{
										if (!DocumentModel.DeviceBusy
											&& !DocumentModel.DeviceProgramButtonPressed)
										{
											DocumentModel.DeviceProgramButtonPressed = true;
											Thread t = new Thread(new ThreadStart(delegate
											{
												Actions.ProgrammerActions.OnFlashProgramClick(null, null);
											}));
											t.Start();
										}
									}
								}
							}
							break;
					}
				}
			}
		}

		// Called on application shutdown to open and init the device
		internal void DeviceDeinit()
		{
			if (DocumentModel.Device != null)
			{
				if (DocumentModel.Device.ConnectionStatus == ConnectionStatuses.Opened)
				{
					try
					{
						DocumentModel.Device.Close();
					}
					catch
					{
					}
				}
				DocumentModel.Device.ApplicationCommandHandlerEvent -= new Zensys.ZWave.Events.DeviceAppCommandHandlerEventHandler(Device_ApplicationCommandHandlerEvent);
				DocumentModel.Device.Dispose();
				DocumentModel.Device = null;
			}
		}

		// Called before each action to open device for that action
		internal IDevice DeviceOpen()
		{
			if (!DocumentModel.Settings.WriteOnPcbButton || DocumentModel.Device == null)
			{
				if (DocumentModel.Device == null)
				{
					if (DocumentModel.PortInfo == null)
					{
						throw new Exception(Resources.MsgInterfaceNotSelected);
						return null;
					}
					DocumentModel.Device = ZWaveManager.ApplicationLayer.CreateDevice();
					if (DocumentModel.Device != null)
					{
						//assign event hadler before opening the device, because it can be called during open:
						DocumentModel.Device.ApplicationCommandHandlerEvent += new Zensys.ZWave.Events.DeviceAppCommandHandlerEventHandler(Device_ApplicationCommandHandlerEvent);
						DocumentModel.Device.Open(DocumentModel.PortInfo.DeviceID);
					}
				}
			}

			// device init:
			//try
			//{
			//DocumentModel.Device.Memory.SwitchEEPROM(false);
			//DocumentModel.Device.SetProgrammingMode(false);
			//}
			//catch
			//{
			//}
			this.DeviceFirmwareVersionDetect();

			DocumentModel.DeviceBusy = true;
			return DocumentModel.Device;
		}

		// Called after each action to close device for that action
		internal void DeviceClose()
		{
			if (!DocumentModel.Settings.WriteOnPcbButton)
			{
				if (DocumentModel.Device != null)
				{
					if (DocumentModel.Device.ConnectionStatus == ConnectionStatuses.Opened)
					{
						DocumentModel.Device.Close();
					}
					DocumentModel.Device.ApplicationCommandHandlerEvent -= new Zensys.ZWave.Events.DeviceAppCommandHandlerEventHandler(Device_ApplicationCommandHandlerEvent);
					DocumentModel.Device.Dispose();
					DocumentModel.Device = null;
				}
			}
			DocumentModel.DeviceBusy = false;
		}
    }
}
