using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.Programmer.Classes;
using Zensys.ZWave.Programmer.Controllers;
using Zensys.ZWave.Enums;
using Zensys.ZWave.Devices;
using Zensys.ZWave.Programmer.Properties;
using System.IO;
using System.Windows.Forms;
using Zensys.ZWave.Programmer.UI;
using System.Globalization;

namespace Zensys.ZWave.Programmer.Actions
{
    /// <summary>
    /// ZW040xFormActions class. Contains ZW040x Form functions.
    /// </summary>
    public class ZW040xFormActions : BaseAction
    {
        /// <summary>
        /// Initializes a new instance of the <see cref="ZW040xFormActions"/> class.
        /// </summary>
        /// <param name="controller">The controller.</param>
        public ZW040xFormActions(ControllerManager controller)
            : base(controller)
        {

        }
        #region Form
        /// <summary>
        /// Called when Form load.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnFormLoad(object sender, EventArgs e)
        {
            ControllerManager.MainForm.ZW040xToolStripMenuItem.Checked = true;
        }
        /// <summary>
        /// Called when Form closing.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnFormClosing(object sender, EventArgs e)
        {
            ControllerManager.MainForm.ZW040xToolStripMenuItem.Checked = false;
        }
        #endregion

		/// <summary>
		/// Called when MTP browse HEX file click.
		/// </summary>
		/// <param name="sender">The sender.</param>
		/// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
		public void OnMtpBrowseHexFileClick(object sender, EventArgs e)
		{
			if (ControllerManager.ZW040xForm != null && !ControllerManager.ZW040xForm.IsDisposed)
			{
				ControllerManager.DocumentModel.SetMtpHexFileName(ChipTypes.ZW040x, ControllerManager.Actions.CommonActions.BrowseFile());
				ControllerManager.ZW040xForm.MtpHexFileNameTextBox.Text = ControllerManager.DocumentModel.GetMtpHexFileName(ChipTypes.ZW040x);
			}
		}

		/// <summary>
		/// Called when MTP HEX file name changed.
		/// </summary>
		/// <param name="sender">The sender.</param>
		/// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
		public void OnMtpHexFileNameChanged(object sender, EventArgs e)
		{
			if (ControllerManager.ZW040xForm != null && !ControllerManager.ZW040xForm.IsDisposed)
			{
				//ControllerManager.DocumentModel.SetFlashHexFileName(ChipTypes.ZW040x, ControllerManager.ZW040xForm.SramHexFileNameTextBox.Text);
			}
		}

		/// <summary>
		/// Called when SRAM browse HEX file click.
		/// </summary>
		/// <param name="sender">The sender.</param>
		/// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
		public void OnSramBrowseHexFileClick(object sender, EventArgs e)
		{
			if (ControllerManager.ZW040xForm != null && !ControllerManager.ZW040xForm.IsDisposed)
			{
				ControllerManager.DocumentModel.SetSramHexFileName(ChipTypes.ZW040x, ControllerManager.Actions.CommonActions.BrowseFile());
				ControllerManager.ZW040xForm.SramHexFileNameTextBox.Text = ControllerManager.DocumentModel.GetSramHexFileName(ChipTypes.ZW040x);
				ControllerManager.Actions.ZW040xFormActions.SramUpdateUserFlashSettingsToDefault(ChipTypes.ZW040x);
			}
		}

		/// <summary>
		/// Called when SRAM HEX file name changed.
		/// </summary>
		/// <param name="sender">The sender.</param>
		/// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
		public void OnSramHexFileNameChanged(object sender, EventArgs e)
		{
			if (ControllerManager.ZW040xForm != null && !ControllerManager.ZW040xForm.IsDisposed)
			{
				//ControllerManager.DocumentModel.SetFlashHexFileName(ChipTypes.ZW040x, ControllerManager.ZW040xForm.SramHexFileNameTextBox.Text);
			}
		}

		/// <summary>
        /// Called when Flash browse HEX file click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnFlashBrowseHexFileClick(object sender, EventArgs e)
        {
            if (ControllerManager.ZW040xForm != null && !ControllerManager.ZW040xForm.IsDisposed)
            {
                ControllerManager.DocumentModel.SetFlashHexFileName(ChipTypes.ZW040x, ControllerManager.Actions.CommonActions.BrowseFile());
                ControllerManager.ZW040xForm.FlashHexFileNameTextBox.Text = ControllerManager.DocumentModel.GetFlashHexFileName(ChipTypes.ZW040x);
                ControllerManager.Actions.ProgrammerActions.UpdateUserFlashSettingsToDefault(ChipTypes.ZW040x);
            }
        }
        /// <summary>
        /// Called when EEPROM browse HEX file click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnEEPROMBrowseHexFileClick(object sender, EventArgs e)
        {
            if (ControllerManager.ZW040xForm != null && !ControllerManager.ZW040xForm.IsDisposed)
            {
                ControllerManager.DocumentModel.SetEepromHexFileName(ChipTypes.ZW040x, ControllerManager.Actions.CommonActions.BrowseFile());
                ControllerManager.ZW040xForm.EepromHexFileNameTextBox.Text = ControllerManager.DocumentModel.GetEepromHexFileName(ChipTypes.ZW040x);
            }
        }

        /// <summary>
        /// Called when Flash HEX file name changed.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnFlashHexFileNameChanged(object sender, EventArgs e)
        {
            if (ControllerManager.ZW040xForm != null && !ControllerManager.ZW040xForm.IsDisposed)
            {
                //ControllerManager.DocumentModel.SetFlashHexFileName(ChipTypes.ZW040x, ControllerManager.ZW040xForm.FlashHexFileNameTextBox.Text);
            }
        }

        /// <summary>
        /// Called when EEPROM HEX file name changed.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnEepromHexFileNameChanged(object sender, EventArgs e)
        {
            if (ControllerManager.ZW040xForm != null && !ControllerManager.ZW040xForm.IsDisposed)
            {
                //ControllerManager.DocumentModel.SetEepromHexFileName(ChipTypes.ZW040x, ControllerManager.ZW040xForm.EepromHexFileNameTextBox.Text);
            }
        }

        /// <summary>
        /// Return SRAM Working Mode, selected by user
        /// </summary>
        public WorkingModes GetSelectedWorkingMode()
        {
            WorkingModes result = WorkingModes.None;
            if (ControllerManager.ZW040xForm != null)
            {
                if (ControllerManager.ZW040xForm.ModeDevelopmentRadioButton.Checked)
                    result = WorkingModes.Development;
                else if (ControllerManager.ZW040xForm.ModeExecOutOfSRAMRadioButton.Checked)
                    result = WorkingModes.ExecuteOutOfSram;
            }
            return result;
        }

        /// <summary>
        /// Called when SRAM read click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnSramReadClick(object sender, EventArgs e)
        {
            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                WorkingModes workingMode = GetSelectedWorkingMode();
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    SramRead(false, null, workingMode);
                }), Resources.MsgReadSramProgress, false, 0x00);

            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
            }
        }
        /// <summary>
        /// Reads the SRAM.
        /// </summary>
        /// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
        /// <param name="fileName">Name of the file.</param>
        /// <param name="workingMode">Working mode.</param>
        public void SramRead(bool isCommandPrompt, string fileName, WorkingModes workingMode)
        {
            if (isCommandPrompt)
            {
                ControllerManager.ShowMessage(Resources.MsgReadSramProgress, false, isCommandPrompt);
            }
            int fileOffset = 0;
            int sramOffset = 0;
            byte[] sramBuffer = null;
			IDevice device = ControllerManager.DeviceOpen();

            try
            {
                if (device.SetProgrammingMode(true))
                {
                    device.ReadSignatureBits();
                    if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
                    {
                        IDeviceLed busyLed = device.Leds.GetByType(LedTypes.Busy);
                        if (busyLed != null)
                        {
                            busyLed.SetState(true);

                            switch (workingMode)
                            {
                                case WorkingModes.Development:
                                    {
										fileOffset = (int)device.Flash.FlashSize - (int)Constants.SRAM_DEVMODE_SIZE;
										sramOffset = (int)Constants.SRAM_DEVMODE_OFFSET;
                                    }
                                    break;
                            }
                            if (!device.Memory.ReadSRAM())
                            {
                                ControllerManager.ShowMessage(Resources.ErrorCantReadSram, true, isCommandPrompt);
                            }

                            if (device.Memory.Buffer != null)
                            {
                                sramBuffer = HexFileHelper.BlankArray(device.Memory.Buffer.Length - sramOffset, Constants.SRAM_BLANK_VALUE);
                                for (int i = 0; i < device.Memory.Buffer.Length - sramOffset; i++)
                                {
                                    sramBuffer[i] = device.Memory.Buffer[sramOffset + i];
                                }
                            }

                            busyLed.SetState(false);
                        }
                    }
                    else
                    {
                        ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true, isCommandPrompt);
                    }
                    if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
                        device.SetProgrammingMode(false);
                }
                else
                {
                    ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true, isCommandPrompt);
                }
            }
            catch
            {
                ControllerManager.ShowMessage(Resources.ErrorCantReadSram, true, isCommandPrompt);
            }
            finally
            {
				ControllerManager.DeviceClose();
            }
            if (sramBuffer != null)
            {
                if (isCommandPrompt)
                {
                    try
                    {
                        TextWriter writer = new StreamWriter(fileName);
                        writer.Write(HexFileHelper.WriteIntelHexFile(sramBuffer, fileOffset, sramBuffer.Length));
                        writer.Close();
                        ControllerManager.ShowMessage(Resources.MsgFinished, false, isCommandPrompt);
                    }
                    catch (Exception ex)
                    {
                        Console.WriteLine(ex.Message);
                    }
                }
                else
                {
                    ControllerManager.ShowConsoleInfo(HexFileHelper.WriteIntelHexFile(sramBuffer, fileOffset, sramBuffer.Length));
					ControllerManager.ShowMessage(Resources.MsgFinished, false, false, true);
				}
            }
        }

		public bool WirteHEXToSRAM(bool isCommandPrompt, IDevice device, string fileName, IFlashSettings rfSettings)
        {
            return WirteHEXToSRAM(isCommandPrompt, device, fileName, 0, 0, rfSettings);
        }

		private bool WirteHEXToSRAM(bool isCommandPrompt, IDevice device, string fileName, int fileUseFromOffset, int sramWriteOffset, IFlashSettings rfSettings)
        {
            bool result = true;
            if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
            {
                IDeviceLed busyLed = device.Leds.GetByType(LedTypes.Busy);
                if (busyLed != null)
                {
                    busyLed.SetState(true);
                    string hexFileName = fileName;
                    if (!String.IsNullOrEmpty(hexFileName) && File.Exists(hexFileName))
                    {
                        byte[] flashDataRaw = HexFileHelper.BlankArray((int)device.Memory.SramSize, Constants.SRAM_BLANK_VALUE);
                        byte rdres = HexFileHelper.ReadIntelHexFile(hexFileName, flashDataRaw, fileUseFromOffset + (int)device.Memory.SramSize, fileUseFromOffset);
                        if (rdres == HexFileHelper.FILE_READ)
                        {
							//Apply the SRAM RF Settings:
							if (rfSettings != null)
							{
								rfSettings.StoreToBuffer(device.ChipType, (uint)fileUseFromOffset, flashDataRaw);
							}

							//Apply the SRAM Write Offset:
                            if (sramWriteOffset != 0)
                            {
                                int i;
                                for (i = (int)device.Memory.SramSize - 1; i >= sramWriteOffset; i--)
                                {
                                    flashDataRaw[i] = flashDataRaw[i - sramWriteOffset];
                                }
                                for (; i >= 0; i--)
                                {
                                    flashDataRaw[i] = Constants.SRAM_BLANK_VALUE;
                                }
                            }
                            //TODO: if hex file have more than X kB of data - show error (or warning??)

							//Erace the MTP Memory if needed, before programming SRAM.
							// We must did this before SRAM programming, because operations with MTP can corrupt the SRAM.
                            WorkingModes wMode = GetSelectedWorkingMode();
                            if (wMode == WorkingModes.ExecuteOutOfSram)
                            {
                                if (MtpAutoErase(device))
                                {
                                    if (!device.Memory.WriteSRAM(flashDataRaw, true))
                                    {
                                        ControllerManager.ShowMessage(Resources.ErrorCantWriteSram, true, isCommandPrompt);
                                        result = false;
                                    }
                                }
                            }
                            else
                            {
                                if (!device.Memory.WriteSRAM(flashDataRaw, true))
                                {
                                    ControllerManager.ShowMessage(Resources.ErrorCantWriteSram, true, isCommandPrompt);
                                    result = false;
                                }
                            }
                        }
                        else if (rdres == HexFileHelper.ADDRESS_OUT_OF_RANGE)
                        {
                            ControllerManager.ShowMessage(Resources.ErrorHexFileDataOutOfRange, true, isCommandPrompt);
                            result = false;
                        }
                        else
                        {
                            ControllerManager.ShowMessage(Resources.ErrorHexFileNotValid, true, isCommandPrompt);
                            result = false;
                        }
                    }
					else if (String.IsNullOrEmpty(hexFileName))
					{
						ControllerManager.ShowMessage("Hex file is not selected.", true, isCommandPrompt);
						result = false;
					}
					else
					{
						ControllerManager.ShowMessage(String.Format("{0} does not exists.", hexFileName), true, isCommandPrompt);
						result = false;
					}
					busyLed.SetState(false);
                }
            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true, isCommandPrompt);
                result = false;
            }
            return result;
        }

        /// <summary>
        /// Called when SRAM write click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnSramWriteClick(object sender, EventArgs e)
        {
            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                WorkingModes workingMode = GetSelectedWorkingMode();
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    SramWrite(false, null, workingMode);
                }), Resources.MsgWriteSramProgress, false, 0x00);

            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
            }
        }
        /// <summary>
        /// Write the SRAM.
        /// </summary>
        /// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
        /// <param name="fileName">Name of the file.</param>
        /// <param name="workingMode">Working mode.</param>
        public void SramWrite(bool isCommandPrompt, string fileName, WorkingModes workingMode)
        {
            if (isCommandPrompt)
            {
                ControllerManager.ShowMessage(Resources.MsgWriteSramProgress, false, isCommandPrompt);
            }
			IDevice device = ControllerManager.DeviceOpen();

            try
            {
                if (device.SetProgrammingMode(true))
                {
                    device.ReadSignatureBits();

					if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
					{
						string hexFileName;
						if (isCommandPrompt)
						{
							hexFileName = fileName;
						}
						else
						{
							hexFileName = ControllerManager.DocumentModel.GetSramHexFilePath((ChipTypes)device.ChipType);
						}

						int fileOffset = 0;
						int sramOffset = 0;
						IFlashSettings rfSettings = null;
						switch (workingMode)
						{
							case WorkingModes.Development:
								{
									fileOffset = (int)device.Flash.FlashSize - (int)Constants.SRAM_DEVMODE_SIZE;
									sramOffset = (int)Constants.SRAM_DEVMODE_OFFSET;
									if (isCommandPrompt)
									{
										rfSettings = null;			//the flashSettings will be used from the .hex file
									}
									else
									{
										rfSettings = SramGetUserFlashSettings((ChipTypes)device.ChipType);
									}
								}
								break;
						}
						if (WirteHEXToSRAM(isCommandPrompt, device, hexFileName, fileOffset, sramOffset, rfSettings))
						{
							ControllerManager.ShowMessage(Resources.MsgFinished, false, isCommandPrompt, !isCommandPrompt);
						}
					}
					else
					{
						ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true, isCommandPrompt);
					}
					if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
						device.SetProgrammingMode(false);
				}
                else
                {
                    ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true, isCommandPrompt);
                }
            }
            catch
            {
                ControllerManager.ShowMessage(Resources.ErrorCantWriteSram, true, isCommandPrompt);
            }
            finally
            {
				ControllerManager.DeviceClose();
            }
        }

        /// <summary>
        /// Called when SRAM compare click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnSramCompareClick(object sender, EventArgs e)
        {
            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                WorkingModes workingMode = GetSelectedWorkingMode();
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    SramCompare(false, null, workingMode);
                }), Resources.MsgCompareSramProgress, false, 0x00);


            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
            }
        }
        /// <summary>
        /// Compare the SRAM.
        /// </summary>
        /// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
        /// <param name="fileName">Name of the file.</param>
        /// <param name="workingMode">Working mode.</param>
        public void SramCompare(bool isCommandPrompt, string fileName, WorkingModes workingMode)
        {
            if (isCommandPrompt)
            {
                ControllerManager.ShowMessage(Resources.MsgReadSramProgress, false, isCommandPrompt);
            }
            bool compareResult = false;
			IDevice device = ControllerManager.DeviceOpen();
            try
            {

                if (device.SetProgrammingMode(true))
                {
                    device.ReadSignatureBits();
                    if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
                    {
                        IDeviceLed busyLed = device.Leds.GetByType(LedTypes.Busy);
                        if (busyLed != null)
                        {
                            string hexFileName;
                            if (isCommandPrompt)
                            {
                                hexFileName = fileName;
                            }
                            else
                            {
                                hexFileName = ControllerManager.DocumentModel.GetSramHexFilePath((ChipTypes)device.ChipType);
                            }
                            if (!String.IsNullOrEmpty(hexFileName) && File.Exists(hexFileName))
                            {
                                int fileOffset = 0;
                                int sramOffset = 0;
                                switch (workingMode)
                                {
                                    case WorkingModes.Development:
                                        {
											fileOffset = (int)device.Flash.FlashSize - (int)Constants.SRAM_DEVMODE_SIZE;
											sramOffset = (int)Constants.SRAM_DEVMODE_OFFSET;
                                        }
                                        break;
                                }

                                byte[] sramDataRaw = HexFileHelper.BlankArray((int)device.Memory.SramSize - sramOffset, Constants.SRAM_BLANK_VALUE);

                                if (HexFileHelper.ReadIntelHexFile(hexFileName, sramDataRaw, fileOffset + (int)device.Memory.SramSize - sramOffset, fileOffset) == HexFileHelper.FILE_READ)
                                {
                                    busyLed.SetState(true);
                                    if (device.Memory.ReadSRAM())
                                    {
                                        if (device.Memory.Buffer != null)
                                        {
                                            if (device.Memory.Buffer.Length - sramOffset == sramDataRaw.Length)
                                            {
                                                compareResult = true;
                                                for (int i = 0; i < sramDataRaw.Length; i++)
                                                {
                                                    if (device.Memory.Buffer[sramOffset + i] != sramDataRaw[i])
                                                    {
                                                        compareResult = false;
                                                        break;
                                                    }
                                                }
                                            }
                                        }
                                    }
                                    busyLed.SetState(false);
                                }
                                if (compareResult)
                                {
                                    ControllerManager.ShowMessage(Resources.MsgCompareSRAMSuccesful, false, isCommandPrompt);
                                }
                                else
                                {
                                    ControllerManager.ShowMessage(Resources.ErrorCompareSRAMFailed, true, isCommandPrompt);
                                }
                            }
							else if (String.IsNullOrEmpty(hexFileName))
							{
								ControllerManager.ShowMessage("Hex file is not selected.", true, isCommandPrompt);
							}
							else
							{
								ControllerManager.ShowMessage(String.Format("{0} does not exists.", hexFileName), true, isCommandPrompt);
							}
						}
                    }
                    else
                    {
                        ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true, isCommandPrompt);
                    }
                    if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
                        device.SetProgrammingMode(false);
                }
                else
                {
                    ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true, isCommandPrompt);
                }
            }
            catch
            {
                ControllerManager.ShowMessage(Resources.ErrorCantCompareSRAM, true, isCommandPrompt);
            }
            finally
            {
				ControllerManager.DeviceClose();
            }
        }

        /// <summary>
        /// Called when SRAM write and run development mode click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnSramWriteAndRunModeClick(object sender, EventArgs e)
        {
            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                WorkingModes workingMode = GetSelectedWorkingMode();
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    SramWriteAndRunMode(false, null, workingMode);
                }), Resources.MsgWriteSramProgress, false, 0x00);

            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
            }
        }
        /// <summary>
        /// Write the SRAM and run specified mode.
        /// </summary>
        /// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
        /// <param name="fileName">Name of the file.</param>
        /// <param name="workingMode">Working mode.</param>
        public void SramWriteAndRunMode(bool isCommandPrompt, string fileName, WorkingModes workingMode)
        {
            switch (workingMode)
            {
                case WorkingModes.Development:
                    {
                        SramWriteAndRunDevMode(isCommandPrompt, fileName);
                    }
                    break;

                case WorkingModes.ExecuteOutOfSram:
                    {
                        SramWriteAndExecOutOfSram(isCommandPrompt, fileName);
                    }
                    break;
            }
        }
        /// <summary>
        /// Write the SRAM and run development mode.
        /// </summary>
        /// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
        /// <param name="fileName">Name of the file.</param>
        private void SramWriteAndRunDevMode(bool isCommandPrompt, string fileName)
        {
            if (isCommandPrompt)
            {
                ControllerManager.ShowMessage(Resources.MsgWriteSramProgress, false, isCommandPrompt);
            }
			IDevice device = ControllerManager.DeviceOpen();

            try
            {
                if (device.SetProgrammingMode(true))
                {
                    device.ReadSignatureBits();
                    if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
                    {
                        string hexFileName;
						IFlashSettings rfSettings = null;
                        if (isCommandPrompt)
                        {
                            hexFileName = fileName;
							rfSettings = null;			//the flashSettings will be used from the .hex file
                        }
                        else
                        {
                            hexFileName = ControllerManager.DocumentModel.GetSramHexFilePath((ChipTypes)device.ChipType);
							rfSettings = SramGetUserFlashSettings((ChipTypes)device.ChipType);
						}
						if (WirteHEXToSRAM(isCommandPrompt, device, hexFileName, (int)device.Flash.FlashSize - (int)Constants.SRAM_DEVMODE_SIZE, (int)Constants.SRAM_DEVMODE_OFFSET, rfSettings))
                        {
							if (device.SetMode(WorkingModes.Development))
							{
								ControllerManager.ShowMessage(Resources.MsgFinished, false, isCommandPrompt, !isCommandPrompt);
							}
							else
                            {
                                ControllerManager.ShowMessage(Resources.ErrorCantSetChipWorkingMode, true, isCommandPrompt);
                            }
                        }
                    }
                    else
                    {
                        ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true, isCommandPrompt);
                    }
                    if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
                        device.SetProgrammingMode(false);
                }
                else
                {
                    ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true, isCommandPrompt);
                }
            }
            catch
            {
                ControllerManager.ShowMessage(Resources.ErrorCantWriteSram, true, isCommandPrompt);
            }
            finally
            {
				ControllerManager.DeviceClose();
            }
        }

        /// <summary>
        /// Called when SRAM write and execute out of SRAM mode click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnSramWriteAndExecOutOfSramClick(object sender, EventArgs e)
        {
            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    SramWriteAndExecOutOfSram(false, null);
                }), Resources.MsgWriteSramProgress, false, 0x00);

            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
            }
        }
        /// <summary>
        /// Srams the write and exec out of sram.
        /// </summary>
        /// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
        /// <param name="fileName">Name of the file.</param>
        private void SramWriteAndExecOutOfSram(bool isCommandPrompt, string fileName)
        {
            if (isCommandPrompt)
            {
                ControllerManager.ShowMessage(Resources.MsgWriteSramProgress, false, isCommandPrompt);
            }
			IDevice device = ControllerManager.DeviceOpen();

            try
            {
                if (device.SetProgrammingMode(true))
                {
                    device.ReadSignatureBits();
                    if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
                    {
                        string hexFileName;
                        if (isCommandPrompt)
                        {
                            hexFileName = fileName;
                        }
                        else
                        {
                            hexFileName = ControllerManager.DocumentModel.GetSramHexFilePath((ChipTypes)device.ChipType);
                        }
                        if (WirteHEXToSRAM(isCommandPrompt, device, hexFileName, null))
                        {
							if (device.SetMode(WorkingModes.ExecuteOutOfSram))
							{
								ControllerManager.ShowMessage(Resources.MsgFinished, false, isCommandPrompt, !isCommandPrompt);
							}
							else
                            {
                                ControllerManager.ShowMessage(Resources.ErrorCantSetChipWorkingMode, true, isCommandPrompt);
                            }
                        }
                    }
                    else
                    {
                        ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true, isCommandPrompt);
                    }
                    if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
                        device.SetProgrammingMode(false);
                }
                else
                {
                    ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true, isCommandPrompt);
                }
            }
            catch
            {
                ControllerManager.ShowMessage(Resources.ErrorCantWriteSram, true, isCommandPrompt);
            }
            finally
            {
				ControllerManager.DeviceClose();
			}
        }

        internal void OnModeDevelopmentClick(object sender, EventArgs e)
        {
            ControllerManager.ShowMessage("Mode has been changed, please power cycle the device.", false, false);
        }
        internal void OnModeExecOutOfSRAMClick(object sender, EventArgs e)
        {
            ControllerManager.ShowMessage("Mode has been changed, please power cycle the device.", false, false);
        }

		internal void OnModeDevelopmentCheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton)
			{
				Form form = ((RadioButton)sender).FindForm();
				if (form != null)
				{
					if (form is ZW040xForm)
					{
						ControllerManager.ZW040xForm.SramOptionsGroupBox.Enabled = ControllerManager.ZW040xForm.ModeDevelopmentRadioButton.Checked;
						ControllerManager.ZW040xForm.SramReadOptionsButton.Enabled = ControllerManager.ZW040xForm.ModeDevelopmentRadioButton.Checked;
						ControllerManager.ZW040xForm.SramWriteOptionsButton.Enabled = ControllerManager.ZW040xForm.ModeDevelopmentRadioButton.Checked;
                        //ControllerManager.ZW040xForm.MtpAutoEraseCheckBox.Checked = false;
					}
				}
			}
		}

		/// <summary>
		/// Called when Sram Read options click.
		/// </summary>
		/// <param name="sender">The sender.</param>
		/// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
		public void OnSramReadOptionsClick(object sender, EventArgs e)
		{
			IFlashSettings sramSettings = null;
			ChipTypes deviceChipType = ChipTypes.UNKNOWN;
			if (ControllerManager.DocumentModel.PortInfo != null)
			{
				ControllerManager.DoAction(new EventHandler(delegate
				{
                    ReadSRAMRfOptions(ref sramSettings, ref deviceChipType, false);
				}), Resources.MsgReadSramOptionsProgress, false, 0x00);
				if (sramSettings != null && deviceChipType != ChipTypes.UNKNOWN)
				{
					SramUpdateUserFlashSettings(deviceChipType, sramSettings);
					ControllerManager.ShowMessage(Resources.MsgFinished, false, false, true);
				}
			}
			else
			{
				ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
			}
		}

        public void ReadSRAMRfOptions(ref IFlashSettings sramSettings, ref ChipTypes deviceChipType, bool isCommandPrompt)
        {
            IDevice device = ControllerManager.DeviceOpen();
            try
            {
                if (device.SetProgrammingMode(true))
                {
                    device.ReadSignatureBits();
                    if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
                    {
                        deviceChipType = (ChipTypes)device.ChipType;
                        sramSettings = device.Memory.ReadSRAMRfOptions();
                    }
                    else
                    {
                        ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true, isCommandPrompt);
                    }
                    if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
                        device.SetProgrammingMode(false);
                }
                else
                {
                    ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true, isCommandPrompt);
                }
            }
            catch
            {
                ControllerManager.ShowMessage(Resources.ErrorCantReadSramOptions, true, isCommandPrompt);
            }
            finally
            {
                ControllerManager.DeviceClose();
            }
        }

		/// <summary>
		/// Called when Sram Write options click.
		/// </summary>
		/// <param name="sender">The sender.</param>
		/// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
		public void OnSramWriteOptionsClick(object sender, EventArgs e)
		{
			if (ControllerManager.DocumentModel.PortInfo != null)
			{
				ControllerManager.DoAction(new EventHandler(delegate
				{
					SramWriteOptions(false, null);

				}), Resources.MsgWriteSramOptionsProgress, false, 0x00);
			}
			else
			{
				ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
			}
		}
		/// <summary>
		/// Writes the Sram options.
		/// </summary>
		/// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
		/// <param name="frequency">The frequency.</param>
		/// <param name="sramSettings">The SRAM RF Options.</param>
		public void SramWriteOptions(bool isCommandPrompt, IFlashSettings sramSettings)
		{
			WriteRfOptionsStatuses result = WriteRfOptionsStatuses.None;
			if (isCommandPrompt)
			{
				ControllerManager.ShowMessage(Resources.MsgWriteSramOptionsProgress, false, isCommandPrompt);
			}
			IDevice device = ControllerManager.DeviceOpen();
			try
			{
				if (device.SetProgrammingMode(true))
				{
					device.ReadSignatureBits();
					if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
					{
						IFlashSettings sramSettingsInHex = null;
						if (isCommandPrompt)
						{
							sramSettingsInHex = sramSettings;
						}
						else
						{
							sramSettingsInHex = SramGetUserFlashSettings((ChipTypes)device.ChipType);
						}
						result = device.Memory.WriteSRAMRfOptions(sramSettingsInHex);
						if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
							device.SetProgrammingMode(false);
					}
					else
					{
						ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true, isCommandPrompt);
					}
				}
				else
				{
					ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true, isCommandPrompt);
				}
			}
			catch
			{
				ControllerManager.ShowMessage(Resources.ErrorCantWriteSramOptions, true, isCommandPrompt);
			}
			finally
			{
				ControllerManager.DeviceClose();
			}

			switch (result)
			{
				case WriteRfOptionsStatuses.CantReadAppRfSettings:
					{
						ControllerManager.ShowMessage(Resources.ErrorCantReadAppRfSettings, true, isCommandPrompt);
					} break;
				case WriteRfOptionsStatuses.CantReadGeneralRfSettings:
					{
						ControllerManager.ShowMessage(Resources.ErrorCantReadGeneralRfSettings, true, isCommandPrompt);
					} break;
				case WriteRfOptionsStatuses.CantWriteAppRfSettings:
					{
						ControllerManager.ShowMessage(Resources.ErrorCantWriteAppRfSettings, true, isCommandPrompt);
					} break;
				case WriteRfOptionsStatuses.CantWriteGeneralRfSettings:
					{
						ControllerManager.ShowMessage(Resources.ErrorCantWriteGeneralRfSettings, true, isCommandPrompt);
					} break;
				case WriteRfOptionsStatuses.NoErrors:
					{
						ControllerManager.ShowMessage(Resources.MsgWriteRfSettingsNoErrors, false, isCommandPrompt);
					} break;
				case WriteRfOptionsStatuses.None:
					{
						ControllerManager.ShowMessage(Resources.MsgWriteRfSettingsNone, false, isCommandPrompt);
					} break;
				case WriteRfOptionsStatuses.RfFrequencyNotSelected:
					{
						ControllerManager.ShowMessage(Resources.ErrorRfFrequencyNotSelected, true, isCommandPrompt);
					} break;
				case WriteRfOptionsStatuses.UndefinedRfSettings:
					{
						ControllerManager.ShowMessage(Resources.ErrorUndefinedRfSettings, true, isCommandPrompt);
					} break;
			}
		}

		/// <summary>
		/// Updates the flash settings.
		/// </summary>
		/// <param name="chipType">Type of the chip.</param>
		/// <param name="flashSettings">The flash settings.</param>
		public void SramUpdateUserFlashSettings(ChipTypes chipType, IFlashSettings flashSettings)
		{
			if (ControllerManager.ActiveView != null)
			{
				if (chipType == ChipTypes.ZW040x && ControllerManager.ActiveView is ZW040xForm)
				{
					ZW040xForm form = ControllerManager.ActiveView as ZW040xForm;
					form.SramLowTxPowerCh0TextBox.Text = flashSettings.LowPowerCh0.ToString("X");
					form.SramNormalTxPowerCh0TextBox.Text = flashSettings.NormalPowerCh0.ToString("X");
                    form.SramLowTxPowerCh1TextBox.Text = flashSettings.LowPowerCh1.ToString("X");
                    form.SramNormalTxPowerCh1TextBox.Text = flashSettings.NormalPowerCh1.ToString("X");
                    form.SramLowTxPowerCh2TextBox.Text = flashSettings.LowPowerCh2.ToString("X");
                    form.SramNormalTxPowerCh2TextBox.Text = flashSettings.NormalPowerCh2.ToString("X");
				}
			}
		}

		/// <summary>
		/// Updates the flash settings to Default.
		/// </summary>
		/// <param name="chipType">Type of the chip.</param>
		public void SramUpdateUserFlashSettingsToDefault(ChipTypes chipType)
		{
			if (ControllerManager.ActiveView != null)
			{
				if (chipType == ChipTypes.ZW040x && ControllerManager.ActiveView is ZW040xForm)
				{
					ZW040xForm form = ControllerManager.ActiveView as ZW040xForm;
					form.SramLowTxPowerCh0TextBox.Text = "";
					form.SramNormalTxPowerCh0TextBox.Text = "";
                    form.SramLowTxPowerCh1TextBox.Text = "";
                    form.SramNormalTxPowerCh1TextBox.Text = "";
                    form.SramLowTxPowerCh2TextBox.Text = "";
                    form.SramNormalTxPowerCh2TextBox.Text = "";
				}
			}
		}

		private IFlashSettings SramGetUserFlashSettings(ChipTypes chipType)
		{
			IFlashSettings result = null;
			if (ControllerManager.ActiveView != null)
			{
				if (chipType == ChipTypes.ZW040x && ControllerManager.ActiveView is ZW040xForm)
				{
					result = ControllerManager.ZWaveManager.ApplicationLayer.CreateFlashSettings();
					ZW040xForm form = ControllerManager.ActiveView as ZW040xForm;

					if (form.SramLowTxPowerCh0TextBox.Text != "")
						result.LowPowerCh0 = byte.Parse(form.SramLowTxPowerCh0TextBox.Text, NumberStyles.HexNumber);
					if (form.SramNormalTxPowerCh0TextBox.Text != "")
						result.NormalPowerCh0 = byte.Parse(form.SramNormalTxPowerCh0TextBox.Text, NumberStyles.HexNumber);

                    if (form.SramLowTxPowerCh1TextBox.Text != "")
                        result.LowPowerCh1 = byte.Parse(form.SramLowTxPowerCh1TextBox.Text, NumberStyles.HexNumber);
                    if (form.SramNormalTxPowerCh1TextBox.Text != "")
                        result.NormalPowerCh1 = byte.Parse(form.SramNormalTxPowerCh1TextBox.Text, NumberStyles.HexNumber);

                    if (form.SramLowTxPowerCh2TextBox.Text != "")
                        result.LowPowerCh2 = byte.Parse(form.SramLowTxPowerCh2TextBox.Text, NumberStyles.HexNumber);
                    if (form.SramNormalTxPowerCh2TextBox.Text != "")
                        result.NormalPowerCh2 = byte.Parse(form.SramNormalTxPowerCh2TextBox.Text, NumberStyles.HexNumber);
				}
			}
			return result;
		}

        /// <summary>
        /// Called when OTP CRC Check run click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnCrcCheckClick(object sender, EventArgs e)
        {
            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    CrcCheckRun(false);
                }), Resources.MsgWriteSramProgress, false, 0x00 /*TODO: <--- change!!!*/);

            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
            }
        }
        /// <summary>
        /// Run OTP CRC Check.
        /// </summary>
        /// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
        public void CrcCheckRun(bool isCommandPrompt)
        {
			/*if (isCommandPrompt)
			{
				ControllerManager.ShowMessage(Resources.MsgWriteSramProgress, true, isCommandPrompt);
			}
			IDevice device = ControllerManager.DeviceOpen();

			try
			{
				if (device.SetProgrammingMode(true))
				{
					device.ReadSignatureBits();
					if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
					{
						string hexFileName;
						if (isCommandPrompt)
						{
							hexFileName = fileName;
						}
						else
						{
							hexFileName = ControllerManager.DocumentModel.GetSramHexFileName((ChipTypes)device.ChipType);
						}
						if (WirteHEXToSRAM(isCommandPromt, device, hexFileName))
						{
							if (device.SetMode(WorkingModes.ExecuteOutOfSram))
							{
			  					ControllerManager.ShowMessage(Resources.MsgFinished, false, isCommandPrompt, !isCommandPrompt);
							}
							else
							{
								ControllerManager.ShowMessage(Resources.ErrorCantSetChipWorkingMode, true, isCommandPrompt);
							}
						}
					}
					else
					{
						ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true, isCommandPrompt);
					}
					device.SetProgrammingMode(false);
				}
				else
				{
					ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true, isCommandPrompt);
				}
			}
			catch
			{
				ControllerManager.ShowMessage(Resources.ErrorCantWriteSram, true, isCommandPrompt);
			}
			finally
			{
				ControllerManager.DeviceClose();
			}*/
		}

		/// <summary>
		/// Called when Set Lock Bits click.
		/// </summary>
		/// <param name="sender">The sender.</param>
		/// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
		public void OnLockBitsSetClick(object sender, EventArgs e)
		{
			ControllerManager.Actions.ProgrammerActions.OnLockBitsSetZW040x(
				ControllerManager.ZW040xForm.DisableReadBackCheckBox.Checked,
				ControllerManager.ZW040xForm.DisableDevModeCheckBox.Checked,
				ControllerManager.ZW040xForm.SetModemCheckBox.Checked);
		}

		/// <summary>
		/// Called when Read Lock Bits click.
		/// </summary>
		/// <param name="sender">The sender.</param>
		/// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
		public void OnLockBitsReadClick(object sender, EventArgs e)
		{
			bool disableReadBack, disableDevMode, setModem;

			ControllerManager.Actions.ProgrammerActions.OnLockBitsReadZW040x(
				out disableReadBack, out disableDevMode, out setModem);

			ControllerManager.ZW040xForm.DisableReadBackCheckBox.Checked = disableReadBack;
			ControllerManager.ZW040xForm.DisableDevModeCheckBox.Checked = disableDevMode;
			ControllerManager.ZW040xForm.SetModemCheckBox.Checked = setModem;
		}


        /// <summary>
        /// Called when DocumentModel state changed.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnDocumentModelStateChanged(object sender, EventArgs e)
        {
            OnDocumentModelStateChanged();
        }

        private delegate void OnDocumentModelStateChangedDelegate();
        private void OnDocumentModelStateChanged()
        {
            if (ControllerManager.ZW040xForm != null && !ControllerManager.ZW040xForm.IsDisposed)
            {

                if (ControllerManager.ZW040xForm.InvokeRequired)
                {
                    ControllerManager.ZW040xForm.Invoke(new OnDocumentModelStateChangedDelegate(OnDocumentModelStateChanged));
                }
                else
                {
                    ControllerManager.ZW040xForm.FlashProgramButton.Enabled =
                    ControllerManager.ZW040xForm.FlashWriteButton.Enabled =
                    ControllerManager.ZW040xForm.FlashCompareButton.Enabled =
                        ControllerManager.DocumentModel.FlashHexFileSelected(ChipTypes.ZW040x);

                    ControllerManager.ZW040xForm.EepromProgramButton.Enabled =
                    ControllerManager.ZW040xForm.EepromCompareButton.Enabled =
                        ControllerManager.DocumentModel.EepromHexFileSelected(ChipTypes.ZW040x);

                    ControllerManager.ZW040xForm.SramCompareButton.Enabled =
                    ControllerManager.ZW040xForm.SramWriteAndRunModeButton.Enabled =
                    ControllerManager.ZW040xForm.SramWriteButton.Enabled =
                        ControllerManager.DocumentModel.SramHexFileSelected(ChipTypes.ZW040x);

					ControllerManager.ZW040xForm.MtpProgramButton.Enabled =
					ControllerManager.ZW040xForm.MtpCompareButton.Enabled =
						ControllerManager.DocumentModel.MtpHexFileSelected(ChipTypes.ZW040x);
				}
            }
        }

        public void OnKeyDown(object sender, System.Windows.Forms.KeyEventArgs e)
        {
            if (e.Control && e.KeyCode == Keys.O)
            {
                OnFlashBrowseHexFileClick(null, null);
            }
            if (e.Control && e.KeyCode == Keys.C)
            {
                ControllerManager.Actions.ZW040xFormActions.OnCrcCheckClick(null, null);
            }
            if (e.Control && e.Alt && e.KeyCode == Keys.R)
            {
                ControllerManager.Actions.ZW040xFormActions.OnSramReadClick(null, null);
            }
            if (e.Control && e.Alt && e.KeyCode == Keys.W)
            {
                ControllerManager.Actions.ZW040xFormActions.OnSramWriteClick(null, null);
            }
            if (e.Control && e.Alt && e.KeyCode == Keys.C)
            {
                ControllerManager.Actions.ZW040xFormActions.OnSramCompareClick(null, null);
            }
			if (e.Control && e.Alt && e.KeyCode == Keys.P)
			{
				ControllerManager.Actions.ZW040xFormActions.OnSramWriteAndRunModeClick(null, null);
			}
			if (e.Control && e.Alt && e.KeyCode == Keys.M)
			{
				ControllerManager.Actions.ZW040xFormActions.OnMtpReadClick(null, null);
			}
		}

		/// <summary>
		/// Return MTP Auto Erase flag, selected by user
		/// </summary>
		public bool IsMtpAutoEraseSelected()
		{
			bool result = false;
			if (ControllerManager.ZW040xForm != null)
			{
				result = ControllerManager.ZW040xForm.MtpAutoEraseCheckBox.Checked;
			}
			return result;
		}

		internal void OnMtpAutoEraseClick(object sender, EventArgs e)
		{
			if (sender is CheckBox)
			{
				Form form = ((CheckBox)sender).FindForm();
				if (form != null)
				{
					if (form is ZW040xForm)
					{
						if (IsMtpAutoEraseSelected())
						{
							ControllerManager.ShowMessage(Resources.MsgMtpAutoEraseNote, false);
						}
					}
				}
			}
		}

		/// <summary>
		/// Perform MTP Auto Erase, if needed.
		/// </summary>
		public bool MtpAutoErase(IDevice device)
		{
			bool result = true;
			if (IsMtpAutoEraseSelected())
			{
				device.Memory.SwitchMTP(true);
				device.Memory.FillMTP(Constants.MTP_BLANK_VALUE);
				device.Memory.SwitchMTP(false);
				result = device.SetProgrammingMode(true);
			}
			return result;
		}

		/// <summary>
		/// Called when MTP read click.
		/// </summary>
		/// <param name="sender">The sender.</param>
		/// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
		public void OnMtpReadClick(object sender, EventArgs e)
		{
			if (ControllerManager.DocumentModel.PortInfo != null)
			{
				ControllerManager.DoAction(new EventHandler(delegate
				{
					MtpRead(false, null);
				}), Resources.MsgReadMtpProgress, false, 0x00);

			}
			else
			{
				ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
			}
		}
		/// <summary>
		/// Reads the MTP.
		/// </summary>
		/// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
		/// <param name="fileName">Name of the file.</param>
		public void MtpRead(bool isCommandPrompt, string fileName)
		{
			if (isCommandPrompt)
			{
				ControllerManager.ShowMessage(Resources.MsgReadMtpProgress, false, isCommandPrompt);
			}
			byte[] mtpBuffer = null;
			IDevice device = ControllerManager.DeviceOpen();

			try
			{
				if (device.SetProgrammingMode(true))
				{
					device.ReadSignatureBits();
					if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
					{
						IDeviceLed busyLed = device.Leds.GetByType(LedTypes.Busy);
						if (busyLed != null)
						{
							busyLed.SetState(true);

							if (device.Memory.SwitchMTP(true))
							{
								if (!device.Memory.ReadMTP())
								{
									ControllerManager.ShowMessage(Resources.ErrorCantReadMtp, true, isCommandPrompt);
								}
								device.Memory.SwitchMTP(false);

								if (device.Memory.Buffer != null)
								{
									mtpBuffer = HexFileHelper.BlankArray(device.Memory.Buffer.Length, Constants.MTP_BLANK_VALUE);
									for (int i = 0; i < device.Memory.Buffer.Length; i++)
									{
										mtpBuffer[i] = device.Memory.Buffer[i];
									}
								}
							}
							else
							{
								ControllerManager.ShowMessage(Resources.ErrorCantInitMtp, true, isCommandPrompt);
							}

							busyLed.SetState(false);
						}
						else
						{
							ControllerManager.ShowMessage(Resources.ErrorCantReadMtp, true, isCommandPrompt);
						}
					}
					else
					{
						ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true, isCommandPrompt);
					}
				}
				else
				{
					ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true, isCommandPrompt);
				}
			}
			catch
			{
				device.Memory.SwitchMTP(false);
				ControllerManager.ShowMessage(Resources.ErrorCantReadMtp, true, isCommandPrompt);
			}
			finally
			{
				if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
					device.SetProgrammingMode(false);
				ControllerManager.DeviceClose();
			}
			if (mtpBuffer != null)
			{
				if (isCommandPrompt)
				{
					try
					{
						TextWriter writer = new StreamWriter(fileName);
						writer.Write(HexFileHelper.WriteIntelHexFile(mtpBuffer, 0, mtpBuffer.Length));
						writer.Close();
						ControllerManager.ShowMessage(Resources.MsgMtpReadFinished, false, isCommandPrompt);
					}
					catch (Exception ex)
					{
						Console.WriteLine(ex.Message);
					}
				}
				else
				{
					ControllerManager.ShowConsoleInfo(HexFileHelper.WriteIntelHexFile(mtpBuffer, 0, mtpBuffer.Length));
					ControllerManager.ShowMessage(Resources.MsgMtpReadFinished, false, false, true);
				}
			}
		}

		/// <summary>
		/// Called when MTP Erase click.
		/// </summary>
		/// <param name="sender">The sender.</param>
		/// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
		public void OnMtpEraseClick(object sender, EventArgs e)
		{
			if (ControllerManager.DocumentModel.PortInfo != null)
			{
				ControllerManager.DoAction(new EventHandler(delegate
				{
					MtpErase(false);
				}), Resources.MsgEraseMtpProgress, false, 0x00);
			}
			else
			{
				ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
			}
		}
		/// <summary>
		/// Erase the EEPROM.
		/// </summary>
		/// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
		public void MtpErase(bool isCommandPrompt)
		{
			if (isCommandPrompt)
			{
				ControllerManager.ShowMessage(Resources.MsgEraseMtpProgress, false, isCommandPrompt);
			}
			IDevice device = ControllerManager.DeviceOpen();
			try
			{
				if (device.SetProgrammingMode(true))
				{
					device.ReadSignatureBits();
					if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
					{
						IDeviceLed busyLed = device.Leds.GetByType(LedTypes.Busy);
						if (busyLed != null)
						{
							busyLed.SetState(true);

							if (device.Memory.SwitchMTP(true))
							{
								if (device.Memory.FillMTP(Constants.MTP_BLANK_VALUE))
								{
									ControllerManager.ShowMessage(Resources.MsgMtpEraseFinished, false, isCommandPrompt, true);
								}
								else
								{
									ControllerManager.ShowMessage(Resources.ErrorCantEraseMtp, true, isCommandPrompt);
								}
							}
							else
							{
								ControllerManager.ShowMessage(Resources.ErrorCantInitMtp, true, isCommandPrompt);
							}
							device.Memory.SwitchMTP(false);
							busyLed.SetState(false);
						}
					}
					else
					{
						ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true, isCommandPrompt);
					}
				}
				else
				{
					ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true, isCommandPrompt);
				}
			}
			catch
			{
				device.Memory.SwitchMTP(false);
				ControllerManager.ShowMessage(Resources.ErrorCantEraseMtp, true, isCommandPrompt);
			}
			finally
			{
				if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
					device.SetProgrammingMode(false);
				ControllerManager.DeviceClose();
			}
		}

		/// <summary>
		/// Called when MTP Program click.
		/// </summary>
		/// <param name="sender">The sender.</param>
		/// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
		public void OnMTPProgramClick(object sender, EventArgs e)
		{
			if (ControllerManager.DocumentModel.PortInfo != null)
			{
				ControllerManager.DoAction(new EventHandler(delegate
				{
					MtpProgram(false, null);
				}), Resources.MsgProgramMtpProgress, false, 0x00);

			}
			else
			{
				ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
			}

		}
		/// <summary>
		/// Program the Mtp.
		/// </summary>
		/// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
		/// <param name="fileName">Name of the file.</param>
		public void MtpProgram(bool isCommandPrompt, string fileName)
		{
			//_isCommandPrompt = isCommandPrompt;
			bool result = false;
			if (isCommandPrompt)
			{
				ControllerManager.ShowMessage(Resources.MsgProgramMtpProgress, false, isCommandPrompt);
			}
			IDevice device = ControllerManager.DeviceOpen();
			//ControllerManager.ChangeProgressInfo(ProgressStatuses.None, 0, 0, _isCommandPrompt);
			try
			{
				if (device.SetProgrammingMode(true))
				{
					device.ReadSignatureBits();
					if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
					{
						string hexFileName;
						if (isCommandPrompt)
						{
							hexFileName = fileName;
						}
						else
						{
							hexFileName = ControllerManager.DocumentModel.GetMtpHexFilePath((ChipTypes)device.ChipType);
						}
						if (!String.IsNullOrEmpty(hexFileName) && File.Exists(hexFileName))
						{
							byte[] mtpDataRaw = HexFileHelper.BlankArray((int)device.Memory.MtpSize, Constants.MTP_BLANK_VALUE);
							if (HexFileHelper.ReadIntelHexFile(hexFileName, mtpDataRaw, (int)device.Memory.MtpSize) == HexFileHelper.FILE_READ)
							{
								IDeviceLed busyLed = device.Leds.GetByType(LedTypes.Busy);
								if (busyLed != null)
								{
									busyLed.SetState(true);

									if (device.Memory.SwitchMTP(true))
									{
										//device.Memory.ProgressChanged += new Zensys.ZWave.Events.ProgressChangedEventHandler(OnDeviceMemoryProgressChanged);
										device.Memory.FillMTP(Constants.MTP_BLANK_VALUE);
										if (device.Memory.WriteMTP(mtpDataRaw, true))
										{
											result = true;
										}
										else
										{
											ControllerManager.ShowMessage(Resources.ErrorProgramMtpFailed, true, isCommandPrompt);
										}
										device.Memory.SwitchMTP(false);
										//device.Memory.ProgressChanged -= new Zensys.ZWave.Events.ProgressChangedEventHandler(OnDeviceMemoryProgressChanged);
									}
									else
									{
										ControllerManager.ShowMessage(Resources.ErrorCantInitMtp, true, isCommandPrompt);
									}

									busyLed.SetState(false);
								}
								else
								{
									ControllerManager.ShowMessage(Resources.ErrorCantProgramMtp, true, isCommandPrompt);
								}
							}
							else
							{
								ControllerManager.ShowMessage(Resources.ErrorHexFileReadingFailed, true, isCommandPrompt);
							}
						}
						else if (String.IsNullOrEmpty(hexFileName))
						{
							ControllerManager.ShowMessage(Resources.ErrorMtpHexFileNotSpecified, true, isCommandPrompt);
						}
                        else
                        {
                            ControllerManager.ShowMessage(String.Format("{0} does not exists.", hexFileName), true, isCommandPrompt);
                        }
					}
					else
					{
						ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true, isCommandPrompt);
					}
				}
				else
				{
					ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true, isCommandPrompt);
				}
			}
			catch
			{
				device.Memory.SwitchMTP(false);
				ControllerManager.ShowMessage(Resources.ErrorCantProgramMtp, true, isCommandPrompt);
			}
			finally
			{
				if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
					device.SetProgrammingMode(false);
				ControllerManager.DeviceClose();
			}
			if (result)
			{
				ControllerManager.ShowMessage(Resources.MsgMtpProgramFinished, false, isCommandPrompt, !isCommandPrompt);
			}
		}

		/// <summary>
		/// Called when MTP Compare click.
		/// </summary>
		/// <param name="sender">The sender.</param>
		/// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
		public void OnMTPCompareClick(object sender, EventArgs e)
		{
			bool compareResult = false;
			if (ControllerManager.DocumentModel.PortInfo != null)
			{
				ControllerManager.DoAction(new EventHandler(delegate
				{
					IDevice device = ControllerManager.DeviceOpen();
					try
					{
						if (device.SetProgrammingMode(true))
						{
							device.ReadSignatureBits();
							if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
							{
								string hexFileName = ControllerManager.DocumentModel.GetMtpHexFilePath((ChipTypes)device.ChipType);
								if (!String.IsNullOrEmpty(hexFileName) && File.Exists(hexFileName))
								{
									byte[] mtpDataRaw = HexFileHelper.BlankArray((int)device.Memory.MtpSize, Constants.MTP_BLANK_VALUE);
									if (HexFileHelper.ReadIntelHexFile(hexFileName, mtpDataRaw, (int)device.Memory.MtpSize) == HexFileHelper.FILE_READ)
									{
										IDeviceLed busyLed = device.Leds.GetByType(LedTypes.Busy);
										if (busyLed != null)
										{
											busyLed.SetState(true);

											if (device.Memory.SwitchMTP(true))
											{
												if (device.Memory.ReadMTP())
												{
													compareResult = device.Memory.Compare(mtpDataRaw);
												}
												else
												{
													ControllerManager.ShowMessage(Resources.ErrorCantReadMtp, true);
												}
												device.Memory.SwitchMTP(false);
												busyLed.SetState(false);
											}
											else
											{
												ControllerManager.ShowMessage(Resources.ErrorCantInitMtp, true);
											}
										}
										else
										{
											ControllerManager.ShowMessage(Resources.ErrorCantCompareMtp, true);
										}
									}
									else
									{
										ControllerManager.ShowMessage(Resources.ErrorHexFileReadingFailed, true);
									}
								}
								else if (String.IsNullOrEmpty(hexFileName))
								{
									ControllerManager.ShowMessage(Resources.ErrorMtpHexFileNotSpecified, true);
								}
                                else
                                {
                                    ControllerManager.ShowMessage(String.Format("{0} does not exists.", hexFileName), true);
                                }
							}
							else
							{
								ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true);
							}
						}
						else
						{
							ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true);
						}
					}
					catch
					{
						device.Memory.SwitchMTP(false);
						ControllerManager.ShowMessage(Resources.ErrorCantCompareMtp, true);
					}
					finally
					{
						if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
							device.SetProgrammingMode(false);
						ControllerManager.DeviceClose();
					}
				}), Resources.MsgReadMtpProgress, false, 0x00);

				if (compareResult)
				{
					ControllerManager.ShowMessage(Resources.MsgCompareMtpSuccesful, false);
				}
				else
				{
					ControllerManager.ShowMessage(Resources.ErrorCompareMtpFailed, true);
				}
			}
			else
			{
				ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
			}
		}





    }
}
