using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.Programmer.Classes;
using Zensys.ZWave.Programmer.Controllers;
using WeifenLuo.WinFormsUI.Docking;
using Zensys.ZWave.Programmer.Layout;
using Zensys.ZWave.Programmer.UI;
using System.Windows.Forms;
using Zensys.ZWave.Programmer.Properties;
using Zensys.ZWave.Enums;
using System.IO;
using Zensys.Framework;

namespace Zensys.ZWave.Programmer.Actions
{
    /// <summary>
    /// MainFormActions class. Contains Main Form functions.
    /// </summary>
    public class MainFormActions : BaseAction
    {
        /// <summary>
        /// Initializes a new instance of the <see cref="MainFormActions"/> class.
        /// </summary>
        /// <param name="controller">The controller.</param>
        public MainFormActions(ControllerManager controller)
            : base(controller)
        {

        }
        #region Form
        /// <summary>
        /// Called when Form load.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnFormLoad(object sender, EventArgs e)
        {
            LayoutManager.OnLoad(ControllerManager, DeserializeDockContent);
            LoadSettings();
			ControllerManager.DeviceInit();
        }

        /// <summary>
        /// Loads the Application settings.
        /// </summary>
        public void LoadSettings()
        {
            if (!String.IsNullOrEmpty(Settings.Default.LastUsedDevice))
            {
                Win32PnPEntityClass portInfo = ComputerSystemHardwareHelper.GetWin32PnPEntityClassSerialPortDevice(Settings.Default.LastUsedDevice);
                if (portInfo != null)
                {
                    ControllerManager.DocumentModel.PortInfo = portInfo;
                    if (Settings.Default.DetectTargetOnStartup == true)
                    {
                        OnDetectTargetClick(null, null);
                    }
                    else
                    {
                        ControllerManager.DocumentModel.SelectedChipType = Settings.Default.SelectedChipType;
                    }
                    if (ControllerManager.DocumentModel.SelectedChipType != (byte)ChipTypes.UNKNOWN)
                    {
                        ControllerManager.SetActiveViewByChipType((ChipTypes)ControllerManager.DocumentModel.SelectedChipType);
                    }
                    ControllerManager.MainForm.SelectedSerialPortStatusLabel.Text = "Serial Port: " + ControllerManager.DocumentModel.PortInfo.DeviceID;
                }
            }
            #region ZW010x
            if (ControllerManager.ZW010xForm != null && !ControllerManager.ZW010xForm.IsDisposed)
            {
                ControllerManager.ZW010xForm.FlashHexFileNameTextBox.Text = ControllerManager.DocumentModel.GetFlashHexFileName(ChipTypes.ZW010x);
                ControllerManager.ZW010xForm.EepromHexFileNameTextBox.Text = ControllerManager.DocumentModel.GetEepromHexFileName(ChipTypes.ZW010x);

                ControllerManager.ZW010xForm.AutoIncrementHomeIdCheckBox.Checked = Settings.Default.ZW010xAutoIncrementHomeId;
                ControllerManager.ZW010xForm.StartHomeIdTextBox.Text = Settings.Default.ZW010xAutoIncrementStartHomeId;
                ControllerManager.ZW010xForm.EndHomeIdTextBox.Text = Settings.Default.ZW010xAutoIncrementEndHomeId;
            }
            #endregion
            #region ZW020x
            if (ControllerManager.ZW020xForm != null && !ControllerManager.ZW020xForm.IsDisposed)
            {
                ControllerManager.ZW020xForm.FlashHexFileNameTextBox.Text = ControllerManager.DocumentModel.GetFlashHexFileName(ChipTypes.ZW020x);
                ControllerManager.ZW020xForm.EepromHexFileNameTextBox.Text = ControllerManager.DocumentModel.GetEepromHexFileName(ChipTypes.ZW020x);

                ControllerManager.ZW020xForm.AutoIncrementHomeIdCheckBox.Checked = Settings.Default.ZW020xAutoIncrementHomeId;
                ControllerManager.ZW020xForm.StartHomeIdTextBox.Text = Settings.Default.ZW020xAutoIncrementStartHomeId;
                ControllerManager.ZW020xForm.EndHomeIdTextBox.Text = Settings.Default.ZW020xAutoIncrementEndHomeId;
            }
            #endregion
            #region ZW030x
            if (ControllerManager.ZW030xForm != null && !ControllerManager.ZW030xForm.IsDisposed)
            {
                ControllerManager.ZW030xForm.FlashHexFileNameTextBox.Text = ControllerManager.DocumentModel.GetFlashHexFileName(ChipTypes.ZW030x);
                ControllerManager.ZW030xForm.EepromHexFileNameTextBox.Text = ControllerManager.DocumentModel.GetEepromHexFileName(ChipTypes.ZW030x);

                ControllerManager.ZW030xForm.AutoIncrementHomeIdCheckBox.Checked = Settings.Default.ZW030xAutoIncrementHomeId;
                ControllerManager.ZW030xForm.StartHomeIdTextBox.Text = Settings.Default.ZW030xAutoIncrementStartHomeId;
                ControllerManager.ZW030xForm.EndHomeIdTextBox.Text = Settings.Default.ZW030xAutoIncrementEndHomeId;
            }
            #endregion
            #region ZW040x
            if (ControllerManager.ZW040xForm != null && !ControllerManager.ZW040xForm.IsDisposed)
            {
                ControllerManager.ZW040xForm.FlashHexFileNameTextBox.Text = ControllerManager.DocumentModel.GetFlashHexFileName(ChipTypes.ZW040x);
                ControllerManager.ZW040xForm.EepromHexFileNameTextBox.Text = ControllerManager.DocumentModel.GetEepromHexFileName(ChipTypes.ZW040x);
				ControllerManager.ZW040xForm.SramHexFileNameTextBox.Text = ControllerManager.DocumentModel.GetSramHexFileName(ChipTypes.ZW040x);
				ControllerManager.ZW040xForm.MtpHexFileNameTextBox.Text = ControllerManager.DocumentModel.GetMtpHexFileName(ChipTypes.ZW040x);

				ControllerManager.ZW040xForm.AutoIncrementHomeIdCheckBox.Checked = !Settings.Default.ZW040xAutoIncrementHomeId;		//for issuing checkbox state changed handler, which disable unneeded controls.
				ControllerManager.ZW040xForm.AutoIncrementHomeIdCheckBox.Checked = Settings.Default.ZW040xAutoIncrementHomeId;
				ControllerManager.ZW040xForm.StartHomeIdTextBox.Text = Settings.Default.ZW040xAutoIncrementStartHomeId;
                ControllerManager.ZW040xForm.EndHomeIdTextBox.Text = Settings.Default.ZW040xAutoIncrementEndHomeId;

                ControllerManager.ZW040xForm.ModeDevelopmentRadioButton.Checked = (Settings.Default.ZW040xSramOperationMode == (int)WorkingModes.Development);
                ControllerManager.ZW040xForm.ModeExecOutOfSRAMRadioButton.Checked = (Settings.Default.ZW040xSramOperationMode == (int)WorkingModes.ExecuteOutOfSram);

				ControllerManager.ZW040xForm.MtpAutoEraseCheckBox.Checked = (Settings.Default.ZW040xMtpAutoErase);
            }
            #endregion
        }

        /// <summary>
        /// Called when Form closing.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnFormClosing(object sender, EventArgs e)
        {
			ControllerManager.DeviceDeinit();
            SaveSettings();
            LayoutManager.OnClose(ControllerManager);
        }

        /// <summary>
        /// Saves the Application settings.
        /// </summary>
        public void SaveSettings()
        {
            if (ControllerManager.ActiveView != null)
            {
                Settings.Default.SelectedChipType = ControllerManager.DocumentModel.SelectedChipType;
                Settings.Default.Save();
            }
            #region ZW010x
            if (ControllerManager.ZW010xForm != null && !ControllerManager.ZW010xForm.IsDisposed)
            {
                Settings.Default.ZW010xAutoIncrementHomeId = ControllerManager.ZW010xForm.AutoIncrementHomeIdCheckBox.Checked;
                Settings.Default.ZW010xAutoIncrementStartHomeId = ControllerManager.ZW010xForm.StartHomeIdTextBox.Text;
                Settings.Default.ZW010xAutoIncrementEndHomeId = ControllerManager.ZW010xForm.EndHomeIdTextBox.Text;
            }
            #endregion
            #region ZW020x
            if (ControllerManager.ZW020xForm != null && !ControllerManager.ZW020xForm.IsDisposed)
            {
                Settings.Default.ZW020xAutoIncrementHomeId = ControllerManager.ZW020xForm.AutoIncrementHomeIdCheckBox.Checked;
                Settings.Default.ZW020xAutoIncrementStartHomeId = ControllerManager.ZW020xForm.StartHomeIdTextBox.Text;
                Settings.Default.ZW020xAutoIncrementEndHomeId = ControllerManager.ZW020xForm.EndHomeIdTextBox.Text;
            }
            #endregion
            #region ZW030x
            if (ControllerManager.ZW030xForm != null && !ControllerManager.ZW030xForm.IsDisposed)
            {
                Settings.Default.ZW030xAutoIncrementHomeId = ControllerManager.ZW030xForm.AutoIncrementHomeIdCheckBox.Checked;
                Settings.Default.ZW030xAutoIncrementStartHomeId = ControllerManager.ZW030xForm.StartHomeIdTextBox.Text;
                Settings.Default.ZW030xAutoIncrementEndHomeId = ControllerManager.ZW030xForm.EndHomeIdTextBox.Text;
            }
            #endregion
            #region ZW040x
            if (ControllerManager.ZW040xForm != null && !ControllerManager.ZW040xForm.IsDisposed)
            {
                Settings.Default.ZW040xAutoIncrementHomeId = ControllerManager.ZW040xForm.AutoIncrementHomeIdCheckBox.Checked;
                Settings.Default.ZW040xAutoIncrementStartHomeId = ControllerManager.ZW040xForm.StartHomeIdTextBox.Text;
                Settings.Default.ZW040xAutoIncrementEndHomeId = ControllerManager.ZW040xForm.EndHomeIdTextBox.Text;
                if (ControllerManager.ZW040xForm.ModeDevelopmentRadioButton.Checked)
                    Settings.Default.ZW040xSramOperationMode = (int)WorkingModes.Development;
                else if (ControllerManager.ZW040xForm.ModeExecOutOfSRAMRadioButton.Checked)
                    Settings.Default.ZW040xSramOperationMode = (int)WorkingModes.ExecuteOutOfSram;
                else
					Settings.Default.ZW040xSramOperationMode = (int)WorkingModes.Development;
				Settings.Default.ZW040xMtpAutoErase = ControllerManager.ZW040xForm.MtpAutoEraseCheckBox.Checked;
            }
            #endregion
        }

        /// <summary>
        /// Called when Active Content changed.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnActiveContentChanged(object sender, EventArgs e)
        {
            if (sender is DockPanel && (sender as DockPanel).ActiveContent != null)
            {
                ControllerManager.ActiveView = (DockableView)(sender as DockPanel).ActiveContent;
                if (ControllerManager.ActiveView is ZW010xForm)
                {
                    if (ControllerManager.DocumentModel.SelectedChipType != (byte)ChipTypes.ZW010x)
                        ControllerManager.DocumentModel.SelectedChipType = (byte)ChipTypes.ZW010x;
                }
                else if (ControllerManager.ActiveView is ZW020xForm)
                {
                    if (ControllerManager.DocumentModel.SelectedChipType != (byte)ChipTypes.ZW020x)
                        ControllerManager.DocumentModel.SelectedChipType = (byte)ChipTypes.ZW020x;
                }
                else if (ControllerManager.ActiveView is ZW030xForm)
                {
                    if (ControllerManager.DocumentModel.SelectedChipType != (byte)ChipTypes.ZW030x)
                        ControllerManager.DocumentModel.SelectedChipType = (byte)ChipTypes.ZW030x;
                }
                else if (ControllerManager.ActiveView is ZW040xForm)
                {
                    if (ControllerManager.DocumentModel.SelectedChipType != (byte)ChipTypes.ZW040x)
                        ControllerManager.DocumentModel.SelectedChipType = (byte)ChipTypes.ZW040x;
                }
                ControllerManager.UpdateFrequencies();
            }
        }

        private IDockContent DeserializeDockContent(string stringId)
        {
            IDockContent ret = null;
            if (stringId == typeof(ZW010xForm).FullName)
            {
                ControllerManager.ZW010xForm = new ZW010xForm();
                ControllerManager.MainForm.ZW010xToolStripMenuItem.Checked = true;
                ret = ControllerManager.ZW010xForm;
            }
            else if (stringId == typeof(ZW020xForm).FullName)
            {
                ControllerManager.ZW020xForm = new ZW020xForm();
                ControllerManager.MainForm.ZW020xToolStripMenuItem.Checked = true;
                ret = ControllerManager.ZW020xForm;
            }
            else if (stringId == typeof(ZW030xForm).FullName)
            {
                ControllerManager.ZW030xForm = new ZW030xForm();
                ControllerManager.MainForm.ZW030xToolStripMenuItem.Checked = true;
                ret = ControllerManager.ZW030xForm;
            }
            else if (stringId == typeof(ZW040xForm).FullName)
            {
                ControllerManager.ZW040xForm = new ZW040xForm();
                ControllerManager.MainForm.ZW040xToolStripMenuItem.Checked = true;
                ret = ControllerManager.ZW040xForm;
            }
            else if (stringId == typeof(ConsoleForm).FullName)
            {
                ControllerManager.ConsoleForm = new ConsoleForm();
                ControllerManager.MainForm.ConsoleToolStripMenuItem.Checked = true;
                ret = ControllerManager.ConsoleForm;
            }
            else if (stringId == typeof(LogForm).FullName)
            {
                ControllerManager.LogForm = new LogForm();
                ControllerManager.MainForm.LogToolStripMenuItem.Checked = true;
                ret = ControllerManager.LogForm;
            }
            else
            {
                ret = new DockContent();
            }
            return ret;
        }
        #endregion

        #region View
        /// <summary>
        /// Called when ZW010x menu item click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnZW010xClick(object sender, EventArgs e)
        {
            ControllerManager.MainForm.Invoke(new EventHandler(delegate
            {
                if (ControllerManager.MainForm.ZW010xToolStripMenuItem.Checked)
                {
                    if (ControllerManager.ZW010xForm != null && !ControllerManager.ZW010xForm.IsDisposed)
                    {
                        ControllerManager.ZW010xForm.Activate();
                    }
                    else
                    {
                        ControllerManager.ZW010xForm = new ZW010xForm();
                        ControllerManager.ZW010xForm.Show(ControllerManager.MainForm.MainDockPanel);
                    }
                }
                else
                {
                    ControllerManager.ZW010xForm.Close();
                    ControllerManager.ZW010xForm = null;
                }
            }));

        }
        /// <summary>
        /// Called when ZW020x menu item click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnZW020xClick(object sender, EventArgs e)
        {
            ControllerManager.MainForm.Invoke(new EventHandler(delegate
            {
                if (ControllerManager.MainForm.ZW020xToolStripMenuItem.Checked)
                {
                    if (ControllerManager.ZW020xForm != null && !ControllerManager.ZW020xForm.IsDisposed)
                    {
                        ControllerManager.ZW020xForm.Activate();
                    }
                    else
                    {
                        ControllerManager.ZW020xForm = new ZW020xForm();
                        ControllerManager.ZW020xForm.Show(ControllerManager.MainForm.MainDockPanel);
                    }
                }
                else
                {
                    ControllerManager.ZW020xForm.Close();
                    ControllerManager.ZW020xForm = null;
                }
            }));
        }
        /// <summary>
        /// Called when ZW030x menu item click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnZW030xClick(object sender, EventArgs e)
        {
            ControllerManager.MainForm.Invoke(new EventHandler(delegate
            {
                if (ControllerManager.MainForm.ZW030xToolStripMenuItem.Checked)
                {
                    if (ControllerManager.ZW030xForm != null && !ControllerManager.ZW030xForm.IsDisposed)
                    {
                        ControllerManager.ZW030xForm.Activate();
                    }
                    else
                    {
                        ControllerManager.ZW030xForm = new ZW030xForm();
                        ControllerManager.ZW030xForm.Show(ControllerManager.MainForm.MainDockPanel);
                    }
                }
                else
                {
                    ControllerManager.ZW030xForm.Close();
                    ControllerManager.ZW030xForm = null;
                }
            }));

        }
        /// <summary>
        /// Called when ZW040x menu item click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnZW040xClick(object sender, EventArgs e)
        {
            ControllerManager.MainForm.Invoke(new EventHandler(delegate
            {
                if (ControllerManager.MainForm.ZW040xToolStripMenuItem.Checked)
                {
                    if (ControllerManager.ZW040xForm != null && !ControllerManager.ZW040xForm.IsDisposed)
                    {
                        ControllerManager.ZW040xForm.Activate();
                    }
                    else
                    {
                        ControllerManager.ZW040xForm = new ZW040xForm();
                        ControllerManager.ZW040xForm.Show(ControllerManager.MainForm.MainDockPanel);
                    }
                }
                else
                {
                    ControllerManager.ZW040xForm.Close();
                    ControllerManager.ZW040xForm = null;
                }
            }));
        }
        /// <summary>
        /// Called when Console menu item click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnConsoleClick(object sender, EventArgs e)
        {
            if (ControllerManager.MainForm.ConsoleToolStripMenuItem.Checked)
            {
                if (ControllerManager.ConsoleForm != null && !ControllerManager.ConsoleForm.IsDisposed)
                {
                    ControllerManager.ConsoleForm.Activate();
                }
                else
                {
                    ControllerManager.ConsoleForm = new ConsoleForm();
                    ControllerManager.ConsoleForm.Show(ControllerManager.MainForm.MainDockPanel);
                }
            }
            else
            {
                ControllerManager.ConsoleForm.Close();
                ControllerManager.ConsoleForm = null;
            }
        }
        /// <summary>
        /// Called when Log menu item click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnLogClick(object sender, EventArgs e)
        {
            if (ControllerManager.MainForm.LogToolStripMenuItem.Checked)
            {
                if (ControllerManager.LogForm != null && !ControllerManager.LogForm.IsDisposed)
                {
                    ControllerManager.LogForm.Activate();
                }
                else
                {
                    ControllerManager.LogForm = new LogForm();
                    ControllerManager.LogForm.Show(ControllerManager.MainForm.MainDockPanel);
                }
            }
            else
            {
                ControllerManager.LogForm.Close();
                ControllerManager.LogForm = null;
            }
        }

        /// <summary>
        /// Called when Exit menu item click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnExitClick(object sender, EventArgs e)
        {
            System.Windows.Forms.Application.Exit();
        }
        #endregion

        #region Other

        /// <summary>
        /// Called when Detect Target menu item click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnDetectTargetClick(object sender, EventArgs e)
        {
            ControllerManager.Actions.ProgrammerActions.DetectDevice();
        }
        /// <summary>
        /// Called when Calibrate menu item click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnCalibrateClick(object sender, EventArgs e)
        {
            ControllerManager.Actions.ProgrammerActions.CalibrateDevice();
        }
        
        /// <summary>
        /// Called when reset Z-Wave module menu item click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnResetZWaveModuleClick(object sender, EventArgs e)
        {
            ControllerManager.Actions.ProgrammerActions.ResetZWaveModule(false);
        }
		/// <summary>
		/// Called when Upgrade Firmware menu item click.
		/// </summary>
		/// <param name="sender">The sender.</param>
		/// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
		public void OnUpgradeFirmwareClick(object sender, EventArgs e)
		{
			ControllerManager.DoAction(new EventHandler(delegate
				{
					if (ControllerManager.Actions.ProgrammerActions.UpgradeLatestFirmware(false) > 0)
					{
						ControllerManager.ShowMessage(Resources.MsgFirmwareUpgradeNotNeeded, false, false);
					}
				}), Resources.MsgUpgradeFirmwareProgress, false, 0x00);
		}
		/// <summary>
		/// Called when Upgrade Firmware menu item click.
		/// </summary>
		/// <param name="sender">The sender.</param>
		/// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
		public void OnUploadFirmwareClick(object sender, EventArgs e)
		{
			string fileName = ControllerManager.Actions.CommonActions.BrowseFile();
			if (!String.IsNullOrEmpty(fileName) && File.Exists(fileName))
			{
				ControllerManager.DoAction(new EventHandler(delegate
					{
						int version = 0;
						int revision = 0;
						if (ControllerManager.Actions.ProgrammerActions.UpgradeFirmware(false, fileName))
						{
							if (ControllerManager.Actions.ProgrammerActions.GetCurrentFirmwareVersion(false, out version, out revision))
							{
								ControllerManager.ShowMessage(Resources.MsgFirmwareUpgradedSuccesful, false, false);
							}
						}
					}), Resources.MsgUpgradeFirmwareProgress, false, 0x00);
			}
			else if (!String.IsNullOrEmpty(fileName))
			{
				ControllerManager.ShowMessage(String.Format("{0} does not exists.", fileName), true, false);
			}
		}
		/// <summary>
        /// Called when Settings menu item click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnSettingsClick(object sender, EventArgs e)
        {
            ControllerManager.MainForm.Cursor = Cursors.WaitCursor;
            SettingsForm settingsForm = new SettingsForm();
            settingsForm.OnLoadInterfaces += new EventHandler(SettingsOnLoadInterfaces);
            settingsForm.CmdApply.Click += new EventHandler(SettingsApplyClick);
            settingsForm.MainTabControl.SelectedIndexChanged += new EventHandler(SettingsTabSelectedIndexChanged);
            settingsForm.SettingsPropertyGrid.SelectedObject = ControllerManager.DocumentModel.Settings;
            if (settingsForm.ShowDialog(ControllerManager.MainForm) == DialogResult.OK)
            {
                SaveSettings(settingsForm);
				ControllerManager.DeviceDeinit();
				ControllerManager.DeviceInit();
            }
            ControllerManager.UpdateFrequencies();
            ControllerManager.MainForm.Cursor = Cursors.Default;
        }

        private void SaveSettings(SettingsForm settingsForm)
        {
            ControllerManager.DocumentModel.PortInfo = settingsForm.SelectedPortInfo;
            ControllerManager.DocumentModel.Settings = (ProgrammerSettings)settingsForm.SettingsPropertyGrid.SelectedObject;
            ControllerManager.DocumentModel.Settings.Save();
            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                Settings.Default.LastUsedDevice = ControllerManager.DocumentModel.PortInfo.DeviceID;
                ControllerManager.MainForm.SelectedSerialPortStatusLabel.Text = "Serial Port: " + ControllerManager.DocumentModel.PortInfo.DeviceID;
            }
            else
            {
                Settings.Default.LastUsedDevice = "";
                ControllerManager.MainForm.SelectedSerialPortStatusLabel.Text = "Serial Port: None";
            }
            Settings.Default.Save();
        }

        void SettingsTabSelectedIndexChanged(object sender, EventArgs e)
        {
            SettingsForm settingsForm = (SettingsForm)(sender as TabControl).FindForm();
            if (settingsForm != null)
            {
                if (settingsForm.MainTabControl.SelectedIndex == 0)
                {
                    SettingsOnLoadInterfaces(settingsForm, EventArgs.Empty);
                }
            }

        }

        void SettingsApplyClick(object sender, EventArgs e)
        {
            SettingsForm settingsForm = (SettingsForm)(sender as Button).FindForm();
            if (settingsForm != null)
            {
                SaveSettings(settingsForm);
            }
        }
        private void SettingsOnLoadInterfaces(object sender, EventArgs e)
        {
            if (sender is SettingsForm)
            {
                List<Win32PnPEntityClass> interfaces = new List<Win32PnPEntityClass>();
                SettingsForm form = sender as SettingsForm;
                if (!String.IsNullOrEmpty(Settings.Default.InterfaceFilter))
                {
                    interfaces = ComputerSystemHardwareHelper.GetWin32PnPEntityClassSerialPortDevices(String.Format(" and Caption Like '%{0}%'", Settings.Default.InterfaceFilter));
                }
                else
                {
                    interfaces = ComputerSystemHardwareHelper.GetWin32PnPEntityClassSerialPortDevices();
                }
                form.InterfacesListBox.Items.Clear();
                form.NoCommInterfacesLabel.Text = "";
                if (interfaces != null && interfaces.Count > 0)
                {
                    form.InterfacesListBox.Enabled = true;
                    foreach (Win32PnPEntityClass serialPortInfo in interfaces)
                    {
                        if (!String.IsNullOrEmpty(Settings.Default.LastUsedDevice) && Settings.Default.LastUsedDevice == serialPortInfo.DeviceID)
                        {
                            form.SelectedPortInfo = serialPortInfo;
                            form.InterfacesListBox.Items.Add(new InterfaceWrapper(serialPortInfo), true);
                        }
                        else
                        {
                            form.InterfacesListBox.Items.Add(new InterfaceWrapper(serialPortInfo), false);
                        }
                    }
                }
                else
                {
                    form.InterfacesListBox.Enabled = false;
                    form.NoCommInterfacesLabel.Text = Resources.MsgNoCommInterfaces;
                }
            }
        }
        #endregion

        /// <summary>
        /// Called when DocumentModel state changed.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnDocumentModelStateChanged(object sender, EventArgs e)
        {
        }

        public void OnKeyDown(object sender, System.Windows.Forms.KeyEventArgs e)
        {
            if (e.KeyCode == Keys.F1) ActivateForm(ControllerManager.ZW010xForm);
            if (e.KeyCode == Keys.F2) ActivateForm(ControllerManager.ZW020xForm);
            if (e.KeyCode == Keys.F3) ActivateForm(ControllerManager.ZW030xForm);
            if (e.KeyCode == Keys.F4) ActivateForm(ControllerManager.ZW040xForm);
        }

        private void ActivateForm(DockableView form)
        {
            if (form != null && !form.IsDisposed)
            {
                form.Activate();
            }
        }
    }
}
