/******************************* dev_ctrl.c  *******************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Z-Wave, the wireless language.
 *
 *              Copyright (c) 2001
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 *
 * Description: Main routines of the simple Development controller...
 *
 * Author:   Henrik Holm
 *
 * Last Changed By:  $Author: efh $
 * Revision:         $Revision: 23833 $
 * Last Changed:     $Date: 2012-11-23 15:01:23 +0100 (fr, 23 nov 2012) $
 *
 ****************************************************************************/

/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/
#include <config_app.h>
#include <p_button.h>
#include <ZW_controller_api.h>
#include <ZW_uart_api.h>
#include <ZW_classcmd.h>
#include <eeprom.h>
#include <ZW_debug_api.h>
#include <ctrl_learn.h>

/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/

/* mainState states */
#define GROUP_CONTROL_STATE       0x00
#define INCLUDE_NODE_STATE        0x01
#define REMOVE_NODE_STATE         0x02
#define NEW_CONTROLLER_STATE      0x03
#define RESET_CONTROLLER_STATE    0x04
#define ASSIGN_ROUTE_STATE        0x05
#define CHECK_BUTTON_STATE        0x06
#define EXECUTE_STATE             0x07
#define DIMMING_STATE             0x08
#define LEARN_STATE               0x09
#define PWR_UP_STATE              0x0A
#define ERROR_STATE               0x0B
#define WAIT_FOR_PB_RELEASE_STATE 0x0C
#define WAIT_FOR_PB_DOWN_STATE    0x0D
#define WAIT_FOR_PB_PRESS_STATE   0x0E
#define WAIT_FOR_PB_DEPRESS_STATE 0x0F
#define RETURN_TO_IDLE_STATE      0x10
/* timeouts */
#define BUSY_BLINK_RATE   20    /* (10ms ticks) Blinking rate during busy states */
#define ADD_TO_GROUP_RATE 10    /* (10ms ticks) Blinking rate during dimming and node/s added to group */
#define DIM_END_TIMEOUT   200   /* (10ms ticks) Wait before assuming that dim set is done */
#define RESET_TIMEOUT     200   /* (10ms ticks) Wait before resetting the controller */
#define LED0_ONTIME       50    /* (10ms ticks) Keep LED0 on for this time after command completed */
#define REPL_TIMEOUT      200		/* (10ms ticks) Replication timeout - wait for this time period before aborting replication */

/* Macros for LED access */
/* Note that LED1 connection on ZW020x/ZW030x requires a strap from RXD to LED1 pin */
/* on development controller board */
#define LED0_OFF PIN_ON(LED0)
#define LED1_OFF PIN_ON(LED1)
#define LED0_ON  PIN_OFF(LED0)
#define LED1_ON  PIN_OFF(LED1)

/* Defines and macros used to update the groupState variable */
#define GROUP_ON_BIT        0x80
#define GROUP_DIR_BIT       0x40

/* Commands for the ControlGroup function */
#define TOGGLE    0x01
#define DIM_START 0x02
#define DIM_STOP  0x03
#define DIM_SET   0x04

#define GROUP_CTRL_ACTIVATION   IS_DOWN_PB0(glKey)
#define ADD_NODE_ACTIVATION     IS_DOWN_PB1(glKey)
#define RESET_NODE_ACTIVATION   IS_DOWN_PB2(glKey)

#ifndef ZW_DEBUG
#ifndef ZW_DEBUG_CMD
#ifndef ZW_DEBUG_PROTOCOL
#define ASSIGN_ROUTE_ACTIVATION IS_DOWN_PB3(glKey)
#define ADD_CTRL_ACTIVATION     IS_DOWN_PB4(glKey)
#define RESET_ACTIVATION        IS_DOWN_PB0(glKey)&&IS_DOWN_PB4(glKey)
#endif
#endif
#endif

/* When using the RS232 interface for debug PB4 jumper should be removed */
#if defined(ZW_DEBUG)||defined(ZW_DEBUG_CMD)
#define ASSIGN_ROUTE_ACTIVATION FALSE  /* No assign route when debug */
#define ADD_CTRL_ACTIVATION     IS_DOWN_PB3(glKey)
#define RESET_ACTIVATION        IS_DOWN_PB0(glKey)&&IS_DOWN_PB3(glKey)
#else
#ifdef ZW_DEBUG_PROTOCOL
/* When using the RS232 interface for debug PB4 jumper should be removed */
#define ASSIGN_ROUTE_ACTIVATION FALSE  /* No assign route when debug */
#define ADD_CTRL_ACTIVATION     IS_DOWN_PB3(glKey)
#define RESET_ACTIVATION        IS_DOWN_PB0(glKey)&&IS_DOWN_PB3(glKey)
#endif /* ZW_DEBUG_PROTOCOL */
#endif /* ZW_DEBUG */

#define UPDATE_TIMEOUT 100      /* 100 * 10 ms */
#define MIN_DELAY  (WORD)(15*60)   /* minimum delay for the route update to be started */
#define MAX_DELAY  (WORD)(30*60)

/* Binary Switch definitions (They were defined in ZW_classcmd.h earlier) */
#define SWITCHED_ON                                     0xFF
#define SWITCHED_OFF                                    0x00

/****************************************************************************/
/*                              PRIVATE DATA                                */
/****************************************************************************/
static IBYTE mainState;                /*State variable for the main loop*/
static IBYTE glKey;                    /*PushButton variable*/

static BYTE lastLearnedNodeType = 0;  /*Store the last learned node type*/
static BYTE lastLearnedNodeID = 0;
/* Separate handles for all timers.. This number can be cut down if RAM */
/* is getting low */
static BYTE led0TimerHandle = 0;
//static BYTE led1TimerHandle = 0;
static BYTE dimEndTimerHandle = 0;
static BYTE resetTimerHandle = 0;
static BYTE replicationTimerHandle = 0;

/*Keeps track of the state of the group (ON/OFF and DIM direction)*/
static IBYTE groupState;

/*Last set DIM level*/
static IBYTE groupLevel;

/*Number of nodes in the group. 0 if no nodes*/
static IBYTE nodesInGroup;

/*Index used to run through the groupNodeList*/
static IBYTE nextNode = 0;

/*Flag used to check if were waiting for a dim level report.*/
static BOOL receivedReport = TRUE;

/*Flag that keeps track of the LED0 Blinker*/
static BYTE goON;

/*Contains the node IDs and node types used when assigning a route*/
static BYTE routingNode, routeToNode;

/*Buffer used to hold commands*/
ZW_APPLICATION_TX_BUFFER txBuffer;

/*The command classes that this device supports besides the ones specified by
 its generic type*/
static BYTE supportedCmdClasses[2] = {COMMAND_CLASS_CONTROLLER_REPLICATION,COMMAND_CLASS_VERSION};

static IBYTE  staticCtrlNodeID = 0x00;   /* hold the node id of the static update controller*/
static IBYTE timerHandle = 0xFF;
static WORD timerCount = 0;
BOOL receivingRemote =FALSE;
BOOL bStartUpdateTimer = FALSE;

/*Global used when getting nodeinformation from protocol*/
static NODEINFO snodeInfo;

BOOL txInProgress = FALSE;

/* Group replication */
BYTE bOtherController = 0;

BYTE bNWIIsStarted = FALSE;
BYTE bNWIStartup = FALSE;

/****************************************************************************/
/*                              PRIVATE FUNCTIONS                           */
/****************************************************************************/
void ControlGroup(BYTE bCommand, void code *func(BYTE bStatus));
void TxCompleteIgnoreAck(BYTE bStatus);
void StopResetTimer(void);

static void SetSUCDone(BYTE txStaus);
static void RequestUpdateTimeOut();
//void NewControllerReceiveStarted(LEARN_INFO *learnNodeInfo);
void StopUpdateTimer(void);
void StopDimEndTimer(void);

void ReplicationSendGroup(BYTE bStatus);


/*============================   GroupSend  =================================
**    Function description
**      Sends out multicast message to the group. If no nodes are in the group
**      it will call the callback func with TRANSMIT_COMPLETE_NO_ACK.
**    Side effects:
**      updates RAMGroupList
**--------------------------------------------------------------------------*/
void
GroupSend(
  BYTE *pData,                /*IN  Data buffer pointer           */
  BYTE dataLength,            /*IN  Data buffer length            */
  BYTE txOptions,             /*IN  Transmit option flags         */
  BYTE bCommand,
  void (code *func)(BYTE))    /*IN  Transmit completed call back function  */
{
  /* TO#01645 fix  Make a group doesn't work correct. */
  static BYTE TxGroupMask[MAX_NODES/8];

  /* TO#1687 fix */
  if (bCommand == TOGGLE)
  {
    memcpy(TxGroupMask, groupMask, MAX_NODES/8);
  }
  else
  {
    memcpy(TxGroupMask, groupMask, MAX_NODES/8);
    nodesInGroup = GetGroupListDimmers(TxGroupMask);  /* Strip binary switches from TxGroupMask */
  }
  /* TO#1687*/

  if (nodesInGroup)
  {
    if (!ZW_SEND_DATA_MULTI(TxGroupMask, pData, dataLength, txOptions, func))
    {
      func(TRANSMIT_COMPLETE_NO_ACK);
    }
  }
  else
  {
    /* TO#1965 fix */
    func(TRANSMIT_COMPLETE_NO_ACK);
  }
}


/*============================   LED0Blinker   ==============================
**    Function description
**      Handles the blinking of LED0
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
LED0Blinker(void)
{
  if (goON = !goON)
  {
    LED0_OFF;
  }
  else
  {
    LED0_ON;
  }
}


/*============================   StopLED0Blink   =============================
**    Function description
**      Stops the LED0Blinker timer and turns of LED0
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
StopLED0Blink(void)
{
  LED0_OFF;
  if (led0TimerHandle)
  {
    ZW_TIMER_CANCEL(led0TimerHandle);
  }
  led0TimerHandle = 0;
}


/*============================   StartLED0Blink   ===========================
**    Function description
**      (Re)starts the timer that runs LED0 blinking..
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
StartLED0Blink(
  BYTE blinkRate)       /*IN blink rate in 10ms*/
{
  StopLED0Blink();
  goON = TRUE;    /*Allways start with LED on*/
  led0TimerHandle = ZW_TIMER_START(LED0Blinker, blinkRate, TIMER_FOREVER);
}

/*============================   SetGroupDimLevel   ==========================
**    Function description
**      Sets the dim level of the nodes in the group to the last groupDimLevel
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
SetGroupDimLevel(void)
{
  StopDimEndTimer();
  ControlGroup(DIM_SET,TxCompleteIgnoreAck);
}


/*============================   StopDimEndTimer   ======================
**    Function description
**      Stops the timeout function that is run when Dimming is ended..
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
StopDimEndTimer(void)
{
  if (dimEndTimerHandle)
  {
    ZW_TIMER_CANCEL(dimEndTimerHandle);
  }
  dimEndTimerHandle = 0;
}


/*============================   StartDimEndTimer   ======================
**    Function description
**      Starts the timeout that is running when dim is ended
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
StartDimEndTimer(void)
{
  StopDimEndTimer();
  dimEndTimerHandle = ZW_TIMER_START(SetGroupDimLevel, DIM_END_TIMEOUT, TIMER_ONE_TIME);
}


/*=========================   RemoveNodeCompleted   ========================
**    Function description
**        Callback function, used when removing switches from the network.
**          bStatus BYTE,     IN  Status of learn process
**          bSource BYTE,     IN  Node id of the node that send node info
**          BYTE* pCmd,       IN  Pointer to Node information
**          BYTE bLen));      IN  Node info length
**
**    Side effects: Shuts down RF receive mode when completed and removes the
**                  node from the group.
**--------------------------------------------------------------------------*/
void
RemoveNodeCompleted(
  LEARN_INFO *learnNodeInfo)
{
  ZW_DEBUG_SEND_BYTE('r');
  ZW_DEBUG_SEND_NUM(learnNodeInfo->bStatus);
  ZW_DEBUG_SEND_BYTE('.');
  ZW_DEBUG_SEND_NUM((*learnNodeInfo).bSource);

  if ((*learnNodeInfo).bStatus == REMOVE_NODE_STATUS_NODE_FOUND)
  {
    ZW_DEBUG_SEND_BYTE('P');
    StartLED0Blink(BUSY_BLINK_RATE);
    mainState = EXECUTE_STATE;
  }
  else if (((*learnNodeInfo).bStatus == REMOVE_NODE_STATUS_REMOVING_SLAVE)||
           ((*learnNodeInfo).bStatus == REMOVE_NODE_STATUS_REMOVING_CONTROLLER))
  {
    RemoveNodeFromGroup((*learnNodeInfo).bSource);
  }
  else if ((learnNodeInfo->bStatus == REMOVE_NODE_STATUS_DONE) ||
           (learnNodeInfo->bStatus == REMOVE_NODE_STATUS_FAILED) )
  {
    ZW_DEBUG_SEND_BYTE('D');

    ZW_REMOVE_NODE_FROM_NETWORK(REMOVE_NODE_STOP, NULL);

    mainState = WAIT_FOR_PB_RELEASE_STATE;
    ZW_TIMER_START(StopLED0Blink,LED0_ONTIME,TIMER_ONE_TIME);
    if (((*learnNodeInfo).bStatus == ADD_NODE_STATUS_FAILED))
    {
      LED1_ON;
    }
  }
}

/*=======================   AssociationAddToGroupComplete   ==================
**    Function description
**      Called when Association add to group has completed
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
AssociationAddToGroupComplete(
  BYTE bStatus)             /*Transmit status*/
{
  ZW_TIMER_START(StopLED0Blink, LED0_ONTIME, TIMER_ONE_TIME);
  ZW_DEBUG_SEND_BYTE('c');
  LED1_OFF;
  if (bStatus != TRANSMIT_COMPLETE_OK)
  {
    mainState = ERROR_STATE;
  }
  else
  {
    mainState = RETURN_TO_IDLE_STATE;
  }
}


/*============================   AssociationAddToGroup   =====================
**    Function description
**    Callback function called when Assign Return Route has finished. This
**    function sends an Association Set frame to the routing slave node
**    routingNode, which inserts the routeToNode node into association group 1.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                    /*RET Nothing */
AssociationAddToGroup(
  BYTE bStatus)         /* IN Status of assign return route */
{
  if (bStatus == TRANSMIT_COMPLETE_OK)
  {
    ZW_DEBUG_SEND_BYTE('9');
    txBuffer.ZW_AssociationSet1byteFrame.cmdClass = COMMAND_CLASS_ASSOCIATION;
    txBuffer.ZW_AssociationSet1byteFrame.cmd = ASSOCIATION_SET;
    txBuffer.ZW_AssociationSet1byteFrame.groupingIdentifier = 1;
    txBuffer.ZW_AssociationSet1byteFrame.nodeId1 = routeToNode;
    if (!ZW_SEND_DATA(routingNode, (BYTE *)&txBuffer,
                      sizeof(ZW_ASSOCIATION_SET_1BYTE_FRAME),
                      TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_EXPLORE, AssociationAddToGroupComplete))
    {
      AssociationAddToGroupComplete(TRANSMIT_COMPLETE_NO_ACK);
    }
  }
  else
  {
    /* Assign Return Route failed somehow - skip Association Set and report */
    AssociationAddToGroupComplete(TRANSMIT_COMPLETE_NO_ACK);
  }
}


/*========================   AssignRouteNodeCompleted   ======================
**    Function description
**    Callback function for Assign route. This function fetches the node ids.
**
**          bStatus BYTE,     IN  Status of learn process
**          bSource BYTE,     IN  Node id of the node that send node info
**          BYTE* pCmd,       IN  Pointer to Node information
**          BYTE bLen));      IN  Node info length
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
AssignRouteNodeCompleted(
  LEARN_INFO *learnNodeInfo)  /*IN Status*/
{
  if ((*learnNodeInfo).bLen)
  {
    /*Only store ids and info when bLen is not Zero*/
    lastLearnedNodeType = ((NODE_TYPE *)(*learnNodeInfo).pCmd)->generic;
    lastLearnedNodeID = (*learnNodeInfo).bSource;
  }
  if ((*learnNodeInfo).bStatus == ADD_NODE_STATUS_LEARN_READY)
  {
    /*If Button is pressed exit learnmode*/
    mainState = WAIT_FOR_PB_DOWN_STATE;
  }
  else if ((*learnNodeInfo).bStatus == ADD_NODE_STATUS_ADDING_SLAVE)
  {
    mainState = EXECUTE_STATE;
    if (!routeToNode)
    {
      /* This node ID is the device that calculated routes should be to */
      routeToNode = lastLearnedNodeID;
    }
    else if (!routingNode)
    {
      /* This node ID is the device that routes should be sent to*/
      routingNode = lastLearnedNodeID;
    }
    StartLED0Blink(BUSY_BLINK_RATE);
  }
  else if (((*learnNodeInfo).bStatus == ADD_NODE_STATUS_PROTOCOL_DONE))
  {
    /*Stop learnmode. Handle completed in Done*/
    ZW_ADD_NODE_TO_NETWORK(ADD_NODE_STOP, AssignRouteNodeCompleted);
  }
  else if (((*learnNodeInfo).bStatus == ADD_NODE_STATUS_DONE))
  {
    /*If either is zero go back and look for new*/
    if (!routingNode)
    {
      StopLED0Blink();
      LED1_ON;
      ZW_ADD_NODE_TO_NETWORK(ADD_NODE_EXISTING, AssignRouteNodeCompleted);
    }
    else
    {
      /* This node is the node that should receive the routes */
      StartLED0Blink(BUSY_BLINK_RATE);
      ZW_DEBUG_SEND_BYTE('7');
      ZW_ADD_NODE_TO_NETWORK(ADD_NODE_STOP, NULL);

      if (ZW_ASSIGN_RETURN_ROUTE(routingNode, routeToNode, AssociationAddToGroup))
      {
        mainState = EXECUTE_STATE;
      }
      else
      {
        StopLED0Blink();
        mainState = ERROR_STATE;
      }
    }
  }
  else if ((*learnNodeInfo).bStatus == ADD_NODE_STATUS_FAILED)
  {
    StopLED0Blink();
    mainState = ERROR_STATE;
  }
}

/*============================   ControlGroup   ======================
**    Function description
**      Executes a group command. This can be one of the following:
**        TOGGLE
**        DIM_START
**        DIM_STOP
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
ControlGroup(                     /*RET Nothing*/
  BYTE bCommand,                  /* IN What to do*/
  void code *func(BYTE bStatus))  /* IN Callback function.*/
{
  static BYTE bRouting, length;

  nodesInGroup = UpdateRAMGroupList();

  nextNode = 0;
  switch(bCommand)
  {
    case TOGGLE:
      /* We use Explore frame Route Resolution if all alse fails */
      bRouting = TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_EXPLORE;
      txBuffer.ZW_BasicSetFrame.cmdClass = COMMAND_CLASS_BASIC;
      txBuffer.ZW_BasicSetFrame.cmd = BASIC_SET;
      groupState ^= GROUP_ON_BIT;  /* Toggle */
      if (groupState & GROUP_ON_BIT)
      {
        txBuffer.ZW_BasicSetFrame.value = SWITCHED_ON;   /* Switch ON */
      }
      else
      {
        txBuffer.ZW_BasicSetFrame.value = SWITCHED_OFF;  /* Switch OFF */
      }
      length = sizeof(ZW_BASIC_SET_FRAME);
      break; /*TOGGLE*/

    case DIM_START:
      bRouting = TRANSMIT_OPTION_ACK;
      txBuffer.ZW_SwitchMultilevelStartLevelChangeFrame.cmdClass = COMMAND_CLASS_SWITCH_MULTILEVEL;
      txBuffer.ZW_SwitchMultilevelStartLevelChangeFrame.cmd = SWITCH_MULTILEVEL_START_LEVEL_CHANGE;
      /* Set dim direction in command and toggle direction bit in state */
      txBuffer.ZW_SwitchMultilevelStartLevelChangeFrame.level = groupState & GROUP_DIR_BIT;
      groupState ^= GROUP_DIR_BIT;
      txBuffer.ZW_SwitchMultilevelStartLevelChangeFrame.startLevel = groupLevel;
      length = sizeof(ZW_SWITCH_MULTILEVEL_START_LEVEL_CHANGE_FRAME);
      break; /*DIM_START*/

    case DIM_STOP:
      receivedReport = FALSE;
      bRouting = TRANSMIT_OPTION_ACK;
      txBuffer.ZW_SwitchMultilevelStopLevelChangeFrame.cmdClass = COMMAND_CLASS_SWITCH_MULTILEVEL;
      txBuffer.ZW_SwitchMultilevelStopLevelChangeFrame.cmd = SWITCH_MULTILEVEL_STOP_LEVEL_CHANGE;
      length = sizeof(ZW_SWITCH_MULTILEVEL_STOP_LEVEL_CHANGE_FRAME);
      break; /*DIM_STOP*/

    case DIM_SET:
      /* Do routing because we want all lamps to turn on/off */
      /* We use Explore frame Route Resolution if all alse fails */
      bRouting = TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_EXPLORE;
      txBuffer.ZW_SwitchMultilevelSetFrame.cmdClass = COMMAND_CLASS_SWITCH_MULTILEVEL;
      txBuffer.ZW_SwitchMultilevelSetFrame.cmd = SWITCH_MULTILEVEL_SET;
      txBuffer.ZW_SwitchMultilevelSetFrame.value = groupLevel;
      length = sizeof(ZW_SWITCH_MULTILEVEL_SET_FRAME);
      break;
  }
  GroupSend((BYTE *)&txBuffer, length, bRouting, bCommand, func);
}


/*============================   DimStopped   ===============================
**    Function description
**      Dim stop has been received. This function fetches a DIM level
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
DimStopped(
  BYTE bStatus)
{
  /*We do not care if the dim stop was succesfull or not. We want a level from a switch in the group*/
  BYTE temp = nextNode;
  ZW_DEBUG_SEND_BYTE(',');
  ZW_DEBUG_SEND_NUM(nodesInGroup);
  /* Any nodes at all? */
  if (!nodesInGroup)
  {
    mainState = ERROR_STATE;
    return;
  }

  /* Locate multilevel switches only */
  do
	{
	  temp = ZW_NodeMaskGetNextNode(temp, groupMask);
	  if (temp > 0)
	  {
      ZW_GET_NODE_STATE(temp, &snodeInfo);
    }
  } while ((temp > 0) && (snodeInfo.nodeType.generic != GENERIC_TYPE_SWITCH_MULTILEVEL));

  if (temp > 0)   /* If we have located next multilevel switch node */
  {
    txBuffer.ZW_SwitchMultilevelGetFrame.cmdClass = COMMAND_CLASS_SWITCH_MULTILEVEL;
    txBuffer.ZW_SwitchMultilevelGetFrame.cmd = SWITCH_MULTILEVEL_GET;
    if ((!nextNode) ||  /* first node In list */
        ((bStatus != TRANSMIT_COMPLETE_OK)))
    {
      nextNode = temp;
      ZW_DEBUG_SEND_NUM(nextNode);
      ZW_SEND_DATA(nextNode, (BYTE *)&txBuffer,
                   sizeof(ZW_SWITCH_MULTILEVEL_GET_FRAME), TRANSMIT_OPTION_ACK, DimStopped);
    }
    else
    {
      /* A node acked, or we timed out.. Wait for a response. */
      StartDimEndTimer();
    }
  }
  else
  {
    /* No node acked, or we timed out.. Wait for a response. */
    StartDimEndTimer();
  }
}


/*============================   TxCompleteCheckAck   ======================
**    Function description
**      This function checks transmit status and indicates if any
**      error occoured
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
TxCompleteCheckAck(
  BYTE bStatus)             /*IN Transmit status*/
{
  txInProgress = FALSE;

  ZW_TIMER_START(StopLED0Blink, LED0_ONTIME, TIMER_ONE_TIME);
  if (bStatus != TRANSMIT_COMPLETE_OK)
  {
    mainState = ERROR_STATE;
  }
  else
  {
    mainState = RETURN_TO_IDLE_STATE;
  }
}


/*============================   TxCompleteIgnoreAck   ======================
**    Function description
**      Callback function that ignores Transmit status and returns to idle state
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
TxCompleteIgnoreAck(
  BYTE bStatus)             /*IN Transmit status*/
{
  ZW_TIMER_START(StopLED0Blink, LED0_ONTIME, TIMER_ONE_TIME);
  mainState = RETURN_TO_IDLE_STATE;
}


/*============================   ToggleGroup   ==============================
**    Function description
**      Toggles the group on or off
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
ToggleGroup(void)
{
  /*Toggle state of the switches*/
  StartLED0Blink(BUSY_BLINK_RATE);
  ControlGroup(TOGGLE,TxCompleteCheckAck);
}


/*============================   DimStarted   ===============================
**    Function description
**        Call back called when Dim has been started. Ignores transmit status
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
DimStarted(
  BYTE bStatus)       /*IN Transmit status*/
{
  ZW_DEBUG_SEND_BYTE('s');
  mainState = DIMMING_STATE;
}


/*============================   ZW_ResetDone   ======================
**    Function description
**      Restarts from PWR_UP_STATE
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
ZW_ResetDone(void)
{
  mainState = PWR_UP_STATE;
  StopUpdateTimer();
  ZW_TIMER_START(StopLED0Blink,LED0_ONTIME,TIMER_ONE_TIME);
}


/*============================   ApplicationResetDone   ======================
**    Function description
**        Starts Protocol reset
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
ApplicationResetDone(void)
{
  ZW_SET_DEFAULT(ZW_ResetDone);
}


/*==================================   DoReset   ============================
**    Function description
**      Starts controller reset..
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
DoReset(void)
{
    StartLED0Blink(BUSY_BLINK_RATE);
    LED1_ON;
    /*Make sure to exit any learnstate before reseting*/

    ZW_DEBUG_SEND_BYTE('8');
    ZW_ADD_NODE_TO_NETWORK(ADD_NODE_STOP, NULL);
    // ZW_SetLearnMode(FALSE, NULL);

    staticCtrlNodeID = DEFAULT_SUC_NODE_ID;
    ClearGroupTable(ApplicationResetDone);
    mainState = EXECUTE_STATE;
}


/*============================   StopResetTimer  ===========================
**    Function description
**      Stops the reset activate timeout
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
StopResetTimer(void)
{
  if(resetTimerHandle)
    ZW_TIMER_CANCEL(resetTimerHandle);
}

/*============================   StartResetTimer   ==========================
**    Function description
**      Starts a timeout. When it runs out the controller will be reset.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
StartResetTimer(void)
{
  StopResetTimer();
  resetTimerHandle = ZW_TIMER_START(DoReset,RESET_TIMEOUT,TIMER_ONE_TIME);
}


/*===========================   GetRandomNodeTime   ===========================
 *
 *  Function description
 *    Get a pseudo-random time
 *
 *  Side effects:
 *
 *--------------------------------------------------------------------------*/
static WORD                    /*RET A pseudo random nodeID */
GetRandomTime( void ) /* IN Nothing */
{
  WORD  delay;
  delay = MIN_DELAY + (ZW_RANDOM()*(((MAX_DELAY-MIN_DELAY)/255)+1));
  if(delay > MAX_DELAY)
    delay = MAX_DELAY;
  return delay;
}

/*============================   StopUpdateTimer   ======================
**    Function description
**      Used to stop the request update timer.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
StopUpdateTimer(void)
{
  if(timerHandle!=0xff)
  {
    ZW_TIMER_CANCEL(timerHandle);
    timerHandle = 0xff;
  }
}


/*============================   StartUpdateTimer   =========================
**    Function description
**      Starts a random timer that transmits update requests to the SUC
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
StartUpdateTimer(void)
{
  StopUpdateTimer();
  timerHandle = ZW_TIMER_START(RequestUpdateTimeOut,UPDATE_TIMEOUT, TIMER_FOREVER);
  if (timerHandle != 0xFF)
  {
    timerCount = GetRandomTime();
    bStartUpdateTimer = FALSE;
  }
  else
  {
    bStartUpdateTimer = TRUE;
  }
}


/*============================   SetSucNodeDelay   =========================
**    Function description
**      Timer call back used to delay the setSUCNodeID function to give both
**       controllers (Tx and RX) to analyze the routing table.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
SetSucNodeDelay( void)
{
  /* TO#1825 fix */
  if (!ZW_SET_SUC_NODEID(staticCtrlNodeID, TRUE, FALSE, ZW_SUC_FUNC_NODEID_SERVER, SetSUCDone))
  {
    SetSUCDone(ZW_SUC_SET_FAILED);
  }
}


/*============================   NewControllerAllDone   ======================
**    Function description
**        Called when replication is all done
**        BYTE bStatus,   IN  Status of learn process
**        BYTE bSource,  IN  Node id of the node that send node info
**        BYTE* pCmd,    IN  Pointer to Node information
**        BYTE bLen.     IN  Node info length

**    Side effects:
**
**--------------------------------------------------------------------------*/
void
NewControllerAllDone(
  LEARN_INFO *learnNodeInfo)    /*IN Learn node info*/
{
  ZW_DEBUG_SEND_BYTE('a');
  ZW_DEBUG_SEND_NUM((*learnNodeInfo).bSource);
  ZW_DEBUG_SEND_BYTE('-');
  ZW_DEBUG_SEND_NUM((*learnNodeInfo).bStatus);
  ZW_DEBUG_SEND_BYTE('-');
  ZW_DEBUG_SEND_NUM((*learnNodeInfo).bLen);
  ZW_GET_NODE_STATE((*learnNodeInfo).bSource, &snodeInfo);
  staticCtrlNodeID = ZW_GET_SUC_NODEID();
/*#if 0*/  /* TO#01646 fix  Development can't set SUC/SIS. */
  if (snodeInfo.nodeType.basic == BASIC_TYPE_STATIC_CONTROLLER)
  /*If learnNodeInfo is not zero and equal the static ctrlID then assign it as SUC*/
  {
    if (!staticCtrlNodeID || (staticCtrlNodeID == (*learnNodeInfo).bSource))
    {
      staticCtrlNodeID = (*learnNodeInfo).bSource;
      mainState = EXECUTE_STATE; /*Do not allow user to interrupt now*/
      if (ZW_TIMER_START(SetSucNodeDelay, 100, TIMER_ONE_TIME) == 0xFF)
      {
        SetSUCDone(ZW_SUC_SET_FAILED);
      }
      return;
    }
  }
/*#endif*/  /* TO#01646 fix  Development can't set SUC/SIS. */
  ZW_TIMER_START(StopLED0Blink, LED0_ONTIME, TIMER_ONE_TIME);
  mainState = WAIT_FOR_PB_RELEASE_STATE;
}


/*============================   NewControllerSendStarted   ==================
**    Function description
**        Called by the protocol as the replication progresses.
**        When NEW_CONTROLLER_LEARNED has been received the user can not abort
**        BYTE bStatus,   IN  Status of learn process
**        BYTE bSource,  IN  Node id of the node that send node info
**        BYTE* pCmd,    IN  Pointer to Node information
**        BYTE bLen.     IN  Node info length

**    Side effects:
**
**--------------------------------------------------------------------------*/
void
NewControllerSendStarted(
  LEARN_INFO *learnNodeInfo)
{
  ZW_DEBUG_SEND_BYTE('@');
  ZW_DEBUG_SEND_NUM((*learnNodeInfo).bStatus);
  ZW_DEBUG_SEND_BYTE('-');
  ZW_DEBUG_SEND_NUM((*learnNodeInfo).bSource);
  ZW_DEBUG_SEND_BYTE('-');
  ZW_DEBUG_SEND_NUM((*learnNodeInfo).bLen);
  if (learnNodeInfo->bStatus == ADD_NODE_STATUS_LEARN_READY)
  {
    /*We are ready to add a node*/
    StopLED0Blink();
    LED0_ON;
    mainState = LEARN_STATE;
    /* Used for group replication */
    bOtherController = 0;
  }
  else if ((learnNodeInfo->bStatus == ADD_NODE_STATUS_NODE_FOUND) ||
           (learnNodeInfo->bStatus == ADD_NODE_STATUS_ADDING_SLAVE) ||
           (learnNodeInfo->bStatus == ADD_NODE_STATUS_ADDING_CONTROLLER))
  {
    /* Store controller node id for group replication */
    if (learnNodeInfo->bStatus == ADD_NODE_STATUS_ADDING_CONTROLLER)
    {
      bOtherController = learnNodeInfo->bSource;
    }

    StartLED0Blink(BUSY_BLINK_RATE);
    mainState = EXECUTE_STATE;
  }
  else if (learnNodeInfo->bStatus == ADD_NODE_STATUS_PROTOCOL_DONE)
  {
    ZW_DEBUG_SEND_BYTE('1');


    /* If it was a controller we added then start group replication */
    if (bOtherController)
    {
      nextNode = 1;
      ReplicationSendGroup(TRANSMIT_COMPLETE_OK);
    }
    else  /* If not a controller then just stop */
    {
      ZW_ADD_NODE_TO_NETWORK(ADD_NODE_STOP, NewControllerSendStarted);
    }

  }
  else if (learnNodeInfo->bStatus == ADD_NODE_STATUS_FAILED)
  {
#if 1
    /* Just keep it in autoinclusion */
    learnNodeInfo->bStatus = ADD_NODE_STATUS_DONE;
    NewControllerAllDone(learnNodeInfo);
#else
    ZW_DEBUG_SEND_BYTE('2');
    /*We have added a new device to the controller*/
    ZW_ADD_NODE_TO_NETWORK(ADD_NODE_STOP, NULL);
    mainState = ERROR_STATE;
#endif
  }
  else if (learnNodeInfo->bStatus == ADD_NODE_STATUS_DONE)
  {
//    ZW_ADD_NODE_TO_NETWORK(ADD_NODE_STOP, NULL);
    NewControllerAllDone(learnNodeInfo);
  }
}



/*==========================   ApplicationControllerUpdate ==================
**    Function description
**      This function is called when a Static Update Controller
**      command have been received by the protocol.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
ApplicationControllerUpdate(
  BYTE bStatus,             /*IN  Status of learn process*/
  BYTE bSource,             /*IN  Node id of the node that send node info*/
  BYTE* pCmd,               /*IN  Pointer to application node information*/
  BYTE bLen)                /*IN  Node info length*/
{
  ZW_DEBUG_SEND_BYTE('?');
  ZW_DEBUG_SEND_NUM(bStatus);
  if (bLen != 0)
  {
    lastLearnedNodeType = ((NODE_TYPE *)pCmd)->generic;
    ZW_DEBUG_SEND_NUM(lastLearnedNodeType);
    ZW_DEBUG_SEND_NUM(bSource);
  }
  if (bStatus == UPDATE_STATE_ADD_DONE)
  {
    /* TO#2009 Fix */
#if 0
    if ((lastLearnedNodeType == GENERIC_TYPE_SWITCH_BINARY) ||
        (lastLearnedNodeType == GENERIC_TYPE_SWITCH_MULTILEVEL))
    {
    /* For demonstration purpuses every switch learned from the SUC are
       added to the hotkey group*/
      AddNodeToGroup(bSource);
    }
    /*In any case we want to turn off learn mode..*/
    ZW_DEBUG_SEND_BYTE('n');
#endif
  }
  else if (bStatus == UPDATE_STATE_DELETE_DONE)
  {
    RemoveNodeFromGroup(bSource);
  }
  else if (bStatus == UPDATE_STATE_SUC_ID)
  {
    /* OK we got a SUC */
  }
  else if (bStatus == UPDATE_STATE_NODE_INFO_RECEIVED)
  {
    /* Only add to group if we are dimming */
    if (mainState == DIMMING_STATE)
    {
      /* TO#2136 fix - No reason for only including some specific nodetypes */
      /* into group - we differentiate at group activation */
#if 0
      /* Only store it in the group if it was a switch type we learned */
      if ((lastLearnedNodeType == GENERIC_TYPE_SWITCH_BINARY) ||
          (lastLearnedNodeType == GENERIC_TYPE_SWITCH_MULTILEVEL))
#endif
      {
        /* If it is already in the group then remove it */
        if (IsNodeInGroup(bSource))
        {
          RemoveNodeFromGroup(bSource);
        }
        else
        {
          AddNodeToGroup(bSource);
          StartLED0Blink(ADD_TO_GROUP_RATE);
        }
      }
    }
  }
}


/*============================   RxOffDelay   ================================
**    Function description
**      RX off is delayed a bit to give the other end a chance to retransmit if
**     it missed the ack on its frame to this controller.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
RxOffDelay(void)
{
  LED0_OFF;
  /*If we still have a SUC start the update requests*/
  if (staticCtrlNodeID)
  {
    bStartUpdateTimer = TRUE;
  }

  /*RX is switched off by CHECK_BUTTON_STATE*/
  if (mainState == EXECUTE_STATE)
  {
    LED1_OFF;
    mainState = RETURN_TO_IDLE_STATE;
  }
}


/*============================   SucUpdateDone   ======================
**    Function description
**    Called when a SUC update have completed.
**
**    The call back function parameter values can be:
**        ZW_SUC_UPDATE_DONE        The update process ended successfully
**        ZW_SUC_UPDATE_ABORT       The update process was aborted
**        ZW_SUC_UPDATE_WAIT        The SUC node is busy, try again later
**        ZW_SUC_UPDATE_DISABLED    The SUC functionality have been disabled
**        ZW_SUC_UPDATE_OVERFLOW    Too many changes to handle by automatic update
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
SucUpdateDone(
  BYTE txStatus) /*IN callback status from protocol*/
{
  /*Regardless of status we get the SUC node ID again, it might have been changed by
  the update..*/
  staticCtrlNodeID = ZW_GET_SUC_NODEID();
  if((txStatus == ZW_SUC_UPDATE_OVERFLOW) ||
     (txStatus == ZW_SUC_UPDATE_DISABLED))
  {
    /*If state was disabled or overflow enter ERROR state so user know that a manual replication is needed*/
    mainState = ERROR_STATE;
  }

  if(ZW_TIMER_START(RxOffDelay,50,TIMER_ONE_TIME)==0xFF);
  {
    RxOffDelay();
  }
}


/*============================   RequestUpdateTimeOut   ======================
**    Function description
**      Timer based function which will request updates from a Static Update
**      controller if one is available
**    Side effects:
**      Disables user input while ongoing
**--------------------------------------------------------------------------*/
static void
RequestUpdateTimeOut(void)
{
  if (--timerCount == 0)
  {
    timerCount = GetRandomTime();
    if (mainState == CHECK_BUTTON_STATE)
    {
      StopUpdateTimer();
      ZW_SET_RX_MODE(TRUE);
      LED0_ON;
      mainState = EXECUTE_STATE;
      if (ZW_REQUEST_NETWORK_UPDATE(SucUpdateDone) == 0)
      {
        SucUpdateDone(ZW_SUC_UPDATE_DONE);
      }
    }
  }
}


/*================================   SetSUCDone  ============================
**    Function description
**      Called when a SUC NodeID have been set.
**    Side effects:
**
**--------------------------------------------------------------------------*/
static void
SetSUCDone(
  BYTE txStaus)
{
  /*txStaus == ZW_SUC_SET_SUCCEEDED if successful, but we handle failed the same way*/
  ZW_TIMER_START(StopLED0Blink,LED0_ONTIME, TIMER_ONE_TIME);
  mainState = RETURN_TO_IDLE_STATE;
  if (staticCtrlNodeID = ZW_GET_SUC_NODEID())
  {
    bStartUpdateTimer = TRUE;
  }
}


/*=====================   LearnCompleted  ======================
**    Function description
**        Called by the protocol as the learn process proceedes
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
extern void LearnCompleted(BYTE nodeID,                  /*IN The nodeID assigned*/
                           BYTE bStatus)
{
  bNWIIsStarted = FALSE;

  if (bStatus == FALSE)
  {
    receivingRemote = FALSE;
    mainState = ERROR_STATE;
  }
  else if (bStatus == TRUE)
  {
    ZW_DEBUG_SEND_BYTE('k');
    /*All protocol data have been transmitted.
      Any application data should be sent before setting NEW_CTRL_STATE_STOP_OK*/
    receivingRemote = FALSE;
    staticCtrlNodeID = ZW_GET_SUC_NODEID();
    if (staticCtrlNodeID )
    {
      bStartUpdateTimer = TRUE;
    }
    ZW_TIMER_START(StopLED0Blink,LED0_ONTIME, TIMER_ONE_TIME);
    /* Update group list in case a group replication was done */
    mainState = WAIT_FOR_PB_RELEASE_STATE;
  }
}

/****************************************************************************/
/*                           EXPORTED FUNCTIONS                             */
/****************************************************************************/

/*==============================   ApplicationInitHW  =======================
**    Non Z-Wave hardware initialization
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/

BYTE                   /*RET  TRUE       */
ApplicationInitHW(BYTE bWakeupReason)  /*IN Nothing */
{
  /* hardware initialization
     If hardware platform changes these needs to be looked over again.
     This can be solved by use the macros created for pin access
     (PIN_OUT, PIN_IN etc.) and defines from ZW_GRCdefs.h.*/
  BYTE x;

  /*Setup the relevant IO settings.*/
  /*Set up push buttons*/
  PIN_IN(PB0, 1);
  PIN_OFF(PB0);
  PIN_IN(PB1, 1);
  PIN_OFF(PB1);
  PIN_IN(PB2, 1);
  PIN_OFF(PB2);
  PIN_IN(PB3, 1);
  PIN_OFF(PB3);
#ifndef ZW_DEBUG_PROTOCOL
#ifndef ZW_DEBUG
  PIN_IN(PB4, 1);
  PIN_OFF(PB4);
#endif
#endif
  /*Set up LEDs*/
  PIN_OUT(LED0);
  PIN_OUT(LED1);

  for(x = 0; x < 255; x++); /*Run through this loop a number of times, to make sure inputs are stable!*/
  if ((PIN_GET(PB0) == 0) && (PIN_GET(PB4) == 0))
  {
    /*PB0 and PB4 is down enter production test*/
    LED0_ON;
    LED1_ON;
    return(FALSE);
  }
  /* If production test is wanted, check for conditions and return FALSE from here */
  return(TRUE);
}


/*===========================   ApplicationInitSW   =========================
**    Initialization of the Application Software
**
**
**--------------------------------------------------------------------------*/
BYTE                   /*RET  TRUE       */
ApplicationInitSW( void )  /*IN Nothing */
{
 /*Start keypad timer. No need to save return value. We wont stop it ever again*/
  ZW_TIMER_START(TimerCheckButtons, 1, TIMER_FOREVER);
  mainState = PWR_UP_STATE;
#ifndef ZW_DEBUG_PROTOCOL
  ZW_DEBUG_INIT(1152);
#endif
  ZW_DEBUG_CMD_INIT(384);
  staticCtrlNodeID = ZW_GET_SUC_NODEID();
  ZW_DEBUG_SEND_NL();
  ZW_DEBUG_SEND_BYTE('D');
  ZW_DEBUG_SEND_BYTE('e');
  ZW_DEBUG_SEND_BYTE('v');
  ZW_DEBUG_SEND_BYTE(' ');
  if (staticCtrlNodeID)
  {
    bStartUpdateTimer = TRUE;
  }
  ZW_SET_RX_MODE(TRUE);
#if 0
  /* TO#2071 fix - Do not set RoutingMAX to 2 - must be library default */
  /* Set MAX number of route tries before resorting to Explore Frame Route Resolution */
  ZW_SetRoutingMAX(2);
#endif
  /* Set Nwtwork wide inclusion active if we dont have aa node ID */

  if (ZW_IsPrimaryCtrl() && (ZW_GetControllerCapabilities() & NO_NODES_INCUDED))
    bNWIStartup = TRUE;
  else
    bNWIStartup = FALSE;


  return(TRUE);
}


BOOL productionTestFirstRun = TRUE;


/*============================   ApplicationTestPoll   ======================
**    Function description
**      This function is called when the controller enters test mode.
**      It can be extended with the functionalities needed to verify the hardware
**      In this application it is kept very simple.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
ApplicationTestPoll( void )
{
  if(productionTestFirstRun)
  {
    ZW_SET_RX_MODE(TRUE);
    productionTestFirstRun = FALSE;
  }

  /* If PB0 and PB4 is released. Enter inifinte transmit */
  if ((PIN_GET(PB0) != 0) && (PIN_GET(PB4) != 0))
  {
    /* key not pressed, transmit */
    LED0_ON;
    LED1_OFF;
    ZW_SEND_CONST();
    while(1);
  }
  LED0_OFF;
  LED1_ON;
}


/*===========================  ApplicationPoll ===============================
**    Function description
**      Main poll routine. This routine should be registered as the application
**      poll routine.
**      This routine handles menu navigation and executes any functions
**      registered as internalfunctions.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
ApplicationPoll( void )
{
  ZW_DEBUG_CMD_POLL();

  switch (mainState)
  {
    case PWR_UP_STATE:
      nodesInGroup = UpdateRAMGroupList();
      groupLevel = GetGroupLevel();
      if (!ButtonPressed())              /* Stay here until no key is pressed */
      {
        mainState = RETURN_TO_IDLE_STATE;
      }
      LED0_OFF;
      LED1_OFF;
      if (bNWIStartup)
      {
        ZW_SET_RX_MODE(TRUE);
        bNWIIsStarted = TRUE;
        StartLearnModeNow(ZW_SET_LEARN_MODE_NWI);
        LED0_ON;
      }
      break;

    case RETURN_TO_IDLE_STATE:
      // ZW_SET_RX_MODE(FALSE);
      mainState = CHECK_BUTTON_STATE;
      /*Drop trough to next state*/

    case CHECK_BUTTON_STATE:
      if (bNWIStartup)
      {
        if (learnInProgress)
        {
          bNWIStartup = FALSE;
          StartLED0Blink(BUSY_BLINK_RATE);
          mainState = EXECUTE_STATE; /*The user should no longer be able to exit learn mode*/
        }
      }
      if (glKey = ButtonPressed())
      {

        if (bNWIIsStarted)
        {
          StopLearnModeNow();
          bNWIIsStarted = FALSE;
        }
        else
        {
          LED1_OFF;
        }

        /* No RF needed when were in this state. */
        // ZW_SET_RX_MODE(TRUE);                 /*Power up RF.*/
        if (GROUP_CTRL_ACTIVATION)
        {
          mainState = GROUP_CONTROL_STATE;    /*group Toggle, dim and add node to group*/
        }
        if (ADD_NODE_ACTIVATION)
        {
          mainState = INCLUDE_NODE_STATE;     /*Add a new node to network*/
        }
        if (RESET_NODE_ACTIVATION)
        {
          mainState = REMOVE_NODE_STATE;      /*Remove Node from network (and group)*/
        }
        if (ASSIGN_ROUTE_ACTIVATION)
        {
          mainState = ASSIGN_ROUTE_STATE;     /*Assign route from node to node*/
        }
        if (ADD_CTRL_ACTIVATION)
        {
          mainState = NEW_CONTROLLER_STATE;   /*Add a slave controller to the network*/
        }
        if (RESET_ACTIVATION)
        {
          StartResetTimer();
          mainState = RESET_CONTROLLER_STATE; /*Reset this controller*/
        }
      }
      else
      {
        if (bStartUpdateTimer)
        {
          if (!ZW_PRIMARYCTRL() ||
              ((ZW_GET_CONTROLLER_CAPABILITIES() & CONTROLLER_NODEID_SERVER_PRESENT) != 0))
          {
            StartUpdateTimer();
          }
          else
          {
            bStartUpdateTimer = FALSE;
          }
        }
      }
      break;

    case GROUP_CONTROL_STATE:
      /*User hit the Group control button, handle it*/
      glKey = ButtonPressed();
      if (RESET_ACTIVATION)
      {
        StartResetTimer();
        mainState = RESET_CONTROLLER_STATE; /*PB4 is down too! We are in the wrong state*/
        break;
      }
      if (!glKey)
      {
        /*Key was released before we entered dimming state*/
        LED0_ON;
        mainState = EXECUTE_STATE;
        ToggleGroup();
        txInProgress = TRUE;
      }
      else if(IS_PB_HELD(glKey))
      {
        /* If the key is being held. Start the dimming routine */
        mainState = EXECUTE_STATE;
        LED0_ON;
        ControlGroup(DIM_START, DimStarted);
      }
      break; /*GROUP_CONTROL_STATE*/

    case INCLUDE_NODE_STATE:
      /* User hit the Include control button, handle it */
      glKey = ButtonPressed();
      if (!glKey)
      {
        mainState = EXECUTE_STATE;
        LED0_ON;
        /* User released before we entered Include state - send nodeinformation */
        ZW_SEND_NODE_INFO(NODE_BROADCAST, 0, TxCompleteIgnoreAck);
      }
      else if (IS_PB_HELD(glKey))
      {
        /* User wants to include a new node to the network. Handle it */
        StartLED0Blink(BUSY_BLINK_RATE);
        mainState = LEARN_STATE;
        /* Add nodes with Normal Power */
        /* Accept networkwide inclusion requests */
        ZW_ADD_NODE_TO_NETWORK((ADD_NODE_ANY | ADD_NODE_OPTION_HIGH_POWER | ADD_NODE_OPTION_NETWORK_WIDE), NewControllerSendStarted);
        ZW_DEBUG_SEND_BYTE('*');
      }
      break; /*INCLUDE_NODE_STATE*/

    case LEARN_STATE:
      /* Internal state. Handling of learn state */
      if (!(ButtonPressed() & PB_MASK))
      {
        ZW_TIMER_START(StopLED0Blink, LED0_ONTIME,TIMER_ONE_TIME);
        mainState = WAIT_FOR_PB_RELEASE_STATE;
      }
      break; /*LEARN_STATE*/

    case REMOVE_NODE_STATE:
      LED0_ON;
      /* Remove nodes with Normal Power */
      /* Accept networkwide exclusion requests */
      ZW_REMOVE_NODE_FROM_NETWORK((REMOVE_NODE_ANY | ADD_NODE_OPTION_HIGH_POWER | ADD_NODE_OPTION_NETWORK_WIDE), RemoveNodeCompleted);
      ZW_DEBUG_SEND_BYTE('_');
      mainState = LEARN_STATE;
      break; /*REMOVE_NODE_STATE*/

    case ASSIGN_ROUTE_STATE:
      /* Get nodeID of the device the routing device should receive a route to */
      if (!ButtonPressed())
      {
        /*When Button released do it*/
        ZW_SET_RX_MODE(TRUE);
        LED0_ON;
        mainState = EXECUTE_STATE;
        routeToNode = 0;
        routingNode = 0;

        ZW_ADD_NODE_TO_NETWORK(ADD_NODE_EXISTING, AssignRouteNodeCompleted);
        mainState = WAIT_FOR_PB_DOWN_STATE;
      }
      break; /* ASSIGN_ROUTE_STATE */

    case NEW_CONTROLLER_STATE:
      glKey = ButtonPressed();
      if (!glKey)
      {
        mainState = ERROR_STATE;
      }
      else if (IS_PB_HELD(glKey))
      {
        /* If the key is being held. Start the New controller receive */
        LED0_ON;
        StopUpdateTimer();
        receivingRemote = TRUE;
        ClearGroupTable(NULL);
        StartLearnModeNow(ZW_SET_LEARN_MODE_CLASSIC);
        //ZW_SetLearnMode(TRUE, NewControllerReceiveStarted);
        mainState = WAIT_FOR_PB_RELEASE_STATE;
      }
      break; /*NEW_CONTROLLER_STATE*/

    case RESET_CONTROLLER_STATE:
      if (!ButtonPressed())
      {
        /*If button is released do not reset!!*/
        StopResetTimer();
        mainState = RETURN_TO_IDLE_STATE;
      }

      break; /*RESET_CONTROLLER_STATE*/

    /*****************************************************************/
    /*Sub states. These are not executed directly from a button press*/
    /*****************************************************************/
    case DIMMING_STATE: /* Wait for key to be released*/
      if (!ButtonPressed())
      {
        ZW_DEBUG_SEND_BYTE('d');
        StartLED0Blink(BUSY_BLINK_RATE);
        mainState = EXECUTE_STATE;

        ControlGroup(DIM_STOP,DimStopped);
      }
      if (RESET_ACTIVATION)
      {
        StartResetTimer();
        mainState = RESET_CONTROLLER_STATE; /*Reset this controller*/
      }
      break; /*DIMMING_STATE*/

    case ERROR_STATE: /*Last operation failed..*/
      LED1_ON;
      StopLED0Blink();
      mainState = WAIT_FOR_PB_RELEASE_STATE;
      break;

    case WAIT_FOR_PB_DOWN_STATE:
      if (ButtonPressed())
      {
        ZW_DEBUG_SEND_BYTE('s');
        mainState = RETURN_TO_IDLE_STATE;
        StopLED0Blink();
        LED1_OFF;
      }
      break; /*WAIT_FOR_PB_DOWN_STATE*/

    case WAIT_FOR_PB_RELEASE_STATE:
      /* Wait until no key is pressed, then return to idle. */
      if (!ButtonPressed())
      {
        LED0_OFF;
        ZW_DEBUG_SEND_BYTE('3');
        ZW_ADD_NODE_TO_NETWORK(ADD_NODE_STOP, NULL);
        //ZW_SetLearnMode(FALSE, NULL);
        mainState = RETURN_TO_IDLE_STATE;
        ZW_DEBUG_SEND_BYTE('q');
      }
      if (RESET_ACTIVATION)
      {
        StartResetTimer();
        mainState = RESET_CONTROLLER_STATE; /*Reset this controller*/
      }

      /* TO#2935 fix - Rearm LearnModeTimeout if still being updated/included */
      if (receivingRemote)
      {
        /* We are still trying to include/update this controller */
        ReArmLearnModeTimeout();
      }

      /* Start busy blink if learn is in progress */
      if (learnInProgress)
      {
        StartLED0Blink(BUSY_BLINK_RATE);
        mainState = EXECUTE_STATE; /*The user should no longer be able to exit learn mode*/
      }


      break;

    case WAIT_FOR_PB_PRESS_STATE:
      if (ButtonPressed())
      {
        ZW_DEBUG_SEND_BYTE('4');
        ZW_ADD_NODE_TO_NETWORK(ADD_NODE_STOP, NULL);
        //ZW_SetLearnMode(FALSE, NULL);

        mainState = WAIT_FOR_PB_DEPRESS_STATE;
      }
      break;

    case WAIT_FOR_PB_DEPRESS_STATE:
      if (!ButtonPressed())
      {
        LED0_OFF;
        mainState = RETURN_TO_IDLE_STATE;
      }
      break;

    case EXECUTE_STATE: /* We are executing a command. State changes should be done by callbacks*/
      if (txInProgress)
      {
        glKey = ButtonPressed();

        if (GROUP_CTRL_ACTIVATION)
        {
          ZW_SendDataAbort();
          txInProgress = FALSE;
        }
      }
      break;

    default: /*Something went horribly wrong.. Return to idle*/
      mainState = RETURN_TO_IDLE_STATE;
      break;
  }
}


/*========================  ApplicationCommandHandler  ======================
**    Function description
**        This callback function is called by the Z-Wave system when a
**        RF message that should be handled by the application is received
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
ApplicationCommandHandler(        /*RET nothing*/
  BYTE  rxStatus,                 /* IN Frame header info */
#if defined(ZW_CONTROLLER) && !defined(ZW_CONTROLLER_STATIC) && !defined(ZW_CONTROLLER_BRIDGE)
  /* TO#1692 */
  BYTE  destNode,                 /* IN  Frame destination ID, only valid when frame is not Multicast */
#endif
  BYTE  sourceNode,               /* IN Command sender Node ID */
  ZW_APPLICATION_TX_BUFFER *pCmd, /* IN Payload from the received frame,   */
                                  /*    the command is the very first byte */
  BYTE cmdLength)                 /* IN Number of command bytes including the command */
{
  switch(pCmd->ZW_Common.cmdClass)
  {
    case COMMAND_CLASS_SWITCH_MULTILEVEL:
      if (pCmd->ZW_Common.cmd == SWITCH_MULTILEVEL_REPORT)
      {
        groupLevel = pCmd->ZW_SwitchMultilevelReportFrame.value;
        StoreGroupLevel(groupLevel);
        StopDimEndTimer();
        if (!receivedReport)
        {
          SetGroupDimLevel();
        }
        receivedReport = TRUE;
      }
      break;

    case COMMAND_CLASS_SWITCH_BINARY:
      break;

    case COMMAND_CLASS_BASIC:
      break;

    case COMMAND_CLASS_CONTROLLER_REPLICATION:
      ZW_DEBUG_SEND_BYTE('R');
      /* Handle replication frames */
      if (receivingRemote)
      {
        /* We are only interested in group replication here */
        /* but all other replication can be handled in the same way. */
        /* In this application we ignore everything else and just */
        /* send the command complete reply */
        if (pCmd->ZW_Common.cmd == CTRL_REPLICATION_TRANSFER_GROUP)
        {
          /* We are only interested in group 1 */
          if (pCmd->ZW_CtrlReplicationTransferGroupFrame.groupId == 1)
          {
            /* Add the node ID to the group */
            AddNodeToGroup(pCmd->ZW_CtrlReplicationTransferGroupFrame.nodeId);
          }
        }
        /* Send command complete to other controller */
        /* NOTE: This call should be done when ready to receive */
        /* the next replication frame */
        ZW_REPLICATION_COMMAND_COMPLETE();
      }
      break;

    case COMMAND_CLASS_VERSION:
      if (pCmd->ZW_Common.cmd == VERSION_GET)
      {
        txBuffer.ZW_VersionReportFrame.cmdClass = COMMAND_CLASS_VERSION;
        txBuffer.ZW_VersionReportFrame.cmd = VERSION_REPORT;
        txBuffer.ZW_VersionReportFrame.zWaveLibraryType = ZW_TYPE_LIBRARY();
        txBuffer.ZW_VersionReportFrame.zWaveProtocolVersion = ZW_VERSION_MAJOR;
        txBuffer.ZW_VersionReportFrame.zWaveProtocolSubVersion = ZW_VERSION_MINOR;
        txBuffer.ZW_VersionReportFrame.applicationVersion = APP_VERSION;
        txBuffer.ZW_VersionReportFrame.applicationSubVersion = APP_REVISION;
        ZW_SEND_DATA(sourceNode, (BYTE *)&txBuffer, sizeof(txBuffer.ZW_VersionReportFrame),
          TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_EXPLORE, NULL);
      }
      break;

    default:
      /*Ignore any unknown commands*/
      break;
  }
}


/*======================ApplicationNodeInformation   =========================
**    Request Node information and current status
**    Called by the the Z-Wave application layer before transmitting a
**    "Node Information" frame.
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
extern void                 /*RET Nothing */
ApplicationNodeInformation(
  BYTE   *deviceOptionsMask,         /*OUT Bitmask with application options     */
  APPL_NODE_TYPE *nodeType, /*OUT  Device type                        */
  BYTE  **nodeParm,         /*OUT  Device parameter buffer pointer    */
  BYTE  *parmLength )       /*OUT  Number of Device parameter bytes   */
{
  /* this is a NON listening node and it supports optional CommandClasses*/
  *deviceOptionsMask = APPLICATION_NODEINFO_NOT_LISTENING|APPLICATION_NODEINFO_OPTIONAL_FUNCTIONALITY;
  nodeType->generic = GENERIC_TYPE_GENERIC_CONTROLLER; /* Generic Device Type */
  nodeType->specific = SPECIFIC_TYPE_PORTABLE_REMOTE_CONTROLLER;
  *nodeParm = supportedCmdClasses;
  *parmLength = sizeof(supportedCmdClasses);
}


/*============================   StopReplicationTimer   ======================
**    This function stops the replication timer
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  <completion code>       */
StopReplicationTimer(void)
{
	if (replicationTimerHandle)
		ZW_TIMER_CANCEL(replicationTimerHandle);
}


/*============================   AbortReplication   ======================
**    This callback function is used to abort replication when the
**		replication timer times out. Replication is aborted if no
**		TRANSMISSION_COMPLETE is received
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  <completion code>       */
AbortReplication(void)
{
	StopReplicationTimer();
	ReplicationSendGroup(TRANSMIT_COMPLETE_FAIL);
}


/*============================   StartReplicationTimer   ======================
**    This function starts the replication timer used when
**		detecting replication attempts that fails.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  <completion code>       */
StartReplicationTimer(void)
{
	replicationTimerHandle = ZW_TIMER_START(AbortReplication,REPL_TIMEOUT,TIMER_ONE_TIME);
	if (replicationTimerHandle == 0xFF)
	{
		// Should do something about this
	}
}


/*======================== ReplicationSendGroup ===========================
**    This functions handles sending group replication information as part of the
**    Controller replication process
**
**
**--------------------------------------------------------------------------*/
void
ReplicationSendGroup(
  BYTE bStatus)
{
  static BYTE bReplicationSuccess = ADD_NODE_STOP;
  register BYTE bTempVar;

	StopReplicationTimer();
  /* Transmit failed stop replication */
  if (bStatus != TRANSMIT_COMPLETE_OK)
  {
    nextNode = MAX_NODES;
    bReplicationSuccess = ADD_NODE_STOP_FAILED;
  }

  /* Find the next node that belongs to the group */
  for (; nextNode < MAX_NODES; nextNode++)
  {
    if (IsNodeInGroup(nextNode))
    {
    	ZW_DEBUG_SEND_BYTE('4');
      break;
    }
  }

  /* Check if we are finished with transferring the group */
  if (nextNode == MAX_NODES)
  {
    /* Here we could transfer a group name, start the next group or start
       transferring scenes, but we dont have any of that in this controller so
       we will just stop */
    ZW_ADD_NODE_TO_NETWORK(bReplicationSuccess, NewControllerAllDone);
    /* TO#1986 fixed, bReplicationSuccess would always be FAIL after the first failure */
    bReplicationSuccess = ADD_NODE_STOP;
    return;
  }
  /* Send next node ID to other controller */
  txBuffer.ZW_CtrlReplicationTransferGroupFrame.cmdClass = COMMAND_CLASS_CONTROLLER_REPLICATION;
  txBuffer.ZW_CtrlReplicationTransferGroupFrame.cmd      = CTRL_REPLICATION_TRANSFER_GROUP;
  txBuffer.ZW_CtrlReplicationTransferGroupFrame.groupId  = 1; /* We only have one group in this controller */
  txBuffer.ZW_CtrlReplicationTransferGroupFrame.nodeId   = nextNode;

  /* Go to next node ID */
  nextNode++;

  /* Send the node id to the other controller */
  bTempVar = ZW_REPLICATION_SEND_DATA(bOtherController, (BYTE*)&txBuffer, sizeof(ZW_CTRL_REPLICATION_TRANSFER_GROUP_FRAME), TRANSMIT_OPTION_ACK, ReplicationSendGroup);
	StartReplicationTimer();

  /* Check if we could send replication frame */
  if (bTempVar == FALSE)
  {
    ReplicationSendGroup(TRANSMIT_COMPLETE_NO_ACK); /* Fail replication */
  }
}
