/********************************  LEDDIM.C  ********************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Z-Wave, the wireless language.
 *
 *              Copyright (c) 2001
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 * Copyright Zensys A/S, 2001
 *
 * Description: Slave application for evaluation kit LED dimmer
 *
 * Author:   Peter Shorty
 *
 * Last Changed By:  $Author: efh $
 * Revision:         $Revision: 23825 $
 * Last Changed:     $Date: 2012-11-23 14:50:30 +0100 (fr, 23 nov 2012) $
 *
 ****************************************************************************/
/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/
#include <config_app.h>
#ifdef ZW_CONTROLLER
#include <ZW_controller_api.h>
#else
#include <ZW_slave_api.h>
#endif
#include <ZW_pindefs.h>
#include <ZW_evaldefs.h>
#include <ZW_debug_api.h>
#include <LEDdim.h>
#include <eeprom.h>
#include <slave_learn.h>
#include <one_button.h>
#include <ZW_uart_api.h>
#include <ZW_mem_api.h>
#ifdef ZW_ISD51_DEBUG
#include "ISD51.h"
#endif

#include <ZW_TransportLayer.h>
#include <association.h>
//#include "Stubs.h"

/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/
#define METER_UNSOLICITED_TIMER_STOP       0xFF
#define METER_NODE_BROADCAST               0xFF
#define METER_UNSOLICITED_DEFAULT_TIME_OUT 300
#define METER_SIM_MAX                      60
#define METER_SIM_MIN                      0
#define NO_PREV_METER_VALUE                0xFF

/* SetLightLevel() command defines */
enum
{
  CURRENT_LEVEL = 0,
  TOGGLE,
  ON,
  OFF,
  LEVEL
};

/* SetLightLevel level defines */
#define LEVEL_DONTCARE    0 /* Level is don't care - the command says it all */


/* Poll function button state defines */
#define POLLIDLE          0
#define POLLDIMMING       1 /* we are dimming dimming */
#define POLLLEARNMODE     3 /* We're in learnmode */


/* Debounce count on button press detection. How many times must we see button */

#define DEBOUNCECOUNT       100


/* Timer repeat define */
#define FOREVER             TIMER_FOREVER
#define ONE_SHOT            TIMER_ONE_TIME

/* DIM direction defines */
#define DIM_TOGGLE          0xFF

/* DIM defines */
#define LEVEL_MAX           99  /* Maximum light level */
#define LEVEL_MIN           1   /* Minimum light level */
#if defined(ZW020x) || defined(ZW030x)
#define DIM_SPEED           3   /* DIM timer routine inc/dec with this value */
#endif /* defined(ZW010x) || defined(ZW020x) */

#define DIM_TIMER_FREQ      10  /* DIM timer routine is called every 100ms */

#define DIM_DIRECTION_DOWN  0
#define DIM_DIRECTION_UP    1

/* Define for setting the time which button must be pressed before starting DIM */
/* and when in childprotect mode its the max time between button presses */
#define DIM_PROTECT         50 /* 1/2 second */
#define CHILD_PROTECT       50 /* max 500ms between button presses */

#define CHILD_PROTECT_COUNT 3  /* number of button press before toggle */
/* when in childprotect mode */


#define LIGHTIDLE           0x00
#define LIGHTTOGGLE         0x01
#define LIGHTDIM            0x02
#define LIGHTNOP            0x04

/* Productiontest pin definition */

#ifdef ZW020x
/* SSNJ pin now input */
#define SET_PRODUCTIONTEST_PIN  PIN_IN(SSN, 0)
#define IN_PRODUCTIONTEST       (!PIN_GET(SSN))
#endif  /* ZW020x */

#ifdef ZW030x
/* SSNJ pin now input */
#define SET_PRODUCTIONTEST_PIN  PIN_IN(SSN, 0)
#define IN_PRODUCTIONTEST       (!PIN_GET(SSN))
#endif  /* ZW030x */



#define SECONDS_PER_MIN 60
#define TICK_PER_SECOND TIMER_ONE_SECOND
#define TICK_PER_MINUTE (TICK_PER_SECOND * SECONDS_PER_MIN)
#define FACTORY_DEFAULT_DIMMING_DURATION 100  //Factory default diming duration value (in timer ticks) 100 -> 1 Sec
#define TIMER_DURATION_ENCODED_FACTORY_DEFAULT_DIMMING_DURATION 0xFF
#define TIMER_DURATION_ENCODED_MINUTES_DIMMING_DURATION_LIMIT 0x7F

#define DEFAULT_PRIMARY_SWITCH_TYPE     0x00
#define DEFAULT_SECONDARY_SWITCH_TYPE   0x00
#define FACTORY_DEFAULT_STEP_SIZE       0x64

/* Values used for Protection Report command */
#define PROTECTION_REPORT_UNPROTECTED_V2                                             0x00
#define PROTECTION_REPORT_PROTECTION_BY_SEQUENCE_V2                                  0x01
#define PROTECTION_REPORT_NO_OPERATION_POSSIBLE_V2                                   0x02

#define RF_PROTECTION_REPORT_UNPROTECTED_V2                                          0x00
#define RF_PROTECTION_REPORT_NO_RF_CONTROL_V2                                        0x01
#define RF_PROTECTION_REPORT_NO_RF_RESPONCE_V2                                       0x02

#define RF_PROTETION_MODE_INFINITELY                                                 0xFFFF

#define TIMER_DURATION_ENCODED_FACTORY_DEFAULT_DIMMING_DURATION 0xFF
#define TIMER_DURATION_ENCODED_MINUTES_DIMMING_DURATION_LIMIT 0x7F

#define TIMER_DURATION_ENCODED_SECONDS_PRORECTION_TO_HI_LIMIT 0x3C
#define TIMER_DURATION_ENCODED_MINUTES_PRORECTION_TO_LO_LIMIT 0x41
#define TIMER_DURATION_ENCODED_MINUTES_PRORECTION_TO_HI_LIMIT 0xFE
#define TIMER_DURATION_ENCODED_PROTECTION_INFINITELY 0xFF



#define PROTECTION_CHILD_NUM 6
#define PROTECTION_CHILD_TIMEOUT 0xFF

/****************************************************************************/
/*                              PRIVATE DATA                                */
/****************************************************************************/

/* Meter command class variables */
static WORD MeterUnsolicitedAddr = 0;
static WORD MeterUnsolicitedTimeOut = 0;
static WORD MeterUnsolicitedTimeOutMax = METER_UNSOLICITED_DEFAULT_TIME_OUT;
static BYTE MeterUnsolicitedTimerHandle = 0xFF;
static BYTE MeterValue = 0;
static BYTE MeterValuePrev = NO_PREV_METER_VALUE;
static BYTE MeterUnsolicitedReportNode = METER_NODE_BROADCAST;
BOOL MeterValueDirection = 0;
BOOL MeterUnsolicitedStart = 0;

/* node status */
volatile t_nodeStatus dimmerNodeStatus;

/* A list of the known command classes. Except the basic class which allways */
/* should be supported. Used when node info is send */
static code t_nodeInfo nodeInfo[] =
{
  COMMAND_CLASS_SWITCH_MULTILEVEL,
  COMMAND_CLASS_SWITCH_ALL,
  COMMAND_CLASS_MANUFACTURER_SPECIFIC,
  COMMAND_CLASS_VERSION,
  COMMAND_CLASS_POWERLEVEL,
#ifdef SECURITY
  COMMAND_CLASS_SECURITY,
#endif
  COMMAND_CLASS_PROTECTION,
  COMMAND_CLASS_METER_V3,
// Only implemented for test purposes and must be removed before certification of product
#ifndef SECURITY
  COMMAND_CLASS_CONFIGURATION,
#endif
// End of //Only implemented...
  COMMAND_CLASS_ASSOCIATION_V2
};

#ifdef SECURITY

extern  BYTE nodeSecure;

extern  BYTE secureCmdHandling;
#define TSEC_CMD_HANDLING_SECURE    1

static code BYTE nodeInfoAfterIncluded[] =
{
  COMMAND_CLASS_VERSION,
  COMMAND_CLASS_MANUFACTURER_SPECIFIC,
  COMMAND_CLASS_SECURITY
};

static code BYTE nodeInfoForTransport[] =
{
  COMMAND_CLASS_SWITCH_MULTILEVEL,
  COMMAND_CLASS_SWITCH_ALL,
  COMMAND_CLASS_MANUFACTURER_SPECIFIC,
  COMMAND_CLASS_VERSION,
  COMMAND_CLASS_POWERLEVEL,
};
#else
#define nodeInfoForTransport nodeInfo
#endif /* SECURITY */


BYTE timerDimHandle = 0;
BYTE timerPollHandle = 0;
BYTE TimerProtectionTimeoutHandle = 0;
BYTE pollState = POLLIDLE;
BYTE pollDimProtect = FALSE;
BYTE toggleDone = 0;
ZW_APPLICATION_TX_BUFFER txBuf;

BYTE ignoreAllOnOff = SWITCH_ALL_SET_INCLUDED_IN_THE_ALL_ON_ALL_OFF_FUNCTIONALITY; /* Default we do allow ALL ON/OFF */

static BYTE protChildCntr = 0;

static BYTE protected_On_Off = PROTECTION_REPORT_UNPROTECTED_V2; /* 0 means not protected and 1 child protected */
static BYTE rf_Protected_On_Off = RF_PROTECTION_REPORT_UNPROTECTED_V2; /* 0 means not protected and 1 child protected */
BYTE txOption;

#if defined(ZW020x) || defined(ZW030x)
BYTE timerPowerLevelHandle = 0;
BYTE timerPowerLevelSec = 0;

#if 0
IBYTE testNodeID = ZW_TEST_NOT_A_NODEID;
IBYTE testSourceNodeID;
IBYTE testPowerLevel;
IWORD testFrameCount;
IWORD testFrameSuccessCount;
IBYTE testState = 0;
IBYTE  currentPower = normalPower;
#else
BYTE testNodeID = ZW_TEST_NOT_A_NODEID;
BYTE testSourceNodeID;
BYTE testPowerLevel;
WORD testFrameCount;
WORD testFrameSuccessCount;
BYTE testState = 0;
BYTE  currentPower = normalPower;
#endif

#endif


#ifdef METADATA_DUT
IWORD metaframecount = 0;
IWORD metaframerecount = 0;
IWORD timestamp = 0;
IWORD timespan = 0;
BOOL metarecount = FALSE;
BOOL metadocount = FALSE;
BYTE metaDataReportNode;
#endif

BYTE bNWIStartup;
BYTE bMyNodeID;

BYTE myNodeID = 0;
BYTE myHomeID[4];

static long ticks;
static BYTE endLevelPoint;
static long dTime;
static long err;
static BYTE dLevel;
static signed char direction;


static BYTE protExclusiveNodeID = 0;
static WORD protTimeout = 0; //SDS11060-11.pdf Ch 4.67.6.1 "Factory default setting for the Timeout parameter must be 0x00" //RF_PROTETION_MODE_INFINITELY;


BYTE currentGroupIndex;
BYTE currentGroupNodeIndex;
/****************************************************************************/
/*                              EXPORTED DATA                               */
/****************************************************************************/

/****************************************************************************/
/*                            PRIVATE FUNCTIONS                             */
/****************************************************************************/

#define METER_RATE_TYPE_IMPORT 0x20
#define METER_TYPE_ELECTRIC    0x01
#define METER_SCALE2_WATT      0x0
#define METER_PECISION         0
#define METER_SCALE_WATT       0x10
#define METER_SIZE_1BYTE             0x1

#define METER_REPORT_V3_PROP1 (METER_SCALE2_WATT | METER_RATE_TYPE_IMPORT | METER_TYPE_ELECTRIC)
#define METER_REPORT_V3_PROP2 (METER_PECISION | METER_SCALE_WATT | METER_SIZE_1BYTE)

#define METER_SUPPORTED_REPORT_V3_PROP1 (METER_SCALE2_WATT | METER_RATE_TYPE_IMPORT | METER_TYPE_ELECTRIC)

BOOL AssociationSend(BYTE broadcast);
void AssociationSendNext(BYTE bStatus);
void SendCompleted(BYTE txStatus);
BOOL HasAssociatedNode ( void );
void AssociationSendToGroup(void);

/*==============================   cbVoidByte   ============================
**
**  Function:  stub for callback
**
**  Side effects: None
**
**--------------------------------------------------------------------------*/
static void cbVoidByte(BYTE b)
{
}

/*==================   SendMeterUnsolicedFrame   ==================
**    Send meter report frame to all associated nodes
**
**    Side effects:
**
**---------------------------------------------------------------*/
void SendMeterUnsolicedFrame()
{
  if (MeterUnsolicitedStart == TRUE)
  {
    if (HasAssociatedNode())
    {

      txBuf.ZW_MeterReport1byteV3Frame.cmdClass     = COMMAND_CLASS_METER_V3;
      txBuf.ZW_MeterReport1byteV3Frame.cmd          = METER_REPORT_V3;
      txBuf.ZW_MeterReport1byteV3Frame.properties1  = METER_REPORT_V3_PROP1;
      txBuf.ZW_MeterReport1byteV3Frame.properties2  = METER_REPORT_V3_PROP2;
      txBuf.ZW_MeterReport1byteV3Frame.meterValue1  = MeterValue;

      txBuf.ZW_MeterReport1byteV3Frame.deltaTime1   = 0;
      if (MeterValuePrev == NO_PREV_METER_VALUE)
      {
        txBuf.ZW_MeterReport1byteV3Frame.deltaTime2   = 0;
      }
      else
      {
        if (MeterUnsolicitedTimeOutMax < 15)
        {
          txBuf.ZW_MeterReport1byteV3Frame.deltaTime2   = 1;
        }
        else
        {
          txBuf.ZW_MeterReport1byteV3Frame.deltaTime2 = (BYTE)((MeterUnsolicitedTimeOutMax +5)/10);
        }
      }

      txBuf.ZW_MeterReport1byteV3Frame.previousMeterValue1 = MeterValuePrev;
      MeterValuePrev = MeterValue;
      AssociationSendToGroup();
    }
  }
}


/*===================   MeterUnsolicitedTimer   ===================
**    Timeout function handling the transmission of meter report
**    frames
**
**    Side effects:
**
**---------------------------------------------------------------*/
static void
MeterUnsolicitedTimer()
{
  MeterUnsolicitedTimeOut++;
  if (MeterUnsolicitedTimeOut >= MeterUnsolicitedTimeOutMax)
  {
    SendMeterUnsolicedFrame();
    MeterUnsolicitedTimeOut = 0;
  }
  if (MeterValueDirection == TRUE)
  {
    MeterValue++;
    if (MeterValue == METER_SIM_MAX)
    {
      MeterValueDirection = FALSE;
    }
  }
  else
  {
    MeterValue--;
    if (MeterValue == METER_SIM_MIN)
    {
      MeterValueDirection = TRUE;
    }
  }
}

/*=================   StartMeterUnsolicitedTimer   =================
**    Start the timer handling the transmission of meter report
**    frames
**
**    Side effects:
**
**---------------------------------------------------------------*/
static void
StartMeterUnsolicitedTimer(WORD timeOut)
{
  MeterUnsolicitedTimeOutMax = timeOut;
  if (MeterUnsolicitedTimerHandle == 0xff)
  {
    MeterUnsolicitedTimerHandle = ZW_TIMER_START(MeterUnsolicitedTimer, 10, TIMER_FOREVER);
  }
}

/*=================   StopMeterUnsolicitedTimer   =================
**    Stop the timer handling the transmission of meter report
**    frames
**
**    Side effects:
**
**---------------------------------------------------------------*/
static void
StopMeterUnsolicitedTimer()
{
  MeterValuePrev = NO_PREV_METER_VALUE;
  if (MeterUnsolicitedTimerHandle != 0xff)
  {
    ZW_TIMER_CANCEL(MeterUnsolicitedTimerHandle);
    MeterUnsolicitedTimerHandle = 0xff;
  }
}



/*=====================   HasAssociatedNode   ====================
**    Checks if any nodes have been associated with this node.
**
**    Side effects:
**
**---------------------------------------------------------------*/
BOOL                   /*RET TRUE if any nodes have been associated */
HasAssociatedNode ( void )
{
  BYTE nodeIndex;

  for (nodeIndex = 0; nodeIndex < MAX_ASSOCIATION_IN_GROUP; nodeIndex++)
  {
    if (groups[currentGroupIndex].nodeID[nodeIndex])
    {
      return TRUE;
    }
  }
  return FALSE;
}


/*============================   AssociationStoreAll   ======================
**    Function description
**      Stores all groups in external Nonvolatile memory. Used by association.c
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
AssociationStoreAll( void )
{
  MemoryPutBuffer(EEOFFSET_ASSOCIATION_START, (BYTE *)&groups[0], ASSOCIATION_SIZE, NULL);
}

/*============================   AssociationClearAll   ======================
**    Function description
**      Clears the Association area in Nonvolatile memory. Used by association.c
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
AssociationClearAll( void )
{
  MemoryPutBuffer(EEOFFSET_ASSOCIATION_START, NULL, ASSOCIATION_SIZE, NULL);
  ZW_MEM_PUT_BYTE(EEOFFSET_ASSOCIATION_MAGIC, MAGIC_VALUE); /* Now ASSOCIATION should be OK */
}

/*============================   AssociationInit   ==========================
**    Function description
**      Reads the groups stored in the Nonvolatile memory
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
AssociationInit(void)
{
  if (ZW_MEM_GET_BYTE(EEOFFSET_ASSOCIATION_MAGIC) != MAGIC_VALUE)
  {
    AssociationClearAll();
  }
  MemoryGetBuffer(EEOFFSET_ASSOCIATION_START, (BYTE *)&groups[0], ASSOCIATION_SIZE);
}




/*============================   AssociationSend   ======================
**    Function description
**      Sends Basic Set frame to the associated node currentGroupNodeIndex (if exist)
**      in the current currentGroupIndex group.
**    Side effects:
**
**--------------------------------------------------------------------------*/
BOOL            /*RET TRUE if there may be more associated nodes to send to */
AssociationSend(
BYTE broadcast)
{
  for (; currentGroupNodeIndex < MAX_ASSOCIATION_IN_GROUP; currentGroupNodeIndex++)
  {
    if (groups[currentGroupIndex].nodeID[currentGroupNodeIndex])
    {
      Transport_SendRequest(
        groups[currentGroupIndex].nodeID[currentGroupNodeIndex],
        (BYTE*)&txBuf,
        sizeof(ZW_METER_REPORT_1BYTE_V3_FRAME), //frameSize,
        (TRANSMIT_OPTION_RETURN_ROUTE | TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_EXPLORE),
        AssociationSendNext,
        FALSE);
      return TRUE;
    }
  }
  if (broadcast)
  {
    Transport_SendRequest(
      NODE_BROADCAST,
      (BYTE*)&txBuf,
      sizeof(ZW_METER_REPORT_1BYTE_V3_FRAME), //frameSize,
      (TRANSMIT_OPTION_RETURN_ROUTE | TRANSMIT_OPTION_ACK),
      SendCompleted,
      FALSE);

    return TRUE;
  }
  return FALSE;
}


/*============================   AssociationSendNext   ======================
**    Function description
**      Callback used when sending to association command
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
AssociationSendNext(
  BYTE bStatus)
{
//  VerifyAssociatedTransmit(bStatus, groups[currentGroupIndex].nodeID[currentGroupNodeIndex]);
  currentGroupNodeIndex++;
  if (!AssociationSend(FALSE))
  {
    SendCompleted(bStatus);  /* Now done... */
  }
}



/*============================   SendCompleted   ======================
**    Callback for association sends...
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                    /*RET  Nothing       */
SendCompleted(
BYTE txStatus)          /*IN   Transmission result           */
{
//  currentState = STATE_APPL_IDLE;
}

/*============================   AssociationSendToGroup   ====================
**    Function description
**      Initiate transmission of a preprepared Basic Set frame to nodes in
**      first association group.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
AssociationSendToGroup(void)
{
  /* Setup to start transmit to first group (for now we have only one group!) */
  currentGroupIndex = 0;
  /* Setup to start with first node */
  currentGroupNodeIndex = 0;
  if (!AssociationSend(TRUE))
  {
    SendCompleted(TRANSMIT_COMPLETE_OK);  /* Now done... */
  }
}


#define DECODE_TIMER_DURATION(c_time)\
  (c_time == TIMER_DURATION_ENCODED_FACTORY_DEFAULT_DIMMING_DURATION)?\
    FACTORY_DEFAULT_DIMMING_DURATION:\
      (c_time <= TIMER_DURATION_ENCODED_MINUTES_DIMMING_DURATION_LIMIT)?\
       (long)c_time * TICK_PER_SECOND:\
       ((long)c_time - TIMER_DURATION_ENCODED_MINUTES_DIMMING_DURATION_LIMIT) * TICK_PER_MINUTE


static void CalculateStopPoint( BYTE settings)
{
  if ((settings & SWITCH_MULTILEVEL_START_LEVEL_CHANGE_PROPERTIES1_UP_DOWN_BIT_MASK_V2) == 0)
  {
    //Up
    ticks *= (LEVEL_MAX - dimmerNodeStatus.level);
    endLevelPoint = LEVEL_MAX;
  }
  else
  {
    //Down
    ticks *= (dimmerNodeStatus.level - LEVEL_MIN);
    endLevelPoint = LEVEL_MIN;
  }

  ticks /= (LEVEL_MAX - LEVEL_MIN);
}


static void InitDimming(BYTE stopLevel)
{
  direction = (stopLevel < dimmerNodeStatus.level) ? -1 : 1;
  dLevel = (stopLevel > dimmerNodeStatus.level) ? (stopLevel - dimmerNodeStatus.level) : (dimmerNodeStatus.level - stopLevel);
  dTime = ticks;
  err = -dTime;
}


static void TimerDimCancel();
void LEDUpdate();
static void onDimingTimer()
{
  if ( ticks > 0 )
  {
    ticks--;
    err += 2 * dLevel;
    if ( err > 0 )
    {
      err -= 2 * dTime;
      dimmerNodeStatus.level += direction;
    }
    LEDUpdate();  /* Update LEDs */
  }
  else
  {
    TimerDimCancel();
    ZW_MEM_PUT_BYTE( EEOFFSET_LEVEL, dimmerNodeStatus.level );
  }
  if ( dimmerNodeStatus.level == endLevelPoint )
  {
    TimerDimCancel();
    ZW_MEM_PUT_BYTE( EEOFFSET_LEVEL, dimmerNodeStatus.level );
  }
}


static void DIMStop( BYTE saveInE2PROM );



/*=============================   LEDallON   ===============================
**    Turn all LED ON
**
**    Side effects :
**
**--------------------------------------------------------------------------*/
static void                   /*RET  Nothing                  */
LEDallON( void )              /* IN  Nothing                  */
{
  LED_ON( 1 );
  LED_ON( 2 );
  LED_ON( 3 );
}


/*=============================   LEDallOFF   ==============================
**    Turn all LED OFF
**
**    Side effects :
**
**--------------------------------------------------------------------------*/
static void                   /*RET  Nothing                  */
LEDallOFF( void )             /* IN  Nothing                  */
{
  LED_OFF( 1 );
  LED_OFF( 2 );
  LED_OFF( 3 );
}


/*==============================   LEDUpdate   ==============================
**    Update LEDs according to current level
**
**    Side effects :
**
**--------------------------------------------------------------------------*/
static void                   /*RET  Nothing                  */
LEDUpdate( void )             /* IN  Nothing                  */
{
  if ( dimmerNodeStatus.status & M_DS_ON_OFF )
  {
    BYTE n;
#if defined(ZW020x) || defined(ZW030x)
    n = dimmerNodeStatus.level / ( 100 / 4 );
#endif  /* defined(ZW020x) || defined(ZW030x) */

    switch ( n )
    {
#if defined(ZW020x) || defined(ZW030x)
      case 0:
        LED_ON( 1 );
        LED_OFF( 2 );
        LED_OFF( 3 );
        break;

      case 1:
        LED_ON( 1 );
        LED_ON( 2 );
        LED_OFF( 3 );
        break;

      case 2:
        LED_ON( 1 );
        LED_ON( 2 );
        LED_ON( 3 );
        break;

      default:
        LEDallON();
        break;
#endif  /* defined(ZW020x) || defined(ZW030x) */

    }
  }
  else
  {
    LEDallOFF();
  }
}


/*===========================   ValidateLevel   =============================
**    Validate current level value
**
**    Side effects
**
**--------------------------------------------------------------------------*/
static void              /*RET  Nothing                  */
ValidateLevel( void )    /* IN  Nothing                  */
{
  if ( dimmerNodeStatus.level > LEVEL_MAX )
  {
    dimmerNodeStatus.level = LEVEL_MAX;
  }
  else
  {
    if ( dimmerNodeStatus.level < LEVEL_MIN )
    {
      dimmerNodeStatus.level = LEVEL_MIN;
    }
  }
}


/*============================   TimerChangeLevel   =========================
**    Change level
**
**    Side effects
**
**--------------------------------------------------------------------------*/
static void                /*RET  Nothing                  */
TimerChangeLevel( void )   /* IN  Nothing                  */
{
  if ( dimmerNodeStatus.status & M_DS_CHANGING_DIM ) /* Are we still DIMming? */
  {
    if (( dimmerNodeStatus.status & M_DS_UP_DOWN ) == 0 )  /* Dim UP - more light */
    {
      if ( dimmerNodeStatus.level < LEVEL_MAX )
      {
        dimmerNodeStatus.level += DIM_SPEED;
        if ( dimmerNodeStatus.level >= LEVEL_MAX )
        {
          if ( dimmerNodeStatus.status & M_DS_ROLLOVER )
          {
            dimmerNodeStatus.status ^= M_DS_UP_DOWN;
          }
        }
      }
      else
      {
        if ( dimmerNodeStatus.status & M_DS_ROLLOVER )
        {
          dimmerNodeStatus.status ^= M_DS_UP_DOWN;
        }
      }
    }
    else  /* Dim DOWN - less light */
    {
      if ( dimmerNodeStatus.level > LEVEL_MIN )
      {
        dimmerNodeStatus.level -= DIM_SPEED;
        if ( dimmerNodeStatus.level <= LEVEL_MIN )
        {
          if ( dimmerNodeStatus.status & M_DS_ROLLOVER )
          {
            dimmerNodeStatus.status ^= M_DS_UP_DOWN;
          }
        }
      }
      else
      {
        if ( dimmerNodeStatus.status & M_DS_ROLLOVER )
        {
          dimmerNodeStatus.status ^= M_DS_UP_DOWN;
        }
      }
    }
    LEDUpdate();  /* Update LEDs */
  }
}


/*===========================   TimerPollCancel   ============================
**    Function description
**      cancels timer, if still valid, used in CHILD_PROTECT
**
**    Side effects :
**
**--------------------------------------------------------------------------*/
static void             /*RET  Nothing                  */
TimerPollCancel( void ) /*IN  Nothing                   */
{
  if ( timerPollHandle )
  {
    ZW_TIMER_CANCEL( timerPollHandle );
    timerPollHandle = 0;
  }
}


/*===========================   TimerDimCancel   =============================
**    Function description
**      cancels timer, if still valid, used in DIM
**
**    Side effects :
**
**--------------------------------------------------------------------------*/
static void             /*RET  Nothing                  */
TimerDimCancel( void )  /*IN  Nothing                   */
{
  if ( timerDimHandle )
  {
    ZW_TIMER_CANCEL( timerDimHandle );
    timerDimHandle = 0;
  }
}


static void                 /*RET  Nothing                  */
TimerProtectionTimeout( void ) /* IN  Nothing                  */
{
  if ((protTimeout == RF_PROTETION_MODE_INFINITELY) || (protTimeout == 0))
  {
    ZW_TIMER_CANCEL(TimerProtectionTimeoutHandle);
  }
  else
  {
    protTimeout--;
  }
}



/*===============================   DIMStart   ==============================
**    Start DIMming
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
static void         /*RET  Nothing                */
DIMStart(
  BYTE level,         /* IN  starting DIM level     */
  BYTE dimBits )      /* IN  DIMming config bits    */
{
  /* are we dimming allready ? */
  if ( dimmerNodeStatus.status & M_DS_CHANGING_DIM )
  {
    DIMStop( FALSE );
  }
  if (dimBits & SWITCH_MULTILEVEL_START_LEVEL_CHANGE_PROPERTIES1_IGNORE_START_LEVEL_BIT_MASK_V2 == 0)
  {
    dimmerNodeStatus.level = level;
  }
  ValidateLevel();
  dimmerNodeStatus.status |= M_DS_ON_OFF;    /* Now ON */
  /* Save new status in EEPROM */
  ZW_MEM_PUT_BYTE( EEOFFSET_STATUS, dimmerNodeStatus.status );

  if ( dimBits == DIM_TOGGLE )
  {
    dimmerNodeStatus.status ^= M_DS_UP_DOWN;   /* Toggle DIM direction */
    dimmerNodeStatus.status |= M_DS_ROLLOVER;  /* Make sure its set */
  }
  else
  {
    dimmerNodeStatus.status &= ~( M_DS_UP_DOWN | M_DS_ROLLOVER );
    dimmerNodeStatus.status |= dimBits & ( M_DS_UP_DOWN | M_DS_ROLLOVER );
  }
  dimmerNodeStatus.status |= M_DS_CHANGING_DIM;  /* Indicate DIMming... */
  TimerDimCancel(); /* Cancel timer if needed... */
  /* Allocate timer for calling TimerChangeLevel every DIM_TIMER_FREQ, forever */
  timerDimHandle = ZW_TIMER_START( TimerChangeLevel, DIM_TIMER_FREQ, FOREVER );
}


/*===============================   DIMStop   ==============================
**    Stop DIMming
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
static void           /*RET  Nothing            */
DIMStop(
  BYTE saveInE2PROM )   /* IN  Should the current level be saved in EEPROM */
{
  TimerDimCancel(); /* Cancel timer if needed... */
  dimmerNodeStatus.status &= ~( M_DS_CHANGING_DIM ); /* STOP DIMming */
  ValidateLevel();
  if ( saveInE2PROM )
  {
    ZW_MEM_PUT_BYTE( EEOFFSET_LEVEL, dimmerNodeStatus.level ); /* Save the last value */
//    ZW_MEM_PUT_BYTE(EEOFFSET_STATUS, dimmerNodeStatus.status);
  }
}


/*=============================   SetLightLevel   ============================
**    Set the light level to:
**
**    CURRENT_LEVEL - The current light level stored in eeprom
**    LEVEL         - Use parameter bLevel as light level
**    TOGGLE        - Toggle the light level
**    ON            - Full on
**    OFF           - Off
**
**--------------------------------------------------------------------------*/
static void       /*RET  Nothing           */
SetLightLevel(
  BYTE bAction,     /* IN  Action to be done */
  BYTE bLevel )     /* IN  Lightlevel to set */
{
  if ( dimmerNodeStatus.status & M_DS_CHANGING_DIM )  /* If we're dimming */
  {
    DIMStop( FALSE );                               /* then STOP that */
  }
  switch ( bAction )
  {
    case TOGGLE:
      if (( dimmerNodeStatus.status & M_DS_ON_OFF ) == 0 )   /* Is switch OFF ? */

#ifdef __C51__
      /* C51 don't like recursive functions */
      {
        dimmerNodeStatus.status |= M_DS_ON_OFF;
        LEDUpdate();
      }
      else
      {
        dimmerNodeStatus.status &= ~( M_DS_ON_OFF );
        LEDallOFF(); /* Just do it! */
      }
#endif
      break;

    case LEVEL:
      dimmerNodeStatus.level = bLevel;
      dimmerNodeStatus.status |= M_DS_ON_OFF;
      ZW_MEM_PUT_BYTE( EEOFFSET_LEVEL, dimmerNodeStatus.level );
      LEDUpdate();
      break;

    case CURRENT_LEVEL:
      dimmerNodeStatus.status |= M_DS_ON_OFF;
      LEDUpdate();
      break;

    case ON:
      dimmerNodeStatus.level = LEVEL_MAX;
      dimmerNodeStatus.status |= M_DS_ON_OFF;
      ZW_MEM_PUT_BYTE( EEOFFSET_LEVEL, dimmerNodeStatus.level );
      LEDallON(); /* Just do it! */
      break;

    case OFF:
      dimmerNodeStatus.status &= ~( M_DS_ON_OFF );
      LEDallOFF(); /* Just do it! */
      break;
  }
  /* Save new status in EEPROM */
  ZW_MEM_PUT_BYTE( EEOFFSET_STATUS, dimmerNodeStatus.status );
}


/*=============================  ApplicationPoll   =========================
**    Application poll function for LED dimmer
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                    /*RET  Nothing                  */
ApplicationPoll( void ) /* IN  Nothing                  */
{
  BYTE lastAction;
#ifdef ZW_ISD51_DEBUG         /* init ISD51 only when the uVision2 Debugger tries to connect */
  ISDcheck();               /* initialize uVision2 Debugger and continue program run */
#endif
  lastAction = OneButtonLastAction();
  ZW_DEBUG_CMD_POLL();

  if ( bNWIStartup )
  {
    StartLearnModeNow( ZW_SET_LEARN_MODE_NWI );
    bNWIStartup = FALSE;
    pollState = POLLLEARNMODE;
  }

  switch ( pollState )
  {
    case POLLIDLE:
      if ( lastAction == BUTTON_TRIPLE_PRESS )
      {
        StartLearnModeNow( ZW_SET_LEARN_MODE_CLASSIC );
        pollState = POLLLEARNMODE;
      }

      if ( protected_On_Off == PROTECTION_SET_PROTECTION_BY_SEQUENCE )
      {
        if (( timerPollHandle == 0 ) && ( lastAction == BUTTON_WAS_PRESSED ) )
        {
          timerPollHandle = ZW_TIMER_START( TimerPollCancel, PROTECTION_CHILD_TIMEOUT, 1 );
          protChildCntr = 0;
        }

        if (( lastAction == BUTTON_IS_HELD ) && ( timerPollHandle == 0 ) )
        {
          toggleDone = LIGHTDIM;  /* Now dimming has been started */
          DIMStart( dimmerNodeStatus.level, DIM_TOGGLE );
          pollState = POLLDIMMING;
        }

        if (( lastAction == BUTTON_WAS_PRESSED ) && ( timerPollHandle != 0 ) )
        {
          protChildCntr++;
          if ( protChildCntr > PROTECTION_CHILD_NUM )
          {
            SetLightLevel( TOGGLE, LEVEL_DONTCARE );
            protChildCntr = 0;
            TimerPollCancel();
          }
        }
      }

      else if ( protected_On_Off == PROTECTION_SET_UNPROTECTED )
      {
        if ( lastAction == BUTTON_WAS_PRESSED )
        {
          if ( toggleDone == LIGHTIDLE )
          {
            SetLightLevel( TOGGLE, LEVEL_DONTCARE );
            // toggleDone = LIGHTTOGGLE;
          }
        }
        else if ( lastAction == BUTTON_IS_HELD )
        {
          toggleDone = LIGHTDIM;  /* Now dimming has been started */
          DIMStart( dimmerNodeStatus.level, DIM_TOGGLE );
          pollState = POLLDIMMING;
        }
        break;

      }
      break;

    case POLLDIMMING:
      if ( lastAction == BUTTON_WAS_RELEASED )
      {
        DIMStop( TRUE );
        toggleDone = LIGHTIDLE;
        pollState = POLLIDLE;
      }
      break;

    case POLLLEARNMODE:
      /* Just wait... */
      if ( lastAction == BUTTON_WAS_PRESSED )
      {
        /* Stop learn mode */
        StartLearnModeNow( FALSE );
        pollState = POLLIDLE;
      }
      break;
  }
}


/*==============================   SendReport   ==============================
**
**  Function:  SendReport
**
**    Sends a report frame to source node.
**
**  Side effects: None
**
**--------------------------------------------------------------------------*/
void              /*RET Nothing */
SendReport(
  BYTE sourceNode,  /* IN Source node - to whom report is to be send */
  BYTE classcmd,    /* IN Command class, which the report belongs to */
  BYTE reportparam ) /* IN paramter to include in report frame */
{
  txBuf.ZW_BasicReportFrame.cmdClass = classcmd;
  /* Report command definition - BASIC/ALL/PROTECTION/MULTI report use the same value ;-) */
  txBuf.ZW_BasicReportFrame.cmd      = BASIC_REPORT;
  txBuf.ZW_BasicReportFrame.value    = reportparam;
  /* Size of Report (BASIC/ALL/PROTECTION/MULTI GET response) frame are equal for BASIC/ALL/PROTECTION/MULTI */
  Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_BasicReportFrame), txOption, cbVoidByte, FALSE);
}


#if defined(ZW020x) || defined(ZW030x)
/*===========================   SendTestReport   ============================
**    Send current Powerlevel test results
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void
SendTestReport( void )
{
  txBuf.ZW_PowerlevelTestNodeReportFrame.cmdClass = COMMAND_CLASS_POWERLEVEL;
  txBuf.ZW_PowerlevelTestNodeReportFrame.cmd = POWERLEVEL_TEST_NODE_REPORT;
  txBuf.ZW_PowerlevelTestNodeReportFrame.testNodeid = testNodeID;
  txBuf.ZW_PowerlevelTestNodeReportFrame.statusOfOperation = testState;
  txBuf.ZW_PowerlevelTestNodeReportFrame.testFrameCount1 = ( BYTE )( *(( BYTE* ) & testFrameSuccessCount ) );
  txBuf.ZW_PowerlevelTestNodeReportFrame.testFrameCount2 = ( BYTE ) * ( &testFrameSuccessCount );
  /* Send Report - we do not care if it gets there or not - if needed report can be requested again */
  Transport_SendReport(testSourceNodeID, (BYTE *)&txBuf, sizeof(txBuf.ZW_PowerlevelTestNodeReportFrame), txOption, cbVoidByte, FALSE);
}


/*=============================   SendTestDone   ============================
**    Test frame has been transmitted to DUT and the result is noted for
**    later reporting. If not finished then another Test frame is transmitted.
**    If all test frames has been transmitted then the test is stopped
**    and the final result is reported to the PowerlevelTest initiator.
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void
SendTestDone(
  BYTE bStatus)
{
  if ( bStatus == TRANSMIT_COMPLETE_OK )
  {
    testFrameSuccessCount++;
  }

  if ( testFrameCount && ( --testFrameCount ) )
  {
    ZW_SEND_TEST_FRAME( testNodeID, testPowerLevel, SendTestDone );
  }
  else
  {
    if ( testFrameSuccessCount )
    {
      testState = ZW_TEST_SUCCES;
    }
    else
    {
      testState = ZW_TEST_FAILED;
    }
    ZW_RF_POWERLEVEL_SET( currentPower );  /* Back to previous setting */
    SendTestReport();
  }
}


/*=======================   PowerLevelTimerCancel   ==========================
**    Cancels PowerLevel timer
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void
PowerLevelTimerCancel( void )
{
  ZW_TIMER_CANCEL( timerPowerLevelHandle );
  timerPowerLevelHandle = 0;
}


/*===============================   StartTest   ==============================
**    Timer callback which maked sure that the RF transmit powerlevel is set
**    back to normalPower after the designated time period.
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void
PowerLevelTimeout( void )
{
  if ( !--timerPowerLevelSec )
  {
    ZW_RF_POWERLEVEL_SET( normalPower );  /* Reset powerlevel to normalPower */
    PowerLevelTimerCancel();
  }
}


/*===============================   StartTest   ==============================
**    Start the powerlevel test run
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void
StartTest( void )
{
  testState = ZW_TEST_INPROGRESS;
  currentPower = ZW_RF_POWERLEVEL_GET();  /* Get current (normalPower) */
  ZW_RF_POWERLEVEL_SET( testPowerLevel );
  ZW_SEND_TEST_FRAME( testNodeID, testPowerLevel, SendTestDone );
}
#endif


#ifdef METADATA_DUT
void
SendMetaTestReportComplete(
  BYTE bStatus )
{
  if ( metadocount && !metarecount )
  {
    metarecount = TRUE;
    timestamp = tickTime;
  }
}


void
SendMetaTestReport(
  BYTE sourceNode )
{
  txBuf.ZW_ManufacturerProprietary2ByteFrame.cmdClass = COMMAND_CLASS_MANUFACTURER_PROPRIETARY;
  txBuf.ZW_ManufacturerProprietary2ByteFrame.manufacturerId1 = 0x00;
  txBuf.ZW_ManufacturerProprietary2ByteFrame.manufacturerId2 = 0x00;
  txBuf.ZW_ManufacturerProprietary2ByteFrame.value1 = 0x03;
  txBuf.ZW_ManufacturerProprietary2ByteFrame.value2 = 0x81;
  *(( &txBuf.ZW_ManufacturerProprietary2ByteFrame.value2 ) + 1 ) = ( BYTE ) * (( BYTE* ) & timespan );
  *(( &txBuf.ZW_ManufacturerProprietary2ByteFrame.value2 ) + 2 ) = ( BYTE ) * ( &timespan );
  *(( &txBuf.ZW_ManufacturerProprietary2ByteFrame.value2 ) + 3 ) = ( BYTE ) * (( BYTE* ) & metaframecount );
  *(( &txBuf.ZW_ManufacturerProprietary2ByteFrame.value2 ) + 4 ) = ( BYTE ) * ( &metaframecount );
  if ( metarecount )
  {
    *(( &txBuf.ZW_ManufacturerProprietary2ByteFrame.value2 ) + 5 ) = ( BYTE ) * (( BYTE* ) & metaframerecount );
    *(( &txBuf.ZW_ManufacturerProprietary2ByteFrame.value2 ) + 6 ) = ( BYTE ) * ( &metaframerecount );
  }
  ZW_SEND_DATA( sourceNode, ( BYTE * )&txBuf, sizeof( txBuf.ZW_ManufacturerProprietary2ByteFrame ) + (( metarecount ) ? 6 : 4 ), txOption, SendMetaTestReportComplete );
#if 0
  txBuf.ZW_MetaDataReportFrame.cmdClass = COMMAND_CLASS_META_DATA;
  txBuf.ZW_MetaDataReportFrame.cmd      = META_DATA_REPORT;
  /* We use the MetaData report to report our findings */
  txBuf.ZW_MetaDataReportFrame.lastAndReportNo1 = ( BYTE ) * (( BYTE* ) & timespan );
  txBuf.ZW_MetaDataReportFrame.reportNo2 = ( BYTE ) * ( &timespan );
  txBuf.ZW_MetaDataReportFrame.data1 = ( BYTE ) * (( BYTE* ) & metaframecount );
  *(( &txBuf.ZW_MetaDataReportFrame.data1 ) + 1 ) = ( BYTE ) * ( &metaframecount );
  if ( metarecount )
  {
    *(( &txBuf.ZW_MetaDataReportFrame.data1 ) + 2 ) = ( BYTE ) * (( BYTE* ) & metaframerecount );
    *(( &txBuf.ZW_MetaDataReportFrame.data1 ) + 3 ) = ( BYTE ) * ( &metaframerecount );
  }
  ZW_SEND_DATA( sourceNode, ( BYTE * )&txBuf, sizeof( txBuf.ZW_MetaDataReportFrame ) + (( metarecount ) ? 3 : 1 ), txOption, SendMetaTestReportComplete );
#endif
}
#endif


/*========================   ApplicationCommandHandler   ====================
**    Handling of a received application commands and requests
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/

void                              /*RET Nothing                  */
Transport_ApplicationCommandHandler(
  BYTE  rxStatus,                 /* IN Frame header info */
  BYTE  sourceNode,               /* IN Command sender Node ID */
  ZW_APPLICATION_TX_BUFFER *pCmd, /* IN Payload from the received frame, the union */
  /*    should be used to access the fields */
  BYTE   cmdLength )              /* IN Number of command bytes including the command */
{
#ifdef METADATA_DUT
  BYTE frametype = rxStatus & RECEIVE_STATUS_TYPE_MASK;
#endif
  ZW_APPLICATION_TX_BUFFER CmdPacket;
  __ZW_memcpy( cmdLength, &CmdPacket, pCmd );
  txOption = (( rxStatus & RECEIVE_STATUS_LOW_POWER ) ?
              TRANSMIT_OPTION_LOW_POWER : 0) | TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_EXPLORE | TRANSMIT_OPTION_AUTO_ROUTE;

  if ( CmdPacket.ZW_Common.cmdClass == COMMAND_CLASS_PROTECTION )
  {
    pollState = POLLIDLE;
    switch ( CmdPacket.ZW_Common.cmd )
    {
      case PROTECTION_SET_V2:
        if ( cmdLength == sizeof( ZW_PROTECTION_SET_FRAME ) )
        {
          CmdPacket.ZW_ProtectionSetV2Frame.level2 = RF_PROTECTION_REPORT_UNPROTECTED_V2;
        }

        if ( CmdPacket.ZW_ProtectionSetV2Frame.level <= PROTECTION_REPORT_NO_OPERATION_POSSIBLE_V2 )
        {
          protected_On_Off = CmdPacket.ZW_ProtectionSetV2Frame.level;
          ZW_MEM_PUT_BYTE( EEOFFSET_PROTECTED, protected_On_Off );
        }

        if ( CmdPacket.ZW_ProtectionSetV2Frame.level2 <= RF_PROTECTION_REPORT_NO_RF_RESPONCE_V2 )
        {
          rf_Protected_On_Off = CmdPacket.ZW_ProtectionSetV2Frame.level2;
          ZW_MEM_PUT_BYTE( EEOFFSET_PROTECTED_RF, rf_Protected_On_Off );
        }
        break;

      case PROTECTION_GET_V2:
      {
        txBuf.ZW_ProtectionReportV2Frame.cmdClass = COMMAND_CLASS_PROTECTION_V2;
        txBuf.ZW_ProtectionReportV2Frame.cmd      = PROTECTION_REPORT_V2;
        txBuf.ZW_ProtectionReportV2Frame.level    = protected_On_Off;
        txBuf.ZW_ProtectionReportV2Frame.level2   = rf_Protected_On_Off;
        /* Size of Report (BASIC/ALL/PROTECTION/MULTI GET response) frame are equal for BASIC/ALL/PROTECTION/MULTI */
        ZW_SEND_DATA( sourceNode, ( BYTE * )&txBuf, sizeof( txBuf.ZW_ProtectionReportV2Frame ), txOption, cbVoidByte );
      }
      break;

      case PROTECTION_SUPPORTED_GET_V2:
      {
        union
        {
          WORD word;
          struct
          {
            BYTE b1;
            BYTE b2;
          } byte;
        } state;

        txBuf.ZW_ProtectionSupportedReportV2Frame.cmdClass = COMMAND_CLASS_PROTECTION_V2;
        txBuf.ZW_ProtectionSupportedReportV2Frame.cmd      = PROTECTION_SUPPORTED_REPORT_V2;

        txBuf.ZW_ProtectionSupportedReportV2Frame.level = PROTECTION_SUPPORTED_REPORT_LEVEL_TIMEOUT_BIT_MASK_V2 | PROTECTION_SUPPORTED_REPORT_LEVEL_EXCLUSIVE_CONTROL_BIT_MASK_V2;

        state.word = 0x1 << protected_On_Off;
        txBuf.ZW_ProtectionSupportedReportV2Frame.localProtectionState1 = state.byte.b1;
        txBuf.ZW_ProtectionSupportedReportV2Frame.localProtectionState2 = state.byte.b2;

        state.word = 0x1 << rf_Protected_On_Off;
        txBuf.ZW_ProtectionSupportedReportV2Frame.rfProtectionState1    = state.byte.b1;
        txBuf.ZW_ProtectionSupportedReportV2Frame.rfProtectionState2    = state.byte.b2;

        ZW_SEND_DATA( sourceNode, ( BYTE * )&txBuf, sizeof( txBuf.ZW_ProtectionSupportedReportV2Frame ), txOption, cbVoidByte );
      }
      break;

      case PROTECTION_EC_SET_V2:
        if ( CmdPacket.ZW_ProtectionEcSetV2Frame.nodeId <= ZW_MAX_NODES )
        {
          protExclusiveNodeID = CmdPacket.ZW_ProtectionEcSetV2Frame.nodeId;
        }
        break;

      case PROTECTION_EC_GET_V2:
      {
        txBuf.ZW_ProtectionEcReportV2Frame.cmdClass = COMMAND_CLASS_PROTECTION_V2;
        txBuf.ZW_ProtectionEcReportV2Frame.cmd      = PROTECTION_EC_REPORT_V2;
        txBuf.ZW_ProtectionEcReportV2Frame.nodeId   = protExclusiveNodeID;
        ZW_SEND_DATA( sourceNode, ( BYTE * )&txBuf, sizeof( txBuf.ZW_ProtectionEcReportV2Frame ), txOption, cbVoidByte );
      }
      break;

      case PROTECTION_TIMEOUT_SET_V2:
        if ( CmdPacket.ZW_ProtectionTimeoutSetV2Frame.timeout == 0 )
        {
          //No timer is set. All normal operation Commands must be accepted.
          protTimeout = 0;
        }
        else if ( CmdPacket.ZW_ProtectionTimeoutSetV2Frame.timeout <= TIMER_DURATION_ENCODED_SECONDS_PRORECTION_TO_HI_LIMIT )
        {
          //Timeout is set from 1 second (0x01) to 60 seconds (0x3C) in 1-second resolution
          protTimeout = ( WORD )CmdPacket.ZW_ProtectionTimeoutSetV2Frame.timeout;
          ZW_TIMER_CANCEL(TimerProtectionTimeoutHandle);
          TimerProtectionTimeoutHandle = ZW_TIMER_START( TimerProtectionTimeout, TICK_PER_SECOND, TIMER_FOREVER );
        }
        else if (( CmdPacket.ZW_ProtectionTimeoutSetV2Frame.timeout >= TIMER_DURATION_ENCODED_MINUTES_PRORECTION_TO_LO_LIMIT ) && ( CmdPacket.ZW_ProtectionTimeoutSetV2Frame.timeout <= TIMER_DURATION_ENCODED_MINUTES_PRORECTION_TO_HI_LIMIT ) )
        {
          //Timeout is set from 2 minutes (0x41) to 191 minutes (0xFE) in 1-minute resolution
          protTimeout = (( WORD )CmdPacket.ZW_ProtectionTimeoutSetV2Frame.timeout - TIMER_DURATION_ENCODED_MINUTES_PRORECTION_TO_LO_LIMIT + 2 ) * SECONDS_PER_MIN;
          ZW_TIMER_CANCEL(TimerProtectionTimeoutHandle);
          TimerProtectionTimeoutHandle = ZW_TIMER_START( TimerProtectionTimeout, TICK_PER_SECOND, TIMER_FOREVER );
        }
        else if ( CmdPacket.ZW_ProtectionTimeoutSetV2Frame.timeout == TIMER_DURATION_ENCODED_PROTECTION_INFINITELY )
        {
          //No Timeout  The Device will remain in RF Protection mode infinitely
          protTimeout = RF_PROTETION_MODE_INFINITELY;
        }
        else
        {
          //invalid cmd param
        }
        break;

      case PROTECTION_TIMEOUT_GET_V2:
      {
        txBuf.ZW_ProtectionTimeoutReportV2Frame.cmdClass = COMMAND_CLASS_PROTECTION_V2;
        txBuf.ZW_ProtectionTimeoutReportV2Frame.cmd      = PROTECTION_TIMEOUT_REPORT_V2;
        if (protTimeout == RF_PROTETION_MODE_INFINITELY)
        {
          txBuf.ZW_ProtectionTimeoutReportV2Frame.timeout  = TIMER_DURATION_ENCODED_PROTECTION_INFINITELY;
        }
        else
        {
          txBuf.ZW_ProtectionTimeoutReportV2Frame.timeout  = protTimeout > SECONDS_PER_MIN ? ( protTimeout + SECONDS_PER_MIN - 1) / SECONDS_PER_MIN : protTimeout;
        }

        ZW_SEND_DATA( sourceNode, ( BYTE * )&txBuf, sizeof( txBuf.ZW_ProtectionTimeoutReportV2Frame ), txOption, cbVoidByte );
      }
      break;

      // PROTECTION_TIMEOUT_GET_V2

      default:
        break;

    }
    return;
  }

  //    if ((protExclusiveNodeID != 0)&&(sourceNode != protExclusiveNodeID)&& (protTimeout != 0))
  if (( sourceNode != protExclusiveNodeID ) && ( protTimeout != 0 ) )
  {
    if ( rf_Protected_On_Off == RF_PROTECTION_REPORT_NO_RF_CONTROL_V2 )
    {
      //EC protection we should send Application rejected Request Command
      txBuf.ZW_ApplicationRejectedRequestFrame.cmdClass = COMMAND_CLASS_APPLICATION_STATUS;
      txBuf.ZW_ApplicationRejectedRequestFrame.cmd      = APPLICATION_REJECTED_REQUEST;
      txBuf.ZW_ApplicationRejectedRequestFrame.status   = 0;
      ZW_SEND_DATA( sourceNode, ( BYTE * )&txBuf, sizeof( txBuf.ZW_ApplicationRejectedRequestFrame ), txOption, cbVoidByte );
      return;
    }
    else if ( rf_Protected_On_Off == RF_PROTECTION_REPORT_NO_RF_RESPONCE_V2 )
    {
      return;
    }
  }

  if (CmdPacket.ZW_Common.cmdClass == COMMAND_CLASS_MANUFACTURER_SPECIFIC)
  {
    switch (CmdPacket.ZW_Common.cmd)
    {
      case MANUFACTURER_SPECIFIC_GET:
        txBuf.ZW_ManufacturerSpecificReportFrame.cmdClass = COMMAND_CLASS_MANUFACTURER_SPECIFIC;
        txBuf.ZW_ManufacturerSpecificReportFrame.cmd = MANUFACTURER_SPECIFIC_REPORT;
        txBuf.ZW_ManufacturerSpecificReportFrame.manufacturerId1 = 0x00; // Zensys manufacturer ID
        txBuf.ZW_ManufacturerSpecificReportFrame.manufacturerId2 = 0x00;
        txBuf.ZW_ManufacturerSpecificReportFrame.productTypeId1 = 0x00; // Assigned by manufacturer
        txBuf.ZW_ManufacturerSpecificReportFrame.productTypeId2 = 0x00;
        txBuf.ZW_ManufacturerSpecificReportFrame.productId1 = 0x00; // Assigned by manufacturer
        txBuf.ZW_ManufacturerSpecificReportFrame.productId2 = 0x00;
        Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_ManufacturerSpecificReportFrame), txOption, NULL, FALSE);
        break;
    }
  }
#ifndef SECURITY
  /* WARNING - The configuration command class is only implemented for test purposes
     in this sample application and must be removed before certification of product */
  else if (CmdPacket.ZW_Common.cmdClass == COMMAND_CLASS_CONFIGURATION)
  {
    /* Configure command*/

    switch (CmdPacket.ZW_Common.cmd)
    {

      case CONFIGURATION_SET:
        MeterUnsolicitedAddr = sourceNode;

        if (CmdPacket.ZW_ConfigurationSet2byteFrame.level & CONFIGURATION_SET_LEVEL_DEFAULT_BIT_MASK)
        {
          MeterUnsolicitedTimeOutMax = METER_UNSOLICITED_DEFAULT_TIME_OUT;
        }
        else
        {
          if ((CmdPacket.ZW_ConfigurationSet2byteFrame.parameterNumber == 1) &&
              ((CmdPacket.ZW_ConfigurationSet2byteFrame.level & CONFIGURATION_SET_LEVEL_SIZE_MASK) == 2) &&
              (CmdPacket.ZW_ConfigurationSet2byteFrame.configurationValue1 == 0))
          {
            if (CmdPacket.ZW_ConfigurationSet2byteFrame.configurationValue2 == 0)
            {
              MeterUnsolicitedTimeOutMax = 0;
              StopMeterUnsolicitedTimer();
            }
            else
            {
#if 1
              StartMeterUnsolicitedTimer((CmdPacket.ZW_ConfigurationSet2byteFrame.configurationValue2 < 10) ? (0x01 << (CmdPacket.ZW_ConfigurationSet2byteFrame.configurationValue2 - 1)) : METER_UNSOLICITED_DEFAULT_TIME_OUT);
#else
              switch (CmdPacket.ZW_ConfigurationSet2byteFrame.configurationValue2)
              {
                case 1:
                  StartMeterUnsolicitedTimer(1);
                  break;

                case 2:
                  StartMeterUnsolicitedTimer(2);
                  break;
                case 3:
                  StartMeterUnsolicitedTimer(4);
                  break;
                case 4:
                  StartMeterUnsolicitedTimer(8);
                  break;
                case 5:
                  StartMeterUnsolicitedTimer(16);
                  break;
                case 6:
                  StartMeterUnsolicitedTimer(32);
                  break;
                case 7:
                  StartMeterUnsolicitedTimer(64);
                  break;
                case 8:
                  StartMeterUnsolicitedTimer(128);
                  break;
                case 9:
                  StartMeterUnsolicitedTimer(256);
                  break;
                case 10:
                  StartMeterUnsolicitedTimer(METER_UNSOLICITED_DEFAULT_TIME_OUT);
                  break;
              }
#endif
            }
          }

        }
        break;

      case CONFIGURATION_GET:
        txBuf.ZW_ConfigurationReport2byteFrame.cmdClass = COMMAND_CLASS_CONFIGURATION;
        txBuf.ZW_ConfigurationReport2byteFrame.cmd = CONFIGURATION_REPORT;
        txBuf.ZW_ConfigurationReport2byteFrame.parameterNumber = 1; /* parameter */
        txBuf.ZW_ConfigurationReport2byteFrame.level = 2; /* length */
        txBuf.ZW_ConfigurationReport2byteFrame.configurationValue1 = 0;
        switch (MeterUnsolicitedTimeOutMax)
        {
          case 0:
            txBuf.ZW_ConfigurationReport2byteFrame.configurationValue2 = 0;
            break;

          case 1:
            txBuf.ZW_ConfigurationReport2byteFrame.configurationValue2 = 1;
            break;

          case 2:
            txBuf.ZW_ConfigurationReport2byteFrame.configurationValue2 = 2;
            break;

          case 4:
            txBuf.ZW_ConfigurationReport2byteFrame.configurationValue2 = 3;
            break;

          case 8:
            txBuf.ZW_ConfigurationReport2byteFrame.configurationValue2 = 4;
            break;

          case 16:
            txBuf.ZW_ConfigurationReport2byteFrame.configurationValue2 = 5;
            break;

          case 32:
            txBuf.ZW_ConfigurationReport2byteFrame.configurationValue2 = 6;
            break;

          case 64:
            txBuf.ZW_ConfigurationReport2byteFrame.configurationValue2 = 7;
            break;

          case 128:
            txBuf.ZW_ConfigurationReport2byteFrame.configurationValue2 = 8;
            break;

          case 256:
            txBuf.ZW_ConfigurationReport2byteFrame.configurationValue2 = 9;
            break;

          default:
            txBuf.ZW_ConfigurationReport2byteFrame.configurationValue2 = 10;
            break;
        }

        ZW_SEND_DATA(
          sourceNode,
          (BYTE *)&txBuf,
          sizeof(ZW_CONFIGURATION_REPORT_2BYTE_FRAME),
          txOption,
          cbVoidByte);
        break;
    }

  }
#endif /*security*/

// End of //Only implemented...
  else if (CmdPacket.ZW_Common.cmdClass == COMMAND_CLASS_ASSOCIATION)
  {
    switch (pCmd->ZW_Common.cmd)
    {
      case ASSOCIATION_GET:
        AssociationSendReport(CmdPacket.ZW_AssociationGetFrame.groupingIdentifier, sourceNode, txOption);
        break;

      case ASSOCIATION_SET:
        /* If node only support one group. Ignore the group ID requested */
        /* (As defined in Device class specification */
        AssociationAdd(1, (BYTE*)pCmd + OFFSET_PARAM_2, cmdLength - OFFSET_PARAM_2);
        AssociationStoreAll();
        break;

      case ASSOCIATION_REMOVE:
        /*If node only support one group. Ignore the group ID requested (As defined in Device class specification*/
        AssociationRemove(1, (BYTE*)pCmd + OFFSET_PARAM_2, cmdLength - OFFSET_PARAM_2);
        break;

      case ASSOCIATION_GROUPINGS_GET:
        AssociationSendGroupings(sourceNode, txOption);
        break;
    }
  }

  else if (CmdPacket.ZW_Common.cmdClass == COMMAND_CLASS_VERSION)
  {
    switch (CmdPacket.ZW_Common.cmd)
    {
      case VERSION_GET:
        txBuf.ZW_VersionReportFrame.cmdClass = COMMAND_CLASS_VERSION;
        txBuf.ZW_VersionReportFrame.cmd = VERSION_REPORT;
        txBuf.ZW_VersionReportFrame.zWaveLibraryType = ZW_TYPE_LIBRARY();
        txBuf.ZW_VersionReportFrame.zWaveProtocolVersion = ZW_VERSION_MAJOR;
        txBuf.ZW_VersionReportFrame.zWaveProtocolSubVersion = ZW_VERSION_MINOR;
        txBuf.ZW_VersionReportFrame.applicationVersion = APP_VERSION;
        txBuf.ZW_VersionReportFrame.applicationSubVersion = APP_REVISION;
        Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_VersionReportFrame), txOption, NULL, FALSE);
        break;

      case VERSION_COMMAND_CLASS_GET:
        txBuf.ZW_VersionCommandClassReportFrame.cmdClass = COMMAND_CLASS_VERSION;
        txBuf.ZW_VersionCommandClassReportFrame.cmd = VERSION_COMMAND_CLASS_REPORT;
        txBuf.ZW_VersionCommandClassReportFrame.requestedCommandClass = CmdPacket.ZW_VersionCommandClassGetFrame.requestedCommandClass;
        switch (CmdPacket.ZW_VersionCommandClassGetFrame.requestedCommandClass)
        {
          case COMMAND_CLASS_BASIC:
            txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = BASIC_VERSION;
            break;

          case COMMAND_CLASS_SWITCH_MULTILEVEL:
            txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = SWITCH_MULTILEVEL_VERSION_V3;
            break;

          case COMMAND_CLASS_SWITCH_ALL:
            txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = SWITCH_ALL_VERSION;
            break;

          case COMMAND_CLASS_MANUFACTURER_SPECIFIC:
            txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = MANUFACTURER_SPECIFIC_VERSION;
            break;

          case COMMAND_CLASS_VERSION:
            txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = VERSION_VERSION;
            break;

          case COMMAND_CLASS_PROTECTION:
            txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = PROTECTION_VERSION_V2;
            break;

          case COMMAND_CLASS_POWERLEVEL:
            txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = POWERLEVEL_VERSION;
            break;

          default:
            txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = UNKNOWN_VERSION;
            break;
        }
        Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_VersionCommandClassReportFrame), txOption, NULL, FALSE);
        break;
    }
  }

#ifdef SECURITY
  else if (nodeSecure)
    if (secureCmdHandling != TSEC_CMD_HANDLING_SECURE)
    {
//    if(currentState == STATE_COMMAND_HANDLER)
//    {
//      currentState = STATE_APPL_IDLE;
//    }
      return;
    }
#endif


    else if ( CmdPacket.ZW_Common.cmdClass == COMMAND_CLASS_BASIC )
    {
      switch ( CmdPacket.ZW_Common.cmd )
      {
        case BASIC_GET:
          SendReport( sourceNode, CmdPacket.ZW_Common.cmdClass, ( dimmerNodeStatus.status & M_DS_ON_OFF ) ? dimmerNodeStatus.level : SWITCHED_OFF );
          break;
        case BASIC_SET:
          if ( CmdPacket.ZW_BasicSetFrame.value == SWITCHED_ON )
          {
            SetLightLevel( CURRENT_LEVEL, LEVEL_DONTCARE );
          }
          else if ( CmdPacket.ZW_BasicSetFrame.value == SWITCHED_OFF )
          {
            SetLightLevel( OFF, LEVEL_DONTCARE );
          }
          else if ( CmdPacket.ZW_BasicSetFrame.value <= LEVEL_MAX ) /* Check for valid level */
          {
            SetLightLevel( LEVEL, CmdPacket.ZW_BasicSetFrame.value );
          }
          break;
      } /* switch */
    }
    else if ( CmdPacket.ZW_Common.cmdClass == COMMAND_CLASS_SWITCH_MULTILEVEL )
    {
      /* Single switch command */
      switch ( CmdPacket.ZW_Common.cmd )
      {
          /* Start dimming */
        case SWITCH_MULTILEVEL_START_LEVEL_CHANGE:
          if ( cmdLength == sizeof( ZW_SWITCH_MULTILEVEL_START_LEVEL_CHANGE_FRAME ) )
          {
            CmdPacket.ZW_SwitchMultilevelStartLevelChangeV2Frame.dimmingDuration =  TIMER_DURATION_ENCODED_FACTORY_DEFAULT_DIMMING_DURATION;
            CmdPacket.ZW_SwitchMultilevelStartLevelChangeV3Frame.stepSize = FACTORY_DEFAULT_STEP_SIZE;
          }
          else if ( cmdLength == sizeof( CmdPacket.ZW_SwitchMultilevelStartLevelChangeV2Frame ) )
          {
            CmdPacket.ZW_SwitchMultilevelStartLevelChangeV3Frame.stepSize = FACTORY_DEFAULT_STEP_SIZE;
          }
          else if ( cmdLength == sizeof( CmdPacket.ZW_SwitchMultilevelStartLevelChangeV3Frame ) )
          {
            if ((( CmdPacket.ZW_SwitchMultilevelStartLevelChangeV3Frame.properties1 & SWITCH_MULTILEVEL_START_LEVEL_CHANGE_PROPERTIES1_UP_DOWN_MASK_V3 ) >> SWITCH_MULTILEVEL_START_LEVEL_CHANGE_PROPERTIES1_UP_DOWN_SHIFT_V3 )
                == SWITCH_MULTILEVEL_START_LEVEL_CHANGE_UP_DOWN_NONE_V3 )
              // If the field is set to 3 there is no change to the Up/Down level.
              return;
          }

          /* are we dimming allready ? */
          if ( dimmerNodeStatus.status & M_DS_CHANGING_DIM )
          {
            DIMStop( FALSE );
          }

          if (( CmdPacket.ZW_SwitchMultilevelStartLevelChangeV2Frame.properties1 & SWITCH_MULTILEVEL_START_LEVEL_CHANGE_PROPERTIES1_IGNORE_START_LEVEL_BIT_MASK_V2 ) == 0 )
          {
            dimmerNodeStatus.level = CmdPacket.ZW_SwitchMultilevelStartLevelChangeV2Frame.startLevel;
          }

          ValidateLevel();
          dimmerNodeStatus.status |= M_DS_ON_OFF;    /* Now ON */

          /* Save new status in EEPROM */
          ZW_MEM_PUT_BYTE( EEOFFSET_STATUS, dimmerNodeStatus.status );
          ZW_MEM_PUT_BYTE( EEOFFSET_LEVEL, dimmerNodeStatus.level );

          dimmerNodeStatus.status &= ~M_DS_UP_DOWN; //~( M_DS_UP_DOWN | M_DS_ROLLOVER );
          dimmerNodeStatus.status |= CmdPacket.ZW_SwitchMultilevelStartLevelChangeV2Frame.properties1 & M_DS_UP_DOWN; //( M_DS_UP_DOWN | M_DS_ROLLOVER );
          dimmerNodeStatus.status |= M_DS_CHANGING_DIM;  /* Indicate DIMming... */

          TimerDimCancel(); /* Cancel timer if needed... */
          ticks = DECODE_TIMER_DURATION(CmdPacket.ZW_SwitchMultilevelStartLevelChangeV2Frame.dimmingDuration);
          CalculateStopPoint( CmdPacket.ZW_SwitchMultilevelStartLevelChangeV2Frame.properties1 );
          InitDimming( endLevelPoint );
          timerDimHandle = ZW_TIMER_START( onDimingTimer, 1, FOREVER );
          break;

          /* Stop dimming */
        case SWITCH_MULTILEVEL_STOP_LEVEL_CHANGE:
          DIMStop( TRUE );
          break;

          /* Controller wants the level */
        case SWITCH_MULTILEVEL_GET:
          SendReport( sourceNode, CmdPacket.ZW_Common.cmdClass, ( dimmerNodeStatus.status & M_DS_ON_OFF ) ? dimmerNodeStatus.level : SWITCHED_OFF );
          break;

          /* Jump to the specified level param1 */
        case SWITCH_MULTILEVEL_SET:
          if ( cmdLength == sizeof( ZW_SWITCH_MULTILEVEL_SET_FRAME ) )
          {
            CmdPacket.ZW_SwitchMultilevelSetV2Frame.dimmingDuration = 0;
          }

          if ( CmdPacket.ZW_SwitchMultilevelSetV2Frame.value == SWITCHED_ON )
          {
            SetLightLevel( CURRENT_LEVEL, LEVEL_DONTCARE );
          }
          else if ( CmdPacket.ZW_SwitchMultilevelSetV2Frame.value == SWITCHED_OFF )
          {
            SetLightLevel( OFF, LEVEL_DONTCARE );
          }
          else if ( CmdPacket.ZW_SwitchMultilevelSetV2Frame.value <= LEVEL_MAX ) /* Check for valid level */
          {
            if ( CmdPacket.ZW_SwitchMultilevelSetV2Frame.dimmingDuration == 0 )
            {
              // Duration :: Instantly
              SetLightLevel( LEVEL, CmdPacket.ZW_SwitchMultilevelSetV2Frame.value );
            }
            else
            {
              /* are we dimming allready ? */
              if ( dimmerNodeStatus.status & M_DS_CHANGING_DIM )
              {
                DIMStop( FALSE );
              }
              ValidateLevel();
              dimmerNodeStatus.status |= M_DS_ON_OFF;    /* Now ON */

              /* Save new status in EEPROM */
              ZW_MEM_PUT_BYTE( EEOFFSET_STATUS, dimmerNodeStatus.status );
              ZW_MEM_PUT_BYTE( EEOFFSET_LEVEL, dimmerNodeStatus.level );

              dimmerNodeStatus.status |= M_DS_CHANGING_DIM;  /* Indicate DIMming... */
              TimerDimCancel(); /* Cancel timer if needed... */
              if ( CmdPacket.ZW_SwitchMultilevelSetV2Frame.value != dimmerNodeStatus.level )
              {
                ticks = DECODE_TIMER_DURATION(CmdPacket.ZW_SwitchMultilevelSetV2Frame.dimmingDuration);
                endLevelPoint = CmdPacket.ZW_SwitchMultilevelSetV2Frame.value;
                InitDimming( endLevelPoint );
                timerDimHandle = ZW_TIMER_START( onDimingTimer, 1, FOREVER );
              }
            }
          }
          else
          {
            // param2 => 100 ... 255 reserved range
          }
          break;

        case SWITCH_MULTILEVEL_SUPPORTED_GET_V3:
          txBuf.ZW_SwitchMultilevelSupportedReportV3Frame.cmdClass = COMMAND_CLASS_SWITCH_MULTILEVEL;
          txBuf.ZW_SwitchMultilevelSupportedReportV3Frame.cmd = SWITCH_MULTILEVEL_SUPPORTED_REPORT_V3;
          txBuf.ZW_SwitchMultilevelSupportedReportV3Frame.properties1 = DEFAULT_PRIMARY_SWITCH_TYPE;
          txBuf.ZW_SwitchMultilevelSupportedReportV3Frame.properties2 = DEFAULT_SECONDARY_SWITCH_TYPE;
          ZW_SEND_DATA( sourceNode, ( BYTE * )&txBuf, sizeof( txBuf.ZW_SwitchMultilevelSupportedReportV3Frame ), txOption, cbVoidByte );
          break;
      } /* switch */
    } /* if */
#ifdef ZW_CONTROLLER
    else if ( CmdPacket.ZW_Common.cmdClass == COMMAND_CLASS_CONTROLLER_REPLICATION )
    {
      /* Handle replication frames */
      /* We are only interested in group replication here */
      /* but all other replication can be handled in the same way. */
      /* In this application we ignore everything else and just */
      /* send the command complete reply */
#if 0
      if ( pCmd->ZW_Common.cmd == CTRL_REPLICATION_TRANSFER_GROUP )
      {
        /* We are only interested in group 1 */
        if ( pCmd->ZW_TransferGroupFrame.groupID == 1 )
        {
          /* Add the node ID to the group */
          AddNodeToGroup( pCmd->ZW_TransferGroupFrame.nodeID );
        }
      }
#endif
      /* Send command complete to other controller */
      /* NOTE: This call should be done when ready to receive */
      /* the next replication frame */
      ZW_REPLICATION_COMMAND_COMPLETE();
    }
#endif
    else if ( CmdPacket.ZW_Common.cmdClass == COMMAND_CLASS_SWITCH_ALL )
    {
      switch ( CmdPacket.ZW_Common.cmd )
      {
        case SWITCH_ALL_ON:
          if (( ignoreAllOnOff != SWITCH_ALL_SET_EXCLUDED_FROM_THE_ALL_ON_ALL_OFF_FUNCTIONALITY ) && ( ignoreAllOnOff != SWITCH_ALL_SET_EXCLUDED_FROM_THE_ALL_ON_FUNCTIONALITY_BUT_NOT_ALL_OFF ) )
          {
            SetLightLevel( ON, LEVEL_DONTCARE );
          }
          break;

        case SWITCH_ALL_OFF:
#ifdef METADATA_DUT
          if ( frametype == RECEIVE_STATUS_TYPE_BROAD )
          {
            /* We received the broadcasted "ALL OFF" - start counting and sampling */
            timestamp = tickTime;
            metarecount = FALSE;  /* This is the initial metadata frame count */
            metadocount = TRUE; /* Now we count the received metadata frames */
            metaframecount = 0; /* Initially no frames received */
            metaframerecount = 0; /* Initially no frames received */
          }
          else if ( metadocount && !metarecount && ( frametype == RECEIVE_STATUS_TYPE_SINGLE ) )
          {
            timespan = tickTime - timestamp;
            metaDataReportNode = sourceNode;
#if 1
            SendMetaTestReportComplete( TRUE );
#else
            SendMetaTestReport( sourceNode );
#endif
          }
#endif
          if (( ignoreAllOnOff != SWITCH_ALL_SET_EXCLUDED_FROM_THE_ALL_ON_ALL_OFF_FUNCTIONALITY ) && ( ignoreAllOnOff != SWITCH_ALL_SET_EXCLUDED_FROM_THE_ALL_OFF_FUNCTIONALITY_BUT_NOT_ALL_ON ) )
          {
            SetLightLevel( OFF, LEVEL_DONTCARE );
          }
          break;

        case SWITCH_ALL_SET:
          ignoreAllOnOff = CmdPacket.ZW_SwitchAllSetFrame.mode;
          /* Store it in the EEPROM */
          ZW_MEM_PUT_BYTE( EEOFFSET_IGNORE_ALL_ON_OFF, ignoreAllOnOff );
          break;

        case SWITCH_ALL_GET:
          SendReport( sourceNode, COMMAND_CLASS_SWITCH_ALL, ignoreAllOnOff );
          break;

        default:
          break;

      } /* switch */
    } /* else if == COMMAND_CLASS_SWITCH_ALL */
  /* TODO_ZW040x */
    else if ( CmdPacket.ZW_Common.cmdClass == COMMAND_CLASS_POWERLEVEL )
    {
      switch ( CmdPacket.ZW_Common.cmd )
      {
        case POWERLEVEL_SET:
          if (( CmdPacket.ZW_PowerlevelSetFrame.powerLevel == normalPower ) || ( CmdPacket.ZW_PowerlevelSetFrame.powerLevel == minus1dBm ) ||
              ( CmdPacket.ZW_PowerlevelSetFrame.powerLevel == minus2dBm ) || ( CmdPacket.ZW_PowerlevelSetFrame.powerLevel == minus3dBm ) ||
              ( CmdPacket.ZW_PowerlevelSetFrame.powerLevel == minus4dBm ) || ( CmdPacket.ZW_PowerlevelSetFrame.powerLevel == minus5dBm ) ||
              ( CmdPacket.ZW_PowerlevelSetFrame.powerLevel == minus6dBm ) || ( CmdPacket.ZW_PowerlevelSetFrame.powerLevel == minus7dBm ) ||
              ( CmdPacket.ZW_PowerlevelSetFrame.powerLevel == minus8dBm ) || ( CmdPacket.ZW_PowerlevelSetFrame.powerLevel == minus9dBm ) )
          {
            if ( CmdPacket.ZW_PowerlevelSetFrame.timeout || ( CmdPacket.ZW_PowerlevelSetFrame.powerLevel == normalPower ) )  /* Only if timeout defined or powerlevel is normalPower */
            {
              CmdPacket.ZW_PowerlevelSetFrame.powerLevel = ZW_RF_POWERLEVEL_SET( CmdPacket.ZW_PowerlevelSetFrame.powerLevel );  /* Set powerlevel according to command */
              if ( timerPowerLevelHandle )
              {
                PowerLevelTimerCancel();
                timerPowerLevelSec = 0;
              }
              if ( CmdPacket.ZW_PowerlevelSetFrame.powerLevel != normalPower )  /* If normalPower then no timer is started */
              {
                timerPowerLevelSec = CmdPacket.ZW_PowerlevelSetFrame.timeout;
                timerPowerLevelHandle = ZW_TIMER_START( PowerLevelTimeout, TIMER_ONE_SECOND, FOREVER );
              }
            }
          }
          break;

        case POWERLEVEL_GET:
          txBuf.ZW_PowerlevelReportFrame.cmdClass = COMMAND_CLASS_POWERLEVEL;
          txBuf.ZW_PowerlevelReportFrame.cmd = POWERLEVEL_REPORT;
          txBuf.ZW_PowerlevelReportFrame.powerLevel = ZW_RF_POWERLEVEL_GET();
          txBuf.ZW_PowerlevelReportFrame.timeout = timerPowerLevelSec;
          Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_PowerlevelReportFrame), txOption, NULL, FALSE);
          break;

        case POWERLEVEL_TEST_NODE_SET:
          if ( testState != ZW_TEST_INPROGRESS )
          {
            testSourceNodeID = sourceNode;
            testNodeID = CmdPacket.ZW_PowerlevelTestNodeSetFrame.testNodeid;
            testPowerLevel = CmdPacket.ZW_PowerlevelTestNodeSetFrame.powerLevel;
            testFrameCount = ( WORD )(( WORD )CmdPacket.ZW_PowerlevelTestNodeSetFrame.testFrameCount1 << 8 ) + CmdPacket.ZW_PowerlevelTestNodeSetFrame.testFrameCount2;
            testFrameSuccessCount = 0;
            if ( testFrameCount )
            {
              StartTest();
            }
            else
            {
              testState = ZW_TEST_FAILED;
              SendTestReport();
            }
          }
          break;

        case POWERLEVEL_TEST_NODE_GET:
          SendTestReport();
          break;
        default:
          break;

      } /* switch */
    } /* else if == COMMAND_CLASS_POWERLEVEL */

#ifdef METADATA_DUT
    else if ( CmdPacket.ZW_Common.cmdClass == COMMAND_CLASS_MANUFACTURER_PROPRIETARY )
    {
      switch ( pCmd->ZW_ManufacturerProprietary2ByteFrame.value1 )
      {
        case 0x03:
          break;

        case 0x04:
          if ( metadocount )
          {
            if ( metarecount )
            {
              metaframerecount++;
              if ( timespan <= tickTime - timestamp )
              {
                SendMetaTestReport( metaDataReportNode );
                metarecount = FALSE;
                metadocount = FALSE;  /* We are done... */
              }
            }
            else
            {
              metaframecount++;
            }
          }
          break;

        default:
          break;
      }
    }
#endif
}

/****************************************************************************/
/*                           EXPORTED FUNCTIONS                             */
/****************************************************************************/

/*============================   ApplicationInitHW   ========================
**    Non Z-Wave hardware initialization
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
BYTE                       /*RET  TRUE        */
ApplicationInitHW( BYTE bStatus )  /* IN  Nothing     */
{
#ifdef __C51__
  IBYTE i;
#endif
#ifdef ZW_ISD51_DEBUG
  ZW_UART_INIT(1152);
#endif
#ifdef __C51__

  SET_PRODUCTIONTEST_PIN;
  for ( i = 0; i < 10; i++ ) ;  /* Short delay... */
  if ( IN_PRODUCTIONTEST ) /* Anyone forcing it into productiontest mode ? */
  {
    return( FALSE );  /*Enter production test mode*/
  }
#endif

  /* hardware initialization */
  PIN_IN( Button, 1 );

  /* Setup specifics for current dim module */
  PIN_OUT( LED1 );
  PIN_OUT( LED2 );
  PIN_OUT( LED3 );

  Transport_OnApplicationInitHW(bStatus);

  return( TRUE );
}


/*===========================   ApplicationInitSW   =========================
**    Initialization of the Application Software
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
BYTE                      /*RET  TRUE       */
ApplicationInitSW( void ) /* IN   Nothing   */
{
  /* Do not reinitialize the UART if already initialized for ISD51 in ApplicationInitHW() */
#ifndef ZW_ISD51_DEBUG
  ZW_DEBUG_INIT(1152);
#endif
  ZW_DEBUG_SEND_NL();

#ifdef AUTO_INCLUSION
  learnModeActiveHandle = 0xFF;
#endif

  /* get stored values */
  if ( ZW_MEM_GET_BYTE( EEOFFSET_MAGIC ) == MAGIC_VALUE )
  {
    dimmerNodeStatus.level = ZW_MEM_GET_BYTE( EEOFFSET_LEVEL );
    if ( dimmerNodeStatus.level > LEVEL_MAX )
    {
      dimmerNodeStatus.level = LEVEL_MAX;
    }
    dimmerNodeStatus.status = ZW_MEM_GET_BYTE( EEOFFSET_STATUS );
    /* Get Ignore all On/Off frames state */
    ignoreAllOnOff = ZW_MEM_GET_BYTE( EEOFFSET_IGNORE_ALL_ON_OFF );
    /* Get protection state */
    protected_On_Off = ZW_MEM_GET_BYTE(EEOFFSET_PROTECTED);
    rf_Protected_On_Off = ZW_MEM_GET_BYTE(EEOFFSET_PROTECTED_RF);
  }
  else
  { /* Its alive */
    ZW_MEM_PUT_BYTE( EEOFFSET_LEVEL, LEVEL_MAX ); /* Full light */
    ZW_MEM_PUT_BYTE( EEOFFSET_STATUS, 0 );  /* Switch OFF */
    ZW_MEM_PUT_BYTE(EEOFFSET_IGNORE_ALL_ON_OFF, SWITCH_ALL_SET_INCLUDED_IN_THE_ALL_ON_ALL_OFF_FUNCTIONALITY);
    ZW_MEM_PUT_BYTE(EEOFFSET_PROTECTED, PROTECTION_REPORT_UNPROTECTED_V2); /* Not protected */
    ZW_MEM_PUT_BYTE(EEOFFSET_PROTECTED_RF, RF_PROTECTION_REPORT_UNPROTECTED_V2); /* Not protected */
    ZW_MEM_PUT_BYTE( EEOFFSET_MAGIC, MAGIC_VALUE ); /* Now EEPROM should be OK */
    dimmerNodeStatus.status = SWITCHED_OFF;
    dimmerNodeStatus.level  = LEVEL_MAX;          /* Set level to full light */
    /* ignoreAllOnOff and protected_On_Off are initialized on startup */
  }
  // ZW_DEBUG_CMD_INIT(1152);
  LEDUpdate();
  OneButtonInit();
  AssociationInit();

  /* Check if we have a node id */
  MemoryGetID( NULL, &bMyNodeID );

  /* Set Nwtwork wide inclusion active if we dont have aa node ID */
  if ( bMyNodeID )
    bNWIStartup = FALSE;
  else
    bNWIStartup = TRUE;

  StartMeterUnsolicitedTimer(METER_UNSOLICITED_DEFAULT_TIME_OUT);
  Transport_OnApplicationInitSW(nodeInfoForTransport, sizeof(nodeInfoForTransport),
                                FALSE, EEOFFSET_TRANSPORT_SETTINGS_START, EEOFFSET_TRANSPORT_SETTINGS_SIZE, NULL);

  return( TRUE );
}


/*============================   ApplicationTestPoll   ======================
**    Function description
**      This function is called when the dimmer enters test mode.
**    Side effects:
**       Code will not exit until it is reset
**--------------------------------------------------------------------------*/
void ApplicationTestPoll( void )
{
  /*Send constant out*/
  ZW_SEND_CONST();
  while ( 1 );
}


/*======================   ApplicationNodeInformation   =====================
**    Request Node information and current status
**    Called by the the Z-Wave application layer before transmitting a
**    "Node Information" frame.
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
extern void                  /*RET  Nothing */
ApplicationNodeInformation(
  BYTE   *deviceOptionsMask,      /*OUT Bitmask with application options     */
  APPL_NODE_TYPE  *nodeType,  /*OUT  Device type Generic and Specific   */
  BYTE       **nodeParm,      /*OUT  Device parameter buffer pointer    */
  BYTE       *parmLength )    /*OUT  Number of Device parameter bytes   */
{
  /* this is a listening node and it supports optional CommandClasses */
  *deviceOptionsMask = APPLICATION_NODEINFO_LISTENING | APPLICATION_NODEINFO_OPTIONAL_FUNCTIONALITY;
  nodeType->generic = GENERIC_TYPE_SWITCH_MULTILEVEL; /* Generic device type */
  nodeType->specific = SPECIFIC_TYPE_POWER_SWITCH_MULTILEVEL; /* Specific class */
  *nodeParm = nodeInfo;                 /* Send list of known command classes. */
  *parmLength = sizeof( nodeInfo );     /* Set length*/

#ifdef SECURITY
  MemoryGetID(myHomeID, &myNodeID);
  nodeSecure = GetNodeSecure(EEOFFSET_TRANSPORT_SETTINGS_START);
  if (myNodeID)
    if (nodeSecure)
    {
      *nodeParm = nodeInfoAfterIncluded;                 /* Send list of known command classes. */
      *parmLength = sizeof(nodeInfoAfterIncluded);       /* Set length*/
    }
#endif
}


/*=========================   ApplicationSlaveUpdate   =======================
**   Inform a slave application that a node information is received.
**   Called from the slave command handler when a node information frame
**   is received and the Z-Wave protocol is not in a state where it is needed.
**
**--------------------------------------------------------------------------*/
void
ApplicationSlaveUpdate(
  BYTE bStatus,     /*IN  Status event */
  BYTE bNodeID,     /*IN  Node id of the node that send node info */
  BYTE* pCmd,       /*IN  Pointer to Application Node information */
  BYTE bLen )      /*IN  Node info length                        */
{
}


#ifdef ZW_CONTROLLER
/*=====================   ApplicationControllerUpdate   =====================
**   Inform the controller of node information update done through
**   the network managment.
**
**--------------------------------------------------------------------------*/
void
ApplicationControllerUpdate(
  BYTE bStatus,     /*IN  Status event */
  BYTE bNodeID,     /*IN  Node id of the node that send node info */
  BYTE* pCmd,       /*IN  Pointer to Application Node information */
  BYTE bLen )      /*IN  Node info length                        */
{
}
#endif


/*============================   LearnCompleted   ========================
**    Function description
**      Called from learnmode.c when learnmode completed
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                                    /*RET   Nothing */
LearnCompleted(
  BYTE nodeID )                   /* IN resulting nodeID */
{
  if ( nodeID == 0 )    /* Was it reset? */
  {
    ignoreAllOnOff = SWITCH_ALL_SET_INCLUDED_IN_THE_ALL_ON_ALL_OFF_FUNCTIONALITY;
    protected_On_Off = PROTECTION_REPORT_UNPROTECTED_V2;
    rf_Protected_On_Off = RF_PROTECTION_REPORT_UNPROTECTED_V2;
    /* Store it in the EEPROM */
    ZW_MEM_PUT_BYTE( EEOFFSET_IGNORE_ALL_ON_OFF, ignoreAllOnOff );
    ZW_MEM_PUT_BYTE(EEOFFSET_PROTECTED, protected_On_Off);
    ZW_MEM_PUT_BYTE(EEOFFSET_PROTECTED_RF, rf_Protected_On_Off);
    MeterUnsolicitedStart = FALSE;
    AssociationClearAll();
    AssociationInit();
  }
  else
  {
    MeterUnsolicitedStart = TRUE;
  }


  Transport_OnLearnCompleted(nodeID);
}
