using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Collections;
using Zensys.Framework;
using Zensys.ZWave.Logging;
using Zensys.PCController.Models;
using Zensys.ZWave.Devices;

namespace Zensys.PCController.Controls
{
    public class ListRouteSourcesView : ListView
    {
        private RoutesCollection dataSource;
        private void BuildList()
        {
            lock (this)
            {
                BeginUpdate();
                this.Items.Clear();
                foreach (RouteItem route in dataSource.SourceRoutes)
                {
                    ListViewItem item = new ListViewItem(new string[] { route.Device.Id.ToString(), route.Device.Type });
                    item.Tag = route;
                    Items.Add(item);
                }
                EndUpdate();
            }
        }

        protected override void OnSelectedIndexChanged(EventArgs e)
        {
            base.OnSelectedIndexChanged(e);
            if (SelectedItems != null && SelectedItems.Count > 0)
            {
                dataSource.SelectedSourceRoute = GetRow(SelectedItems[0]);
            }
            else
            {
                dataSource.SelectedSourceRoute = null;
            }
        }

        public RouteItem GetRow(ListViewItem listViewItem)
        {
            return listViewItem.Tag as RouteItem;
        }

        public void SelectNode(RouteItem routeItem)
        {
            if (routeItem != null)
            {
                foreach (ListViewItem listViewItem in Items)
                {
                    if (GetRow(listViewItem).Device.Id == routeItem.Device.Id)
                    {
                        if (SelectedItems.Count == 0 || listViewItem != SelectedItems[0])
                        {
                            listViewItem.Selected = true;
                            break;
                        }
                    }
                }
            }
            else
            {
                foreach (ListViewItem listViewItem in Items)
                {
                    listViewItem.Selected = false;
                }
            }
        }

        [Category("Data")]
        public RoutesCollection DataSource
        {
            get
            {
                return dataSource;
            }
            set
            {
                if (value != null)
                {
                    dataSource = value;
                    BuildList();
                    dataSource.RouteCollectionChanged += new EventHandler<EventArgs>(dataSource_RouteCollectionChanged);
                    dataSource.SelectedSourceRouteChanged += new EventHandler<EventArgs>(dataSource_SelectedRouteChanged);
                }
            }
        }

        protected override void Dispose(bool disposing)
        {
            base.Dispose(disposing);
            if (dataSource != null)
            {
                dataSource.RouteCollectionChanged -= new EventHandler<EventArgs>(dataSource_RouteCollectionChanged);
                dataSource.SelectedSourceRouteChanged -= new EventHandler<EventArgs>(dataSource_SelectedRouteChanged);
            }
        }

        void dataSource_SelectedRouteChanged(object sender, EventArgs e)
        {
            ReSelectNode(dataSource.SelectedSourceRoute);
        }

        void dataSource_RouteCollectionChanged(object sender, EventArgs e)
        {
            ReBuildList();
        }

        private delegate void ReBuildListDelegate();
        private void ReBuildList()
        {
            if (InvokeRequired)
            {
                Invoke(new ReBuildListDelegate(ReBuildList));
            }
            else
            {
                BuildList();
            }
        }

        private delegate void ReSelectNodeDelegate(RouteItem routeItem);
        private void ReSelectNode(RouteItem routeItem)
        {
            if (InvokeRequired)
            {
                Invoke(new ReSelectNodeDelegate(ReSelectNode), routeItem);
            }
            else
            {
                SelectNode(routeItem);
            }
        }

    }
}
