using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.Collections.ObjectModel;
using System.Drawing;
using Zensys.PCController.Models;
using System.ComponentModel;
using Zensys.Framework;
using Zensys.PCController.Properties;
using Zensys.ZWave.Devices;

namespace Zensys.PCController.Controls.Drawing
{
    public class TopologyMapCanvas : PictureBox
    {

        #region Constants & Enums

        public const int NODE_HEIGHT = 25;
        public const int NODE_WIDTH = 25;
        public const int ELEMENT_HEIGHT = NODE_HEIGHT * 2 + 1;
        public const int ELEMENT_WIDTH = NODE_WIDTH * 2 + 1;

        #endregion

        #region Constructor
        public TopologyMapCanvas()
        {
            DoubleBuffered = true;
            SetStyle(ControlStyles.AllPaintingInWmPaint, true);
            SetStyle(ControlStyles.OptimizedDoubleBuffer, true);
            SetStyle(ControlStyles.UserPaint, true);
            SetStyle(ControlStyles.ResizeRedraw, true);
        }
        #endregion
        #region Private Fields
        private int mMarginTop = 45;
        private int mMarginLeft = 45;
        private int mMarginRight = 45;
        private int mMarginBottom = 45;

        private int mPaddingTop = 5;
        private int mPaddingLeft = 5;
        private int mPaddingRight = 5;
        private int mPaddingBottom = 5;
        private TopologyMapElement mSelectedElement;

        public TopologyMapElement SelectedElement
        {
            get { return mSelectedElement; }
            set { mSelectedElement = value; }
        }
        public int PaddingBottom
        {
            get { return mPaddingBottom; }
            set { mPaddingBottom = value; }
        }

        public int PaddingRight
        {
            get { return mPaddingRight; }
            set { mPaddingRight = value; }
        }

        public int PaddingLeft
        {
            get { return mPaddingLeft; }
            set { mPaddingLeft = value; }
        }

        public int PaddingTop
        {
            get { return mPaddingTop; }
            set { mPaddingTop = value; }
        }

        private List<TopologyMapNode> mVerticalNodes = new List<TopologyMapNode>();
        private List<TopologyMapNode> mHorizontalNodes = new List<TopologyMapNode>();
        private List<TopologyMapElement> mElements = new List<TopologyMapElement>();

        private Color mBackColor = Color.White;
        private Pen mDefaultPen = new Pen(Color.Gray);

        #endregion
        #region Public Properties
        public int MarginTop
        {
            get { return mMarginTop; }
            set { mMarginTop = value; }
        }
        public int MarginLeft
        {
            get { return mMarginLeft; }
            set { mMarginLeft = value; }
        }
        public int MarginRignt
        {
            get { return mMarginRight; }
            set { mMarginRight = value; }
        }
        public int MarginBottom
        {
            get { return mMarginBottom; }
            set { mMarginBottom = value; }
        }
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        internal List<TopologyMapNode> VerticalNodes
        {
            get { return mVerticalNodes; }
            set { mVerticalNodes = value; }
        }
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        internal List<TopologyMapNode> HorizontalNodes
        {
            get { return mHorizontalNodes; }
            set { mHorizontalNodes = value; }
        }
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        internal List<TopologyMapElement> Elements
        {
            get { return mElements; }
        }
        public override Color BackColor
        {
            get { return mBackColor; }
            set { mBackColor = value; }
        }
        public Pen DefaultPen
        {
            get { return mDefaultPen; }
            set { mDefaultPen = value; }
        }
        private TopologyMapItemCollection mDataSource = new TopologyMapItemCollection();
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public TopologyMapItemCollection DataSource
        {
            get { return mDataSource; }
            set
            {
                mDataSource = value;
                if (mDataSource != null)
                {
                    mDataSource.CollectionChanged -= new EventHandler<NotifyCollectionChangedEventArgs>(mDataSource_CollectionChanged);
                    mDataSource.CollectionChanged += new EventHandler<NotifyCollectionChangedEventArgs>(mDataSource_CollectionChanged);
                }
            }
        }

        void mDataSource_CollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            if (this.InvokeRequired)
            {
                this.Invoke(new LoadItemsDelegate(LoadItems));
            }
            else
            {
                LoadItems();
            }
        }
        private delegate void LoadItemsDelegate();
        private void LoadItems()
        {
            this.HorizontalNodes.Clear();
            this.VerticalNodes.Clear();
            this.Elements.Clear();

            if (DataSource.Count > 0)
            {
                if (this.Width < (DataSource.Count * ELEMENT_WIDTH + this.MarginLeft + this.MarginRignt))
                {
                    this.Width = (DataSource.Count * ELEMENT_WIDTH + this.MarginLeft + this.MarginRignt);
                }
                if (this.Height < (DataSource.Count * ELEMENT_HEIGHT + this.MarginTop + this.MarginBottom))
                {
                    this.Height = (DataSource.Count * ELEMENT_HEIGHT + this.MarginTop + this.MarginBottom);
                }
                foreach (TopologyMapItem item in DataSource)
                {
                    AddNode(item);
                }
                AddElements();
            }
            this.Invalidate(true);
        }

        #endregion
        #region Methods
        public void AddNode(TopologyMapItem item)
        {
            TopologyMapNode vNode = new TopologyMapNode();
            TopologyMapNode hNode = new TopologyMapNode();

            vNode.Index = this.VerticalNodes.Count;
            vNode.Type = TopologyMapNode.NodeTypes.Vertical;
            vNode.Site = this;
            vNode.BackColor = item.GetColorByDeviceType();
            vNode.Label = item.Device.Id.ToString().PadLeft(3, '0');
            vNode.Tag = item;

            hNode.Index = this.HorizontalNodes.Count;
            hNode.Type = TopologyMapNode.NodeTypes.Horizontal;
            hNode.Site = this;
            hNode.BackColor = item.GetColorByDeviceType();
            hNode.Label = item.Device.Id.ToString().PadLeft(3, '0');
            hNode.Tag = item;

            this.HorizontalNodes.Add(hNode);
            this.VerticalNodes.Add(vNode);

        }

        public void AddElements()
        {
            for (int i = 0; i < this.HorizontalNodes.Count; i++)
            {
                for (int j = 0; j < this.VerticalNodes.Count; j++)
                {
                    TopologyMapElement element = new TopologyMapElement();
                    element.BackColor = GetRoutingInfoColor(this.HorizontalNodes[i], this.VerticalNodes[j]);
                    element.Site = this;
                    element.ColumnIndex = i;
                    element.RowIndex = j;
                    element.HorizontalNode = this.HorizontalNodes[i];
                    element.VerticalNode = this.VerticalNodes[j];
                    this.Elements.Add(element);
                }
            }
        }

        private Color GetRoutingInfoColor(TopologyMapNode hTopologyMapNode, TopologyMapNode vTopologyMapNode)
        {
            Color result = Color.Red;
            if (hTopologyMapNode.Tag.Device.Id != vTopologyMapNode.Tag.Device.Id && vTopologyMapNode.Tag.DeviceRoutingInfo != null)
            {
                int nodeIndex = 0;

                foreach (byte b in vTopologyMapNode.Tag.DeviceRoutingInfo)
                {
                    nodeIndex++;
                    for (byte i = 0; i < 8; i++)
                    {
                        int _b = (b >> i) & 0x01;
                        if (_b == 1)
                        {
                            if (hTopologyMapNode.Tag.Device.Id == nodeIndex * 8 - (8 - (i + 1)))
                            {
                                result = Color.Blue;
                            }
                        }
                    }
                }
            }
            else
            {
                result = Color.White;
            }
            return result;
        }
        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);
            foreach (TopologyMapElement element in this.Elements)
            {
                if (element.HitTest(e.Location))
                {
                    element.Hovered = true;
                }
                else
                {
                    element.Hovered = false;
                }
            }
            this.Elements.Sort(new TopologyMapElementComparer());
            this.Invalidate(true);

        }
        protected override void OnMouseClick(MouseEventArgs e)
        {
            base.OnMouseClick(e);
            foreach (TopologyMapElement element in this.Elements)
            {
                if (element.HitTest(e.Location))
                {
                    element.Selected = true;
                    this.SelectedElement = element;
                }
                else
                {
                    element.Selected = false;
                }
            }
            this.Elements.Sort(new TopologyMapElementComparer());
            this.Invalidate(true);
        }
        protected override void OnPaintBackground(PaintEventArgs e)
        {
            e.Graphics.FillRectangle(new SolidBrush(this.BackColor), e.ClipRectangle);
        }
        protected override void OnPaint(PaintEventArgs e)
        {
            DrawCanvas(e.Graphics);
            DrawNodes(e.Graphics);
            DrawElements(e.Graphics);
        }
        private void DrawCanvas(Graphics graphics)
        {
            Rectangle rect = new Rectangle(this.MarginLeft, this.MarginTop, this.Width - (this.MarginLeft + this.MarginRignt), this.Height - (this.MarginTop + this.MarginBottom));
            graphics.DrawRectangle(DefaultPen, rect);
        }
        private void DrawElements(Graphics graphics)
        {
            foreach (TopologyMapElement element in this.Elements)
            {
                element.Paint(graphics);
            }
        }
        private void DrawNodes(Graphics graphics)
        {
            foreach (TopologyMapNode node in this.VerticalNodes)
            {
                node.Paint(graphics);
            }
            foreach (TopologyMapNode node in this.HorizontalNodes)
            {
                node.Paint(graphics);
            }
        }
        #endregion
    }
}
