using System;
using System.Collections.Generic;
using System.Text;
using Zensys.PCController.Classes;
using Zensys.PCController.Controllers;
using Zensys.ZWave.Devices;
using System.Windows.Forms;
using Zensys.ZWave.SerialPortApplication.Devices;
using Zensys.ZWave.Enums;
using Zensys.PCController.Properties;
using System.Threading;
using Zensys.ZWave;
using Zensys.Framework;
using Zensys.ZWave.ZWaveHAL;
using Zensys.PCController.Models;
using Zensys.ZWave.Events;
using Zensys.ZWave.Application;

namespace Zensys.PCController.Actions
{
    public class CommonActions : BaseAction
    {
        /// <summary>
        /// Initializes a new instance of the <see cref="CommonActions"/> class.
        /// </summary>
        /// <param name="controller">The controller.</param>
        public CommonActions(ControllerManager controller)
            : base(controller)
        {
            StartProcessACH();
        }

        #region Changing Connected Controller Device routine

        private void FillDataSources()
        {
            ControllerManager.DocumentModel.Devices.Clear();
            ControllerManager.DocumentModel.AssociativeDevices.Clear();
            ControllerManager.DocumentModel.AssociativeGroups.Clear();
            ControllerManager.DocumentModel.RoutesCollection.ClearRoutes();

            foreach (IDeviceInfo device in ControllerManager.DocumentModel.Controller.IncludedNodes)
            {
                GenericDevice gd = ControllerManager.XmlDataManager.FindGenericDevice(device.Generic);
                if (gd != null)
                {
                    device.Type = (String.IsNullOrEmpty(gd.Text)) ? gd.Name : gd.Text;
                }
                else
                {
                    device.Type = "Generic Device:" + Tools.ToHexString(device.Generic);
                }
                ControllerManager.DocumentModel.Devices.Add(device);
                if (device.SupportedCommandClasses != null &&
                    ControllerManager.DocumentModel.Controller.CommandClassesStore.IsSupported(device.Id,
                    ControllerManager.XmlDataManager.GetCommandClassKey("COMMAND_CLASS_ASSOCIATION")))
                {
                    ControllerManager.DocumentModel.AssociativeDevices.Add(device);
                }
                if (device.Generic == ControllerManager.XmlDataManager.GetGenericDeviceKey("GENERIC_TYPE_SENSOR_BINARY") ||
                    device.Generic == ControllerManager.XmlDataManager.GetGenericDeviceKey("GENERIC_TYPE_SENSOR_MULTILEVEL"))
                {
                    ControllerManager.DocumentModel.RoutesCollection.AddSourceRoute(device);
                }
                if (device.Generic == ControllerManager.XmlDataManager.GetGenericDeviceKey("GENERIC_TYPE_SWITCH_BINARY") ||
                    device.Generic == ControllerManager.XmlDataManager.GetGenericDeviceKey("GENERIC_TYPE_SWITCH_MULTILEVEL") ||
                    device.Generic == ControllerManager.XmlDataManager.GetGenericDeviceKey("GENERIC_TYPE_STATIC_CONTROLLER") ||
                    device.Generic == ControllerManager.XmlDataManager.GetGenericDeviceKey("GENERIC_TYPE_GENERIC_CONTROLLER") ||
                    device.Generic == ControllerManager.XmlDataManager.GetGenericDeviceKey("GENERIC_TYPE_REPEATER_SLAVE") ||
                    device.Generic == ControllerManager.XmlDataManager.GetGenericDeviceKey("GENERIC_TYPE_ENTRY_CONTROL") ||
                    device.Generic == ControllerManager.XmlDataManager.GetGenericDeviceKey("GENERIC_TYPE_THERMOSTAT"))
                {
                    ControllerManager.DocumentModel.RoutesCollection.AddDestinationRoute(device);
                }
            }

        }
        private bool isStartUp = true;
        public void SetUpController()
        {
            ProcessACH.Pause();
            ControllerManager.DoAction(new EventHandler(delegate
            {
                try
                {
                    if (ControllerManager.DocumentModel.Controller.ConnectionStatus == Zensys.ZWave.Enums.ConnectionStatuses.Closed)
                    {
                        ControllerManager.DocumentModel.Controller.Open(ControllerManager.DocumentModel.Controller.SerialPort);
                    }
                    ControllerManager.DocumentModel.Controller.CommandClassesStore.Load();
                    ControllerManager.DocumentModel.Controller.GetNodes();
                    ControllerManager.DocumentModel.Controller.GetCapabilities();
                    ControllerManager.DocumentModel.Controller.GetControllerCapabilities();
                    ControllerManager.DocumentModel.Controller.Memory.GetId();
                    ControllerManager.DocumentModel.Controller.GetSUCNodeID();
                }
                catch (Exception ex)
                {
                    ControllerManager.LogManager.LogMessage("!!!!!!!!!!Exception: " + ex.Message);
                    throw ex;
                }

            }), null, Resources.MsgLoadingNodes, false, (byte)CommandTypes.None);

            if ((ControllerManager.DocumentModel.Controller.NetworkRole & ControllerRoles.SUC) == 0)
            {
                if (ControllerManager.DocumentModel.Controller.IncludedNodes != null)
                {
                    if (ControllerManager.DocumentModel.Controller.IncludedNodes.Count < 2)
                    {
                        if (isStartUp)
                        {
                            ControllerManager.Actions.ControllerFormActions.EnableSucSis(ControllerRoles.SIS);
                        }
                    }
                }
            }
            if (isStartUp)
            {
                isStartUp = false;
            }
            ControllerManager.LogManager.LogMessage("Controller Network Role: " + ControllerManager.DocumentModel.Controller.NetworkRole.ToString());
            InitializeEntryPointHAL();
            FillDataSources();
            ProcessACH.Resume();
        }

        private void InitializeEntryPointHAL()
        {
            ControllerManager.DoAction(new EventHandler(delegate
            {
                try
                {
                    BaseEntryPointHAL ep = CustomLoader.GetEntryPoint();
                    if (ep != null)
                    {
                        ep.Initialize(ControllerManager.DocumentModel.Controller, Settings.Default.LastUsedDevice, ControllerManager.LogManager, ControllerManager.ExceptionManager, ControllerManager.XmlDataManager);
                        ControllerManager.MainForm.Invoke(new EventHandler(delegate
                        {
                            ep.InjectUIControl((ToolStripMenuItem)ControllerManager.MainForm.MenuMain.Items[0], ControllerManager.MainForm.DockPanelMain);
                        }));
                    }
                }
                catch (Exception ex)
                {
                    throw ex;
                }

            }), null, Resources.MsgLoadingNodes, false, (byte)CommandTypes.None);
        }

        public void FinalizeEntryPointHAL()
        {
            ControllerManager.DoAction(new EventHandler(delegate
            {
                try
                {
                    BaseEntryPointHAL ep = CustomLoader.GetEntryPoint();
                    if (ep != null)
                    {
                        ep.Finalize(ControllerManager.DocumentModel.Controller);
                    }
                }
                catch (Exception ex)
                {
                    throw ex;
                }

            }), null, Resources.MsgLoadingNodes, false, (byte)CommandTypes.None);
        }
        #endregion

        #region Controller

        public IController DetectDevice(string deviceId, bool isRemoteSerialApi)
        {
            bool isError = false;
            IController device = null;
            //ControllerManager.ZWaveManager.FrameLayer.IsRemoteSerialApi = isRemoteSerialApi;
            device = ControllerManager.ZWaveManager.ApplicationLayer.CreateController();
            try
            {
                device.Open(deviceId);
                device.GetVersion();
            }
            catch (Exception ex)
            {
                isError = true;
            }
            finally
            {
                if (device != null)
                {
                    device.Close();
                }
            }
            return isError ? null : device;
        }

        #endregion

        public void ControllerApplicationCommandHandlerEvent(DeviceAppCommandHandlerEventArgs args)
        {
            if (args.SourceNodeId != 0)
            {
                CommandClassValue[] values = null;
                List<byte> payload = new List<byte>();
                payload.Add(args.CommandClassKey);
                payload.Add(args.CommandKey);
                if (args.CommandBuffer != null && args.CommandBuffer.Length > 0)
                {
                    payload.AddRange(args.CommandBuffer);
                }
                ControllerManager.XmlDataManager.ParseApplicationObject(payload.ToArray(), out values);
                ProcessACH.Add(args.SourceNodeId, args.IsBroadcast, values);
            }
        }

        Thread processACH = null;
        private void StartProcessACH()
        {
            if (processACH == null)
            {
                processACH = new Thread(new ParameterizedThreadStart(delegate { ProcessACH.Action(ProcessACHInternal); }));
                processACH.IsBackground = true;
                processACH.Start();
            }
        }

        private void ProcessACHInternal(byte sourceNodeId, bool isBroadcast, CommandClassValue[] values)
        {
            if (ControllerManager.Actions.AssociationsFormActions.ActionsHAL != null)
                ControllerManager.Actions.AssociationsFormActions.ActionsHAL.ProcessApplicationCommandHandler(
                    ControllerManager.DocumentModel.Controller,
                    sourceNodeId,
                    isBroadcast,
                    values);
            if (ControllerManager.Actions.NodeFormActions.ActionsHAL != null)
                ControllerManager.Actions.NodeFormActions.ActionsHAL.ProcessApplicationCommandHandler(
                    ControllerManager.DocumentModel.Controller,
                    sourceNodeId,
                    isBroadcast,
                    values);
        }

        public List<IDeviceInfo> GetSelectedDevices(bool excludeConnectedDevice)
        {
            List<IDeviceInfo> ret = new List<IDeviceInfo>();
            foreach (DataGridViewRow row in ControllerManager.NodeForm.nodeGridViewControl.nodesGridView.SelectedRows)
            {
                IDeviceInfo device = row.DataBoundItem as IDeviceInfo;
                if (device != null)
                {
                    if (!excludeConnectedDevice || device.Id != ControllerManager.DocumentModel.Controller.Id)
                    {
                        ret.Add(device);
                    }
                }
            }
            return ret;
        }

    }
}
