using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.InstallerTool.Classes;
using Zensys.ZWave.InstallerTool.Controllers;
using Zensys.ZWave.Devices;
using Zensys.ZWave.InstallerTool.Properties;
using Zensys.ZWave.Enums;
using System.Windows.Forms;
using Zensys.ZWave.SerialPortApplication.Devices;
using System.Drawing;
using Zensys.Framework;
using Zensys.ZWave;
using Zensys.Framework.Helpers;
using Zensys.ZWave.InstallerTool.Models;
using Zensys.ZWave.ZWaveHAL.Actions;
using Zensys.Framework.UI.Controls;
using System.Threading;
using Zensys.ZWave.ZWaveHAL;
using Zensys.ZWave.Application;

namespace Zensys.ZWave.InstallerTool.Actions
{
    public class NodeFormActions : BaseAction
    {
        NodesViewActions NodesHAL;
        public NodeFormActions(ControllerManager controller)
            : base(controller)
        {
            NodesHAL = CustomLoader.CreateNodesViewActionsInstance();
            NodesHAL.Initialize(controller.LogManager, controller.ExceptionManager, controller.XmlDataManager);
            NodesHAL.AddQueueCommandCallback = controller.AddQueueCommand;
            NodesHAL.ExecuteQueueCommandCallback = controller.ExecuteQueueCommand;
            NodesHAL.ItemChanged += new EventHandler(NodesHAL_ItemChanged);
            NodesHAL.NodeAdded += new EventHandler<NodeEventArgs>(NodesHAL_NodeAdded);
            base.ActionsHAL = NodesHAL;
        }

        private void NodesHAL_NodeAdded(object sender, NodeEventArgs e)
        {
            AddControllerNode(e.Device);
        }

        void NodesHAL_ItemChanged(object sender, EventArgs e)
        {
            int position = -1;
            foreach (IDeviceInfo d in ControllerManager.DocumentModel.Devices)
            {
                position++;
                if (d.Id == ((IDeviceInfo)sender).Id)
                    break;
            }
            if (position > -1)
            {
                ControllerManager.DocumentModel.Devices.ResetItem(position);
            }
        }
        #region Form
        public void OnFormLoad(object sender, EventArgs e)
        {
            ControllerManager.NodeForm.nodeGridViewControl.btnBasicToggle.Visible = false;
            ControllerManager.NodeForm.nodeGridViewControl.btnNWInclusion.Visible = false;
            ControllerManager.NodeForm.nodeGridViewControl.btnIsFailed.Visible = false;
            ControllerManager.NodeForm.nodeGridViewControl.btnRemoveFailed.Visible = false;
            ControllerManager.NodeForm.nodeGridViewControl.btnReplaceFailed.Visible = false;
            ControllerManager.NodeForm.nodeGridViewControl.btnNOP.Visible = false;
            ControllerManager.NodeForm.nodeGridViewControl.btnWUISet.Visible = false;
            ControllerManager.NodeForm.nodeGridViewControl.txtNOPId.Visible = false;
            ControllerManager.NodeForm.nodeGridViewControl.txtWUIValue.Visible = false;
            
            
            ControllerManager.NodeForm.nodeGridViewControl.btnReach.Visible = true;
            
            DataGridViewColumn colNodeId = new DataGridViewTextBoxColumn();
            DataGridViewColumn colNodeType = new DataGridViewTextBoxColumn();
            DataGridViewBitImageColumn colIsListening = new DataGridViewBitImageColumn();
            DataGridViewBitImageColumn colIsVirtual = new DataGridViewBitImageColumn();

            colNodeId.Name = "colId";
            colNodeId.HeaderText = "Id";
            colNodeId.DataPropertyName = "Id";
            colNodeId.AutoSizeMode = DataGridViewAutoSizeColumnMode.AllCellsExceptHeader;


            colNodeType.Name = "colType";
            colNodeType.HeaderText = "Type";
            colNodeType.DataPropertyName = "Type";
            colNodeType.MinimumWidth = 100;
            colNodeType.FillWeight = 100;
            colIsListening.AutoSizeMode = DataGridViewAutoSizeColumnMode.Fill;

            colIsListening.Name = "colIsListening";
            colIsListening.HeaderText = "Listening";
            colIsListening.DataPropertyName = "IsListening";
            colIsListening.Images = ControllerManager.NodeForm.nodeGridViewControl.Images;
            colIsListening.TrueImageIndex = 0;
            colIsListening.AutoSizeMode = DataGridViewAutoSizeColumnMode.AllCellsExceptHeader;
            colNodeType.MinimumWidth = 40;

            colIsVirtual.Name = "colIsVirtual";
            colIsVirtual.HeaderText = "Virtual";
            colIsVirtual.DataPropertyName = "IsVirtual";
            colIsVirtual.Images = ControllerManager.NodeForm.nodeGridViewControl.Images;
            colIsVirtual.TrueImageIndex = 1;
            colIsVirtual.AutoSizeMode = DataGridViewAutoSizeColumnMode.AllCellsExceptHeader;
            colNodeType.MinimumWidth = 40;
            ControllerManager.NodeForm.nodeGridViewControl.nodesGridView.AutoSizeColumnsMode = DataGridViewAutoSizeColumnsMode.Fill;
            ControllerManager.NodeForm.nodeGridViewControl.nodesGridView.AutoGenerateColumns = false;
            ControllerManager.NodeForm.nodeGridViewControl.nodesGridView.Columns.AddRange(new DataGridViewColumn[] { colNodeId, colNodeType, colIsListening, colIsVirtual });
            BindData();


            ControllerManager.NodeForm.vcNodeInfoTreeView.ImageList = ControllerManager.NodeForm.nodeGridViewControl.Images;
            ControllerManager.NodeForm.vcNodeInfoTreeView.DataSource = ControllerManager.DocumentModel.Devices;
            ControllerManager.NodeForm.vcNodeInfoTreeView.CreatePrefetchedObject = CreatePrefetchedObject;
            OnDocumentModelStateChanged();
        }

        private void BindData()
        {
            ControllerManager.DocumentModel.Devices.Ctx = SynchronizationContext.Current;
            ControllerManager.NodeForm.nodeGridViewControl.nodesGridView.DataSource = ControllerManager.DocumentModel.Devices;
        }

        public void OnFormClosing(object sender, FormClosingEventArgs e)
        {
            ControllerManager.MainForm.NodeToolStripMenuItem.Checked = false;
            ControllerManager.NodeForm.Hide();
            e.Cancel = true;
        }

        private PrefetchedObject CreatePrefetchedObject(object currentObject)
        {
            IDeviceInfo device = (IDeviceInfo)currentObject;
            List<ParamValue> payloadValues = GetNodeInfoPayloadValues(device);
            PrefetchedObject poRoot = new PrefetchedObject();
            poRoot.Text = string.Format("Id: {0}; {1}", device.Id, device.Type);
            poRoot.ImageIndex = 2;
            foreach (ParamValue paramValue in payloadValues)
            {
                List<string> str = paramValue.TextValueList;
                PrefetchedObject po = new PrefetchedObject();
                po.ImageIndex = 3;
                if (paramValue.TextValueList.Count == 1)
                {
                    po.Text = string.Format("{0}: {1}", paramValue.ParamDefinition.Text, paramValue.TextValueList[0]);
                }
                else if (paramValue.TextValueList.Count > 1)
                {
                    po.Text = string.Format("{0}: {1}", paramValue.ParamDefinition.Text, Tools.GetHex(paramValue.ByteValueList.ToArray(), " ", false));
                    foreach (string var in paramValue.TextValueList)
                    {
                        PrefetchedObject subpo = new PrefetchedObject();
                        subpo.Text = var;
                        subpo.ImageIndex = 4;
                        po.Children.Add(subpo);
                    }
                }
                else
                {
                    po.Text = string.Format("{0}: {1}", paramValue.ParamDefinition.Text, Tools.GetHex(paramValue.ByteValueList.ToArray(), " ", false));
                }
                if (!String.IsNullOrEmpty(paramValue.ParamDefinition.GroupName))
                {
                    PrefetchedObject groupPo = FindPrefetchedObject(poRoot, paramValue.ParamDefinition.GroupName);
                    if (groupPo != null)
                    {
                        groupPo.Children.Add(po);
                    }
                    else
                    {
                        groupPo = new PrefetchedObject();
                        groupPo.ImageIndex = 3;
                        groupPo.Text = paramValue.ParamDefinition.GroupName;
                        groupPo.Children.Add(po);
                        poRoot.Children.Add(groupPo);
                    }
                }
                else
                {
                    poRoot.Children.Add(po);
                }
            }

            return poRoot;
        }

        private PrefetchedObject FindPrefetchedObject(PrefetchedObject node, string name)
        {
            PrefetchedObject result = null;
            if (node.Text == name) return node;
            if (node.Children != null && node.Children.Count > 0)
            {
                foreach (PrefetchedObject child in node.Children)
                {
                    result = FindPrefetchedObject(child, name);
                }
            }
            return result;
        }

        private List<ParamValue> GetNodeInfoPayloadValues(IDeviceInfo device)
        {
            CommandClass cmdClass = ControllerManager.XmlDataManager.FindCommandClass("ZWAVE_CMD_CLASS", 1);
            Command cmd = ControllerManager.XmlDataManager.FindCommand("ZWAVE_CMD_CLASS", 1, "NODE_INFO");
            List<byte> payload = new List<byte>();
            payload.Add(cmdClass.KeyId);
            payload.Add(cmd.KeyId);
            payload.AddRange(new byte[] { device.Capability, device.Security, device.Reserved, /*device.Basic,*/ device.Generic, device.Specific });
            if (device.SupportedCommandClasses != null)
            {
                payload.AddRange(device.SupportedCommandClasses);
            }
            CommandClassValue[] cmdClassValues = null;
            ControllerManager.XmlDataManager.ParseApplicationObject(payload.ToArray(), out cmdClassValues);
            if (cmdClassValues != null)
            {
                return cmdClassValues[0].CommandValue.ParamValues;
            }
            return null;
        }

        #endregion

        #region DocumentModel
        public void OnDocumentModelStateChanged(object sender, EventArgs e)
        {
            OnDocumentModelStateChanged();
        }

        private delegate void OnDocumentModelStateChangedDelegate();
        private void OnDocumentModelStateChanged()
        {
            if (ControllerManager.NodeForm != null && !ControllerManager.NodeForm.IsDisposed)
            {
                if (ControllerManager.NodeForm.InvokeRequired)
                {
                    ControllerManager.NodeForm.Invoke(new OnDocumentModelStateChangedDelegate(OnDocumentModelStateChanged));
                }
                else
                {
                    ControllerManager.NodeForm.vcAddNodeToolStripButton.Enabled =
                      ControllerManager.DocumentModel.Controller != null &&
                      ((ControllerManager.DocumentModel.Controller.NetworkRole & ControllerRoles.Secondary) == 0 ||
                      (ControllerManager.DocumentModel.Controller.NetworkRole & ControllerRoles.Inclusion) != 0);

                    ControllerManager.NodeForm.vcRemoveNodeToolStripButton.Enabled =
                        ControllerManager.DocumentModel.Controller != null &&
                      ((ControllerManager.DocumentModel.Controller.NetworkRole & ControllerRoles.Secondary) == 0 ||
                      (ControllerManager.DocumentModel.Controller.NetworkRole & ControllerRoles.Inclusion) != 0);

                    ControllerManager.NodeForm.vcNodesListView.Enabled =
                        ControllerManager.DocumentModel.Controller != null;

                    ControllerManager.NodeForm.vcNodeInfoTreeView.Enabled =
                        ControllerManager.DocumentModel.Controller != null;

                    ControllerManager.NodeForm.vcSetAllOnToolStripButton.Enabled =
                        ControllerManager.DocumentModel.Controller != null &&
                        ControllerManager.DocumentModel.Devices.Count > 0;

                    ControllerManager.NodeForm.vcSetOffToolStripButton.Enabled =
                        ControllerManager.DocumentModel.Controller != null &&
                        ControllerManager.DocumentModel.CurrentDevice != null;

                    ControllerManager.NodeForm.vcSetOnToolStripButton.Enabled =
                        ControllerManager.DocumentModel.Controller != null &&
                        ControllerManager.DocumentModel.CurrentDevice != null;

                    ControllerManager.NodeForm.vcToolStripMain.Enabled =
                        ControllerManager.DocumentModel.Controller != null;
                }
            }
        }

        #endregion

        #region Toolbar
        public void RemoveControllerNode(byte removedNodeId)
        {
            RemoveControllerNodeInternal(removedNodeId);
        }
        private void RemoveControllerNodeInternal(byte removedNodeId)
        {
            if (removedNodeId > 0)
            {
                ControllerManager.DocumentModel.RemoveDevice(removedNodeId);
            }
        }
        List<byte> nodesAddedFailedList = new List<byte>();
        public void AddControllerNode(IDeviceInfo addedNode)
        {
            if (addedNode != null)
            {
                ControllerManager.DocumentModel.RemoveDevice(addedNode.Id);
                if (!nodesAddedFailedList.Contains(addedNode.Id))
                {
                    GenericDevice gd = ControllerManager.XmlDataManager.FindGenericDevice(addedNode.Generic);
                    if (gd != null)
                    {
                        addedNode.Type = (String.IsNullOrEmpty(gd.Text)) ? gd.Name : gd.Text;
                    }
                    else
                    {
                        addedNode.Type = "Generic Device:" + Tools.ToHexString(addedNode.Generic);
                    }
                    ControllerManager.DocumentModel.Devices.Add(addedNode);
                }
            }
            if (addedNode != null && !nodesAddedFailedList.Contains(addedNode.Id))
            {
                ControllerManager.LogManager.LogMessage("Node added to the network...");
            }
            else
            {
                ControllerManager.LogManager.LogMessage("Add Node to the network failed...");
            }
        }
        public void OnNodeInfoClick(object sender, EventArgs e)
        {
            IDeviceInfo device = ControllerManager.DocumentModel.CurrentDevice;
            QueueAction action = new QueueAction(
                            new FunctionCaller<IDeviceInfo, IController, byte, IDeviceInfo>(NodesHAL.NodeInfoAction),
                            new object[] 
                        { 
                    ControllerManager.DocumentModel.Controller, 
                    device.Id, 
                    device },
                         Resources.MsgGetNodeReport, true, (byte)CommandTypes.CmdZWaveRequestNodeInfo, "Request Node Information", 0);

            if (device.IsListening || device.IsCommandQueueOverrided || (device.Security & 0x20) != 0 || (device.Security & 0x40) != 0)
            {
                IDeviceInfo deviceInfo = (IDeviceInfo)ControllerManager.DoAction(action);
                device = deviceInfo;
                //ControllerManager.DocumentModel.Devices.Remove(device);
                //ControllerManager.DocumentModel.Devices.Add(deviceInfo);
            }
            else
            {
                ControllerManager.AddQueueCommand(device.Id, action);
                ControllerManager.LogManager.LogMessage(string.Format("{0} added to command queue for Node {1}.", "Request Node Information", device.Id));
            }
        }

        public void OnAddNodeClick(object sender, EventArgs e)
        {
            nodesAddedFailedList.Clear();
            IDeviceInfo addedNode = (IDeviceInfo)ControllerManager.DoAction(
                new FunctionCaller<IDeviceInfo, IController>(NodesHAL.AddNodeAction),
                new object[] { ControllerManager.DocumentModel.Controller },
                Resources.MsgAddNode, true, (byte)CommandTypes.CmdZWaveAddNodeToNetwork);

            ControllerManager.Actions.RoutingTableFormActions.OnReloadTopologyClick(null, null);
        }
        public void OnRemoveNodeClick(object sender, EventArgs e)
        {
            byte removedNodeId = (byte)ControllerManager.DoAction(
                new FunctionCaller<byte, IController>(NodesHAL.RemoveNodeAction),
                new object[] { ControllerManager.DocumentModel.Controller },
                Resources.MsgRemoveNode, true, (byte)CommandTypes.CmdZWaveRemoveNodeFromNetwork);
            if (removedNodeId > 0)
            {
                RemoveControllerNodeInternal(removedNodeId);
                ControllerManager.Actions.RoutingTableFormActions.OnReloadTopologyClick(null, null);
            }
        }
        public void OnReachNodeClick(object sender, EventArgs e)
        {
            ControllerManager.DoAction(new EventHandler(delegate
            {
                if (ControllerManager.DocumentModel.Controller != null)
                {
                    if (ControllerManager.DocumentModel.Controller.ConnectionStatus == Zensys.ZWave.Enums.ConnectionStatuses.Closed)
                    {
                        ControllerManager.DocumentModel.Controller.Open(ControllerManager.DocumentModel.Controller.SerialPort);
                    }
                    if (ControllerManager.DocumentModel.CurrentDevice != null)
                    {
                        ControllerManager.DocumentModel.Controller.SendData(
                                        ControllerManager.DocumentModel.CurrentDevice.Id,
                                        new byte[] { 0x00, 0x00 },
                                        TransmitOptions.TransmitOptionAutoRoute | TransmitOptions.TransmitOptionAcknowledge);
                    }


                }
            }), null, Resources.MsgReachNodeProgress, false, (byte)CommandTypes.None);
        }
        public void OnSetOnClick(object sender, EventArgs e)
        {
            SendBasicSetOnOff(true, GetSelectedDevices());
        }

        private List<IDeviceInfo> GetSelectedDevices()
        {
            return GetSelectedDevices(false);
        }
        private List<IDeviceInfo> GetSelectedDevices(bool excludeConnectedDevice)
        {
            List<IDeviceInfo> ret = new List<IDeviceInfo>();
            foreach (DataGridViewRow row in ControllerManager.NodeForm.nodeGridViewControl.nodesGridView.SelectedRows)
            {
                IDeviceInfo device = row.DataBoundItem as IDeviceInfo;
                if (device != null)
                {
                    if (!excludeConnectedDevice || device.Id != ControllerManager.DocumentModel.Controller.Id)
                    {
                        ret.Add(device);
                    }
                }
            }
            return ret;
        }

        public void OnSetOffClick(object sender, EventArgs e)
        {
            SendBasicSetOnOff(false, GetSelectedDevices());
        }
        public void OnSetAllOnClick(object sender, EventArgs e)
        {
            SendSwitchOnOff(true);
        }
        public void OnSetAllOffClick(object sender, EventArgs e)
        {
            SendSwitchOnOff(false);
        }
        private void SendSwitchOnOff(bool isOn)
        {
            ControllerManager.DoAction(
              new ProcedureCaller<IController, IList<IDeviceInfo>, bool>(NodesHAL.SendSwitchOnOffAction),
              new object[]{
                  ControllerManager.DocumentModel.Controller,
                  ControllerManager.DocumentModel.Devices,
                  isOn},
              Resources.MsgSetOn, true, (byte)CommandTypes.None);
        }
        private void SendBasicSetOnOff(bool isOn, List<IDeviceInfo> selectedNodes)
        {
            ControllerManager.DoAction(
             new ProcedureCaller<IController, IList<IDeviceInfo>, bool>(NodesHAL.SendBasicSetOnOffAction),
             new object[]{
                  ControllerManager.DocumentModel.Controller,
                  selectedNodes,
                  isOn},
             Resources.MsgSetOn, true, (byte)CommandTypes.None);
        }
        public void OnBasicToggleToolStripButtonClick(object sender, EventArgs e)
        {
            List<IDeviceInfo> nodes = GetSelectedDevices(true);
            ControllerManager.DoAction(
              new ProcedureCaller<IController, IList<IDeviceInfo>, bool>(NodesHAL.BasicGetAction),
              new object[]{
                  ControllerManager.DocumentModel.Controller,
                  nodes,
                  ControllerManager.NodeForm.vcBasicToggleToolStripButton.Checked},
              Resources.MsgSetOn, true, (byte)CommandTypes.None);
        }
        #endregion
        public void OnControllerNodeStatusChanged(Zensys.ZWave.Events.NodeStatusChangedEventArgs args)
        {
            switch (args.Status)
            {
                case NodeStatuses.AddingRemovingController:
                    break;
                case NodeStatuses.AddingRemovingSlave:
                    break;
                case NodeStatuses.Done:
                    break;
                case NodeStatuses.Failed:
                    ControllerManager.LogManager.LogOperationFailure(string.Format("NodesStatuses.Failed received Node:{0}...", args.SourceNodeId));
                    nodesAddedFailedList.Add(args.SourceNodeId);
                    break;
                case NodeStatuses.LearnReady:
                    break;
                case NodeStatuses.NodeFound:
                    break;
                case NodeStatuses.ProtocolDone:
                    break;
                case NodeStatuses.Unknown:
                    break;
                default:
                    break;
            }
        }
    }
}